/*
 * rcal - a "rolling" calculator / v1.0.1
 * Copyright 2018-2024 (C) Mateusz Viste
 *
 * http://rcal.sourceforge.net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "apm\apm.h"

#include <string.h> /* strlen() */

enum keys_p {
  KEY_ESCAPE = 0x1B,
  KEY_BCKSPC = 0x08,
  KEY_RETURN = 0x0D
};

#define COL_DECIMAL 11
#define COL_RESULT  21
#define DIV_PREC 20


/* get a single key from DOS (INT 21h/AH=8, AL=input char, 0 if ext key) */
static int io_getkey(void);
#pragma aux io_getkey = \
"mov ah, 0x08" \
"int 0x21" \
"xor ah, ah" \
"test al, al" \
"jnz DONE" \
"mov ah, 0x08" \
"int 0x21" \
"mov ah, 1" \
"DONE:" \
value [ax];


/* INT 21h/AH=09h   DOS 1+ - WRITE STRING TO STANDARD OUTPUT
 * DS:DX -> '$'-terminated string */
static void io_printdosstr(char *s);
#pragma aux io_printdosstr = \
"mov ah, 9" \
"int 0x21" \
modify [ax] \
parm [dx];


/* INT 21h/AH=02h  DOS 1+ - WRITE CHARACTER TO STANDARD OUTPUT
 * DL = character to write */
static void io_printdoschar(char c);
#pragma aux io_printdoschar = \
"mov ah, 2" \
"int 0x21" \
modify [ax] \
parm [dl];


/* INT 10h, AH=02h
 * BH = page number
 * DH = row (0 is top)
 * DL = column (0 is left) */
static void io_setcursorpos(int row, int col);
#pragma aux io_setcursorpos = \
"mov ah, 2" \
"xor bh, bh" \
"int 0x10" \
modify [ah bh] \
parm [dh] [dl];


/* INT 10h, AH=03h
 * BH = page number
 * Returns:
 *   AX = 0
 *   CH = start scan line
 *   CL = end scan line
 *   DH = row (0 is top)
 *   DL = column (0 is left) */
static void io_getcursorpos(unsigned char *row, unsigned char *col);
#pragma aux io_getcursorpos = \
"mov ah, 3" \
"xor bh, bh" \
"int 0x10" \
"mov [di], dh" \
"mov [si], dl" \
modify [ax bh cx dx] \
parm [di] [si];


/* INT 21h, AH=38h  DOS 2.11+ - GET COUNTRY-SPECIFIC INFORMATION
 * AL = 00h get current-country info
 * BX = 16-bit country code
 * DS:DX -> buffer for returned info
 * Return:
 *   CF set on error, clear otherwise
 *   AX = error code (02h)
 * Format of country info:
 * 00h    WORD    date format
 * 02h  5 BYTEs   ASCIZ currency symbol string
 * 07h  2 BYTEs   ASCIZ thousands separator
 * 09h  2 BYTEs   ASCIZ decimal separator
 * 0Bh  2 BYTEs   ASCIZ date separator
 * 0Dh  2 BYTEs   ASCIZ time separator
 * 0Fh    BYTE    currency format
 * ...
 * 18h 10 BYTEs   reserved
 */
static int getcountryparms(void *buf);
#pragma aux getcountryparms = \
"mov ax, 0x3800" \
"int 0x21" \
"xor ax, ax" \
"jnc DONE" \
"inc ax" \
"DONE:" \
parm [dx] \
modify [bx] \
value [ax];


/* DOS 3.0+ CREATE NEW FILE */
static int io_fcreat(char *f, unsigned short *handle);
#pragma aux io_fcreat = \
"mov ah, 0x5b" \
"xor cx, cx" \
"int 0x21" \
"jc DONE" \
"mov [bx], ax" \
"xor ax, ax" \
"DONE:" \
parm [dx] [bx] \
modify [cx] \
value [ax];


/* DOS 2+ WRITE - WRITE TO FILE OR DEVICE (returns 0 on error) */
static unsigned short io_dos_fwrite(unsigned short handle, void *s, unsigned short len);
#pragma aux io_dos_fwrite = \
"mov ah, 0x40" \
"int 0x21" \
"jnc DONE" \
"xor ax, ax" \
"DONE:" \
parm [bx] [dx] [cx] \
value [ax];


static int io_fwrite(unsigned short handle, void *s, unsigned short len) {
  unsigned short l;
  while (len > 0) {
    l = io_dos_fwrite(handle, s, len);
    if (l == 0) return(-1);
    len -= l;
    s = (char *)s + l;
  }
  return(0);
}


/* DOS 2+ CLOSE - CLOSE FILE */
static int io_fclose(unsigned short handle);
#pragma aux io_fclose = \
"mov ah, 0x3e" \
"int 0x21" \
"xor ax, ax" \
"jnc DONE" \
"inc ax" \
"DONE:" \
parm [bx] \
value [ax];


static void io_setcursorxpos(unsigned char col) {
  unsigned char x, y;
  io_getcursorpos(&y, &x);
  io_setcursorpos(y, col);
}


static void io_movcursorx(signed char delta) {
  unsigned char x, y;
  io_getcursorpos(&y, &x);
  io_setcursorpos(y, x + delta);
}


/* fills s with a valid operand and returns an operator */
static int getoperand(char *s, int ilen, int dlen, char decsep) {
  int c, slen = 0, decpos = -1;
  s++;         /* leave one byte for signum */
  s[-1] = ' '; /* signum placeholder */
  for (;;) {
    s[slen] = 0;
    s[slen+1] = '$';
    if (decpos >= 0) {
      io_setcursorxpos(COL_DECIMAL - decpos);
    } else {
      io_setcursorxpos(COL_DECIMAL - slen);
    }
    io_printdosstr(" $");
    io_printdosstr(s - 1);
    io_movcursorx(-1); /* back by one */

    c = io_getkey();
    switch (c) {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
        if (decpos >= 0) { /* decimal */
          if (((slen - 1) - decpos) < dlen) s[slen++] = c;
        } else { /* integer */
          if (slen < ilen) s[slen++] = c;
        }
        break;
      case '.':
      case ',':
        if (decpos < 0) {
          decpos = slen;
          s[slen++] = decsep;
        }
        break;
      case KEY_RETURN:
        return(' ');
      case 's':
      case 'S':
        if (s[-1] == '-') {
          s[-1] = ' ';
        } else {
          s[-1] = '-';
        }
        break;
      case '%':
        io_printdoschar('%');
      case KEY_ESCAPE:
      case '+':
      case '-':
      case '*':
      case '/':
      case '^':
        return(c);
      case KEY_BCKSPC: /* bkspc */
        if (slen > 0) slen--;
        if (decpos >= slen) decpos = -1;
        break;
    }
  }
}

/* converts a null-terminated string to a DOS '$'-terminated string and
 * replaces the decimal separator by decsep */
static void processstr2dos(char *s, char decsep) {
  while (*s != 0) {
    if (*s == '.') *s = decsep;
    s++;
  }
  *s = '$';
}

/* computes res = a ^ b by running a multiplication multiple times. b must be
 * an integer */
static int powmul(APM res, APM a, APM b) {
  APM r, t, i;
  int x;
  r = apmNew(10);
  t = apmNew(10);
  i = apmNew(10);
  /* special case: anything to power 0 results in 1 */
  x = apmCompare(b, t);
  if (x == 0) {
    apmAssignString(res, "1", 10);
    apmDispose(r);
    apmDispose(t);
    apmDispose(i);
    return(0);
  }
  x = 0;
  /* */
  apmAssignString(t, "2", 10); /* t = 2 */
  apmAbsoluteValue(r, b);      /* r = abs(b) */
  apmSubtract(i, r, t);        /* i = abs(b) - t */
  apmAssignString(t, "1", 10); /* t = 1 */
  apmAssign(r, a);             /* r = a */
  while (apmSign(i) > 0) {
    x = apmMultiply(res, r, a);
    if (x != 0) break;
    apmSubtract(r, i, t); /* r = i - 1 */
    apmAssign(i, r);      /* i = r     */
    apmAssign(r, res);    /* r = res   */
  }
  /* if b was negative, then actual result is a reciprocal (x^-n == 1/x^n) */
  if ((x == 0) && (apmSign(b) < 0)) {
    x = apmDivide(r, DIV_PREC, (APM)0, t, res); /* r = 1 / res */
    apmAssign(res, r); /* res = r */
  }
  apmDispose(r);
  apmDispose(t);
  apmDispose(i);
  return(x);
}

static char lcase(char s) {
  if ((s >= 'A') && (s <= 'Z')) s += 'a' - 'A';
  return(s);
}

static int strstartswith(char *s1, char *s2) {
  for (;;) {
    if (*s2 == 0) return(0);
    if (lcase(*s1) != lcase(*s2)) return(-1);
    s1++;
    s2++;
  }
}

static int processargv(int argc, char **argv, char **logfile) {
  while (--argc > 0) {
    argv++;
    if (strstartswith(*argv, "/log=") == 0) {
      *logfile = *argv + 5;
    } else {
      return(-1);
    }
  }
  return(0);
}

static int dosstrlen(char *s) {
  int i = 0;
  while (*s != '$') {
    i++;
    s++;
  }
  return(i);
}

/* write num in s into file handle fh, using fixed format of i integer len
 * and d decimal positions. decimal separator is decsep. also prints % if op
 * is set to that */
static void writenumtofile(unsigned short fh, char *s, int i, int d, char decsep, int op) {
  int dpos;
  /* find out position of decimal separator */
  for (dpos = 0; (s[dpos] != decsep) && (s[dpos] != 0); dpos++);
  /* write as many spaces as necessary to align integer part */
  for (i -= dpos; i > 0; i--) io_fwrite(fh, " ", 1);
  /* write the integer part */
  io_fwrite(fh, s, dpos);
  /* continue writing content of s (including decsep) until end of string */
  s += dpos;
  for (dpos = 0; s[dpos] != 0; dpos++) io_fwrite(fh, s + dpos, 1);
  /* add the percent char if op is a percent */
  if (op == '%') {
    io_fwrite(fh, "%", 1);
    dpos++;
  }
  /* append as many spaces as necessary to align decimal part to d lenght */
  for (; dpos <= d; dpos++) io_fwrite(fh, " ", 1);
}


/* replace all occurences of character c in string s by character n */
static void str_tr(char *s, char c, char n) {
  for (; *s != 0; s++) if (*s == c) *s = n;
}


int main(int argc, char **argv) {
  APM opword1, opword2, opwordbackup, opwordres;
  unsigned char op, lastop = ' ';
  char decsep = '.';
  static char s[57];
  char *logfile = (void *)0;
  unsigned short loghandle;

  io_printdosstr("rcal v1.0.1         a 'rolling' calculator         (C) 2018-2024 Mateusz Viste\r\n\r\n$");

  if (processargv(argc, argv, &logfile) != 0) {
    io_printdosstr("rcal is an interactive calculator that loosely mimics the behavior of \"rolling\r\n"
                   "paper\" calculators. It supports floating point operations and handles huge\r\n"
                   "numbers. Requires only DOS and an 8086 compatible CPU (no FPU needed).\r\n"
                   "\r\n"
                   "usage: rcal [/log=file.log]\r\n"
                   "\r\n"
                   "rcal is released under the terms of the MIT license. It is using the Arbitrary\r\n"
                   "Precision Math library by Lloyd Zusman, 1988. This library is licensed under\r\n"
                   "the terms of a different license, the APM GPL.\r\n"
                   "\r\n"
                   "See RCAL.TXT for details.\r\n"
                   "$");
    return(1);
  }

  /* open the log file, if any defined */
  if (logfile != (void *)0) {
    if (io_fcreat(logfile, &loghandle) != 0) {
      io_printdosstr("ERROR: failed to create log file.\r\n$");
      return(1);
    }
  }

  /* load the system-wide decimal separator */
  if (getcountryparms(s) == 0) decsep = s[9];

  opword1 = apmNew(10);
  opword2 = apmNew(10);
  opwordres = apmNew(10);
  opwordbackup = apmNew(10);

  io_printdosstr("Enter an operand, followed by an operator, followed by an operand. ESC to quit.\r\n"
                 "Operators: + add  - sub  / div  * mul  % perc  ^ exp.  Press S to change sign.\r\n"
                 "\r\n$");

  for (;;) {
    int res;

    for (;;) {
      io_setcursorxpos(0);
      io_printdoschar(lastop);
      op = getoperand(s, 10, 6, decsep);
      if (op == KEY_ESCAPE) break;
      if (s[1] == 0) {
        lastop = op;
      } else {
        break;
      }
    }
    /* convert input to an APM struct - take care to convert the decimal
     * separator to '.' before, as that's the only separator APM understands */
    if (decsep != '.') str_tr(s, decsep, '.');
    apmAssignString(opword2, s, 10);
    if (decsep != '.') str_tr(s, '.', decsep);

    io_setcursorxpos(COL_RESULT);
    io_printdosstr("= $");

    if (logfile != (void *)0) {
      io_fwrite(loghandle, &lastop, 1);
      io_fwrite(loghandle, " ", 1);
      writenumtofile(loghandle, s, 10, 6, decsep, op);
      io_fwrite(loghandle, " = ", 3);
    }

    if (op == KEY_ESCAPE) {
      io_printdosstr("Goodbye\r\n$");
      if (logfile != (void *)0) io_fwrite(loghandle, "Goodbye\r\n", 9);
      break;
    }

    /* percents are a very specific case */
    if (op == '%') {
      op = ' ';
      /* scale opword2 by /100 */
      apmScale(opwordres, opword2, -2);
      apmAssign(opword2, opwordres);
      /* if lastop is - or +, then opword2 should rather contain opword1*opword2 */
      if ((lastop == '+') || (lastop == '-')) {
        apmMultiply(opwordres, opword1, opword2);
        apmAssign(opword2, opwordres);
      }
    }

    /* save current result in case next operation fails */
    apmAssign(opwordbackup, opword1);

    switch (lastop) {
      case '+': res = apmAdd(opwordres, opword1, opword2); break;
      case '-': res = apmSubtract(opwordres, opword1, opword2); break;
      case '*': res = apmMultiply(opwordres, opword1, opword2); break;
      case '/': res = apmDivide(opwordres, DIV_PREC, (APM)0, opword1, opword2); break;
      case '^': res = powmul(opwordres, opword1, opword2); break;
      case ' ':
        res = 0;
        apmAssign(opwordres, opword2);
        break;
    }

    lastop = op;

    if (res != 0) {
      switch (res) {
        case APM_WDIVBYZERO: io_printdosstr("DIV BY ZERO$"); break;
        case APM_ENOMEM: io_printdosstr("OUT OF MEM$"); break;
        default: io_printdosstr("OPERATION FAILED$"); break;
      }
      apmAssign(opword1, opwordbackup); /* revert saved result */
    } else {
      apmAssign(opword1, opwordres); /* update result */
      res = apmConvert(s, sizeof(s), -1, 0, 1, opword1);
      if ((res == 0) || (res == APM_WTRUNC)) {
        processstr2dos(s, decsep);
        if (res == APM_WTRUNC) {
          s[sizeof(s) - 4] = '.';
          s[sizeof(s) - 3] = '.';
          s[sizeof(s) - 2] = '.';
          s[sizeof(s) - 1] = '$';
        }
        io_printdosstr(s);
        if (loghandle != (void *)0) io_fwrite(loghandle, s, dosstrlen(s));
      } else {
        io_printdosstr("INTERNAL ERROR$");
      }
    }
    io_printdosstr("\r\n$");
    if (loghandle != (void *)0) io_fwrite(loghandle, "\r\n", 2);

  }

  /* APM cleanup */
  apmDispose(opwordres);
  apmDispose(opword1);
  apmDispose(opword2);
  apmDispose(opwordbackup);
  apmGarbageCollect();

  /* close log file if any defined */
  if ((logfile != (void *)0) && (io_fclose(loghandle) != 0)) {
    io_printdosstr("ERROR: failed to close log file.\r\n$");
    return(1);
  }

  return(0);
}
