/*
	ListVESA - VESA screen mode listing utility
	Copyright (C) 2021 - 2023 Mercury Thirteen

	This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>



// function prototypes
void				PrintAppHeader();
void				PrintHelp();
unsigned short		VESAModeInfoGet(struct tVESAModeInfo far *thisModeInfoPtr, unsigned short far *modePtr);





// global definitions
unsigned char verMajor = 1;
unsigned char verMinor = 10;



struct tVESAInfoBlock
{
	unsigned char VBESignature[4];
	unsigned char VBEVersionMinor;
	unsigned char VBEVersionMajor;
	unsigned long OEMStringPtr;
	unsigned long Capabilities;
	unsigned long VideoModeListPtr;
	unsigned short TotalMemory;
	unsigned char OEMSoftwareRevMajor;
	unsigned char OEMSoftwareRevMinor;
	unsigned long VendorNamePtr;
	unsigned long ProductNamePtr;
	unsigned long ProductRevPtr;
	unsigned char Reserved[222];
	unsigned char OEMData[256];
};





struct tVESAModeInfo
{
	unsigned short ModeAttributes;
	unsigned char WinAAttributes;
	unsigned char WinBAttributes;
	unsigned short WinGranularity;
	unsigned short WinSize;
	unsigned short WinASegment;
	unsigned short WinBSegment;
	unsigned short WinFuncPtrOffset;
	unsigned short WinFuncPtrSegment;
	unsigned short BytesPerScanline;
	unsigned short XResolution;
	unsigned short YResolution;
	unsigned char XCharSize;
	unsigned char YCharSize;
	unsigned char NumberOfPlanes;
	unsigned char BitsPerPixel;
	unsigned char NumberOfBanks;
	unsigned char MemoryModel;
	unsigned char BankSize;
	unsigned char NumberOfImagePages;
	unsigned char ReservedA;
	unsigned char RedMaskSize;
	unsigned char RedFieldPosition;
	unsigned char GreenMaskSize;
	unsigned char GreenFieldPosition;
	unsigned char BlueMaskSize;
	unsigned char BlueFieldPosition;
	unsigned char RsvdMaskSize;
	unsigned char RsvdFieldPosition;
	unsigned char DirectColorModeInfo;
	unsigned long PhysBasePtr;
	unsigned long OffScreenMemOffset;
	unsigned short OffScreenMemSize;
	unsigned char ReservedB[206];
};





int main (int optionCount, char **options)
{
	unsigned long bytesPerFrame, bytesPerPixel, totalMemory;
	unsigned short modesMatching = 0, modesSkipped = 0;
	unsigned short index, matchColorDepth = 0, matchMode = 0;
	bool flagLFB = false, flagAdapterOnly = false, flagMatchColorDepth = false, flagMatchMode = false, flagSummaryOnly = false, flagVerbose = false;
	bool flagUnknownOption = true, flagModesSkipped = false, flagDisplayThisMode, flagIllogical;

	struct tVESAInfoBlock thisVESAInfo;
	struct tVESAInfoBlock far *thisVESAInfoPtr;

	struct tVESAModeInfo thisModeInfo;

	unsigned short outAX, charsWide, charsHigh;
	unsigned short far *modePtr;



	thisVESAInfoPtr = &thisVESAInfo;
	strcpy(thisVESAInfo.VBESignature, "VBE2");



	// hello world!
	PrintAppHeader();


	_asm
	{
		push es
		mov ax, 0x4F00
		mov edi, thisVESAInfoPtr
		shr edi, 16
		mov es, di
		mov edi, thisVESAInfoPtr
		int 0x10
		pop es
	}


	if (strncmp(thisVESAInfo.VBESignature, "VESA", 4))
	{
		printf("VESA is not supported on this machine. Exiting.");
	} else{
		if (optionCount > 1)
		{
			// if we get here, the number of options passed were greater than 1
			// let's process them!

			for (index = 1; index < optionCount; index = index + 1)
			{
				if (!strcasecmp(options[index], "/a"))
				{
					flagAdapterOnly = true;
					flagUnknownOption = false;
				}


				if (!strcasecmp(options[index], "/d"))
				{
					if (index + 1 < optionCount)
					{
						flagMatchColorDepth = true;
						matchColorDepth = strtol(options[index + 1], 0, 0);
						index++;
					} else {
						break;
					}
					flagUnknownOption = false;
				}


				if (!strcasecmp(options[index], "/l"))
				{
					flagLFB = true;
					flagUnknownOption = false;
				}


				if (!strcasecmp(options[index], "/m"))
				{
					if (index + 1 < optionCount)
					{
						flagMatchMode = true;
						flagVerbose = true;
						matchMode = strtol(options[index + 1], 0, 0);
						index++;
					} else {
						break;
					}
					flagUnknownOption = false;
				}


				if (!strcasecmp(options[index], "/s"))
				{
					flagSummaryOnly = true;
					flagUnknownOption = false;
				}

				if (!strcasecmp(options[index], "/v"))
				{
					flagVerbose = true;
					flagUnknownOption = false;
				}
			}

			if (flagUnknownOption)
			{
				PrintHelp();
				return(0);
			}
		}



		printf("VESA BIOS version %u.%u found.\n", thisVESAInfo.VBEVersionMajor, thisVESAInfo.VBEVersionMinor);
		printf("OEM String: %Ws\n", thisVESAInfo.OEMStringPtr);
		printf("OEM Vendor Name String: %Ws\n", thisVESAInfo.VendorNamePtr);
		printf("OEM Product Name String: %Ws\n", thisVESAInfo.ProductNamePtr);
		printf("OEM Product Revision String: %Ws\n", thisVESAInfo.ProductRevPtr);
		printf("Capabilities: %.32b\n", thisVESAInfo.Capabilities);

		if (thisVESAInfo.Capabilities & 0x00000001) printf ("\tDAC can be switched into 8-bit mode (Bit 0)\n");

		if (thisVESAInfo.Capabilities & 0x00000002) printf ("\tNon-VGA controller (Bit 1)\n");

		if (thisVESAInfo.Capabilities & 0x00000004) printf ("\tProgrammed DAC with blank bit (Bit 2)\n");

		if (thisVESAInfo.Capabilities & 0xFFFFFFF8) printf ("\tReserved (Bits 8 - 31)\n");

		totalMemory = 0;
		printf("Total Video Memory: %lu KiB\n", (unsigned long)thisVESAInfo.TotalMemory * 64);



		// handle summary
		if ((!flagAdapterOnly || flagSummaryOnly) && !flagMatchMode)
		{
			modesMatching = 0;
			modesSkipped = 0;

			printf("\n\n\nMode Summary:\n");
			printf("Mode     Width   Height   Depth   Type       Frame size (bytes)\n");

			modePtr = (unsigned short far *)thisVESAInfo.VideoModeListPtr;

			while (*modePtr != 0xFFFF)
			{
				outAX = VESAModeInfoGet(&thisModeInfo, modePtr);

				// default to showing the mode
				flagDisplayThisMode = true;
				flagIllogical = false;


				// let's see if it's appropriate to show this mode or not
				if (outAX != 0x004F) flagDisplayThisMode = false;

				if (flagLFB && !thisModeInfo.PhysBasePtr) flagDisplayThisMode = false;

				if (flagMatchColorDepth && matchColorDepth != thisModeInfo.BitsPerPixel) flagDisplayThisMode = false;

				if (flagMatchMode && matchMode != *modePtr) flagDisplayThisMode = false;

				if (thisModeInfo.XResolution == 0 || thisModeInfo.YResolution == 0 || thisModeInfo.BitsPerPixel == 0) flagIllogical = true;

				if (!flagVerbose && flagIllogical)
				{
					flagDisplayThisMode = false;
					flagModesSkipped = true;
					modesSkipped++;
				}

				if (flagDisplayThisMode)
				{
					bytesPerPixel = (unsigned long)thisModeInfo.BitsPerPixel / 8;
					if ((unsigned long)thisModeInfo.BitsPerPixel % 8) bytesPerPixel++;

					charsWide = 0;
					charsHigh = 0;

					if (thisModeInfo.XCharSize) charsWide = thisModeInfo.XResolution / thisModeInfo.XCharSize;
					if (thisModeInfo.YCharSize) charsHigh = thisModeInfo.YResolution / thisModeInfo.YCharSize;


					printf("0x%-4X   %-5u   %-6u   %-5u   ", *modePtr, thisModeInfo.XResolution, thisModeInfo.YResolution, thisModeInfo.BitsPerPixel);

	 				if (thisModeInfo.ModeAttributes & 0x0010)
	 				{
	 					printf("Graphics");
	 					bytesPerFrame = (unsigned long)thisModeInfo.XResolution * (unsigned long)thisModeInfo.YResolution * bytesPerPixel;
	 				} else {
	 					printf("Text    ");
	 					bytesPerFrame = charsWide * charsHigh;
	 				}
					printf("   %-7lu", bytesPerFrame);
					if (flagIllogical) printf("\t!");
					printf("\n");

					modesMatching++;
				}

				modePtr++;
			}
		}





		// handle detailed list
		if (!flagAdapterOnly && !flagSummaryOnly)
		{
			modesMatching = 0;
			modesSkipped = 0;

			modePtr = (unsigned short far *)thisVESAInfo.VideoModeListPtr;

			while (*modePtr != 0xFFFF)
			{
				outAX = VESAModeInfoGet(&thisModeInfo, modePtr);

				// default to showing the mode
				flagDisplayThisMode = true;
				flagIllogical = false;

				// shall we show this mode?
				if (outAX != 0x004F) flagDisplayThisMode = false;

				if (flagLFB && !thisModeInfo.PhysBasePtr) flagDisplayThisMode = false;

				if (flagMatchColorDepth && matchColorDepth != thisModeInfo.BitsPerPixel) flagDisplayThisMode = false;

				if (flagMatchMode && matchMode != *modePtr) flagDisplayThisMode = false;

				if (thisModeInfo.XResolution == 0 || thisModeInfo.YResolution == 0 || thisModeInfo.BitsPerPixel == 0) flagIllogical = true;

				if (!flagVerbose && flagIllogical)
				{
					flagDisplayThisMode = false;
					flagModesSkipped = true;
					modesSkipped++;
				}

				if (flagDisplayThisMode)
				{
					printf("\n\n\n");

					bytesPerPixel = (unsigned long)thisModeInfo.BitsPerPixel / 8;
					if ((unsigned long)thisModeInfo.BitsPerPixel % 8) bytesPerPixel++;


					charsWide = 0;
					charsHigh = 0;

					if (thisModeInfo.XCharSize) charsWide = thisModeInfo.XResolution / thisModeInfo.XCharSize;
					if (thisModeInfo.YCharSize) charsHigh = thisModeInfo.YResolution / thisModeInfo.YCharSize;


	 				if (thisModeInfo.ModeAttributes & 0x0010)
	 				{
	 					bytesPerFrame = (unsigned long)thisModeInfo.XResolution * (unsigned long)thisModeInfo.YResolution * bytesPerPixel;
	 				} else {
	 					bytesPerFrame = charsWide * charsHigh;
	 				}

					printf("Mode 0x%X - %ux%u, %u bit color (Frame size: %lu bytes)\n", *modePtr, thisModeInfo.XResolution, thisModeInfo.YResolution, thisModeInfo.BitsPerPixel, bytesPerFrame);
					if (flagIllogical) printf("WARNING: Illogical information reported for this mode\n");


		 			printf("Mode Attributes: %.16b\n", thisModeInfo.ModeAttributes);

	 				if (thisModeInfo.ModeAttributes & 0x0001)
	 				{
	 					printf("\tMode supported by present hardware configuration (Bit 0)\n");
	 				} else {
	 					printf("\tMode not supported by present hardware configuration (Bit 0)\n");
	 				}

	 				if (thisModeInfo.ModeAttributes & 0x0002) printf("\tOptional information available (Bit 1)\n");
	 				if (thisModeInfo.ModeAttributes & 0x0004) printf("\tBIOS output supported (Bit 2)\n");

	 				if (thisModeInfo.ModeAttributes & 0x0008)
	 				{
	 					printf("\tColor mode (Bit 3)\n");
	 				} else {
	 					printf("\tMonochrome mode (Bit 3)\n");
	 				}

	 				if (thisModeInfo.ModeAttributes & 0x0010)
	 				{
	 					printf("\tGraphics mode (Bit 4)\n");
	 				} else {
	 					printf("\tText mode (Bit 4)\n");
	 				}

	 				if (thisModeInfo.ModeAttributes & 0x0020) printf("\tMode is not VGA-compatible (Bit 5)\n");
	 				if (thisModeInfo.ModeAttributes & 0x0040) printf("\tBank-switched mode not supported (Bit 6)\n");
	 				if (thisModeInfo.ModeAttributes & 0x0080) printf("\tLinear framebuffer mode supported (Bit 7)\n");
	 				if (thisModeInfo.ModeAttributes & 0x0100) printf("\tDouble-scan mode available (Bit 8)\n");
	 				if (thisModeInfo.ModeAttributes & 0x0200) printf("\tInterlaced mode available (Bit 9)\n");
	 				if (thisModeInfo.ModeAttributes & 0x0400) printf("\tHardware supports triple buffering (Bit 10)\n");
	 				if (thisModeInfo.ModeAttributes & 0x0800) printf("\tHardware supports stereoscopic display (Bit 11)\n");
	 				if (thisModeInfo.ModeAttributes & 0x1000) printf("\tDual display start address support (Bit 12)\n");
	 				if (thisModeInfo.ModeAttributes & 0xE000) printf("\tReserved (Bits 13-15)\n");


					printf("Window A Attributes: ");
					if (thisModeInfo.WinASegment)
					{
						printf("%.8b\n", thisModeInfo.WinAAttributes);
	 					if (thisModeInfo.WinAAttributes & 0x01) printf("\tExists (Bit 0)\n");
	 					if (thisModeInfo.WinAAttributes & 0x02) printf("\tReadable (Bit 1)\n");
	 					if (thisModeInfo.WinAAttributes & 0x04) printf("\tWritable (Bit 2)\n");
	 					if (thisModeInfo.WinAAttributes & (unsigned char)0xF8) printf("\tReserved (Bits 3-7)\n");
	 				} else {
						printf("Not supported\n");
					}


					printf("Window B Attributes: ");
					if (thisModeInfo.WinBSegment)
					{
						printf("%.8b\n", thisModeInfo.WinBAttributes);
	 					if (thisModeInfo.WinBAttributes & 0x01) printf("\tExists (Bit 0)\n");
	 					if (thisModeInfo.WinBAttributes & 0x02) printf("\tReadable (Bit 1)\n");
	 					if (thisModeInfo.WinBAttributes & 0x04) printf("\tWritable (Bit 2)\n");
	 					if (thisModeInfo.WinBAttributes & 0xF8) printf("\tReserved (Bits 3-7)\n");
					} else {
						printf("Not supported\n");
					}


					printf("Window Granularity: %d KiB\n", thisModeInfo.WinGranularity);
					printf("Window Size: %d KiB\n", thisModeInfo.WinSize);


					printf("Window A Segment: ");
					if (thisModeInfo.WinASegment)
					{
						printf("0x%.4X", thisModeInfo.WinASegment);
					} else {
						printf("Not supported");
					}
					printf("\n");


					printf("Window B Segment: ");
					if (thisModeInfo.WinBSegment)
					{
						printf("0x%.4X\n", thisModeInfo.WinBSegment);
					} else {
						printf("Not supported");
					}
					printf("\n");


					printf("Far Pointer to Window Function: %.4X:%.4X\n", thisModeInfo.WinFuncPtrSegment, thisModeInfo.WinFuncPtrOffset);
					printf("Bytes Per Scanline: %d\n", thisModeInfo.BytesPerScanline);
					printf("Character Width: %d pixels (%u columns)\n", thisModeInfo.XCharSize, charsWide);
					printf("Character Height: %d pixels (%u rows)\n", thisModeInfo.YCharSize, charsHigh);
					printf("Number of Planes: %d\n", thisModeInfo.NumberOfPlanes);
					printf("Number of Banks: %d\n", thisModeInfo.NumberOfBanks);


					printf("Memory Model: ");
					switch (thisModeInfo.MemoryModel)
					{
						case 0x00:
							printf("Text");
							break;

						case 0x01:
							printf("CGA graphics");
							break;

						case 0x02:
							printf("HGC graphics");
							break;

						case 0x03:
							printf("16-color (EGA) graphics");
							break;

						case 0x04:
							printf("Packed pixel graphics");
							break;

						case 0x05:
							printf("Sequential 256 (non-chain 4) graphics");
							break;

						case 0x06:
							printf("Direct color");
							break;

						case 0x07:
							printf("YUV/YIQ luma-chroma");
							break;

						default:
							printf("Unknown");
					}
					printf(" (0x%.2X)\n", thisModeInfo.MemoryModel);


					printf("Bank Size: %d KiB\n", thisModeInfo.BankSize);
					printf("Number of Pages: %d\n", thisModeInfo.NumberOfImagePages + 1);
					printf("Red Mask Size: %u\n", thisModeInfo.RedMaskSize);
					printf("Red Field Position: %u\n", thisModeInfo.RedFieldPosition);
					printf("Green Mask Size: %u\n", thisModeInfo.GreenMaskSize);
					printf("Green Field Position: %u\n", thisModeInfo.GreenFieldPosition);
					printf("Blue Mask Size: %u\n", thisModeInfo.BlueMaskSize);
					printf("Blue Field Position: %u\n", thisModeInfo.BlueFieldPosition);
					printf("Reserved Mask Size: %u\n", thisModeInfo.RsvdMaskSize);
					printf("Reserved Field Position: %u\n", thisModeInfo.RsvdFieldPosition);
					printf("Direct Color Mode Info: %b\n", thisModeInfo.DirectColorModeInfo);
					printf("Physical Address of LFB: 0x%.8lX\n", thisModeInfo.PhysBasePtr);
					printf("Offscreen Memory Offset: 0x%.8lX\n", thisModeInfo.OffScreenMemOffset);
					printf("Offscreen Memory Size: 0x%.4X\n", thisModeInfo.OffScreenMemSize);

					modesMatching++;
				}

				modePtr++;
			}
		}
	}


	// print the wrap-up messages
	if (!flagAdapterOnly && (!flagMatchMode || (flagMatchMode && modesMatching == 0))) printf("\n\n\n");

	if (flagMatchMode)
	{
		if (modesMatching == 0)
		{
			printf("Mode 0x%X does not exist on this system.\n", matchMode);
		}
	} else {
		if (!flagAdapterOnly) printf("Modes listed: %u\n", modesMatching);
	}

	if (flagModesSkipped)
	{
		printf("\n\nYour video BIOS returned ");
		if (modesSkipped == 1)
		{
			printf("one");
		} else {
			printf("%u", modesSkipped);

		}
		printf(" screen mode");
		if (modesSkipped != 1) printf("s");
		printf(" containing illogical information,\n");
		printf("such as a width, height, or color depth of zero, which have been omitted here\n");
		printf("as this typically should not occur on properly implemented hardware. Run\n");
		printf("ListVESA with the verbose switch (/V) to see the complete list of modes which\n");
		printf("your video BIOS reported.\n");
	}

	printf("\n\n");

	return (0);
}





void PrintAppHeader()
{
	printf("ListVESA    VESA screen mode listing utility.\n");
	printf("v%u.%.2u       2021 - 2023 by Mercury Thirteen (mercury0x0d@protonmail.com)\n\n", verMajor, verMinor);
}





void PrintHelp()
{
	printf("Usage:\n");
	printf(" /A   \tShows adapter info only, no data on specific modes.\n");
	printf(" /D n \tShows only modes with a color depth of n.\n");
	printf(" /L   \tShows only modes which support a linear frame buffer.\n");
	printf(" /M n \tShows info only for the specified mode n. Implies /V.\n");
	printf(" /S   \tShows a summary of modes instead of detailed mode information.\n");
	printf(" /V   \tLists all screen modes, no matter how illogical they may seem.\n");
	printf("\n");
	printf("Numbers for all options may be entered in decimal, hexadecimal, or octal.\n\n");
	printf("Options may be combined. For example, LISTVESA /D 8 /L would list only all\n");
	printf("screen modes with 8-bit color depth and which support a linear frame buffer.\n\n");
}





unsigned short VESAModeInfoGet(struct tVESAModeInfo far *thisModeInfoPtr, unsigned short far *modePtr)
{
	unsigned short mode, outAX;

	// get mode data
	mode = *modePtr;
	_asm
	{
		push es
		mov ax, 0x4F01
		mov cx, mode
		mov edi, thisModeInfoPtr
		shr edi, 16
		mov es, di
		mov edi, thisModeInfoPtr
		int 0x10
		mov outAX, ax
		pop es
	}

	return outAX;
}
