//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: ffmpeg video playing (main scheduler)

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUGOUT_WARNING stdout
#define DISPQT_DEBUGOUT_QUEUE  stdout
#define DISPQT_DEBUGOUT_TIMED  NULL // stdout
#define DISPQT_DEBUGOUT_RENDER NULL // stdout
#define DISPQT_DEBUGOUT_DISPLAY stdout
#define DISPQT_DEBUGOUT_DEINT  NULL // stdout
#define DISPQT_DEBUGOUT_SUBTITLE NULL //stdout
#define DISPQT_DEBUGOUT_VFCC  stdout
#define DISPQT_DEBUGOUT_SEEKV NULL // stdout
#define DISPQT_DEBUGOUT_SEEKP NULL // stdout
#define DISPQT_DEBUGOUT_VIDEOWALL NULL // stdout

#include "moc_video_qt.h"
#include "moc_mainwindow.h"

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

#define INFFMPG_MUTEX_COMMAND_TIMEOUT       5300
#define INFFMPG_MUTEX_GETTIMESYNC_TIMEOUT   21   // ignore sync timestamp if it cannot be fetched within this time
#define DISPQT_VIDEOWORKER_DRAWFRAME_RETRY  3

#define MPXPLAY_VIDEODECODER_THREAD_AFFINITY_MASK 0 // MPXPLAY_AUDIODECODER_THREAD_AFFINITY_MASK //0x00000004

extern "C" {
 extern unsigned long mpxplay_config_dvbepg_control_flags, playcontrol;
}

extern unsigned int mpxplay_dispqt_videofilter_videowall_nb_descriptors, mpxplay_dispqt_videofilter_workerprocess_nb_descriptors;
extern struct dispqt_videofilter_processdescriptor_s mpxplay_dispqt_videofilter_videowall_descriptors_all[];
extern struct dispqt_videofilter_processdescriptor_s mpxplay_dispqt_videofilter_workerprocess_descriptors_all[];

static void videoworker_video_surface_refresh(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, unsigned int refresh_type);
static mpxp_bool_t videoworker_video_frame_display(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, int video_index, unsigned int flags, AVFrame *videoout_frame, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int refresh_type);
static mpxp_bool_t videoworker_video_reset_display(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, ffmpegvideo_decoder_s *fvdt, int video_index, unsigned int refresh_type);
static void videoworker_video_reset_display_from_cb(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos);

#if defined(PDS_THREADS_POSIX_THREAD)
static void *videoworker_video_worker_thread(void *decoder_data);
#else
static unsigned int videoworker_video_worker_thread(void *decoder_data);
#endif

FFMpegVideoDecoderWorker::FFMpegVideoDecoderWorker(MainWindow *main_window, dispqt_video_surface_info_s *video_surface_infos)
{
	this->main_window = main_window;
	pds_memset(&this->video_worker_infos, 0, sizeof(this->video_worker_infos));
	this->video_worker_infos.main_window = main_window;
	this->video_worker_infos.video_surface_infos = video_surface_infos;
	pds_threads_mutex_new(&this->video_worker_infos.mutexhnd_workerinfo);
	this->videodworker_thread_id = pds_threads_thread_create((mpxp_thread_func)videoworker_video_worker_thread, (void *)&this->video_worker_infos, MPXPLAY_THREAD_PRIORITY_HIGHER, MPXPLAY_VIDEODECODER_THREAD_AFFINITY_MASK);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD
	this->videodecoder_thread_id = pds_threads_thread_create((mpxp_thread_func)mpxplay_dispqt_video_decoder_worker_thread, &this->video_worker_infos, MPXPLAY_THREAD_PRIORITY_NORMAL, MPXPLAY_VIDEODECODER_THREAD_AFFINITY_MASK);
#endif
}

FFMpegVideoDecoderWorker::~FFMpegVideoDecoderWorker()
{
	this->decoderworker_video_scheduler_control(DISPQT_VIDEOSCHEDULER_CONTROL_REQ_CLOSE);
	void *mutexhnd = this->video_worker_infos.mutexhnd_workerinfo;
	int lockerror = PDS_THREADS_MUTEX_LOCK(&mutexhnd, DISPQT_VIDEO_MUTEX_TIMEOUT);
	this->video_worker_infos.mutexhnd_workerinfo = NULL;
	this->video_worker_infos.is_paused = true;
	this->cb_queue_clear(&this->video_worker_infos);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD
	this->video_worker_infos.video_decoderthread_control = DISPQT_VIDEOSCHEDULER_CONTROL_REQ_CLOSE;
	pds_threads_thread_close(this->videodecoder_thread_id);
#endif
	pds_threads_thread_close(this->videodworker_thread_id);
	if(lockerror == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mutexhnd);
	pds_threads_mutex_del(&mutexhnd);
	mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "FFMpegVideoDecoderWorker END");
}

void FFMpegVideoDecoderWorker::cb_queue_videostreamdecoderthread_set(ffmpegvideo_callback_list_s *cbelem, int streamtype_index)
{
	ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[streamtype_index];

	pds_memset(fvdt, 0, sizeof(*fvdt));
	mpxplay_ffmpgdec_packetqueue_init(&fvdt->videoframe_temp_queue, MPXPLAY_STREAMTYPEINDEX_TEMP);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
	mpxplay_ffmpgdec_packetqueue_init(&fvdt->videoframe_output_queue, streamtype_index);
#endif
	if((streamtype_index >= MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW) && funcbit_test(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_FILTER_WINDOW))
		fvdt->video_frame_filter_workerprocess = new DispQtVideoFrameFilter(this->main_window, this, &mpxplay_dispqt_videofilter_videowall_descriptors_all[0], mpxplay_dispqt_videofilter_videowall_nb_descriptors);
	fvdt->streamtype_index = streamtype_index;
	funcbit_enable(fvdt->flags, (DISPQT_FFMPGVIDEOCALLBACKFLAG_FIRSTFRAME | DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME | DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP));
}

void FFMpegVideoDecoderWorker::cb_queue_videostreamdecoderthread_clear(ffmpegvideo_callback_list_s *cbelem, int streamtype_index)
{
	ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[streamtype_index];

	mpxplay_ffmpgdec_packetqueue_close(&fvdt->videoframe_temp_queue);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
	mpxplay_ffmpgdec_packetqueue_close(&fvdt->videoframe_output_queue);
#endif
	MPXPLAY_SAFE_DELETE_OBJECT(fvdt->video_frame_filter_workerprocess);
	mpxplay_dispqt_videotrans_ffmpeg_swsctx_close(&fvdt->frame_conv_infos);
	pds_memset(fvdt, 0, sizeof(*fvdt));
}

int FFMpegVideoDecoderWorker::cb_queue_put_entry(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, mpxp_ptrsize_t cbk_func, mpxp_ptrsize_t passdata, bool prepend)
{
	ffmpegvideo_callback_list_s *cbelem_new;
	mpxp_uint32_t is_videowall_available, is_vwall_enabled;

	cbelem_new = (ffmpegvideo_callback_list_s *)av_mallocz(sizeof(ffmpegvideo_callback_list_s));
	if(!cbelem_new)
		goto err_out_put_entry;

	cbelem_new->infile_passdata = passdata;
	if(prepend)
		funcbit_enable(cbelem_new->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO);

	if(cbk_func){
		for(int i = MPXPLAY_STREAMTYPEINDEX_VIDEO; i < MPXPLAY_STREAMTYPEINDEX_MAX; i++)
			this->cb_queue_videostreamdecoderthread_set(cbelem_new, i);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD
		cbelem_new->videodecoder_thread_id = pds_threads_thread_create((mpxp_thread_func)mpxplay_dispqt_video_decoder_worker_thread, cbelem_new, MPXPLAY_THREAD_PRIORITY_NORMAL, MPXPLAY_VIDEODECODER_THREAD_AFFINITY_MASK);
		if(!cbelem_new->videodecoder_thread_id)
			goto err_out_put_entry;
#endif
	}

	cbelem_new->video_frame_filter_workerprocess = new DispQtVideoFrameFilter(this->main_window, this, &mpxplay_dispqt_videofilter_workerprocess_descriptors_all[0], mpxplay_dispqt_videofilter_workerprocess_nb_descriptors);
	if(!cbelem_new->video_frame_filter_workerprocess)
		goto err_out_put_entry;

#ifdef DISPQT_VIDEO_SCHEDULER_USE_COND_SIGNAL
	cbelem_new->videodec_thread_condition_signal_handler = pds_threads_cond_new();
#endif

	if(!workinfos->cblist_last){
		workinfos->cblist_first = cbelem_new;
		workinfos->cblist_last = cbelem_new;
	}else if(prepend){
		funcbit_disable(workinfos->cblist_first->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO);
		cbelem_new->next = workinfos->cblist_first;
		workinfos->cblist_first = cbelem_new;
	}else{ // TODO: cb_queue_del_entry DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO flag at append mode
		workinfos->cblist_last->next = cbelem_new;
		workinfos->cblist_last = cbelem_new;
	}
	mpxplay_dispqt_ffmpegvideo_seekpreview_send(NULL);
	cbelem_new->videodelays_decoder.video_init_framecounter = DISPQT_FFMPGVIDEOCALLBACK_INITFRAMECOUNTER_START;
	cbelem_new->videodelays_scheduler.video_init_framecounter = DISPQT_FFMPGVIDEOCALLBACK_INITFRAMECOUNTER_START;

	// videowall settings ---------------------------------------------------------------------------------------------------------------
	is_videowall_available = FALSE;
	if(cbk_func)
		((mpxplay_infile_callback_t *)cbk_func)(passdata, MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLISAVAILABLE, (mpxp_ptrsize_t)&is_videowall_available, 0, INFFMPG_MUTEX_COMMAND_TIMEOUT + 1);

	emit this->main_window->qt_video_player->signal_video_set_value(DispQtVideoPlayer::VideoCtrlValue_VideoWallAvailable, is_videowall_available);
	workinfos->is_videowall_enabled = false;
	if(is_videowall_available)
	{
		funcbit_enable(cbelem_new->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_AVAILABLE);

		is_vwall_enabled = FALSE;
		((mpxplay_infile_callback_t *)cbk_func)(passdata, MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLISENABLED, (mpxp_ptrsize_t)&is_vwall_enabled, 0, INFFMPG_MUTEX_COMMAND_TIMEOUT + 2);
		workinfos->is_videowall_enabled = (bool)is_vwall_enabled;
		emit this->main_window->qt_video_player->signal_video_set_value(DispQtVideoPlayer::VideoCtrlValue_VideoWallEnabled, is_vwall_enabled);

		if(funcbit_test(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_KEYFRAMES_ONLY))
			funcbit_enable(cbelem_new->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY);
	}
	cbelem_new->selected_stream_index = MPXPLAY_STREAMTYPEINDEX_VIDEO;
	cbelem_new->videowall_nb_streams = 4;   // !!! initial fake number
	cbelem_new->videowall_nb_windows_x = 2; //
	cbelem_new->videowall_nb_windows_y = 2; //
	// ----------------------------------------------------------------------------------------------------------------------------------

	cbelem_new->infile_callback = (mpxplay_infile_callback_t *)cbk_func;
	workinfos->nb_callbacks++;

	mpxplay_debugf(DISPQT_DEBUGOUT_QUEUE, "cb_queue_put n:%d f:%8.8X l:%8.8X cb:%8.8X pd:%8.8X", workinfos->nb_callbacks,
			(mpxp_ptrsize_t)workinfos->cblist_first, (mpxp_ptrsize_t)workinfos->cblist_last, (mpxp_ptrsize_t)cbk_func, (mpxp_ptrsize_t)passdata);

	return 0;

err_out_put_entry:
	this->cb_queue_elem_clear(cbelem_new);
	mpxplay_debugf(DISPQT_DEBUGOUT_QUEUE, "cb_queue_put FAILED");
	return -1;
}

void FFMpegVideoDecoderWorker::cb_queue_elem_decoder_close(struct ffmpegvideo_callback_list_s *cbelem)
{
	mpxplay_debugf(DISPQT_DEBUGOUT_QUEUE, "cb_queue_elem_decoder_close START cb:%8.8X", (mpxp_ptrsize_t)cbelem);
	if(!cbelem)
		return;

#if defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD) || defined(DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD)
#if defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD)
	if(cbelem->videodecoder_thread_id)
#endif
	{
		int retry = DISPQT_VIDEO_MUTEX_TIMEOUT / MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT;
		do {
			funcbit_enable(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_EXITEVENT);
			if(funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_THEXITDONE))
				break;
			pds_threads_sleep(MPXPLAY_VIDEO_WORKDEC_THREAD_SLEEPTIME_SHORT);
		} while(--retry);
#if defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD)
		pds_threads_thread_close(cbelem->videodecoder_thread_id);
#endif
	}
#else
	mpxplay_dispqt_video_decoder_worker_close(cbelem);
#endif
	mpxplay_debugf(DISPQT_DEBUGOUT_QUEUE, "cb_queue_elem_decoder_close END cb:%8.8X", (mpxp_ptrsize_t)cbelem);
}

void FFMpegVideoDecoderWorker::cb_queue_elem_clear(struct ffmpegvideo_callback_list_s *cbelem)
{
	if(!cbelem)
		return;
	mpxplay_debugf(DISPQT_DEBUGOUT_QUEUE, "cb_queue_elem_clear START cb:%8.8X pd:%8.8X", (mpxp_ptrsize_t)cbelem, (mpxp_ptrsize_t)cbelem->infile_passdata);

	for(int i = MPXPLAY_STREAMTYPEINDEX_VIDEO; i < MPXPLAY_STREAMTYPEINDEX_MAX; i++)
		this->cb_queue_videostreamdecoderthread_clear(cbelem, i);
	cbelem->infile_callback = NULL;
	cbelem->infile_passdata = (mpxp_ptrsize_t)0;
	mpxplay_decoderworker_subtitle_reset(cbelem);
	mpxplay_dispqt_videotrans_ffmpeg_swsctx_close(&cbelem->subtitle_bitmap_cv_infos);
	if(cbelem->subtitle_infos)
		av_freep(&cbelem->subtitle_infos);
	MPXPLAY_SAFE_DELETE_OBJECT(cbelem->video_frame_filter_workerprocess);
#ifdef DISPQT_VIDEO_SCHEDULER_USE_COND_SIGNAL
	pds_threads_cond_del(&cbelem->videodec_thread_condition_signal_handler);
#endif
	av_free(cbelem);
	mpxplay_debugf(DISPQT_DEBUGOUT_QUEUE, "cb_queue_elem_clear END");
}

int FFMpegVideoDecoderWorker::cb_queue_del_entry(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, mpxp_ptrsize_t cbk_func, mpxp_ptrsize_t passdata)
{
	ffmpegvideo_callback_list_s *cbelem = workinfos->cblist_first, *cbe_prev = NULL;

	do{
		if(!cbelem)
			return -1;
		if((cbelem->infile_callback == (mpxplay_infile_callback_t *)cbk_func) && (cbelem->infile_passdata == passdata)){
			if( (workinfos->nb_callbacks <= 1)
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
//			 || (workinfos->cblist_first->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO].videoframe_output_queue.nb_packets <= 0) // FIXME: it's not good at stillpict
#endif
			){
				videoworker_video_reset_display_from_cb(workinfos);
			}
			cb_queue_elem_decoder_close(cbelem);
			if(cbe_prev)
				cbe_prev->next = cbelem->next;
			else if(cbelem == workinfos->cblist_first)
				workinfos->cblist_first = cbelem->next;
			if(cbelem == workinfos->cblist_last)
				workinfos->cblist_last = cbe_prev;
			cb_queue_elem_clear(cbelem);
			workinfos->nb_callbacks--;
			break;
		}
		cbe_prev = cbelem;
		cbelem = cbelem->next;
	}while(1);

	mpxplay_debugf(DISPQT_DEBUGOUT_QUEUE, "cb_queue_del_entry n:%d f:%8.8X l:%8.8X cb:%8.8X pd:%8.8X", workinfos->nb_callbacks,
			(mpxp_ptrsize_t)workinfos->cblist_first, (mpxp_ptrsize_t)workinfos->cblist_last, (mpxp_ptrsize_t)cbk_func, (mpxp_ptrsize_t)passdata);

	return 0;
}

void FFMpegVideoDecoderWorker::cb_queue_clear(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos)
{
	ffmpegvideo_callback_list_s *cbelem;
	if(!workinfos)
	{
		return;
	}
	videoworker_video_reset_display_from_cb(workinfos);
	cbelem = workinfos->cblist_first;
	while(cbelem){
		ffmpegvideo_callback_list_s *nextcbe = cbelem->next;
		cb_queue_elem_decoder_close(cbelem);
		cb_queue_elem_clear(cbelem);
		cbelem = nextcbe;
	}
	workinfos->cblist_first = NULL;
	workinfos->cblist_last = NULL;
	workinfos->nb_callbacks = 0;
}

static void videoworker_videosurface_is_open(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos)
{
	bool is_videosurface_open = ((workinfos->nb_callbacks > 0) && (workinfos->is_video_stream || (mpxplay_config_video_audiovisualization_type != DISPQT_AUDIOVISUALIZATIONTYPE_NONE)))? true : false;
	if(is_videosurface_open != workinfos->is_videosurface_open)
	{
		if(!is_videosurface_open)
			workinfos->video_surface_refresh_request = DISPQT_VIDEOSURFACE_REFRESHTYPE_CLOSE;
		workinfos->is_videosurface_open = is_videosurface_open;
	}
}

bool FFMpegVideoDecoderWorker::videothread_ffmpeg_config_callback(unsigned int cfgcmd, mpxp_ptrsize_t cbk_func, mpxp_ptrsize_t passdata)
{
	struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos = &this->video_worker_infos;
	ffmpegvideo_callback_list_s *cbelem;
	mpxplay_packetlist_t *videoctx_pktelem;
	int video_streamtypeindex, lockerror;
	bool is_video_stream = false, is_videosurface_open;

	mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videothread_ffmpeg_config_callback BEGIN cmd:%8.8X cbk:%8.8X pd:%8.8X", cfgcmd, (unsigned int)cbk_func, (unsigned int)passdata);

	if(cfgcmd == MPXPLAY_INFILE_CBKCFG_SRVFFMV_CLEAR_CALLBACKS)
	{
		workinfos->is_paused = true;
		workinfos->is_video_stream = false;
		workinfos->is_videosurface_open = false;
		return workinfos->is_videosurface_open;
	}
	else if((cfgcmd == MPXPLAY_INFILE_CBKCFG_SRVFFMV_PAUSE) || (cfgcmd == MPXPLAY_INFILE_CBKCFG_SRVFFMV_UNPAUSE) || (cfgcmd == MPXPLAY_INFILE_CBKCFG_SRVFFMV_RECHECK_SURFACE))
	{
		switch(cfgcmd)
		{
			case MPXPLAY_INFILE_CBKCFG_SRVFFMV_PAUSE:
				workinfos->is_paused = true;
				break;
			case MPXPLAY_INFILE_CBKCFG_SRVFFMV_UNPAUSE:
				workinfos->is_paused = false;
				break;
		}
		videoworker_videosurface_is_open(workinfos);
		return workinfos->is_videosurface_open;
	}

	lockerror = PDS_THREADS_MUTEX_LOCK(&workinfos->mutexhnd_workerinfo, DISPQT_VIDEO_MUTEX_TIMEOUT);
	mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videothread_ffmpeg_config_callback LOCKED cmd:%8.8X cbk:%8.8X pd:%8.8X lock:%d", cfgcmd, (unsigned int)cbk_func, (unsigned int)passdata, lockerror);

	switch(cfgcmd)
	{
//		case MPXPLAY_INFILE_CBKCFG_SRVFFMV_CLEAR_CALLBACKS: // FIXME: ?
//			this->cb_queue_clear(workinfos);
//			break;
		case MPXPLAY_INFILE_CBKCFG_SRVFFMV_OPEN_CALLBACK:
			this->cb_queue_put_entry(workinfos, cbk_func, passdata, true); // newly opened video is always the primary (putting it on the begin of cb queue)
			workinfos->is_paused = (playcontrol & PLAYC_RUNNING)? false : true; // TODO: a nicer solution
			break;
		case MPXPLAY_INFILE_CBKCFG_SRVFFMV_CLOSE_CALLBACK:
			this->cb_queue_del_entry(workinfos, cbk_func, passdata);
			if(workinfos->nb_callbacks < 0)
				workinfos->nb_callbacks = 0;
			break;
		case MPXPLAY_INFILE_CBKCFG_SRVFFMV_REFRESH_CALLBACK:
			break;
	}

	if(workinfos->nb_callbacks > 0)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videothread_ffmpeg_config_callback SEARCH cmd:%8.8X pd:%8.8X", cfgcmd, (unsigned int)passdata);
		cbelem = workinfos->cblist_first;
		while(cbelem)
		{
			int retval = MPXPLAY_ERROR_INFILE_NODATA;
			video_streamtypeindex = MPXPLAY_STREAMTYPEINDEX_VIDEO;
			videoctx_pktelem = NULL;
			if(cbelem->infile_callback)
				retval = cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_CODECCTX_FFMPEG, (mpxp_ptrsize_t)&videoctx_pktelem, video_streamtypeindex, INFFMPG_MUTEX_COMMAND_TIMEOUT + 3);
			mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videothread_ffmpeg_config_callback infile_callback FOUND ret:%d pd:%8.8X ctx:%8.8X next:%8.8X", retval,
							(mpxp_ptrsize_t)cbelem->infile_passdata, (mpxp_ptrsize_t)videoctx_pktelem, (mpxp_ptrsize_t)cbelem->next);
			if((retval == MPXPLAY_ERROR_OK) && videoctx_pktelem)
			{
				is_video_stream = true;
			}
			//cbelem = cbelem->next;
			break; // currently we check the primary video only
		}
	}

	workinfos->is_video_stream = is_video_stream;
	videoworker_videosurface_is_open(workinfos);

	mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videothread_ffmpeg_config_callback END open:%d nb:%d cmd:%8.8X pd:%8.8X lock:%d", (int)workinfos->is_videosurface_open, workinfos->nb_callbacks, cfgcmd, (int)passdata, lockerror);

	if(lockerror == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&workinfos->mutexhnd_workerinfo);

	return workinfos->is_videosurface_open;
}

//------------------------------------------------------------------------------------------------------------------------------------------------------

int FFMpegVideoDecoderWorker::videothread_videowall_search_windowitem(ffmpegvideo_callback_list_s *cbelem, int streamtypeindex, int globalpos_x, int globalpos_y, bool halve_win_width)
{
	struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEOWALL];

	if(!cbelem->is_videowall_enabled || cbelem->is_videowall_disabled_by_decoder)
	{
		return streamtypeindex;
	}
	if((globalpos_x < 0) || (globalpos_y < 0))
	{
		return streamtypeindex;
	}

	for(int i = MPXPLAY_STREAMTYPEINDEX_VIDEOWALL, w = 0; i < MPXPLAY_STREAMTYPEINDEX_MAX; i++, fvdt++)
	{
		if( (fvdt->videowall_window_width > 0) && (fvdt->videowall_window_height > 0)
		 && (globalpos_x >= fvdt->videowall_window_pos_x) && (globalpos_x < (fvdt->videowall_window_pos_x + (fvdt->videowall_window_width / ((halve_win_width)? 2 : 1)))) // EPGTooltipWindow is waked up only by the left half of the sub-window
		 && (globalpos_y >= fvdt->videowall_window_pos_y) && (globalpos_y < (fvdt->videowall_window_pos_y + fvdt->videowall_window_height))
		){
			streamtypeindex = i;
			mpxplay_debugf(DISPQT_DEBUGOUT_VIDEOWALL, "videowall_search gx:%d xy:%d wx:%d wy:%d ww:%d wh:%d", globalpos_x, globalpos_y, fvdt->videowall_window_pos_x,
					fvdt->videowall_window_pos_y, fvdt->videowall_window_width, fvdt->videowall_window_height);
			break;
		}
	}

	if(streamtypeindex < 0)
	{
		fvdt = &cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEOWALL];
		mpxplay_debugf(DISPQT_DEBUGOUT_VIDEOWALL, "videowall_search gx:%d xy:%d wx:%d wy:%d ww:%d wh:%d", globalpos_x, globalpos_y, fvdt->videowall_window_pos_x,
							fvdt->videowall_window_pos_y, fvdt->videowall_window_width, fvdt->videowall_window_height);
	}

	return streamtypeindex;
}

void FFMpegVideoDecoderWorker::videothread_ffmpeg_callback_call(int globalpos_x, int globalpos_y, unsigned long command, mpxp_ptrsize_t arg1, mpxp_ptrsize_t arg2)
{
	struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos = &this->video_worker_infos;
	if((workinfos->video_scheduler_control == DISPQT_VIDEOSCHEDULER_CONTROL_RUN) && this->main_window->qt_video_player)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videothread_ffmpeg_callback_call BEGIN");
		int lockerror = PDS_THREADS_MUTEX_LOCK(&workinfos->mutexhnd_workerinfo, DISPQT_VIDEO_MUTEX_TIMEOUT);
		ffmpegvideo_callback_list_s *cbelem = workinfos->cblist_first; // TODO: select by globalpos
		if(cbelem && cbelem->infile_callback)
		{
			switch(command)
			{
				case MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLCURRSTI:
				case MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLCURRSTI_EW:
					if((arg1 = videothread_videowall_search_windowitem(cbelem, (int)arg1, globalpos_x, globalpos_y, false)) < 0) // select program by globalpos
						break;
					if(command == MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLCURRSTI_EW)
					{
						if(arg1 == cbelem->selected_stream_index) // only exit from videowall, if the program hasn't changed
						{
							command = MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLMODE;
							arg1 = 0;
						}
						workinfos->is_videowall_enabled = 0;
						emit this->main_window->qt_video_player->signal_video_set_value(DispQtVideoPlayer::VideoCtrlValue_VideoWallEnabled, 0);
					}
					// @suppress("No break at end of case")
				case MPXPLAY_INFILE_CBKCTRL_SET_SELECT_PROGRAM:
				case MPXPLAY_INFILE_CBKCTRL_SET_SELECT_STREAM:
					cbelem->callback_command.cmd = command;
					cbelem->callback_command.arg1 = arg1;
					cbelem->callback_command.arg2 = arg2;
					break;
				case MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLPRGEPGLIST:
					if((arg1 = videothread_videowall_search_windowitem(cbelem, (int)arg1, globalpos_x, globalpos_y, true)) >= 0) // select program by globalpos
					{
						cbelem->infile_callback(cbelem->infile_passdata, command, arg1, arg2, INFFMPG_MUTEX_COMMAND_TIMEOUT + 4);
						struct mpxplay_dvb_epgeventlist_t *epg_list = *((struct mpxplay_dvb_epgeventlist_t **)arg2);
						if(epg_list)
						{
							struct ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[arg1];
							epg_list->videowall_window_pos_x = fvdt->videowall_window_pos_x;
							epg_list->videowall_window_pos_y = fvdt->videowall_window_pos_y;
							epg_list->videowall_window_width = fvdt->videowall_window_width;
							epg_list->videowall_window_height = fvdt->videowall_window_height;
						}
					}
					break;
				case MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLMODE:
					if(!arg1 || (funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_AVAILABLE) && !cbelem->is_videowall_disabled_by_decoder))
					{
						workinfos->is_videowall_enabled = arg1;
						cbelem->infile_callback(cbelem->infile_passdata, command, arg1, arg2, INFFMPG_MUTEX_COMMAND_TIMEOUT + 5);
						emit this->main_window->qt_video_player->signal_video_set_value(DispQtVideoPlayer::VideoCtrlValue_VideoWallEnabled, (int)arg1);
					}
					break;
				default:
					cbelem->infile_callback(cbelem->infile_passdata, command, arg1, arg2, INFFMPG_MUTEX_COMMAND_TIMEOUT + 6);
					break;
			}
		}
		if(lockerror == MPXPLAY_ERROR_OK)
			PDS_THREADS_MUTEX_UNLOCK(&workinfos->mutexhnd_workerinfo);
		mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videothread_ffmpeg_callback_call END");
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------------------

bool FFMpegVideoDecoderWorker::decoderworker_video_scheduler_control(unsigned int control_req)
{
	struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos = &this->video_worker_infos;
	unsigned int expected_status = (control_req == DISPQT_VIDEOSCHEDULER_CONTROL_REQ_CLOSE)? DISPQT_VIDEOSCHEDULER_STATUS_CLOSED : DISPQT_VIDEOSCHEDULER_STATUS_SUSPENDED;
	mpxp_int64_t end_time = pds_gettimem() + DISPQT_VIDEO_MUTEX_TIMEOUT;
	bool success = false;

	do{
		if(control_req == DISPQT_VIDEOSCHEDULER_CONTROL_RUN)
		{
			success = true;
		}
		else if(workinfos->video_scheduler_control == control_req)
		{
			if(workinfos->video_scheduler_status == expected_status)
			{
				success = true;
			}
			else if(workinfos->video_scheduler_status != DISPQT_VIDEOSCHEDULER_CONTROL_RUN)
			{
				workinfos->video_scheduler_status = DISPQT_VIDEOSCHEDULER_CONTROL_RUN;
			}
		}
		workinfos->video_scheduler_control = control_req;
		if(success)
		{
			break;
		}
		pds_threads_sleep(MPXPLAY_THREADS_SHORTTASKSLEEP);
	}while(pds_gettimem() < end_time);

	if(control_req == DISPQT_VIDEOSCHEDULER_CONTROL_REQ_SUSPEND)
	{
		decoderworker_video_surface_refresh(DISPQT_VIDEOSURFACE_REFRESHTYPE_CLEAR);
	}

	return success;
}

static void videoworker_video_surface_refresh(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, unsigned int refresh_type)
{
	if(refresh_type > workinfos->video_surface_refresh_request)
		workinfos->video_surface_refresh_request = refresh_type;
}

void FFMpegVideoDecoderWorker::decoderworker_video_surface_refresh(unsigned int refresh_type)
{
	videoworker_video_surface_refresh(&this->video_worker_infos, refresh_type);
}

//------------------------------------------------------------------------------------------------------------------------------------------------------

static int videoworker_video_load_next_frame(ffmpegvideo_callback_list_s *cbelem, ffmpegvideo_decoder_s *fvdt, bool *resetvideo)
{
	int retval_video = 0;

#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
	mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_pktlist_elem);

	retval_video = mpxplay_ffmpgdec_packetqueue_get(cbelem->infile_passdata, &fvdt->videoframe_output_queue, &fvdt->videoframe_pktlist_elem, 0);
#endif
	if(fvdt->videoframe_pktlist_elem)
	{
		if(fvdt->videoframe_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)
			*resetvideo = true;
		if(fvdt->videoframe_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_STILLPICT)
			funcbit_enable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_STILLPICT);
		else
			funcbit_disable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_STILLPICT);
		if(fvdt->videoframe_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_RESET)
			funcbit_enable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_FIRSTFRAME);
		if((fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) && !funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_STILLPICT))
			cbelem->subtitle_vframe_infos = mpxplay_decoderworker_subtitle_get(cbelem, fvdt->videoframe_pktlist_elem->timestamp_us, *resetvideo);
	}

	return retval_video;
}

static mpxp_bool_t videoworker_video_frame_display(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, int video_index, unsigned int flags, AVFrame *videoout_frame, ffmpegvideo_subtitle_info_s *subtitle_infos, unsigned int refresh_type)
{
	mpxp_bool_t success = FALSE;
	if(funcbit_test(workinfos->video_surface_infos->video_render_infos->renderer_capabilities, DISPQT_VIDEORENDERER_CAPFLAG_DIRECTRENDER))
	{   // call directly rendering
		success = mpxplay_dispqt_videowidget_direct_frame_display(workinfos->video_surface_infos, flags, videoout_frame, subtitle_infos, refresh_type);
	}
	else // send frame through Qt's API
	{
		if(mpxplay_dispqt_ffmpegvideo_frame_display(video_index, flags, videoout_frame, subtitle_infos, refresh_type) == MPXPLAY_ERROR_OK)
		{
			success = TRUE;
		}
	}
	return success;
}

static mpxp_bool_t videoworker_video_reset_display(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, ffmpegvideo_decoder_s *fvdt, int video_index, unsigned int refresh_type)
{
	unsigned int sch_flags = (fvdt)? fvdt->flags : (workinfos->video_surface_infos->scheduler_flags & ~DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_PAUSEDPLAY);
	if(workinfos->is_paused)
		funcbit_enable(sch_flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_PAUSEDPLAY);
	mpxplay_debugf(DISPQT_DEBUGOUT_VFCC, "videoworker_video_reset_display %d", refresh_type);
	return videoworker_video_frame_display(workinfos, video_index, sch_flags, NULL, NULL, refresh_type);
}

static void videoworker_video_reset_display_from_cb(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos)
{
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	// dealloc avframe stored in the video_widget, this shall be called before closing related hw decoder (avcodec)
	if(workinfos->video_surface_infos->video_source_infos)
	{
		AVFrame *avframe = workinfos->video_surface_infos->video_source_infos->ffmpeg_decoded_frame;
		workinfos->video_surface_infos->video_source_infos->ffmpeg_decoded_frame = NULL;
		av_frame_free(&avframe);
	}
	if(workinfos->video_scheduler_status != DISPQT_VIDEOSCHEDULER_STATUS_CLOSED)
		videoworker_video_surface_refresh(workinfos, DISPQT_VIDEOSURFACE_REFRESHTYPE_CLEAR); // it has no effect at stop playing (REFRESHTYPE_CLOSE is initiated from videoworker_videosurface_is_open)
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
}

static void videoworker_video_reset(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, ffmpegvideo_callback_list_s *cbelem, ffmpegvideo_decoder_s *fvdt, int video_index)
{
	if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
	{
		cbelem->video_render_status = DISPQT_VIDEOSCHEDULER_RENDERSTAT_NONE;
		// cbelem->video_last_sch_timestamp = 0; // FIXME: needed?
		if(!funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME))
		{
			videoworker_video_reset_display(workinfos, fvdt, video_index, DISPQT_VIDEOSURFACE_REFRESHTYPE_RESET);
			funcbit_disable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_FIRSTFRAME);
		}
		mpxplay_dispqt_videoworkdec_videodelay_reset(&cbelem->videodelays_scheduler, TRUE);
		mpxplay_decoderworker_subtitle_reset(cbelem);
		mpxplay_debugf(DISPQT_DEBUGOUT_SEEKV, "TimedEvent RESETVIDEO 1 %d", fvdt->streamtype_index);
	}

	funcbit_enable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP);
	if(workinfos->is_paused || (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW))
		funcbit_enable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME);
	fvdt->is_deinterlacedup_2ndpass = false;
	mpxplay_debugf(DISPQT_DEBUGOUT_SEEKV, "TimedEvent RESETVIDEO 2 i:%d dof:%d p:%d", fvdt->streamtype_index, ((fvdt->flags & DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME)? 1:0),
			(int)workinfos->is_paused);
}

static inline mpxp_bool_t videoworker_is_frame_duplication_enabled(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos, ffmpegvideo_callback_list_s *cbelem)
{
	if( funcbit_test(workinfos->video_surface_infos->video_render_infos->renderer_capabilities, DISPQT_VIDEORENDERER_CAPENABLE_DEINTERDUP) // duplicate interlaced frames is supported and enabled by render output
	 && (cbelem->video_average_frame_interval >= DISPQT_FFMPGVIDEOCALLBACK_VIDEOFRAME_INTERVAL_33FPS) // FPS of video source is less than 33
	 && ((cbelem->video_delay_us < cbelem->video_average_frame_interval) || funcbit_test(workinfos->main_window->gui_config->selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESKIPFRAME)) // video delay is not more than 1 frame
	){
		return TRUE;
	}
	return FALSE;
}

static mpxp_bool_t videoworker_video_frame_scheduler(struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos,
		ffmpegvideo_callback_list_s *cbelem, ffmpegvideo_decoder_s *fvdt, mpxp_int64_t timestamp_sync_us,
		unsigned int *sleep_length)
{
#ifdef MPXPLAY_USE_DEBUGF
	static mpxp_int64_t average_present_time;
#endif
	AVFrame *videoout_frame = (AVFrame *)(fvdt->videoframe_pktlist_elem->frame_pkt);
	mpxp_bool_t videoframe_presented = FALSE;

	if(!videoout_frame && !fvdt->is_deinterlacedup_2ndpass)
		return videoframe_presented;

	const mpxp_int64_t render_begintime_ms = pds_gettimem(), render_time_diff_ms = render_begintime_ms - cbelem->video_last_present_time;
	mpxp_int64_t timestamp_vpacket_us = fvdt->videoframe_pktlist_elem->timestamp_us;
	const mpxp_uint32_t renderer_capabilities = workinfos->video_surface_infos->video_render_infos->renderer_capabilities;
	unsigned int refreshtype_scheduler = 0, renderflags_scheduler = (workinfos->is_paused)? DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_PAUSEDPLAY : 0;
	mpxp_bool_t doIgnoreTimestamp = (funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP) || funcbit_test(fvdt->videoframe_pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_STILLPICT))? TRUE : FALSE;
	mpxp_bool_t do_present_frame = FALSE, doFrameDrop = FALSE;

	if(fvdt->is_deinterlacedup_2ndpass)
	{
		//mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "DEINT avg:%5d rtd:%3d", cbelem->video_average_frame_interval, (render_time_diff_ms * 1000));
		if((render_time_diff_ms * 1000) >= (cbelem->video_average_frame_interval >> 1))
		{
			fvdt->is_deinterlacedup_2ndpass = false;
			funcbit_enable(renderflags_scheduler, DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_DEINT2NDPASS);
			refreshtype_scheduler = DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE;
			do_present_frame = TRUE;
			videoout_frame = NULL;
			mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "DEINT 2ND PASS start");
		}
	}
	else
	{
		cbelem->video_delay_us = timestamp_sync_us - timestamp_vpacket_us;

		if(!doIgnoreTimestamp && (timestamp_sync_us > 0) && (cbelem->video_last_sch_timestamp > 0))
		{
			// correct rendering delay (gpu deinterlace)
			if(timestamp_vpacket_us > cbelem->video_last_sch_timestamp)
			{
				mpxp_uint32_t render_frame_delay = mpxplay_video_render_infos.render_function_get_frame_delay(mpxplay_video_render_infos.renderer_private_data);
				if(render_frame_delay)
				{
					mpxp_int64_t render_time_delay_us = (mpxp_int64_t)render_frame_delay * (timestamp_vpacket_us - cbelem->video_last_sch_timestamp);
					if(render_time_delay_us <= 200000) // the delay should be max 2-3 frames, more than a 0.2 sec rendering delay is invalid (or seek forward has happened)
					{
						timestamp_sync_us += render_time_delay_us;
						//mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "render_frame_delay: %d", render_frame_delay);
					}
				}
			}
			// detect invalid timestamp
			if(timestamp_vpacket_us > (timestamp_sync_us + DISPQT_FFMPGVIDEO_AVSYNC_MAX_VALID_DIFF_US))
			{
				mpxplay_debugf(DISPQT_DEBUGOUT_TIMED, "TimedEvent vd:%lld rtd:%lld", (timestamp_vpacket_us - cbelem->video_last_sch_timestamp), render_time_diff_ms);
				if((timestamp_vpacket_us - cbelem->video_last_sch_timestamp) < (render_time_diff_ms * 1000))
				{
					doIgnoreTimestamp = TRUE; // to start video with invalid PTS too
					mpxplay_debugf(DISPQT_DEBUGOUT_WARNING, "invalid timestamp");
				}
			}
		}

		if( doIgnoreTimestamp
		 || (!workinfos->is_paused && (render_time_diff_ms >= INFFMPEG_VIDEO_RENDER_MAX_INTERVAL_MS)) // draw min 1 fps
		 || ((timestamp_sync_us > 0) && (timestamp_vpacket_us <= timestamp_sync_us))
		 || (cbelem->video_last_sch_timestamp && (
					(timestamp_vpacket_us < cbelem->video_last_sch_timestamp)
				 || (timestamp_vpacket_us > (cbelem->video_last_sch_timestamp + DISPQT_FFMPGVIDEO_VIDEO_MAX_FRAME_DIFF_US)))
			)
		){
			do_present_frame = TRUE;
			//mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "do_present_frame ignore:%d t:%lld v:%lld l:%lld", doIgnoreTimestamp, timestamp_curr_us, timestamp_vpacket_us, cbelem->video_last_timestamp);
		}

		if( !doIgnoreTimestamp && do_present_frame && !funcbit_test(videoout_frame->flags, DISPQT_AVFRAME_FLAG_NOSKIPFRAME)
			&& (cbelem->video_average_frame_interval < DISPQT_FFMPGVIDEOCALLBACK_VIDEOFRAME_INTERVAL_33FPS) // 33fps limit to drop frames on the output TODO: should move this into process_output_drop()
		){
			doFrameDrop = mpxplay_dispqt_videoworkdec_videodelay_process_output_drop(cbelem, workinfos->main_window->gui_config, &cbelem->videodelays_scheduler, DISPQT_FFMPGVIDEOCALLBACK_VIDEOFRAME_INTERVAL_24FPS);
		}
	}

	if( !doFrameDrop 
	 && do_present_frame && (render_time_diff_ms >= INFFMPEG_VIDEO_RENDER_MIN_INTERVAL_MS)
	){
		ffmpegvideo_subtitle_info_s *subtitle = cbelem->subtitle_vframe_infos;

		if(!doIgnoreTimestamp && videoout_frame && !fvdt->is_deinterlacedup_2ndpass && videoworker_is_frame_duplication_enabled(workinfos, cbelem))
		{
			fvdt->is_deinterlacedup_2ndpass = true;
			mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "DEINT 2ND PASS enable");
		}
		if(videoout_frame && funcbit_test(fvdt->videoframe_pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_STILLPICT))  // TODO: check: still pictures always use pixel aspect ratio override
		{
			videoout_frame->sample_aspect_ratio.num = videoout_frame->sample_aspect_ratio.den = 1;
		}
		if(workinfos->video_surface_refresh_request > DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE)
		{
			refreshtype_scheduler = workinfos->video_surface_refresh_request;
		}
		workinfos->video_surface_refresh_request = DISPQT_VIDEOSURFACE_REFRESHTYPE_NONE;

		if(!videoworker_video_frame_display(workinfos, fvdt->streamtype_index, (fvdt->flags | renderflags_scheduler), videoout_frame, subtitle, refreshtype_scheduler))
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_WARNING, "videoworker_video_frame_display failed");
			return videoframe_presented;
		}

		fvdt->videoframe_pktlist_elem->frame_pkt = NULL;  // to keep AVFrame for videowidget_ffmpeg_frame_display
		funcbit_disable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_FIRSTFRAME);
		cbelem->video_last_present_time = render_begintime_ms;
		videoframe_presented = TRUE;
#ifdef MPXPLAY_USE_DEBUGF
		mpxp_uint64_t render_len = pds_gettimem() - render_begintime_ms;
		mpxp_int64_t rel_time = average_present_time - render_time_diff_ms;
		average_present_time = (average_present_time + render_time_diff_ms) / 2;
		//if((rel_time < -9) || (rel_time > 9) || (cbelem->video_delay_us >= 100000))
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_DISPLAY, "DISPLAY END rl:%3d gap:%2d spr:%2d vd:%2d pk:%d ps:%d rs:%d dr:%d sl:%d",
				(int)render_len, (int)render_time_diff_ms, (int)rel_time, (int)(cbelem->video_delay_us / 1000),
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
				cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO].videoframe_output_queue.nb_packets,
#else
				0,
#endif
				(int)fvdt->is_deinterlacedup_2ndpass, cbelem->video_render_status,
				cbelem->videodelays_scheduler.video_delay_framedrop_value, *sleep_length);
		}
#endif
	}

	if((doFrameDrop == TRUE) || videoframe_presented)
	{
		if(!fvdt->is_deinterlacedup_2ndpass)
		{
			mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_pktlist_elem);
		}
		cbelem->video_last_sch_timestamp = timestamp_vpacket_us;
		funcbit_disable(fvdt->flags, (DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME | DISPQT_FFMPGVIDEOCALLBACKFLAG_IGNORETIMESTAMP));
		if(doFrameDrop == TRUE)
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "CLEAR pktelem DROP vd:%5d pk:%d dv:%d dr:%d", (int)(cbelem->video_delay_us / 1000),
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
				cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO].videoframe_output_queue.nb_packets,
#else
				0,
#endif
				cbelem->videodelays_scheduler.video_delay_framedrop_value, cbelem->videodelays_scheduler.video_delay_framedrop_restore_retry);
		}
	}


#ifdef DISPQT_VIDEO_SCHEDULER_USE_COND_SIGNAL
	if( videoframe_presented && videoout_frame
	 && (cbelem->video_average_frame_interval >= DISPQT_FFMPGVIDEOCALLBACK_VIDEOFRAME_INTERVAL_33FPS)
	 && (videoout_frame->format == mpxplay_video_render_infos.hwdevice_avpixfmt)
	 && !mpxplay_video_render_infos.render_function_poolbufelem_validity_check(videoout_frame)
	){
		pds_threads_cond_signal(cbelem->videodec_thread_condition_signal_handler); // TODO: this doesn't help at better scheduling
	}
#endif

	return videoframe_presented;
}

#if defined(PDS_THREADS_POSIX_THREAD)
static void *videoworker_video_worker_thread(void *decoder_data)
#else
static unsigned int videoworker_video_worker_thread(void *decoder_data)
#endif
{
	struct mpxplay_dispqt_ffmpegvideo_worker_info_s *workinfos = (struct mpxplay_dispqt_ffmpegvideo_worker_info_s *)decoder_data;

	do{
		unsigned int video_index = 0, sleep_time = MPXPLAY_VIDEO_WORKER_THREAD_SLEEPTIME_DEFAULT, retry;
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		int mutex_error = -1;
#endif
		ffmpegvideo_callback_list_s *cbelem;
		mpxp_bool_t video_frame_presented = FALSE;

		if(!workinfos || !workinfos->mutexhnd_workerinfo)
			break;

		if(workinfos->video_scheduler_control != DISPQT_VIDEOSCHEDULER_CONTROL_RUN)
		{
			if(workinfos->video_scheduler_control == DISPQT_VIDEOSCHEDULER_CONTROL_REQ_CLOSE)
			{
				break;
			}
			workinfos->video_scheduler_status = DISPQT_VIDEOSCHEDULER_STATUS_SUSPENDED;
			goto do_sleep_thread;
		}

		if(workinfos->main_window->gui_config->selected_videorenderer_type != workinfos->main_window->gui_config->video_renderer_type)
			goto do_sleep_thread;

		cbelem = workinfos->cblist_first;
		if(cbelem)
			cbelem->is_videowall_enabled = workinfos->is_videowall_enabled;

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif

		while(cbelem)
		{
			ffmpegvideo_callback_command_s *ccmd = &cbelem->callback_command;
			if(ccmd->cmd)
			{
				if(cbelem->infile_callback)
					cbelem->infile_callback(cbelem->infile_passdata, ccmd->cmd, ccmd->arg1, ccmd->arg2, INFFMPG_MUTEX_COMMAND_TIMEOUT + 7);
				ccmd->cmd = 0;
			}
			cbelem = cbelem->next;
		}

		workinfos->video_scheduler_status = DISPQT_VIDEOSCHEDULER_CONTROL_RUN;

		if(!workinfos->is_videosurface_open)
		{
			goto err_out_refresh;
		}

		//mpxplay_debugf(DISPQT_DEBUGOUT_TIMED, "decoderworker_TimedEvent BEGIN");

		retry = DISPQT_VIDEOWORKER_DRAWFRAME_RETRY;
		cbelem = workinfos->cblist_first;
		while(cbelem)
		{
			int decoder_thread_num, retval_video = -1;

			if(!workinfos->main_window->qt_video_player)
				break;

			cbelem->video_wall_artype = workinfos->main_window->qt_video_player->video_get_value(DispQtVideoPlayer::VideoCtrlValue_VideoWallARtype);

			if(workinfos->is_videowall_enabled)
			{
				if(funcbit_test(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_KEYFRAMES_ONLY))
					funcbit_enable(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY);
				else
					funcbit_disable(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_VW_KEYFRAMES_ONLY);
			}

			for(decoder_thread_num = MPXPLAY_STREAMTYPEINDEX_VIDEO; decoder_thread_num < MPXPLAY_STREAMTYPEINDEX_NUM; decoder_thread_num++)
			{
				ffmpegvideo_decoder_s *fvdt = &cbelem->video_decoder_threads[decoder_thread_num];

				if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
				{
					continue;
				}

//				mpxplay_debugf(DISPQT_DEBUGOUT_TIMED, "TimedEvent BEGIN n:%d f:%8.8X l:%8.8X cb:%8.8X pd:%8.8X qn:%d", workinfos->nb_callbacks,
//					(int)workinfos->cblist_first, (int)workinfos->cblist_last, (unsigned long)workinfos->cblist_first->infile_callback,
//					(unsigned long)workinfos->cblist_first->infile_passdata, cbelem->videoframe_output_queue.nb_packets);
				if(cbelem->infile_callback && cbelem->infile_passdata)
				{
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
					bool resetvideo = (mpxplay_ffmpgdec_packetqueue_clear(&fvdt->videoframe_output_queue, MPXPLAY_PACKETLISTFLAG_CLEARBUF))? true : false;
					if(!fvdt->videoframe_pktlist_elem || resetvideo)
#else
					bool resetvideo = false;
					if(fvdt->videoframe_pktlist_elem)
#endif
					{
						int retval_nextframe = 0;
						if(funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME) || (fvdt->streamtype_index != MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW) || resetvideo)
						{
							retval_nextframe = videoworker_video_load_next_frame(cbelem, fvdt, &resetvideo);
							if((retval_nextframe == 0) && (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW) && fvdt->videoframe_pktlist_elem->frame_pkt)
							{
								mpxplay_debugf(DISPQT_DEBUGOUT_SEEKP, "TimedEvent SEEKP GOT FRAME pts:%lld res:%d nb:%d fl:%8.8X", ((AVFrame *)fvdt->videoframe_pktlist_elem->frame_pkt)->pts, resetvideo,
#ifdef DISPQT_VIDEO_SCHEDULER_USE_OUTPUT_QUEUE
										fvdt->videoframe_output_queue.nb_packets,
#else
										0,
#endif
										fvdt->flags);
							}
						}
						if(resetvideo && (cbelem == workinfos->cblist_first) && funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO))
						{
							videoworker_video_reset(workinfos, cbelem, fvdt, video_index);
							if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
								retval_video = retval_nextframe;
						}
					}
					if((workinfos->is_paused || (fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)) && !funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME))
					{
//						mpxplay_debugf(DISPQT_DEBUGOUT_TIMED, "TimedEvent PAUSED");
						continue;
					}
//					mpxplay_debugf(DISPQT_DEBUGOUT_TIMED, "infile_callback END pd:%8.8X r:%d vf:%8.8X pf:%8.8X", (mpxp_ptrsize_t)cbelem->infile_passdata, retval_video,
//							(mpxp_ptrsize_t)fvdt->videoframe_pktlist_elem, ((fvdt->videoframe_pktlist_elem)? fvdt->videoframe_pktlist_elem->flags : 0));
				}
				if((cbelem == workinfos->cblist_first) && funcbit_test(cbelem->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_PRIMARYVIDEO))
				{
					if(fvdt->videoframe_pktlist_elem)
					{
						if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO)
						{
							mpxp_int64_t timestamp_sync_us = fvdt->videoframe_pktlist_elem->timestamp_us;
							if(!funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAGS_IGNORE_TS))
								cbelem->infile_callback(cbelem->infile_passdata, MPXPLAY_INFILE_CBKCTRL_GET_TIMESYNC, (mpxp_ptrsize_t)&timestamp_sync_us, MPXPLAY_STREAMTYPEINDEX_VIDEO, INFFMPG_MUTEX_GETTIMESYNC_TIMEOUT);
							video_frame_presented |= videoworker_video_frame_scheduler(workinfos, cbelem, fvdt, timestamp_sync_us, &sleep_time);
						}
						else if(fvdt->streamtype_index == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
						{
							if(funcbit_test(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME) && fvdt->videoframe_pktlist_elem->frame_pkt)
							{
								mpxplay_debugf(DISPQT_DEBUGOUT_SEEKP, "TimedEvent SEEKP SEND FRAME pts:%lld", ((AVFrame *)fvdt->videoframe_pktlist_elem->frame_pkt)->pts);
								mpxplay_dispqt_ffmpegvideo_seekpreview_send(fvdt->videoframe_pktlist_elem->frame_pkt);
								fvdt->videoframe_pktlist_elem->frame_pkt = NULL;
								funcbit_disable(fvdt->flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_DISPLAYONEFRAME);
							}
						}
						else
						{
							if(fvdt->videoframe_pktlist_elem->frame_pkt)
								av_frame_free((AVFrame **)&fvdt->videoframe_pktlist_elem->frame_pkt);
						}
						if(fvdt->videoframe_pktlist_elem && !fvdt->videoframe_pktlist_elem->frame_pkt && !fvdt->is_deinterlacedup_2ndpass)
						{
							mpxplay_ffmpgdec_queuelistelem_clear(&fvdt->videoframe_pktlist_elem);
						}
					}
				}
				else // TODO: currently we simply drop non primary video packets (later can be video crossfade)
				{

				}
			}
#if !defined(DISPQT_VIDEO_SCHEDULER_USE_DECODER_THREAD) && !defined(DISPQT_VIDEO_SCHEDULER_USE_CONST_DECODER_THREAD)
			mpxplay_dispqt_video_decoder_worker_process(cbelem, &sleep_time);
#endif
			if(!cbelem->video_decoder_threads[MPXPLAY_STREAMTYPEINDEX_VIDEO].videoframe_pktlist_elem && (retval_video >= 0) && (--retry))
				continue;
			retry = DISPQT_VIDEOWORKER_DRAWFRAME_RETRY;
//			mpxplay_debugf(DISPQT_DEBUGOUT_TIMED, "TimedEvent END n:%d f:%8.8X l:%8.8X cb:%8.8X pd:%8.8X", workinfos->nb_callbacks,
//							(int)workinfos->cblist_first, (int)workinfos->cblist_last, (unsigned long)workinfos->cblist_first->infile_callback, 
//							(unsigned long)workinfos->cblist_first->infile_passdata);
			cbelem = cbelem->next;
			video_index++;
		}

err_out_refresh:
		if(workinfos->video_surface_refresh_request != DISPQT_VIDEOSURFACE_REFRESHTYPE_NONE)
		{
			//mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "rr:%d sp:%d", workinfos->video_surface_refresh_request, ((funcbit_test(sch_flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_STILLPICT)? 1 :0)));
			if( (workinfos->is_paused || (workinfos->video_surface_refresh_request > DISPQT_VIDEOSURFACE_REFRESHTYPE_UPDATE) || funcbit_test(workinfos->video_surface_infos->scheduler_flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_STILLPICT))
			 && (videoworker_video_reset_display(workinfos, NULL, 0, workinfos->video_surface_refresh_request) == TRUE)) // initial clear delay
			{
				workinfos->video_surface_refresh_request = DISPQT_VIDEOSURFACE_REFRESHTYPE_NONE;
			}
		}
do_sleep_thread:
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
		if(mutex_error == MPXPLAY_ERROR_OK)
			PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
		if(sleep_time > 0)
			pds_threads_sleep(sleep_time);
	}while(workinfos->video_scheduler_control != DISPQT_VIDEOSCHEDULER_CONTROL_REQ_CLOSE);
	workinfos->video_scheduler_status = DISPQT_VIDEOSCHEDULER_STATUS_CLOSED;
	pds_threads_sleep(DISPQT_VIDEO_MUTEX_TIMEOUT);  // required for pds_threads_thread_close -> TerminateThread
#ifndef PDS_THREADS_POSIX_THREAD
	return 0;
#endif
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
