//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: Vulkan video render output

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT NULL //stdout
#define DISPQT_DEBUG_ERROR stderr

#include "moc_config.h"

#if MPXPLAY_DISPQT_ENABLE_RENDER_VULKAN

#include "moc_video_qt.h"
#include "moc_mainwindow.h"

#define VK_USE_PLATFORM_WIN32_KHR 1
#include <vulkan/vulkan.h>

#define DISPQT_VULKAN_TRANSPARENT_BASE 0 // transparent or black area for non-video part of the window

/* Amount of time, in nanoseconds, to wait for a command buffer to complete */
#define DISPQT_VULKAN_FENCE_TIMEOUT 100000000
#define DISPQT_VULKAN_GETBUFFER_TIMEOUT (DISPQT_VULKAN_FENCE_TIMEOUT * 10)

#define DISPQT_VULKAN_FUNCPTR_DECLARE(FUNCNAME) PFN_##FUNCNAME FUNCNAME##_func
#define DISPQT_VULKAN_FUNCPTR_LOAD(FUNCNAME) { \
	vlk_api->FUNCNAME##_func = (PFN_##FUNCNAME)GetProcAddress(vlk_api->vulkan_dll_hnd, #FUNCNAME); \
	if(!vlk_api->FUNCNAME##_func) \
		return false; \
}
#define DISPQT_VULKAN_FUNCPTR_CALL(FUNCNAME) vulkan_data->vlk_api.FUNCNAME##_func

typedef struct dispqt_vulkan_api_s{
	HMODULE      vulkan_dll_hnd;
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateInstance);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkEnumeratePhysicalDevices);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetPhysicalDeviceQueueFamilyProperties);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetPhysicalDeviceMemoryProperties);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateWin32SurfaceKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetPhysicalDeviceSurfaceSupportKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateDevice);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetDeviceQueue);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateCommandPool);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkAllocateCommandBuffers);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateSemaphore);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateFence);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkBeginCommandBuffer);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetPhysicalDeviceFormatProperties);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCmdPipelineBarrier);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetPhysicalDeviceSurfaceFormatsKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetPhysicalDeviceSurfaceCapabilitiesKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateSwapchainKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetSwapchainImagesKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkEndCommandBuffer);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateImageView);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCreateImage);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetImageMemoryRequirements);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkAllocateMemory);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkBindImageMemory);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkQueueSubmit);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkWaitForFences);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkGetImageSubresourceLayout);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkMapMemory);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkResetCommandBuffer);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkUnmapMemory);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkFreeMemory);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroyImage);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkQueueWaitIdle);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDeviceWaitIdle);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroyImageView);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroySwapchainKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkFreeCommandBuffers);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroyCommandPool);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroySemaphore);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroyFence);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroyDevice);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroySurfaceKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkDestroyInstance);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkFlushMappedMemoryRanges);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkAcquireNextImageKHR);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCmdClearColorImage);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkCmdBlitImage);
	DISPQT_VULKAN_FUNCPTR_DECLARE(vkQueuePresentKHR);
}dispqt_vulkan_api_s;

typedef struct vulkan_swap_chain_buffers {
    VkImage image;
    VkImageView view;
    bool clear_picture_flag;
} vulkan_swap_chain_buffers;

typedef struct dispqt_render_vulkan_data_s{
	HWND parent_window_handler;
	VkInstance vkinst;
	VkPhysicalDevice gpu_device;
	uint32_t queue_family_count;
	VkQueueFamilyProperties *queue_props;
	VkQueue graphics_queue, present_queue;
	VkPhysicalDeviceMemoryProperties memory_properties;
	VkSurfaceKHR surface;
	uint32_t graphics_queue_family_index, present_queue_family_index;
	VkDevice vulkan_device;
	VkCommandPool cmd_pool;
	VkCommandBuffer cmd_buffer;
	VkSemaphore imageAcquiredSemaphore;
	VkFence queue_fence;
	VkFormat surface_vk_format;
	enum AVPixelFormat surface_fmt_format;
	uint32_t swapchain_width, swapchain_height;
	VkSwapchainKHR swap_chain;
	uint32_t swapchainImageCount, swapchan_buf_idx;
	vulkan_swap_chain_buffers *swap_buffers;
	VkImage src_image_handler;
	VkDeviceMemory src_image_mem_handler;
	void *src_image_mem_ptr;
	VkDeviceSize src_image_mem_size;
	uint32_t src_image_width, src_image_height, src_image_rowpitch;
	dispqt_vulkan_api_s vlk_api;
}dispqt_render_vulkan_data_s;

static const VkImageSubresourceRange vulkan_clearRange = { .aspectMask = VK_IMAGE_ASPECT_COLOR_BIT, .baseMipLevel = 0, .levelCount = VK_REMAINING_MIP_LEVELS, .baseArrayLayer = 0, .layerCount = VK_REMAINING_ARRAY_LAYERS};
#if DISPQT_VULKAN_TRANSPARENT_BASE
static const VkClearColorValue vulkan_clearColor = {0};
#else
static const VkClearColorValue vulkan_clearColor = {0UL,0UL,0UL,255UL};
#endif

//-----------------------------------------------------------------------------------------------------------

static bool dispqt_vulkan_library_load(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	struct dispqt_vulkan_api_s *vlk_api = &vulkan_data->vlk_api;

	vlk_api->vulkan_dll_hnd = LoadLibrary("vulkan-1.dll");
	if(!vlk_api->vulkan_dll_hnd)
		return false;

	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateInstance);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateInstance);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkEnumeratePhysicalDevices);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetPhysicalDeviceQueueFamilyProperties);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetPhysicalDeviceMemoryProperties);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateWin32SurfaceKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetPhysicalDeviceSurfaceSupportKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateDevice);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetDeviceQueue);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateCommandPool);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkAllocateCommandBuffers);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateSemaphore);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateFence);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkBeginCommandBuffer);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetPhysicalDeviceFormatProperties);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCmdPipelineBarrier);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetPhysicalDeviceSurfaceFormatsKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetPhysicalDeviceSurfaceCapabilitiesKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateSwapchainKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetSwapchainImagesKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkEndCommandBuffer);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateImageView);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCreateImage);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetImageMemoryRequirements);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkAllocateMemory);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkBindImageMemory);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkQueueSubmit);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkWaitForFences);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkGetImageSubresourceLayout);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkMapMemory);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkResetCommandBuffer);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkUnmapMemory);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkFreeMemory);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroyImage);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkQueueWaitIdle);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDeviceWaitIdle);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroyImageView);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroySwapchainKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkFreeCommandBuffers);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroyCommandPool);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroySemaphore);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroyFence);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroyDevice);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroySurfaceKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkDestroyInstance);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkFlushMappedMemoryRanges);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkAcquireNextImageKHR);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCmdClearColorImage);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkCmdBlitImage);
	DISPQT_VULKAN_FUNCPTR_LOAD(vkQueuePresentKHR);

	return true;
}

static bool dispqt_vulkan_init_create_instance(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	const char *required_instance_ext_names [2] = {VK_KHR_SURFACE_EXTENSION_NAME, VK_KHR_WIN32_SURFACE_EXTENSION_NAME};
	const struct VkApplicationInfo app_info = {
		.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO,
		.pNext = NULL,
		.pApplicationName = "MMC",
		.applicationVersion = 310,
		.pEngineName = "MMC",
		.engineVersion = 1,
		.apiVersion = VK_API_VERSION_1_0};

	const struct VkInstanceCreateInfo inst_info = {
		.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO,
		.pNext = NULL,
		.flags = 0,
		.pApplicationInfo = &app_info,
		.enabledLayerCount = 0,
		.ppEnabledLayerNames = NULL,
		.enabledExtensionCount = 2,
		.ppEnabledExtensionNames = required_instance_ext_names};

	if (DISPQT_VULKAN_FUNCPTR_CALL(vkCreateInstance)(&inst_info, NULL, &vulkan_data->vkinst) != VK_SUCCESS)
		return false;

	return true;
}

static bool dispqt_vulkan_init_enumerate_device(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	VkPhysicalDevice *physical_devices = NULL;
	uint32_t gpu_count = 0;
	bool retval = false;
	VkResult result;

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkEnumeratePhysicalDevices)(vulkan_data->vkinst, &gpu_count, NULL);
	if ((result != VK_SUCCESS) || !gpu_count)
		goto err_out_init;

	physical_devices = (VkPhysicalDevice *)malloc(sizeof(VkPhysicalDevice) * gpu_count);
	if(!physical_devices)
		goto err_out_init;

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkEnumeratePhysicalDevices)(vulkan_data->vkinst, &gpu_count, physical_devices);
	if ((result != VK_SUCCESS) || !gpu_count)
		goto err_out_init;

	vulkan_data->gpu_device = physical_devices[0];

	DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceQueueFamilyProperties)(vulkan_data->gpu_device, &vulkan_data->queue_family_count, NULL);
	if(!vulkan_data->queue_family_count)
		goto err_out_init;

	vulkan_data->queue_props = (VkQueueFamilyProperties *)malloc(vulkan_data->queue_family_count * sizeof(VkQueueFamilyProperties));
	if(!vulkan_data->queue_props)
		goto err_out_init;

	DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceQueueFamilyProperties)(vulkan_data->gpu_device, &vulkan_data->queue_family_count, vulkan_data->queue_props);
	if(!vulkan_data->queue_family_count)
		goto err_out_init;

	DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceMemoryProperties)(vulkan_data->gpu_device, &vulkan_data->memory_properties);

	retval = true;

err_out_init:
	if(physical_devices)
		free(physical_devices);
	return retval;
}

static bool dispqt_vulkan_init_create_surface(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	const HWND hwnd = (HWND)vulkan_data->parent_window_handler;
	const HINSTANCE hInst = GetModuleHandle(NULL);
	VkWin32SurfaceCreateInfoKHR surfCrInfo;

	surfCrInfo.sType = VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR;
	surfCrInfo.pNext = NULL;
	surfCrInfo.flags = 0;
	surfCrInfo.hinstance = hInst;
	surfCrInfo.hwnd = hwnd;

	if (DISPQT_VULKAN_FUNCPTR_CALL(vkCreateWin32SurfaceKHR)(vulkan_data->vkinst, &surfCrInfo, NULL, &vulkan_data->surface) != VK_SUCCESS)
		return false;

	return true;
}

static bool dispqt_vulkan_init_get_presenter_queue(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	VkBool32 *pSupportsPresent = (VkBool32 *)malloc(vulkan_data->queue_family_count * sizeof(VkBool32));

	if(!pSupportsPresent)
		return false;

	for (uint32_t i = 0; i < vulkan_data->queue_family_count; i++)
		DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceSurfaceSupportKHR)(vulkan_data->gpu_device, i, vulkan_data->surface, &pSupportsPresent[i]);

	vulkan_data->graphics_queue_family_index = UINT32_MAX;
	vulkan_data->present_queue_family_index = UINT32_MAX;

	for (uint32_t i = 0; i < vulkan_data->queue_family_count; ++i) {
		if ((vulkan_data->queue_props[i].queueFlags & VK_QUEUE_GRAPHICS_BIT) != 0) {
			if (vulkan_data->graphics_queue_family_index == UINT32_MAX)
				vulkan_data->graphics_queue_family_index = i;

			if (pSupportsPresent[i] == VK_TRUE) {
				vulkan_data->graphics_queue_family_index = i;
				vulkan_data->present_queue_family_index = i;
				break;
			}
		}
	}

	if (vulkan_data->present_queue_family_index == UINT32_MAX) {
		for (size_t i = 0; i < vulkan_data->queue_family_count; ++i) {
			if (pSupportsPresent[i] == VK_TRUE) {
				vulkan_data->present_queue_family_index = i;
				break;
			}
		}
	}
	free(pSupportsPresent);

	if (vulkan_data->graphics_queue_family_index == UINT32_MAX || vulkan_data->present_queue_family_index == UINT32_MAX)
		return false;

	return true;
}

static bool dispqt_vulkan_init_create_device(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	const char *required_device_ext_names[1] = {VK_KHR_SWAPCHAIN_EXTENSION_NAME};
	const float queue_priorities[1] = {0.0};
	VkDeviceQueueCreateInfo queue_info = {};
	VkDeviceCreateInfo device_info = {};
	VkResult result;

	queue_info.sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
	queue_info.pNext = NULL;
	queue_info.flags = 0;
	queue_info.queueFamilyIndex = vulkan_data->graphics_queue_family_index;
	queue_info.queueCount = 1;
	queue_info.pQueuePriorities = &queue_priorities[0];

	device_info.sType = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO;
	device_info.pNext = NULL;
	device_info.flags = 0;
	device_info.queueCreateInfoCount = 1;
	device_info.pQueueCreateInfos = &queue_info;
	device_info.enabledLayerCount = 0;
	device_info.ppEnabledLayerNames = NULL;
	device_info.enabledExtensionCount = 1;
	device_info.ppEnabledExtensionNames = required_device_ext_names;
	device_info.pEnabledFeatures = NULL;

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkCreateDevice)(vulkan_data->gpu_device, &device_info, NULL, &vulkan_data->vulkan_device);
	if(result != VK_SUCCESS)
		return false;

	DISPQT_VULKAN_FUNCPTR_CALL(vkGetDeviceQueue)(vulkan_data->vulkan_device, vulkan_data->graphics_queue_family_index, 0, &vulkan_data->graphics_queue);
	if (vulkan_data->graphics_queue_family_index == vulkan_data->present_queue_family_index) {
		vulkan_data->present_queue = vulkan_data->graphics_queue;
	} else {
		DISPQT_VULKAN_FUNCPTR_CALL(vkGetDeviceQueue)(vulkan_data->vulkan_device, vulkan_data->present_queue_family_index, 0, &vulkan_data->present_queue);
	}

	return true;
}

static bool dispqt_vulkan_init_create_command_buffer(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	VkCommandPoolCreateInfo cmd_pool_info = {};
	cmd_pool_info.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
	cmd_pool_info.pNext = NULL;
	cmd_pool_info.queueFamilyIndex = vulkan_data->graphics_queue_family_index;
	cmd_pool_info.flags = VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkCreateCommandPool)(vulkan_data->vulkan_device, &cmd_pool_info, NULL, &vulkan_data->cmd_pool) != VK_SUCCESS)
		return false;

	VkCommandBufferAllocateInfo cmd = {};
	cmd.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
	cmd.pNext = NULL;
	cmd.commandPool = vulkan_data->cmd_pool;
	cmd.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
	cmd.commandBufferCount = 1;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkAllocateCommandBuffers)(vulkan_data->vulkan_device, &cmd, &vulkan_data->cmd_buffer) != VK_SUCCESS)
		return false;

	VkSemaphoreCreateInfo imageAcquiredSemaphoreCreateInfo = {};
	imageAcquiredSemaphoreCreateInfo.sType = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO;
	imageAcquiredSemaphoreCreateInfo.pNext = NULL;
	imageAcquiredSemaphoreCreateInfo.flags = 0;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkCreateSemaphore)(vulkan_data->vulkan_device, &imageAcquiredSemaphoreCreateInfo, NULL, &vulkan_data->imageAcquiredSemaphore) != VK_SUCCESS)
		return false;

	VkFenceCreateInfo fenceInfo = {};
	fenceInfo.sType = VK_STRUCTURE_TYPE_FENCE_CREATE_INFO;
	fenceInfo.pNext = NULL;
	fenceInfo.flags = 0;
	if(DISPQT_VULKAN_FUNCPTR_CALL(vkCreateFence)(vulkan_data->vulkan_device, &fenceInfo, NULL, &vulkan_data->queue_fence)!= VK_SUCCESS)
		return false;

	return true;
}

static bool dispqt_vulkan_execute_begin_command_buffer(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	VkCommandBufferBeginInfo cmd_buf_info = {};
	cmd_buf_info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
	cmd_buf_info.pNext = NULL;
	cmd_buf_info.flags = 0;
	cmd_buf_info.pInheritanceInfo = NULL;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkBeginCommandBuffer)(vulkan_data->cmd_buffer, &cmd_buf_info) != VK_SUCCESS)
		return false;

	return true;
}

//-----------------------------------------------------------------------------------------------------------

static enum AVPixelFormat vulkan_surfaceformat_to_ffmpegfmt(VkFormat surface_vk_format)
{
	enum AVPixelFormat ff_fmt = AV_PIX_FMT_NONE;
	switch(surface_vk_format)
	{
		case VK_FORMAT_R8G8B8_UNORM: ff_fmt = AV_PIX_FMT_RGB24; break;
		case VK_FORMAT_B8G8R8_UNORM: ff_fmt = AV_PIX_FMT_BGR24; break;
		case VK_FORMAT_R8G8B8A8_UNORM: ff_fmt = AV_PIX_FMT_RGBA; break;
		case VK_FORMAT_B8G8R8A8_UNORM: ff_fmt = AV_PIX_FMT_BGRA; break;
	}
	return ff_fmt;
}

static bool dispqt_vulkan_select_surface_format(struct dispqt_render_vulkan_data_s *vulkan_data, VkSurfaceFormatKHR *surfFormats, uint32_t formatCount)
{
	bool formatIsSelected = false;
	VkFormatProperties formatProps;

	if((formatCount == 1) && (surfFormats[0].format == VK_FORMAT_UNDEFINED))
	{
		vulkan_data->surface_vk_format = VK_FORMAT_B8G8R8A8_UNORM;
		vulkan_data->surface_fmt_format = AV_PIX_FMT_BGRA;
		formatIsSelected = true;
	}
	else
	{
		for(int i = 0; i < formatCount; i++)
		{
			VkFormat vk_sff = surfFormats[i].format;
			if(vk_sff == VK_FORMAT_B8G8R8A8_UNORM) // this is tested, surely works (and 32-bit RGB format is preferred, not 24)
			{
				pds_memset(&formatProps, 0, sizeof(formatProps));
				DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceFormatProperties)(vulkan_data->gpu_device, vk_sff, &formatProps);
				if(formatProps.linearTilingFeatures & VK_FORMAT_FEATURE_BLIT_SRC_BIT) // format can be used as transfer source
				{
					vulkan_data->surface_vk_format = vk_sff;
					vulkan_data->surface_fmt_format = AV_PIX_FMT_BGRA;
					formatIsSelected = true;
				}
				break;
			}
		}
		if(!formatIsSelected)
		{
			for(int i = 0; i < formatCount; i++) // search for an other supported RGB format
			{
				VkFormat vk_sff = surfFormats[i].format;
				pds_memset(&formatProps, 0, sizeof(formatProps));
				DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceFormatProperties)(vulkan_data->gpu_device, vk_sff, &formatProps);
				if(formatProps.linearTilingFeatures & VK_FORMAT_FEATURE_BLIT_SRC_BIT) // format can be used as transfer source
				{
					enum AVPixelFormat ff_fmt = vulkan_surfaceformat_to_ffmpegfmt(vk_sff);
					if(ff_fmt != AV_PIX_FMT_NONE)
					{
						vulkan_data->surface_vk_format = vk_sff;
						vulkan_data->surface_fmt_format = ff_fmt;
						formatIsSelected = true;
						break;
					}
				}
			}
		}
	}
	return formatIsSelected;
}

static bool dispqt_vulkan_set_image_layout(struct dispqt_render_vulkan_data_s *vulkan_data, VkImage image, VkImageAspectFlags aspectMask, VkImageLayout old_image_layout,
                      VkImageLayout new_image_layout, VkPipelineStageFlags src_stages, VkPipelineStageFlags dest_stages)
{
	if(!vulkan_data->cmd_buffer)
		return false;
	if(!vulkan_data->graphics_queue)
		return false;

	VkImageMemoryBarrier image_memory_barrier = {};
	image_memory_barrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
	image_memory_barrier.pNext = NULL;
	image_memory_barrier.srcAccessMask = 0;
	image_memory_barrier.dstAccessMask = 0;
	image_memory_barrier.oldLayout = old_image_layout;
	image_memory_barrier.newLayout = new_image_layout;
	image_memory_barrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
	image_memory_barrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
	image_memory_barrier.image = image;
	image_memory_barrier.subresourceRange.aspectMask = aspectMask;
	image_memory_barrier.subresourceRange.baseMipLevel = 0;
	image_memory_barrier.subresourceRange.levelCount = 1;
	image_memory_barrier.subresourceRange.baseArrayLayer = 0;
	image_memory_barrier.subresourceRange.layerCount = 1;

	switch (old_image_layout) {
		case VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL:
			image_memory_barrier.srcAccessMask = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
			break;
		case VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL:
			image_memory_barrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
			break;
		case VK_IMAGE_LAYOUT_PREINITIALIZED:
			image_memory_barrier.srcAccessMask = VK_ACCESS_HOST_WRITE_BIT;
			break;
		default:
			break;
	}

	switch (new_image_layout) {
		case VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL:
			image_memory_barrier.dstAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
			break;
		case VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL:
			image_memory_barrier.dstAccessMask = VK_ACCESS_TRANSFER_READ_BIT;
			break;
		case VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL:
			image_memory_barrier.dstAccessMask = VK_ACCESS_SHADER_READ_BIT;
			break;
		case VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL:
			image_memory_barrier.dstAccessMask = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
			break;
		case VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL:
			image_memory_barrier.dstAccessMask = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT;
			break;
		default:
			break;
	}

	DISPQT_VULKAN_FUNCPTR_CALL(vkCmdPipelineBarrier)(vulkan_data->cmd_buffer, src_stages, dest_stages, 0, 0, NULL, 0, NULL, 1, &image_memory_barrier);

	return true;
}

static bool dispqt_vulkan_init_prepare_surface(struct dispqt_render_vulkan_data_s *vulkan_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	const VkCompositeAlphaFlagBitsKHR compositeAlphaFlags[4] = {
			VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR,
			VK_COMPOSITE_ALPHA_PRE_MULTIPLIED_BIT_KHR,
			VK_COMPOSITE_ALPHA_POST_MULTIPLIED_BIT_KHR,
			VK_COMPOSITE_ALPHA_INHERIT_BIT_KHR,
	};
	VkCompositeAlphaFlagBitsKHR compositeAlpha = VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR;
	VkPresentModeKHR swapchainPresentMode = VK_PRESENT_MODE_FIFO_KHR;
	VkSurfaceTransformFlagBitsKHR preTransform;
	VkSwapchainCreateInfoKHR swapchain_ci = {};
	VkSurfaceCapabilitiesKHR surfCapabilities = {};
	VkSurfaceFormatKHR *surfFormats = NULL;
	VkExtent2D swapchainExtent;
	VkImage *swapchainImages = NULL;
	uint32_t formatCount, desiredNumberOfSwapChainImages, queueFamilyIndices[2];
	VkFormatProperties formatProps = {};
	bool retval = false;
	VkResult result;

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceSurfaceFormatsKHR)(vulkan_data->gpu_device, vulkan_data->surface, &formatCount, NULL);
	if(result != VK_SUCCESS)
		return false;

	surfFormats = (VkSurfaceFormatKHR *)malloc(formatCount * sizeof(VkSurfaceFormatKHR));
	result = DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceSurfaceFormatsKHR)(vulkan_data->gpu_device, vulkan_data->surface, &formatCount, surfFormats);
	if(result != VK_SUCCESS)
		goto err_out_init;

	if(!dispqt_vulkan_select_surface_format(vulkan_data, surfFormats, formatCount))
		goto err_out_init;

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkGetPhysicalDeviceSurfaceCapabilitiesKHR)(vulkan_data->gpu_device, vulkan_data->surface, &surfCapabilities);
	if(result != VK_SUCCESS)
	   goto err_out_init;

	if (surfCapabilities.currentExtent.width == 0xFFFFFFFF) {
		swapchainExtent.width = video_surface_infos->window_size_x;
		swapchainExtent.height = video_surface_infos->window_size_y;
		if (swapchainExtent.width < surfCapabilities.minImageExtent.width) {
			swapchainExtent.width = surfCapabilities.minImageExtent.width;
		} else if (swapchainExtent.width > surfCapabilities.maxImageExtent.width) {
			swapchainExtent.width = surfCapabilities.maxImageExtent.width;
		}

		if (swapchainExtent.height < surfCapabilities.minImageExtent.height) {
			swapchainExtent.height = surfCapabilities.minImageExtent.height;
		} else if (swapchainExtent.height > surfCapabilities.maxImageExtent.height) {
			swapchainExtent.height = surfCapabilities.maxImageExtent.height;
		}
	} else {
		// If the surface size is defined, the swap chain size must match
		swapchainExtent = surfCapabilities.currentExtent;
	}


	desiredNumberOfSwapChainImages = surfCapabilities.minImageCount;

	if (surfCapabilities.supportedTransforms & VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR) {
		preTransform = VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR;
	} else {
		preTransform = surfCapabilities.currentTransform;
	}

	for (uint32_t i = 0; i < sizeof(compositeAlphaFlags) / sizeof(compositeAlphaFlags[0]); i++) {
		if (surfCapabilities.supportedCompositeAlpha & compositeAlphaFlags[i]) {
			compositeAlpha = compositeAlphaFlags[i];
			break;
		}
	}

	swapchain_ci.sType = VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR;
	swapchain_ci.pNext = NULL;
	swapchain_ci.surface = vulkan_data->surface;
	swapchain_ci.minImageCount = desiredNumberOfSwapChainImages;
	swapchain_ci.imageFormat = vulkan_data->surface_vk_format;
	swapchain_ci.imageExtent.width = swapchainExtent.width;
	swapchain_ci.imageExtent.height = swapchainExtent.height;
	swapchain_ci.preTransform = preTransform;
	swapchain_ci.compositeAlpha = compositeAlpha;
	swapchain_ci.imageArrayLayers = 1;
	swapchain_ci.presentMode = swapchainPresentMode;
	swapchain_ci.oldSwapchain = VK_NULL_HANDLE;
	swapchain_ci.clipped = true;
	swapchain_ci.imageColorSpace = VK_COLORSPACE_SRGB_NONLINEAR_KHR;
	swapchain_ci.imageUsage = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT | VK_IMAGE_USAGE_TRANSFER_DST_BIT;
	swapchain_ci.imageSharingMode = VK_SHARING_MODE_EXCLUSIVE;
	swapchain_ci.queueFamilyIndexCount = 0;
	swapchain_ci.pQueueFamilyIndices = NULL;
	queueFamilyIndices[0] = vulkan_data->graphics_queue_family_index;
	queueFamilyIndices[1] = vulkan_data->present_queue_family_index;
	if (vulkan_data->graphics_queue_family_index != vulkan_data->present_queue_family_index) {
		swapchain_ci.imageSharingMode = VK_SHARING_MODE_CONCURRENT;
		swapchain_ci.queueFamilyIndexCount = 2;
		swapchain_ci.pQueueFamilyIndices = queueFamilyIndices;
	}

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkCreateSwapchainKHR)(vulkan_data->vulkan_device, &swapchain_ci, NULL, &vulkan_data->swap_chain);
	if(result != VK_SUCCESS)
		goto err_out_init;

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkGetSwapchainImagesKHR)(vulkan_data->vulkan_device, vulkan_data->swap_chain, &vulkan_data->swapchainImageCount, NULL);
	if(result != VK_SUCCESS)
		goto err_out_init;

	swapchainImages = (VkImage *)malloc(vulkan_data->swapchainImageCount * sizeof(VkImage));
	if(!swapchainImages)
		goto err_out_init;

	result = DISPQT_VULKAN_FUNCPTR_CALL(vkGetSwapchainImagesKHR)(vulkan_data->vulkan_device, vulkan_data->swap_chain, &vulkan_data->swapchainImageCount, swapchainImages);
	if(result != VK_SUCCESS)
		goto err_out_init;

	vulkan_data->swap_buffers = (vulkan_swap_chain_buffers *)malloc(vulkan_data->swapchainImageCount * sizeof(vulkan_swap_chain_buffers));
	if(!vulkan_data->swap_buffers)
		goto err_out_init;

	dispqt_vulkan_execute_begin_command_buffer(vulkan_data);

	for (uint32_t i = 0; i < vulkan_data->swapchainImageCount; i++) {
		vulkan_data->swap_buffers[i].image = swapchainImages[i];
		if(!dispqt_vulkan_set_image_layout(vulkan_data, vulkan_data->swap_buffers[i].image, VK_IMAGE_ASPECT_COLOR_BIT, VK_IMAGE_LAYOUT_UNDEFINED,
										 VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT, VK_PIPELINE_STAGE_TRANSFER_BIT))
			goto err_out_init;
	}

	DISPQT_VULKAN_FUNCPTR_CALL(vkEndCommandBuffer)(vulkan_data->cmd_buffer);

	for (uint32_t i = 0; i < vulkan_data->swapchainImageCount; i++) {
		VkImageViewCreateInfo color_image_view = {};
		color_image_view.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
		color_image_view.pNext = NULL;
		color_image_view.flags = 0;
		color_image_view.image = vulkan_data->swap_buffers[i].image;
		color_image_view.viewType = VK_IMAGE_VIEW_TYPE_2D;
		color_image_view.format = vulkan_data->surface_vk_format;
		color_image_view.components.r = VK_COMPONENT_SWIZZLE_R;
		color_image_view.components.g = VK_COMPONENT_SWIZZLE_G;
		color_image_view.components.b = VK_COMPONENT_SWIZZLE_B;
		color_image_view.components.a = VK_COMPONENT_SWIZZLE_A;
		color_image_view.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
		color_image_view.subresourceRange.baseMipLevel = 0;
		color_image_view.subresourceRange.levelCount = 1;
		color_image_view.subresourceRange.baseArrayLayer = 0;
		color_image_view.subresourceRange.layerCount = 1;

		result = DISPQT_VULKAN_FUNCPTR_CALL(vkCreateImageView)(vulkan_data->vulkan_device, &color_image_view, NULL, &vulkan_data->swap_buffers[i].view);
		if(result != VK_SUCCESS)
			goto err_out_init;
	}

	DISPQT_VULKAN_FUNCPTR_CALL(vkEndCommandBuffer)(vulkan_data->cmd_buffer);

	vulkan_data->swapchain_width = swapchainExtent.width;
	vulkan_data->swapchain_height = swapchainExtent.height;

	retval = true;

err_out_init:
	if(surfFormats)
		free(surfFormats);
	if(swapchainImages)
		free(swapchainImages);
	return retval;
}

//-----------------------------------------------------------------------------------------------------------

static bool vulkan_memory_type_from_properties(VkPhysicalDeviceMemoryProperties *memory_properties, uint32_t typeBits, VkFlags requirements_mask, uint32_t *typeIndex)
{
	for (uint32_t i = 0; i < memory_properties->memoryTypeCount; i++) {
		if ((typeBits & 1) == 1) {
			if ((memory_properties->memoryTypes[i].propertyFlags & requirements_mask) == requirements_mask) {
				*typeIndex = i;
				return true;
			}
		}
		typeBits >>= 1;
	}
	return false;
}

static bool dispqt_vulkan_src_image_init(struct dispqt_render_vulkan_data_s *vulkan_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	VkPipelineStageFlags pipe_stage_flags = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
	VkImageCreateInfo image_info = {};
	VkMemoryRequirements memReq = {};
	VkMemoryAllocateInfo memAllocInfo = {};
	VkSubmitInfo submit_info = {};
	VkImageSubresource image_subresource = {.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT, .mipLevel = 0, .arrayLayer = 0};
	VkSubresourceLayout image_subres_layout = {};

	if(!dispqt_vulkan_execute_begin_command_buffer(vulkan_data))
		return false;

	image_info.sType = VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO;
	image_info.pNext = NULL;
	image_info.imageType = VK_IMAGE_TYPE_2D;
	image_info.format = vulkan_data->surface_vk_format; // FIXME: surface_vk_format must be constant on this way
	image_info.extent.width = video_surface_infos->ffmpeg_output_frame->width;
	image_info.extent.height = video_surface_infos->ffmpeg_output_frame->height;
	image_info.extent.depth = 1;
	image_info.mipLevels = 1;
	image_info.arrayLayers = 1;
	image_info.samples = VK_SAMPLE_COUNT_1_BIT;
	image_info.queueFamilyIndexCount = 0;
	image_info.pQueueFamilyIndices = NULL;
	image_info.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
	image_info.usage = VK_IMAGE_USAGE_TRANSFER_SRC_BIT;
	image_info.flags = 0;
	image_info.tiling = VK_IMAGE_TILING_LINEAR;
	image_info.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
	if(DISPQT_VULKAN_FUNCPTR_CALL(vkCreateImage)(vulkan_data->vulkan_device, &image_info, NULL, &vulkan_data->src_image_handler) != VK_SUCCESS)
		return false;

	dispqt_vulkan_set_image_layout(vulkan_data, vulkan_data->src_image_handler, VK_IMAGE_ASPECT_COLOR_BIT, VK_IMAGE_LAYOUT_GENERAL, VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL,
						 VK_PIPELINE_STAGE_HOST_BIT, VK_PIPELINE_STAGE_TRANSFER_BIT);

	memAllocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
	memAllocInfo.pNext = NULL;

	DISPQT_VULKAN_FUNCPTR_CALL(vkGetImageMemoryRequirements)(vulkan_data->vulkan_device, vulkan_data->src_image_handler, &memReq);

	if(!vulkan_memory_type_from_properties(&vulkan_data->memory_properties, memReq.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT, &memAllocInfo.memoryTypeIndex))
		goto err_out_srcinit;

	memAllocInfo.allocationSize = memReq.size;
	if(DISPQT_VULKAN_FUNCPTR_CALL(vkAllocateMemory)(vulkan_data->vulkan_device, &memAllocInfo, NULL, &vulkan_data->src_image_mem_handler) != VK_SUCCESS)
		goto err_out_srcinit;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkBindImageMemory)(vulkan_data->vulkan_device, vulkan_data->src_image_handler, vulkan_data->src_image_mem_handler, 0) != VK_SUCCESS)
		goto err_out_srcinit;

	dispqt_vulkan_set_image_layout(vulkan_data, vulkan_data->src_image_handler, VK_IMAGE_ASPECT_COLOR_BIT, VK_IMAGE_LAYOUT_UNDEFINED, VK_IMAGE_LAYOUT_GENERAL,
					 VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT, VK_PIPELINE_STAGE_HOST_BIT);

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkEndCommandBuffer)(vulkan_data->cmd_buffer) != VK_SUCCESS)
		goto err_out_srcinit;

	submit_info.pNext = NULL;
	submit_info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
	submit_info.waitSemaphoreCount = 1;
	submit_info.pWaitSemaphores = &vulkan_data->imageAcquiredSemaphore;
	submit_info.pWaitDstStageMask = &pipe_stage_flags;
	submit_info.commandBufferCount = 1;
	submit_info.pCommandBuffers = &vulkan_data->cmd_buffer;
	submit_info.signalSemaphoreCount = 0;
	submit_info.pSignalSemaphores = NULL;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkQueueSubmit)(vulkan_data->graphics_queue, 1, &submit_info, vulkan_data->queue_fence) != VK_SUCCESS)
		goto err_out_srcinit;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkWaitForFences)(vulkan_data->vulkan_device, 1, &vulkan_data->queue_fence, VK_TRUE, DISPQT_VULKAN_FENCE_TIMEOUT) != VK_SUCCESS)
		goto err_out_srcinit;

	DISPQT_VULKAN_FUNCPTR_CALL(vkGetImageSubresourceLayout)(vulkan_data->vulkan_device, vulkan_data->src_image_handler, &image_subresource, &image_subres_layout);
	vulkan_data->src_image_rowpitch = image_subres_layout.rowPitch;

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkMapMemory)(vulkan_data->vulkan_device, vulkan_data->src_image_mem_handler, 0, memReq.size, 0, &vulkan_data->src_image_mem_ptr) != VK_SUCCESS)
		goto err_out_srcinit;
	if(!vulkan_data->src_image_mem_ptr)
		goto err_out_srcinit;

	vulkan_data->src_image_mem_size = memReq.size;
	vulkan_data->src_image_width = video_surface_infos->ffmpeg_output_frame->width;
	vulkan_data->src_image_height = video_surface_infos->ffmpeg_output_frame->height;

	return true;

err_out_srcinit:
	DISPQT_VULKAN_FUNCPTR_CALL(vkResetCommandBuffer)(vulkan_data->cmd_buffer, 0);
	return false;
}

static void dispqt_vulkan_src_image_deinit(struct dispqt_render_vulkan_data_s *vulkan_data, VkDevice v_device)
{
	vulkan_data->src_image_mem_ptr = NULL;
	vulkan_data->src_image_mem_size = 0;
	if(v_device)
	{
		if(vulkan_data->src_image_mem_handler)
		{
			VkDeviceMemory srcMem = vulkan_data->src_image_mem_handler;
			vulkan_data->src_image_mem_handler = (VkDeviceMemory)NULL;
			DISPQT_VULKAN_FUNCPTR_CALL(vkUnmapMemory)(v_device, srcMem);
			DISPQT_VULKAN_FUNCPTR_CALL(vkFreeMemory)(v_device, srcMem, NULL);
		}
		if(vulkan_data->src_image_handler)
		{
			VkImage srcImage = vulkan_data->src_image_handler;
			vulkan_data->src_image_handler = (VkImage)NULL;
			DISPQT_VULKAN_FUNCPTR_CALL(vkDestroyImage)(v_device, srcImage, NULL);
		}
	}
	vulkan_data->src_image_width = vulkan_data->src_image_height = 0;
}

static void dispqt_vulkan_graphics_queue_submit(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	VkSubmitInfo submit_info = {};
	submit_info.pNext = NULL;
	submit_info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
	submit_info.waitSemaphoreCount = 0;
	submit_info.pWaitSemaphores = NULL;
	submit_info.pWaitDstStageMask = NULL;
	submit_info.commandBufferCount = 1;
	submit_info.pCommandBuffers = &vulkan_data->cmd_buffer;
	submit_info.signalSemaphoreCount = 0;
	submit_info.pSignalSemaphores = NULL;

	DISPQT_VULKAN_FUNCPTR_CALL(vkQueueSubmit)(vulkan_data->graphics_queue, 1, &submit_info, vulkan_data->queue_fence);

	DISPQT_VULKAN_FUNCPTR_CALL(vkQueueWaitIdle)(vulkan_data->graphics_queue);
}

//-----------------------------------------------------------------------------------------------------------

static void dispqt_vulkan_videorenderer_deinit_swapchain(struct dispqt_render_vulkan_data_s *vulkan_data, VkDevice v_device)
{
	if(v_device)
	{
		DISPQT_VULKAN_FUNCPTR_CALL(vkDeviceWaitIdle)(v_device);
		if(vulkan_data->swap_buffers)
		{
			vulkan_swap_chain_buffers *sw_buf = vulkan_data->swap_buffers;
			vulkan_data->swap_buffers = NULL;
			for (uint32_t i = 0; i < vulkan_data->swapchainImageCount; i++)
				DISPQT_VULKAN_FUNCPTR_CALL(vkDestroyImageView)(v_device, sw_buf[i].view, NULL);
			free(sw_buf);
		}
		if(vulkan_data->swap_chain)
		{
			VkSwapchainKHR swc = vulkan_data->swap_chain;
			vulkan_data->swap_chain = (VkSwapchainKHR)NULL;
			DISPQT_VULKAN_FUNCPTR_CALL(vkDestroySwapchainKHR)(v_device, swc, NULL);
		}
	}
	vulkan_data->swapchan_buf_idx = vulkan_data->swapchainImageCount = 0;
	vulkan_data->swapchain_width = vulkan_data->swapchain_height = 0;
}

static void dispqt_vulkan_deinit_vulkan(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	if(vulkan_data->vulkan_device)
	{
		VkDevice v_device = vulkan_data->vulkan_device;
		vulkan_data->vulkan_device = NULL;
		dispqt_vulkan_videorenderer_deinit_swapchain(vulkan_data, v_device);
		dispqt_vulkan_src_image_deinit(vulkan_data, v_device);
		if(vulkan_data->cmd_pool)
		{
			VkCommandPool c_pool = vulkan_data->cmd_pool;
			vulkan_data->cmd_pool = (VkCommandPool)NULL;
			if(vulkan_data->cmd_buffer)
			{
				VkCommandBuffer cmd_bufs[1] = {vulkan_data->cmd_buffer};
				vulkan_data->cmd_buffer = NULL;
				DISPQT_VULKAN_FUNCPTR_CALL(vkFreeCommandBuffers)(v_device, c_pool, 1, cmd_bufs);
			}
			DISPQT_VULKAN_FUNCPTR_CALL(vkDestroyCommandPool)(v_device, c_pool, NULL);
		}
		if(vulkan_data->imageAcquiredSemaphore)
		{
			VkSemaphore sem = vulkan_data->imageAcquiredSemaphore;
			vulkan_data->imageAcquiredSemaphore = (VkSemaphore)NULL;
			DISPQT_VULKAN_FUNCPTR_CALL(vkDestroySemaphore)(v_device, sem, NULL);
		}
		if(vulkan_data->queue_fence)
		{
			VkFence fence = vulkan_data->queue_fence;
			vulkan_data->queue_fence = (VkFence)NULL;
			DISPQT_VULKAN_FUNCPTR_CALL(vkDestroyFence)(v_device, fence, NULL);
		}
		DISPQT_VULKAN_FUNCPTR_CALL(vkDestroyDevice)(v_device, NULL);
	}
	if(vulkan_data->surface)
	{
		VkSurfaceKHR surf = vulkan_data->surface;
		vulkan_data->surface = (VkSurfaceKHR)NULL;
		DISPQT_VULKAN_FUNCPTR_CALL(vkDestroySurfaceKHR)(vulkan_data->vkinst, surf, NULL);
	}
	if(vulkan_data->queue_props)
	{
		VkQueueFamilyProperties *props = vulkan_data->queue_props;
		vulkan_data->queue_props = NULL;
		free(props);
	}
	if(vulkan_data->vkinst)
	{
		VkInstance inst = vulkan_data->vkinst;
		vulkan_data->vkinst = NULL;
		DISPQT_VULKAN_FUNCPTR_CALL(vkDestroyInstance)(inst, NULL);
	}
	if(vulkan_data->vlk_api.vulkan_dll_hnd)
		FreeLibrary(vulkan_data->vlk_api.vulkan_dll_hnd);
	pds_memset(vulkan_data, 0, sizeof(*vulkan_data));
}

static void mpxplay_dispqt_vulkan_videoframe_present(struct dispqt_render_vulkan_data_s *vulkan_data)
{
	VkPresentInfoKHR present;

	pds_memset(&present, 0, sizeof(present));
	present.sType = VK_STRUCTURE_TYPE_PRESENT_INFO_KHR;
	present.pNext = NULL;
	present.swapchainCount = 1;
	present.pSwapchains = &vulkan_data->swap_chain;
	present.pImageIndices = &vulkan_data->swapchan_buf_idx;
	present.pWaitSemaphores = NULL;
	present.waitSemaphoreCount = 0;
	present.pResults = NULL;

	if(vulkan_data->vulkan_device && vulkan_data->vlk_api.vkWaitForFences_func)
	{
		DISPQT_VULKAN_FUNCPTR_CALL(vkWaitForFences)(vulkan_data->vulkan_device, 1, &vulkan_data->queue_fence, VK_TRUE, DISPQT_VULKAN_FENCE_TIMEOUT);
		if(vulkan_data->present_queue && vulkan_data->vlk_api.vkQueuePresentKHR_func)
			DISPQT_VULKAN_FUNCPTR_CALL(vkQueuePresentKHR)(vulkan_data->present_queue, &present);
	}
}

//-----------------------------------------------------------------------------------------------------------
bool MMCDispQtVideoRendererVulkan::videorenderer_vulkan_fill_textures(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_vulkan_data_s *vulkan_data = (struct dispqt_render_vulkan_data_s *)this->private_data;
	AVFrame *output_frame = video_surface_infos->ffmpeg_output_frame;
	VkMappedMemoryRange memRange;
	AVFrame subtitle_frame;

	if(!vulkan_data || !vulkan_data->src_image_mem_ptr || !output_frame || !output_frame->height)
		return false;

	// copy video frame into vulkan memory
	register const int linelen_out = vulkan_data->src_image_rowpitch;
	register int linelen_in =  output_frame->linesize[0];
	register unsigned int copy_len = min(linelen_in, linelen_out);
	register uint8_t *indataptr = output_frame->data[0];
	register uint8_t *outdataptr = (uint8_t *)vulkan_data->src_image_mem_ptr;
	register int linecount = min(output_frame->height, vulkan_data->src_image_height);
	do
	{
		pds_memcpy(outdataptr, indataptr, copy_len);
		outdataptr += linelen_out; indataptr += linelen_in;
	}while(--linecount);

	// render subtitle directly into vulkan src image memory
	pds_memcpy(&subtitle_frame, output_frame, sizeof(subtitle_frame));
	subtitle_frame.width = vulkan_data->src_image_width;
	subtitle_frame.height = vulkan_data->src_image_height;
	subtitle_frame.data[0] = (uint8_t *)vulkan_data->src_image_mem_ptr;
	subtitle_frame.linesize[0] = vulkan_data->src_image_rowpitch;

	video_surface_infos->ffmpeg_output_frame = &subtitle_frame;

	this->renderbase_subtitle_qpaint(video_surface_infos, subtitle_infos, (DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_RENDERTOVIDEO | DISPQT_RENDERBASE_SUBTITLE_CTRLFLAG_DRAWEVERY));

	video_surface_infos->ffmpeg_output_frame = output_frame;

	// flush memory writings
	pds_memset(&memRange, 0, sizeof(memRange));
	memRange.sType = VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE;
	memRange.pNext = NULL;
	memRange.memory = vulkan_data->src_image_mem_handler;
	memRange.offset = 0;
	memRange.size = vulkan_data->src_image_mem_size;
	DISPQT_VULKAN_FUNCPTR_CALL(vkFlushMappedMemoryRanges)(vulkan_data->vulkan_device, 1, &memRange);

	return true;
}

//-----------------------------------------------------------------------------------------------------------
MMCDispQtVideoRendererVulkan::MMCDispQtVideoRendererVulkan(MainWindow *mainwindow, QWidget *parent) : MMCDispQtVideoRendererBase(mainwindow, parent)
{
	struct dispqt_render_vulkan_data_s *vulkan_data = (struct dispqt_render_vulkan_data_s *)pds_calloc(1, sizeof(struct dispqt_render_vulkan_data_s));
	if(!vulkan_data)
		return;

	this->private_data = vulkan_data;
	vulkan_data->parent_window_handler = (HWND)parent->winId();
	vulkan_data->surface_vk_format = VK_FORMAT_UNDEFINED;
	vulkan_data->surface_fmt_format = AV_PIX_FMT_NONE;
	mpxplay_video_render_infos.hwdevice_avpixfmt = AV_PIX_FMT_NONE;
}

MMCDispQtVideoRendererVulkan::~MMCDispQtVideoRendererVulkan()
{
	if(this->private_data)
	{
		struct dispqt_render_vulkan_data_s *vulkan_data = (struct dispqt_render_vulkan_data_s *)this->private_data;
		this->private_data = NULL;
		dispqt_vulkan_deinit_vulkan(vulkan_data);
		pds_free(vulkan_data);
	}
}

bool MMCDispQtVideoRendererVulkan::videorenderer_init(struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_render_vulkan_data_s *vulkan_data = (struct dispqt_render_vulkan_data_s *)this->private_data;
	if(!vulkan_data)
		return false;

	if(!dispqt_vulkan_library_load(vulkan_data))
		goto err_out_init;
	if(!dispqt_vulkan_init_create_instance(vulkan_data))
		goto err_out_init;
	if(!dispqt_vulkan_init_enumerate_device(vulkan_data))
		goto err_out_init;
	if(!dispqt_vulkan_init_create_surface(vulkan_data))
		goto err_out_init;
	if(!dispqt_vulkan_init_get_presenter_queue(vulkan_data))
		goto err_out_init;
	if(!dispqt_vulkan_init_create_device(vulkan_data))
		goto err_out_init;
	if(!dispqt_vulkan_init_create_command_buffer(vulkan_data))
		goto err_out_init;
	if(!dispqt_vulkan_init_prepare_surface(vulkan_data, video_surface_infos))
		goto err_out_init;

	return true;

err_out_init:
	dispqt_vulkan_deinit_vulkan(vulkan_data);
	return false;
}

void MMCDispQtVideoRendererVulkan::videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame)
{
	struct dispqt_render_vulkan_data_s *vulkan_data = (struct dispqt_render_vulkan_data_s *)this->private_data;
	if(!vulkan_data)
		return;
	video_surface_infos->videoout_surface_pix_fmt = vulkan_data->surface_fmt_format;
}

void MMCDispQtVideoRendererVulkan::videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_vulkan_data_s *vulkan_data = (struct dispqt_render_vulkan_data_s *)private_data;
	VkImageMemoryBarrier memBarrier = {};
	VkImageBlit region = {};
	VkImage bltDstImage;

	if(!vulkan_data)
		return;
	if(!vulkan_data->vulkan_device || !vulkan_data->swap_buffers)
		return;
	if(!video_surface_infos->videoout_surface_clearpic && !video_surface_infos->ffmpeg_output_frame)
		return;

	if((vulkan_data->swapchain_width != video_surface_infos->window_size_x) || (vulkan_data->swapchain_height != video_surface_infos->window_size_y))
	{
		dispqt_vulkan_videorenderer_deinit_swapchain(vulkan_data, vulkan_data->vulkan_device);
		dispqt_vulkan_init_prepare_surface(vulkan_data, video_surface_infos);
	}

	if(video_surface_infos->videoout_surface_clearpic)
	{
		video_surface_infos->videoout_surface_clearpic = false;
		for(int i = 0; i < vulkan_data->swapchainImageCount; i++)
			vulkan_data->swap_buffers[i].clear_picture_flag = true;
	}

	DISPQT_VULKAN_FUNCPTR_CALL(vkResetCommandBuffer)(vulkan_data->cmd_buffer, 0);

	if(DISPQT_VULKAN_FUNCPTR_CALL(vkAcquireNextImageKHR)(vulkan_data->vulkan_device, vulkan_data->swap_chain, DISPQT_VULKAN_GETBUFFER_TIMEOUT, vulkan_data->imageAcquiredSemaphore, VK_NULL_HANDLE, &vulkan_data->swapchan_buf_idx) != VK_SUCCESS)
		goto err_out_render;

	if(vulkan_data->swapchan_buf_idx >= vulkan_data->swapchainImageCount)
		goto err_out_render;

	if( video_surface_infos->ffmpeg_output_frame
	 && ((vulkan_data->src_image_width != video_surface_infos->ffmpeg_output_frame->width) || (vulkan_data->src_image_height != video_surface_infos->ffmpeg_output_frame->height))
	){
		dispqt_vulkan_src_image_deinit(vulkan_data, vulkan_data->vulkan_device);
		dispqt_vulkan_src_image_init(vulkan_data, video_surface_infos);
		for(int i = 0; i < vulkan_data->swapchainImageCount; i++)
			vulkan_data->swap_buffers[i].clear_picture_flag = true;
	}

	bltDstImage = vulkan_data->swap_buffers[vulkan_data->swapchan_buf_idx].image;

	if(!dispqt_vulkan_execute_begin_command_buffer(vulkan_data))
		goto err_out_render;

	if(vulkan_data->swap_buffers[vulkan_data->swapchan_buf_idx].clear_picture_flag || !video_surface_infos->ffmpeg_output_frame)
	{
		DISPQT_VULKAN_FUNCPTR_CALL(vkCmdClearColorImage)(vulkan_data->cmd_buffer, bltDstImage, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, &vulkan_clearColor, 1, &vulkan_clearRange);
		vulkan_data->swap_buffers[vulkan_data->swapchan_buf_idx].clear_picture_flag = false;
	}

	if(!video_surface_infos->ffmpeg_output_frame)
		goto label_draw_output;

	if(!this->videorenderer_vulkan_fill_textures(video_surface_infos, subtitle_infos))
		goto err_out_render;

	region.srcSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
	region.srcSubresource.mipLevel = 0;
	region.srcSubresource.baseArrayLayer = 0;
	region.srcSubresource.layerCount = 1;
	region.srcOffsets[0].x = video_surface_infos->sws_crop_x;
	region.srcOffsets[0].y = video_surface_infos->sws_crop_y;
	region.srcOffsets[0].z = 0;
	region.srcOffsets[1].x = video_surface_infos->sws_crop_x + video_surface_infos->sws_crop_w;
	region.srcOffsets[1].y = video_surface_infos->sws_crop_y + video_surface_infos->sws_crop_h;
	region.srcOffsets[1].z = 0;
	region.dstSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
	region.dstSubresource.mipLevel = 0;
	region.dstSubresource.baseArrayLayer = 0;
	region.dstSubresource.layerCount = 1;
	region.dstOffsets[0].x = video_surface_infos->sws_dst_pos_x;
	region.dstOffsets[0].y = video_surface_infos->sws_dst_pos_y;
	region.dstOffsets[0].z = 0;
	region.dstOffsets[1].x = video_surface_infos->sws_dst_pos_x + video_surface_infos->sws_dst_width;
	region.dstOffsets[1].y = video_surface_infos->sws_dst_pos_y + video_surface_infos->sws_dst_height;
	region.dstOffsets[1].z = 0;
	DISPQT_VULKAN_FUNCPTR_CALL(vkCmdBlitImage)(vulkan_data->cmd_buffer, vulkan_data->src_image_handler, VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL, bltDstImage, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, 1, &region, VK_FILTER_LINEAR);

	memBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
	memBarrier.pNext = NULL;
	memBarrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
	memBarrier.dstAccessMask = VK_ACCESS_MEMORY_READ_BIT;
	memBarrier.oldLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
	memBarrier.newLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
	memBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
	memBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
	memBarrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
	memBarrier.subresourceRange.baseMipLevel = 0;
	memBarrier.subresourceRange.levelCount = 1;
	memBarrier.subresourceRange.baseArrayLayer = 0;
	memBarrier.subresourceRange.layerCount = 1;
	memBarrier.image = bltDstImage;
	DISPQT_VULKAN_FUNCPTR_CALL(vkCmdPipelineBarrier)(vulkan_data->cmd_buffer, VK_PIPELINE_STAGE_TRANSFER_BIT, VK_PIPELINE_STAGE_TRANSFER_BIT, 0, 0, NULL, 0, NULL, 1, &memBarrier);

label_draw_output:
	DISPQT_VULKAN_FUNCPTR_CALL(vkEndCommandBuffer)(vulkan_data->cmd_buffer);

	dispqt_vulkan_graphics_queue_submit(vulkan_data);

	mpxplay_dispqt_vulkan_videoframe_present(vulkan_data);

err_out_render:

	DISPQT_VULKAN_FUNCPTR_CALL(vkResetCommandBuffer)(vulkan_data->cmd_buffer, 0);

	return;
}

#endif // MPXPLAY_DISPQT_ENABLE_RENDER_VULCAN
