//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2022 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: Direct3D 11 render output

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_ERROR stderr
#define DISPQT_DEBUG_WARNING stdout
#define DISPQT_DEBUGOUT_MIXER NULL // stdout
#define DISPQT_DEBUGOUT_RENDER NULL // stdout
#define DISPQT_DEBUGOUT_DEINT NULL // stdout
#define DISPQT_DEBUGOUT_SWAPCHAIN NULL // stdout
#define DISPQT_DEBUGOUT_TEXTURE NULL // stdout
#define DISPQT_DEBUGOUT_LUMINANCE NULL // stdout
#define DISPQT_DEBUGOUT_POOL NULL // stdout
#define DISPQT_DEBUGOUT_AUTOCROP NULL // stdout
#define DISPQT_DEBUGOUT_SUBTITLE NULL // stdout
#define DISPQT_DEBUGOUT_PIXSHADER NULL // stdout

#include "moc_config.h"

#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11

#include "moc_video_qt.h"
#include "moc_mainwindow.h"
#include "video_render_dx11.h"

#define MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY 1 // reduces the number of texture copies, seems to be good, needs some testing

typedef struct mpxplay_video_d3d11_static_s{
	ID3D11Device *d3d_device_ptr;
	ID3D11DeviceContext *d3d_device_ctx;
	ID3D11VideoDevice *d3d_video_device;
	ID3D11VideoContext *d3d_video_ctx;
#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
	ID3D10Multithread *d3d_multi_thread;
	ID3D11Query *d3d_device_state;
#endif
	mpxp_ptrsize_t d3d_dll_hnd;          // dll handler for video renderer DLL (D3D11.DLL)
	mpxp_bool_t d3d_forced_reopen;
}mpxplay_video_d3d11_static_s;

static mpxp_bool_t mpxplay_dispqt_videorenderer_d3d11_check_videodecoder_inputformat(const GUID format_guid, int av_pixfmt, int video_width, int video_height);
static bool dispqt_evr_texture2d_create(ID3D11Device *d3d_device, int width, int heigth, DXGI_FORMAT format, ID3D11Texture2D **text_hand_ptr, D3D11_USAGE texture_usage, uint32_t bind_flags);
static bool dispqt_evr_texture2d_shaderresourceview_create(ID3D11Device *d3d_device, ID3D11Texture2D *texture2d, DXGI_FORMAT format, ID3D11ShaderResourceView **shaderview_hand_ptr);
static void dispqt_evr_videoframe_download_deinit(struct dispqt_render_evr_data_s *evr_data);

static struct mpxplay_video_d3d11_static_s d3d11_static_datas;

// ====================================================================================================================
// common static context for video output and FFmpeg hw/gpu video decoder

mpxp_bool_t mpxplay_dispqt_videorender_d3d11_static_init(void)
{
	struct mpxplay_video_d3d11_static_s *d3d11_datas = &d3d11_static_datas;
	struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;
	mpxp_bool_t result = FALSE;

	if(funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD))
		return result;
	if(d3d11_datas->d3d_forced_reopen)
		mpxplay_dispqt_videorender_d3d11_static_close();

	if(!d3d11_datas->d3d_device_ptr || !d3d11_datas->d3d_device_ctx)
	{
		const UINT flags = D3D11_CREATE_DEVICE_SINGLETHREADED|D3D11_CREATE_DEVICE_PREVENT_INTERNAL_THREADING_OPTIMIZATIONS|D3D11_CREATE_DEVICE_VIDEO_SUPPORT;
		//const UINT flags = D3D11_CREATE_DEVICE_VIDEO_SUPPORT;
		PFN_D3D11_CREATE_DEVICE d3d_create_device_func;
		HRESULT hr;

		if(!d3d11_datas->d3d_dll_hnd)
		{
			d3d11_datas->d3d_dll_hnd = (mpxp_ptrsize_t)LoadLibrary("D3D11.DLL");
			if(!d3d11_datas->d3d_dll_hnd)
				return FALSE;
		}

		d3d_create_device_func = (PFN_D3D11_CREATE_DEVICE)GetProcAddress((HMODULE)d3d11_datas->d3d_dll_hnd, "D3D11CreateDevice");
		if(!d3d_create_device_func)
			return FALSE;

		hr = d3d_create_device_func(NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, flags, NULL, 0, D3D11_SDK_VERSION,
								&d3d11_datas->d3d_device_ptr, NULL, &d3d11_datas->d3d_device_ctx);
		if(FAILED(hr) || !d3d11_datas->d3d_device_ptr || !d3d11_datas->d3d_device_ctx)
			return FALSE;

#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
		hr = d3d11_datas->d3d_device_ptr->QueryInterface(IID_ID3D10Multithread, (void **)&d3d11_datas->d3d_multi_thread);
//		if (SUCCEEDED(hr)) {
//			d3d11_datas->d3d_multi_thread->SetMultithreadProtected(TRUE);
//		}
		const D3D11_QUERY_DESC query_event = { .Query = D3D11_QUERY_EVENT };
		d3d11_datas->d3d_device_ptr->CreateQuery(&query_event, &d3d11_datas->d3d_device_state);
#endif

		d3d11_datas->d3d_device_ptr->QueryInterface(IID_ID3D11VideoDevice, (void **)&d3d11_datas->d3d_video_device);
		d3d11_datas->d3d_device_ctx->QueryInterface(IID_ID3D11VideoContext, (void **)&d3d11_datas->d3d_video_ctx);
	}

	if(d3d11_datas->d3d_video_device && d3d11_datas->d3d_video_ctx)
	{
		vri_datas->d3d_device_ptr = d3d11_datas->d3d_device_ptr;
		vri_datas->d3d_device_ctx = d3d11_datas->d3d_device_ctx;
		vri_datas->d3d_video_device = d3d11_datas->d3d_video_device;
		vri_datas->d3d_video_ctx = d3d11_datas->d3d_video_ctx;
		vri_datas->hwdevice_avpixfmt = AV_PIX_FMT_D3D11;
		vri_datas->render_function_check_videodecoder_inputformat = mpxplay_dispqt_videorenderer_d3d11_check_videodecoder_inputformat;
		result = TRUE;
	}

	return result;
}

void mpxplay_dispqt_videorender_d3d11_static_close(void)
{
	struct mpxplay_video_d3d11_static_s *d3d11_datas = &d3d11_static_datas;

#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
	SAFE_RELEASE(d3d11_datas->d3d_multi_thread);
	SAFE_RELEASE(d3d11_datas->d3d_device_state);
#endif
	SAFE_RELEASE(d3d11_datas->d3d_video_ctx);
	SAFE_RELEASE(d3d11_datas->d3d_video_device);
	SAFE_RELEASE(d3d11_datas->d3d_device_ctx);
	SAFE_RELEASE(d3d11_datas->d3d_device_ptr);

	if(d3d11_datas->d3d_dll_hnd)
	{
		HMODULE d3d_dll = (HMODULE)d3d11_datas->d3d_dll_hnd;
		d3d11_datas->d3d_dll_hnd = 0;
		FreeLibrary(d3d_dll);
	}

	d3d11_datas->d3d_forced_reopen = FALSE;
}

#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
void mpxplay_dispqt_videorender_d3d11_flush(mpxp_bool_t wait_for_completition)
{
	struct mpxplay_video_d3d11_static_s *d3d11_datas = &d3d11_static_datas;

	if(d3d11_datas->d3d_device_ctx)
	{
		d3d11_datas->d3d_device_ctx->Flush();

		if(wait_for_completition && d3d11_datas->d3d_device_state)
		{
			int maxWait = 20;

			d3d11_datas->d3d_device_ctx->End((ID3D11Asynchronous*)d3d11_datas->d3d_device_state);

			while((d3d11_datas->d3d_device_ctx->GetData((ID3D11Asynchronous*)d3d11_datas->d3d_device_state, NULL, 0, 0) == S_FALSE) && (--maxWait))
			{
				SleepEx(2, TRUE);
			}
		}
	}
}
#endif

mpxp_bool_t mpxplay_dispqt_videorender_d3d11_refer_d3ddevice(struct mpxplay_video_d3d_device_ref_s *d3d_device)
{
	struct mpxplay_video_d3d11_static_s *d3d11_datas = &d3d11_static_datas;

	if(!mpxplay_dispqt_videorender_d3d11_static_init())
		return FALSE;
	if(!d3d11_datas->d3d_device_ptr || !d3d11_datas->d3d_device_ctx || !d3d11_datas->d3d_video_device || !d3d11_datas->d3d_video_ctx)
		return FALSE;

	d3d11_datas->d3d_device_ptr->AddRef();
	d3d_device->d3d_device_ptr = d3d11_datas->d3d_device_ptr;

	d3d11_datas->d3d_device_ctx->AddRef();
	d3d_device->d3d_device_ctx = d3d11_datas->d3d_device_ctx;

	d3d11_datas->d3d_video_device->AddRef();
	d3d_device->d3d_video_device = d3d11_datas->d3d_video_device;

	d3d11_datas->d3d_video_ctx->AddRef();
	d3d_device->d3d_video_ctx = d3d11_datas->d3d_video_ctx;

	return TRUE;
}

// to detect if d3d11 context has halted
mpxp_bool_t mpxplay_dispqt_videorender_d3d11_detect_device_context_failure(void)
{
	struct mpxplay_video_d3d11_static_s *d3d11_datas = &d3d11_static_datas;

	if(d3d11_datas->d3d_forced_reopen)
	{
		return TRUE;
	}

	if(d3d11_datas->d3d_device_ptr)
	{
		const D3D11_QUERY_DESC query_event = { .Query = D3D11_QUERY_EVENT };
		ID3D11Query *render_state = NULL;
		HRESULT hr = d3d11_datas->d3d_device_ptr->CreateQuery(&query_event, &render_state);

		if(FAILED(hr))
		{
			d3d11_datas->d3d_forced_reopen = TRUE;
			mpxplay_debugf(DISPQT_DEBUG_WARNING, "d3d11_detect_device FAILURE");
			return TRUE;
		}

		SAFE_RELEASE(render_state);
	}

	return FALSE; // no error
}

// ====================================================================================================================
static DXGI_FORMAT d3d11_map_avsw_to_d3dhw_format(int av_pixfmt)
{
	switch (av_pixfmt)
	{
		case AV_PIX_FMT_NV12:       return DXGI_FORMAT_NV12;
		case AV_PIX_FMT_P010:       return DXGI_FORMAT_P010;
		case AV_PIX_FMT_YUV420P:    return DXGI_FORMAT_NV12;
		case AV_PIX_FMT_YUV420P10:  return DXGI_FORMAT_P010;
		case AV_PIX_FMT_X2BGR10:    return DXGI_FORMAT_R10G10B10A2_UNORM;
#if MPXPLAY_USE_FFMPEG_V7X
		case AV_PIX_FMT_RGBAF16:    return DXGI_FORMAT_R16G16B16A16_FLOAT;
#endif
		default:                    return DXGI_FORMAT_UNKNOWN;
	}
}

static mpxp_bool_t mpxplay_dispqt_videorenderer_d3d11_check_videodecoder_inputformat(const GUID format_guid, int av_pixfmt, int video_width, int video_height)
{
	struct mpxplay_video_d3d11_static_s *d3d11_datas = &d3d11_static_datas;
	ID3D11VideoDevice *d3d_video_device;
	DXGI_FORMAT surface_format;
	HRESULT hr;

	if(!mpxplay_dispqt_videorender_d3d11_static_init() || !d3d11_datas->d3d_video_device)
		return FALSE;
	surface_format = d3d11_map_avsw_to_d3dhw_format(av_pixfmt);
	if(surface_format == DXGI_FORMAT_UNKNOWN)
		return FALSE;
	d3d_video_device = d3d11_datas->d3d_video_device;
	if((video_width > 0) && (video_height > 0) && (pds_mswin_getver() >= MPXPLAY_MSWIN_VERSIONID_WIN80))
	{
		struct D3D11_VIDEO_DECODER_DESC desc = {.Guid = format_guid, .SampleWidth = (UINT)video_width, .SampleHeight = (UINT)video_height, .OutputFormat = surface_format};
		UINT cfg_count = 0;
		hr = d3d_video_device->GetVideoDecoderConfigCount(&desc, &cfg_count);
		if(SUCCEEDED(hr) && cfg_count)
			return TRUE;
		if(av_pixfmt != AV_PIX_FMT_YUV420P)
			return FALSE;
		desc.OutputFormat = DXGI_FORMAT_420_OPAQUE;
		hr = d3d_video_device->GetVideoDecoderConfigCount(&desc, &cfg_count);
		return (SUCCEEDED(hr) && cfg_count)? TRUE : FALSE;
	}
	else
	{
		WINBOOL is_supported = FALSE;
		hr = d3d_video_device->CheckVideoDecoderFormat(&format_guid, surface_format, &is_supported);
		if(SUCCEEDED(hr) && is_supported)
			return TRUE;
		if(av_pixfmt != AV_PIX_FMT_YUV420P)
			return FALSE;
		surface_format = DXGI_FORMAT_420_OPAQUE;
		is_supported = FALSE;
		hr = d3d_video_device->CheckVideoDecoderFormat(&format_guid, surface_format, &is_supported);
		return (SUCCEEDED(hr) && is_supported);
	}
}

// ====================================================================================================================
static ID3DBlob *dispqt_render_d3d11_shader_compile(struct dispqt_render_evr_data_s *evr_data,
		const char *shader_code_str, const char *shader_mainfunc_name, const char *shader_type_str)
{
	const D3D_FEATURE_LEVEL level = evr_data->d3d_feature_level;
	const char *shader_version_str = (level >= D3D_FEATURE_LEVEL_10_0)? "4_0" : ((level >= D3D_FEATURE_LEVEL_9_3)? "4_0_level_9_3" : "4_0_level_9_1");
	ID3DBlob *pShaderBlob = NULL;
	char target_profile[64];
	HRESULT hr;

	snprintf(target_profile, sizeof(target_profile), "%s_%s", shader_type_str, shader_version_str);

	hr = evr_data->d3d_api_funcs.d3d_shader_compiler_func(shader_code_str, pds_strlen((char *)shader_code_str), NULL, NULL, NULL, shader_mainfunc_name, target_profile, 0, 0, &pShaderBlob, NULL);
	if (FAILED(hr))
		return NULL;

	return pShaderBlob;
}

// ====================================================================================================================
// common functions to create texture(s) and shader resource(s)

static bool dispqt_evr_texture2darray_create(ID3D11Device *d3d_device, int width, int heigth, DXGI_FORMAT format,
		ID3D11Texture2D **text_hand_ptr, D3D11_USAGE texture_usage, uint32_t bind_flags, unsigned int array_size)
{
	D3D11_TEXTURE2D_DESC d3dTextureDesc;

	pds_memset(&d3dTextureDesc, 0,sizeof(d3dTextureDesc));
	d3dTextureDesc.Width = width;
	d3dTextureDesc.Height = heigth;
	d3dTextureDesc.MipLevels = 1;
	d3dTextureDesc.ArraySize = array_size;
	d3dTextureDesc.Format = format;
	d3dTextureDesc.SampleDesc.Count = 1;
	d3dTextureDesc.Usage = texture_usage;
	switch(texture_usage)
	{
		case D3D11_USAGE_DYNAMIC: d3dTextureDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE; break;
		case D3D11_USAGE_STAGING:
			d3dTextureDesc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
			if(bind_flags & D3D11_BIND_SHADER_RESOURCE)
			{
				funcbit_enable(d3dTextureDesc.CPUAccessFlags, D3D11_CPU_ACCESS_WRITE);
				funcbit_disable(bind_flags, D3D11_BIND_SHADER_RESOURCE);
			}
			break;
	}
	d3dTextureDesc.BindFlags = bind_flags;

	return (FAILED(d3d_device->CreateTexture2D(&d3dTextureDesc, nullptr, text_hand_ptr)))? false : true;
}

static bool dispqt_evr_texture2d_create(ID3D11Device *d3d_device, int width, int heigth, DXGI_FORMAT format,
		ID3D11Texture2D **text_hand_ptr, D3D11_USAGE texture_usage, uint32_t bind_flags)
{
	return dispqt_evr_texture2darray_create(d3d_device, width, heigth, format, text_hand_ptr, texture_usage, bind_flags, 1);
}

// create shader resource view for shader resourced texture
static bool dispqt_evr_texture2d_shaderresourceview_create(ID3D11Device *d3d_device, ID3D11Texture2D *texture2d, DXGI_FORMAT format,
		ID3D11ShaderResourceView **shaderview_hand_ptr)
{
	D3D11_SHADER_RESOURCE_VIEW_DESC shaderViewDesc;

	pds_memset(&shaderViewDesc, 0, sizeof(shaderViewDesc));
	shaderViewDesc.Format = format;
	shaderViewDesc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
	shaderViewDesc.Texture2D.MipLevels = 1;

	return (FAILED(d3d_device->CreateShaderResourceView(texture2d, &shaderViewDesc, shaderview_hand_ptr)))? false : true;
}

static bool dispqt_evr_texture2darray_shaderresourceview_create(ID3D11Device *d3d_device, ID3D11Texture2D *texture2d, DXGI_FORMAT format,
		ID3D11ShaderResourceView **shaderview_hand_ptr, unsigned int array_index)
{
	D3D11_SHADER_RESOURCE_VIEW_DESC shaderViewDesc;
	HRESULT hr;

	pds_memset(&shaderViewDesc, 0, sizeof(shaderViewDesc));
	shaderViewDesc.Format = format;
	shaderViewDesc.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2DARRAY;
	shaderViewDesc.Texture2DArray.MipLevels = 1;
	shaderViewDesc.Texture2DArray.FirstArraySlice = array_index;
	shaderViewDesc.Texture2DArray.ArraySize = 1;

	hr = d3d_device->CreateShaderResourceView(texture2d, &shaderViewDesc, shaderview_hand_ptr);

	return (FAILED(hr))? false : true;
}

static void dispqt_evr_input_textures_with_shaderresourceview_clear(ID3D11Texture2D **texture_2d, ID3D11ShaderResourceView **shader_resource_views)
{
	for(int i = 0; i < DISPQT_EVR_NB_MAX_PLANES; i++)
	{
		if(shader_resource_views)
		{
			SAFE_RELEASE(shader_resource_views[i]);
		}
		SAFE_RELEASE(texture_2d[i]);
	}
}

static bool dispqt_evr_input_textures_with_shaderresourceview_alloc(ID3D11Device *d3d_device, struct dispqt_evr_format_desc_s *format_datas, int tex_width, int tex_height,
		ID3D11Texture2D **texture_2d, ID3D11ShaderResourceView  **shader_resource_views, D3D11_USAGE texture_usage, unsigned int texture_bind_flags)
{
	DXGI_FORMAT texture_dxgi_format;
	bool success = false;

	if(texture_usage != D3D11_USAGE_STAGING)
		funcbit_enable(texture_bind_flags, D3D11_BIND_SHADER_RESOURCE);

	dispqt_evr_input_textures_with_shaderresourceview_clear(texture_2d, shader_resource_views);

	texture_dxgi_format = (format_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)? (DXGI_FORMAT)format_datas->shader_view_formats[0] : (DXGI_FORMAT)format_datas->d3d_texture_format;
	if(!dispqt_evr_texture2d_create(d3d_device, tex_width, tex_height, texture_dxgi_format, texture_2d, texture_usage, texture_bind_flags))
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR, "texture2d_create FAILED w:%d h:%d f:%d u:%d b:%d", tex_width, tex_height, texture_dxgi_format, (int)texture_usage, texture_bind_flags);
		goto err_out_alloc;
	}

	// create multiply planes at the case of unsupported color format
	if(format_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)
		for(int i = 1; i < format_datas->nb_components; i++)
			if(!dispqt_evr_texture2d_create(d3d_device, tex_width >> format_datas->align_mask_x, tex_height >> format_datas->align_mask_y, (DXGI_FORMAT)format_datas->shader_view_formats[i], (texture_2d + i), texture_usage, texture_bind_flags))
				goto err_out_alloc;

	if(texture_usage != D3D11_USAGE_STAGING)
	{
		for(int i = 0; i < format_datas->nb_components; i++)
		{
			if(!dispqt_evr_texture2d_shaderresourceview_create(d3d_device, texture_2d[(format_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)? i : 0], (DXGI_FORMAT)format_datas->shader_view_formats[i], &shader_resource_views[i]))
			{
				mpxplay_debugf(DISPQT_DEBUG_ERROR, "shaderresourceview_create FAILED");
				goto err_out_alloc;
			}
		}
	}

	success = true;

err_out_alloc:
	if(!success)
		dispqt_evr_input_textures_with_shaderresourceview_clear(texture_2d, shader_resource_views);
	return success;
}

// ====================================================================================================================
// wrappers/callbacks for pool buffering (video_render_common.cpp)

static void *dispqt_d3d11_poolbuf_bufptr_get(void)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)mpxplay_video_render_infos.renderer_private_data;
	if(!evr_data)
		return NULL;
	return (void *)&evr_data->decoder_pool_datas[0];
}

static void *dispqt_d3d11_poolbuf_pixform_get(void *format_datas_buf, void *src_avframe)
{
	AVFrame *src_frame = (AVFrame *)src_avframe;
	void *form_datas = mpxplay_dispqt_render_common_input_select_color_format(&dispqt_evr_supported_pixel_formats[0], (dispqt_evr_format_desc_s *)format_datas_buf, src_frame->format, TRUE);

	if(!form_datas)
		form_datas = mpxplay_dispqt_render_common_input_select_native_color_format(&dispqt_evr_supported_pixel_formats[0], (dispqt_evr_format_desc_s *)format_datas_buf, AV_PIX_FMT_NV12, TRUE);
	return form_datas;
}

static mpxp_bool_t dispqt_d3d11_poolbuf_texture_alloc(void *format_datas, int tex_width, int tex_height, void **poolbufelem_texture_2d, void **poolbufelem_shader_resource_view, void *mapped_resource)
{
	struct dispqt_evr_format_desc_s *form_datas = (struct dispqt_evr_format_desc_s *)format_datas;

	if(!format_datas)
		return FALSE;

	if(dispqt_evr_input_textures_with_shaderresourceview_alloc(d3d11_static_datas.d3d_device_ptr, form_datas, tex_width,
					tex_height, (ID3D11Texture2D **)poolbufelem_texture_2d, (ID3D11ShaderResourceView **)poolbufelem_shader_resource_view,
					D3D11_USAGE_DYNAMIC, D3D11_BIND_SHADER_RESOURCE))
	{
		struct dispqt_render_mapped_subresource_s *map_res = (struct dispqt_render_mapped_subresource_s *)mapped_resource;
		map_res->vidmem_beginptr = NULL;
		map_res->vidmem_linesize = 0;
		map_res->vidmem_linenum = tex_height;
		if(form_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)
			for(int i = 1; i < form_datas->nb_components; i++)
				map_res[i].vidmem_linenum = tex_height >> form_datas->align_mask_y;
		return TRUE;
	}

	return FALSE;
}

static void dispqt_d3d11_poolbuf_texture_free(void **poolbufelem_texture_2d, void **poolbufelem_shader_resource_unused)
{
	dispqt_evr_input_textures_with_shaderresourceview_clear((ID3D11Texture2D **)poolbufelem_texture_2d, (ID3D11ShaderResourceView **)poolbufelem_shader_resource_unused);
}

static mpxp_bool_t dispqt_d3d11_poolbuf_texture_map(void *poolbufelem_texture_2d, void *mapped_resource)
{
	struct dispqt_render_mapped_subresource_s *map_res = (struct dispqt_render_mapped_subresource_s *)mapped_resource;
	D3D11_MAPPED_SUBRESOURCE sub_res;
	mpxp_bool_t success = FALSE;
	if(!poolbufelem_texture_2d || !map_res)
		return success;
	pds_memset(&sub_res, 0, sizeof(sub_res));
	if(SUCCEEDED(d3d11_static_datas.d3d_device_ctx->Map((ID3D11Resource *)poolbufelem_texture_2d, 0, D3D11_MAP_WRITE_DISCARD, 0, &sub_res)))
	{
		map_res->vidmem_beginptr = sub_res.pData;
		map_res->vidmem_linesize = sub_res.RowPitch;
		success = TRUE;
	}
	return success;
}

static void dispqt_d3d11_poolbuf_texture_unmap(void *poolbufelem_texture_2d, void *mapped_resource)
{
	struct dispqt_render_mapped_subresource_s *map_res = (struct dispqt_render_mapped_subresource_s *)mapped_resource;
	if(map_res)
	{
		map_res->vidmem_beginptr = NULL;
		map_res->vidmem_linesize = 0;
	}
	if(poolbufelem_texture_2d)
		d3d11_static_datas.d3d_device_ctx->Unmap((ID3D11Resource *)poolbufelem_texture_2d, 0);
}

static void dispqt_d3d11_poolbuf_callbacks_assign(void)
{
	mpxplay_video_render_infos.render_function_poolbuf_bufptr_get = dispqt_d3d11_poolbuf_bufptr_get;
	mpxplay_video_render_infos.render_function_poolbuf_pixform_get = dispqt_d3d11_poolbuf_pixform_get;
	mpxplay_video_render_infos.render_function_poolbuf_texture_alloc = dispqt_d3d11_poolbuf_texture_alloc;
	mpxplay_video_render_infos.render_function_poolbuf_texture_free = dispqt_d3d11_poolbuf_texture_free;
	mpxplay_video_render_infos.render_function_poolbuf_texture_map = dispqt_d3d11_poolbuf_texture_map;
	mpxplay_video_render_infos.render_function_poolbuf_texture_unmap = dispqt_d3d11_poolbuf_texture_unmap;
}

// ====================================================================================================================

static void dispqt_evr_poolbufelem_textures_unmap(struct dispqt_render_poolbufelem_s *bufelem)
{
	for(int i = 0; i < DISPQT_EVR_NB_MAX_PLANES; i++)
	{
		struct dispqt_render_mapped_subresource_s *mapres = &bufelem->poolbufelem_texture_mappedResource[i];
		dispqt_d3d11_poolbuf_texture_unmap(bufelem->poolbufelem_texture_2d[i], mapres);
	}
}

static int dispqt_evr_poolbufelem_check_and_select(ID3D11DeviceContext *d3d_device_context, struct dispqt_evr_input_texture_data_s *texture_datas,
		AVFrame *av_frame, struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_render_poolbufelem_s *bufelem;

	if(!mpxplay_video_render_infos.render_function_poolbufelem_validity_check((void *)av_frame))
		return 0;

	bufelem = (struct dispqt_render_poolbufelem_s *)av_frame->buf[0]->data;
	if(!bufelem->poolbufelem_texture_2d[0])
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "POOL buffer elem no texture");
		return -1;
	}
	dispqt_evr_poolbufelem_textures_unmap(bufelem);
	if(bufelem->bufpoolelem_status != DISPQT_RENDER_BUFPOOLELEM_STATUS_DONE)
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "POOL buffer elem invalid status");
		return -1;
	}

	unsigned int nb_textures = (bufelem->bufpoolelem_format_desc.d3d_texture_format == DXGI_FORMAT_UNKNOWN)? bufelem->bufpoolelem_format_desc.nb_components : 1;
	if((texture_datas->texture_width == bufelem->allocated_width) && (texture_datas->texture_height == bufelem->allocated_height) && (texture_datas->pixel_format_infos->av_pixfmt == bufelem->bufpoolelem_format_desc.av_pixfmt))
	{
		if( bufelem->poolbufelem_shader_resource_views[0]
			&& ((video_surface_infos->gui_config->video_crop_type != DISPQT_VIDEOSCRCROPTYPE_AUTO) || (av_frame->height > video_surface_infos->gui_config->video_autocrop_limit))
			&& !mpxplay_dispqt_render_common_deinterlace_condition(av_frame, video_surface_infos->gui_config)
		){
			pds_memcpy(&texture_datas->d3d_input_shaderview_selector[0], &bufelem->poolbufelem_shader_resource_views[0], sizeof(texture_datas->d3d_input_shaderview_selector));
			mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL buffer elem %d SHADER SOURCE", bufelem->elem_id);
		}
		else
		{
			for(int i = 0; i < nb_textures; i++)
				d3d_device_context->CopyResource(texture_datas->d3d_input_texture_2d[i], (ID3D11Resource *)bufelem->poolbufelem_texture_2d[i]);
			mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL buffer elem %d COPY RESOURCE", bufelem->elem_id);
		}
		texture_datas->input_width = bufelem->allocated_width;
		texture_datas->input_height = bufelem->allocated_height;
	}
	else
	{
		const D3D11_BOX box = {0, 0, 0, (UINT)min(texture_datas->video_width, bufelem->allocated_width), (UINT)min(texture_datas->video_height, bufelem->allocated_height), 1};
		for(int i = 0; i < nb_textures; i++)
			d3d_device_context->CopySubresourceRegion(texture_datas->d3d_input_texture_2d[i], 0, 0,0,0, (ID3D11Resource *)bufelem->poolbufelem_texture_2d[i], 0, &box);
		mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL buffer elem %d COPY SUB RESOURCE tw:%d th:%d bw:%d bh:%d", bufelem->elem_id,
				texture_datas->video_width, texture_datas->video_height, bufelem->allocated_width, bufelem->allocated_height);
	}
	//mpxplay_debugf(DISPQT_DEBUGOUT_POOL, "POOL buffer elem selected %d", bufelem->elem_id);

	return 1;
}

// ====================================================================================================================
static IDXGIAdapter *dispqt_evr_d3d11_get_dxgi_adapter(ID3D11Device *d3ddev)
{
	IDXGIAdapter *dxgi_adapter = NULL;
	IDXGIDevice *dxgi_device = NULL;
	if(SUCCEEDED(d3ddev->QueryInterface(IID_IDXGIDevice, (void **)&dxgi_device)) && dxgi_device)
	{
		if(FAILED(dxgi_device->GetAdapter(&dxgi_adapter)))
			dxgi_adapter = NULL;
		dxgi_device->Release();
	}
	return dxgi_adapter;
}

static bool dispqt_evr_d3d11_init(struct dispqt_render_evr_data_s *evr_data)
{
	struct mpxplay_video_d3d11_static_s *d3d11_datas = &d3d11_static_datas;
	struct dispqt_evr_d3d_api_s *d3d_api = &evr_data->d3d_api_funcs;
	IDXGIAdapter *dxgi_adapter;
	HRESULT hr;

	if(!mpxplay_dispqt_videorender_d3d11_static_init())
		return false;

	d3d11_datas->d3d_device_ptr->AddRef();
	evr_data->d3d_device = d3d11_datas->d3d_device_ptr;
	d3d11_datas->d3d_device_ctx->AddRef();
	evr_data->d3d_device_context = d3d11_datas->d3d_device_ctx;

	if(d3d11_datas->d3d_video_device && d3d11_datas->d3d_video_ctx)
	{
		d3d11_datas->d3d_video_device->AddRef();
		evr_data->d3d_video_device = d3d11_datas->d3d_video_device;
		d3d11_datas->d3d_video_ctx->AddRef();
		evr_data->d3d_video_context = d3d11_datas->d3d_video_ctx;
	}

	// load shader compiler DLL
	for(int version = 47; version > 41; version--)
	{
		char filename[20];
		snprintf(filename, sizeof(filename), "D3DCOMPILER_%d.DLL", version);
		d3d_api->d3dcompiler_dll_hnd = LoadLibrary(filename);
		if (d3d_api->d3dcompiler_dll_hnd)
			break;
	}
	if(!d3d_api->d3dcompiler_dll_hnd)
		return false;

	// get shader compiler function
	d3d_api->d3d_shader_compiler_func = (pD3DCompile)GetProcAddress(d3d_api->d3dcompiler_dll_hnd, "D3DCompile");
	if(!d3d_api->d3d_shader_compiler_func)
		return false;

	evr_data->d3d_feature_level = evr_data->d3d_device->GetFeatureLevel();

	dxgi_adapter = dispqt_evr_d3d11_get_dxgi_adapter(evr_data->d3d_device);
	if(!dxgi_adapter)
		return false;

	hr = dxgi_adapter->GetParent(IID_IDXGIFactory2, (void **)&evr_data->dxgi_factory2);
	if(FAILED(hr) || !evr_data->dxgi_factory2)
	{
		hr = dxgi_adapter->GetParent(IID_IDXGIFactory, (void **)&evr_data->dxgi_factory1);
		SAFE_RELEASE(dxgi_adapter);
		if(FAILED(hr) || !evr_data->dxgi_factory1)
			return false;
	}
	else
	{
		SAFE_RELEASE(dxgi_adapter);
	}

	return true;
}

//------------------------------------------------------------------------------------------------------------
static bool dispqt_evr_d3d11_sampler_state_create(struct dispqt_render_evr_data_s *evr_data)
{
	ID3D11SamplerState  *d3d_sampler_state = NULL;
	D3D11_SAMPLER_DESC SamplerDesc;
	HRESULT hr;
	pds_memset(&SamplerDesc, 0, sizeof(SamplerDesc));
	SamplerDesc.Filter = D3D11_FILTER_MIN_MAG_LINEAR_MIP_POINT;
	SamplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
	SamplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
	SamplerDesc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
	SamplerDesc.MaxLOD = D3D11_FLOAT32_MAX;
	SamplerDesc.ComparisonFunc = D3D11_COMPARISON_NEVER;

	hr = evr_data->d3d_device->CreateSamplerState(&SamplerDesc, &d3d_sampler_state);
	if(FAILED(hr) || !d3d_sampler_state)
		return false;

	evr_data->d3d_device_context->PSSetSamplers(0, 1, &d3d_sampler_state);
	SAFE_RELEASE(d3d_sampler_state);

	return true;
}

static bool dispqt_evr_d3d11_alpha_blend_create(struct dispqt_render_evr_data_s *evr_data)
{
	D3D11_BLEND_DESC spuBlendDesc;

	pds_memset(&spuBlendDesc, 0, sizeof(spuBlendDesc));
	spuBlendDesc.RenderTarget[0].BlendEnable = TRUE;
	spuBlendDesc.RenderTarget[0].SrcBlend = D3D11_BLEND_SRC_ALPHA;
	spuBlendDesc.RenderTarget[0].DestBlend = D3D11_BLEND_INV_SRC_ALPHA;
	spuBlendDesc.RenderTarget[0].BlendOp = D3D11_BLEND_OP_ADD;
	spuBlendDesc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_ONE;
	spuBlendDesc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_ZERO;
	spuBlendDesc.RenderTarget[0].BlendOpAlpha = D3D11_BLEND_OP_ADD;
	spuBlendDesc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;
	if(FAILED(evr_data->d3d_device->CreateBlendState(&spuBlendDesc, &evr_data->d3d_blend_state)))
		return false;

	evr_data->d3d_device_context->OMSetBlendState(evr_data->d3d_blend_state, NULL, 0xFFFFFFFF);

	return true;
}

static void dispqt_evr_d3d11_alpha_blend_clear(struct dispqt_render_evr_data_s *evr_data)
{
	if(evr_data->d3d_blend_state)
	{
		evr_data->d3d_device_context->OMSetBlendState(NULL, NULL, 0xFFFFFFFF);
		SAFE_RELEASE(evr_data->d3d_blend_state);
	}
}

//------------------------------------------------------------------------------------------------------------

static bool dispqt_evr_d3d11_constant_buffers_init(struct dispqt_render_evr_data_s *evr_data)
{
	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	D3D11_BUFFER_DESC constantBufDesc;
	HRESULT hr;

	pds_memset(&constantBufDesc, 0, sizeof(constantBufDesc));
	constantBufDesc.Usage = D3D11_USAGE_DYNAMIC;
	constantBufDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	constantBufDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;

	constantBufDesc.ByteWidth = sizeof(struct dispqt_evr_color_mixer_s);
	hr = evr_data->d3d_device->CreateBuffer(&constantBufDesc, NULL, &texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_COLORMIXER]);
	if(FAILED(hr) || !texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_COLORMIXER])
		return false;

	constantBufDesc.ByteWidth = sizeof(struct dispqt_evr_color_transform_matrixes_s);
	hr = evr_data->d3d_device->CreateBuffer(&constantBufDesc, NULL, &texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_YUVTORGB]);
	if(FAILED(hr) || !texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_YUVTORGB])
		return false;

	constantBufDesc.ByteWidth = sizeof(struct dispqt_evr_dynamic_runtime_data_s);
	hr = evr_data->d3d_device->CreateBuffer(&constantBufDesc, NULL, &texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_RUNTIME]);
	if(FAILED(hr) || !texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_RUNTIME])
		return false;

	return true;
}

static bool dispqt_evr_d3d11_native_format_support_check(struct dispqt_render_evr_data_s *evr_data)
{
	// TODO: maybe we should check D3D11_FORMAT_SUPPORT_VIDEO_PROCESSOR_OUTPUT for deinterlace and D3D11_FORMAT_SUPPORT_BLENDABLE for subtitles
	const UINT required_flags_input = D3D11_FORMAT_SUPPORT_TEXTURE2D | D3D11_FORMAT_SUPPORT_SHADER_LOAD;
	struct dispqt_evr_format_desc_s *format_decs = &dispqt_evr_supported_pixel_formats[0];
	bool supported_native_format_found = false;

	do {
		DXGI_FORMAT format = (format_decs->d3d_texture_format == DXGI_FORMAT_UNKNOWN)? (DXGI_FORMAT)format_decs->shader_view_formats[0] : (DXGI_FORMAT)format_decs->d3d_texture_format;
		UINT supported_flags = 0;
		if(SUCCEEDED(evr_data->d3d_device->CheckFormatSupport(format, &supported_flags)))
		{
			if((supported_flags & required_flags_input) == required_flags_input)
			{
				funcbit_enable(format_decs->is_native_support, DISPQT_EVR_SUPPORTFLAG_INPUT);
				supported_native_format_found = true;
			}
			if(supported_flags & D3D11_FORMAT_SUPPORT_RENDER_TARGET)
			{
				funcbit_enable(format_decs->is_native_support, DISPQT_EVR_SUPPORTFLAG_OUTPUT);
			}
		}
		format_decs++;
	}while(format_decs->av_pixfmt != AV_PIX_FMT_NONE);

	return supported_native_format_found;
}

//============================================================================================================

static void dispqt_evr_swapchain_fill_desc1(struct dispqt_render_evr_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos, DXGI_SWAP_CHAIN_DESC *scd)
{
	pds_memset(scd, 0, sizeof(*scd));
	scd->BufferDesc.Format = evr_data->swapchain_format;
	scd->BufferDesc.Width = video_surface_infos->window_size_x;
	scd->BufferDesc.Height = video_surface_infos->window_size_y;
	scd->BufferUsage = DXGI_USAGE_SHADER_INPUT | DXGI_USAGE_RENDER_TARGET_OUTPUT;
	scd->SampleDesc.Count = 1;
	scd->OutputWindow = evr_data->parent_window_handler;
	scd->Windowed = FALSE;

	if(pds_mswin_getver() >= MPXPLAY_MSWIN_VERSIONID_WIN80)
	{
		scd->SwapEffect = DXGI_SWAP_EFFECT_FLIP_SEQUENTIAL; scd->BufferCount = DISPQT_EVR_SWAPCHAN_BUFFERS;
	}
	else
	{
		scd->SwapEffect = DXGI_SWAP_EFFECT_DISCARD; scd->BufferCount = 1;
	}
}

static void dispqt_evr_swapchain_fill_desc2(struct dispqt_render_evr_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos, DXGI_SWAP_CHAIN_DESC1 *scd)
{
	pds_memset(scd, 0, sizeof(*scd));
	scd->Format = evr_data->swapchain_format;
	scd->Width = video_surface_infos->window_size_x;
	scd->Height = video_surface_infos->window_size_y;
	scd->BufferUsage = DXGI_USAGE_SHADER_INPUT | DXGI_USAGE_RENDER_TARGET_OUTPUT;
	scd->SampleDesc.Count = 1;

	if(pds_mswin_getver() >= MPXPLAY_MSWIN_VERSIONID_WIN80)
	{
		scd->SwapEffect = DXGI_SWAP_EFFECT_FLIP_SEQUENTIAL; scd->BufferCount = DISPQT_EVR_SWAPCHAN_BUFFERS;
	}
	else
	{
		scd->SwapEffect = DXGI_SWAP_EFFECT_DISCARD; scd->BufferCount = 1;
	}
}

static bool dispqt_evr_swapchain_create(struct dispqt_render_evr_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
#if DISPQT_EVR_USE_10BIT_OUTPUT
	const DXGI_FORMAT swapchan_try_list[] = { DXGI_FORMAT_R10G10B10A2_UNORM, DXGI_FORMAT_B8G8R8A8_UNORM, DXGI_FORMAT_R8G8B8A8_UNORM, DXGI_FORMAT_B8G8R8X8_UNORM};
#else
	const DXGI_FORMAT swapchan_try_list[] = { DXGI_FORMAT_B8G8R8A8_UNORM, DXGI_FORMAT_R8G8B8A8_UNORM, DXGI_FORMAT_B8G8R8X8_UNORM};
#endif
	const unsigned int nb_try_list_elems = sizeof(swapchan_try_list) / sizeof(DXGI_FORMAT);
	DXGI_SWAP_CHAIN_DESC swap_chain_desc1;
	DXGI_SWAP_CHAIN_DESC1 swap_chain_desc2;
	bool result = false;
	HRESULT hr = -1;

	SAFE_RELEASE(evr_data->swap_chain_sdr1);
	SAFE_RELEASE(evr_data->swap_chain_sdr2);

	for(int i = 0; i < nb_try_list_elems; i++)
	{
		evr_data->swapchain_format = swapchan_try_list[i];

		if(evr_data->dxgi_factory1)
		{
			dispqt_evr_swapchain_fill_desc1(evr_data, video_surface_infos, &swap_chain_desc1);

			hr = evr_data->dxgi_factory1->CreateSwapChain((IUnknown *)evr_data->d3d_device,
					&swap_chain_desc1, &evr_data->swap_chain_sdr1);

			if(SUCCEEDED(hr))
				evr_data->dxgi_factory1->MakeWindowAssociation(evr_data->parent_window_handler, DXGI_MWA_VALID);
		}
		else if(evr_data->dxgi_factory2)
		{
			dispqt_evr_swapchain_fill_desc2(evr_data, video_surface_infos, &swap_chain_desc2);

			hr = evr_data->dxgi_factory2->CreateSwapChainForHwnd((IUnknown *)evr_data->d3d_device,
					evr_data->parent_window_handler, &swap_chain_desc2, NULL, NULL, &evr_data->swap_chain_sdr2);

			if(SUCCEEDED(hr))
				evr_data->dxgi_factory2->MakeWindowAssociation(evr_data->parent_window_handler, DXGI_MWA_VALID);
		}
		if(SUCCEEDED(hr))
		{
			result = true;
			break;
		}
	}

	return result;
}

static void dispqt_evr_swapchain_deinit(struct dispqt_render_evr_data_s *evr_data)
{
//	static ID3D11RenderTargetView *null_view[1] = {NULL};
//	evr_data->d3d_device_context->OMSetRenderTargets(1, &null_view[0], nullptr);
	SAFE_RELEASE(evr_data->d3d_render_target_view);
	evr_data->swapchain_width = 0;
	evr_data->swapchain_height = 0;
}

static bool dispqt_evr_swapchain_init(struct dispqt_render_evr_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	const int dst_width = video_surface_infos->window_size_x;
	const int dst_height = video_surface_infos->window_size_y;
	ID3D11Texture2D *d3dTexture2d = NULL;
	D3D11_TEXTURE2D_DESC d3dTextureDesc;
	HRESULT hr;

	if((dst_width <= 0) || (dst_height <= 0))
		return false;

	if(!evr_data->swap_chain_sdr1 && !evr_data->swap_chain_sdr2) // TODO: or output/display pixel format change
		if(!dispqt_evr_swapchain_create(evr_data, video_surface_infos))
			return false;

	if((evr_data->swapchain_width == dst_width) && (evr_data->swapchain_height == dst_height))
		return true; // output resolution has not changed

	mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN BEGIN w:%d h:%d f:%d", dst_width, dst_height, evr_data->swapchain_format);

	dispqt_evr_swapchain_deinit(evr_data);

	mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN swapchain_deinit END");

	if(evr_data->swap_chain_sdr1)
	{
		hr = evr_data->swap_chain_sdr1->ResizeBuffers(0, dst_width, dst_height, evr_data->swapchain_format, 0);
		if(FAILED(hr))
			return false;
		// render target view
		hr = evr_data->swap_chain_sdr1->GetBuffer(0, IID_ID3D11Texture2D, (LPVOID *)&d3dTexture2d);
		if(FAILED(hr) || !d3dTexture2d)
			return false;
	}
	else
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN ResizeBuffers begin");
		hr = evr_data->swap_chain_sdr2->ResizeBuffers(0, dst_width, dst_height, evr_data->swapchain_format, 0);
		if(FAILED(hr))
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN ResizeBuffers failed");
			return false;
		}
		mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN GetBuffer begin");
		// render target view
		hr = evr_data->swap_chain_sdr2->GetBuffer(0, IID_ID3D11Texture2D, (LPVOID *)&d3dTexture2d);
		if(FAILED(hr) || !d3dTexture2d)
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN GetBuffer failed");
			return false;
		}
	}

	mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN CreateRenderTargetView begin");

	D3D11_RENDER_TARGET_VIEW_DESC renderDesc = {};
	renderDesc.Format = evr_data->swapchain_format;
	renderDesc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
	renderDesc.Texture2D.MipSlice = 0;

	hr = evr_data->d3d_device->CreateRenderTargetView((ID3D11Resource *)d3dTexture2d, &renderDesc, &evr_data->d3d_render_target_view);
	d3dTexture2d->Release();
	d3dTexture2d = NULL;
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN CreateRenderTargetView failed");
		return false;
	}

	evr_data->swapchain_width = dst_width;
	evr_data->swapchain_height = dst_height;

	mpxplay_debugf(DISPQT_DEBUGOUT_SWAPCHAIN, "SWAPCHAIN END w:%d h:%d", dst_width, dst_height);

	return true;

err_out_init:
	dispqt_evr_swapchain_deinit(evr_data);
	return false;
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_evr_clear_output_pic(struct dispqt_render_evr_data_s *evr_data)
{
	const FLOAT blackRGBA[4] = {0.0f, 0.0f, 0.0f, 1.0f};
	if(!evr_data->d3d_render_target_view)
		return;
	evr_data->d3d_device_context->ClearRenderTargetView(evr_data->d3d_render_target_view, blackRGBA);
}

//============================================================================================================
static void dispqt_evr_d3d11_videoprocessor_deinterlace_reset(struct dispqt_evr_input_texture_data_s *texture_datas)
{
	struct dispqt_evr_deinterlace_data_s *deint_datas = &texture_datas->deinterlace_datas;
	deint_datas->d3d_deinterlace_frame_counter = 0;
	deint_datas->d3d_deinterlace_frdup_pass = 0;
}

static void dispqt_evr_d3d11_videoprocessor_deinterlace_close(struct dispqt_evr_input_texture_data_s *texture_datas)
{
	struct dispqt_evr_deinterlace_data_s *deint_datas = &texture_datas->deinterlace_datas;
	funcbit_disable(mpxplay_video_render_infos.renderer_capabilities, (DISPQT_VIDEORENDERER_CAPFLAG_DEINTERDUP|DISPQT_VIDEORENDERER_CAPENABLE_DEINTERDUP));
	SAFE_RELEASE(deint_datas->d3d_deinterlace_outview);
	dispqt_evr_input_textures_with_shaderresourceview_clear(&deint_datas->d3d_deinterlace_outtexture[0], &deint_datas->d3d_deinterlace_out_shaderview[0]);
	for(int i = 0; i < DISPQT_EVR_DEINTERLACE_REF_FRAMES; i++)
	{
		SAFE_RELEASE(deint_datas->d3d_deinterlace_inviews[i]);
		SAFE_RELEASE(deint_datas->d3d_deinterlace_intextures[i]);
	}
	SAFE_RELEASE(deint_datas->d3d_deinterlace_processor);
	SAFE_RELEASE(deint_datas->d3d_deinterlace_enumerator);
	dispqt_evr_d3d11_videoprocessor_deinterlace_reset(texture_datas);
	deint_datas->d3d_deinterlace_videoplayer_control = 0;
}

static bool dispqt_evr_d3d11_videoprocessor_deinterlace_init(struct dispqt_render_evr_data_s *evr_data,
		struct dispqt_evr_input_texture_data_s *texture_datas, struct dispqt_video_surface_info_s *video_surface_infos,
		AVFrame *av_inframe, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_evr_deinterlace_data_s *deint_datas = &texture_datas->deinterlace_datas;
	struct dispqt_evr_format_desc_s *format_datas;
	D3D11_VIDEO_PROCESSOR_RATE_CONVERSION_CAPS rateCaps;
	D3D11_VIDEO_PROCESSOR_CONTENT_DESC processorDesc;
	D3D11_VIDEO_PROCESSOR_OUTPUT_VIEW_DESC outDesc;
	D3D11_VIDEO_PROCESSOR_INPUT_VIEW_DESC inDesc;
	D3D11_VIDEO_PROCESSOR_CAPS processorCaps;
	UINT flags = 0, selectedRateConvCapIndex = 0, highestDeinterlaceSupport = 0;
	const UINT DEINTERLACESUPPORTMASK = D3D11_VIDEO_PROCESSOR_PROCESSOR_CAPS_DEINTERLACE_BLEND | D3D11_VIDEO_PROCESSOR_PROCESSOR_CAPS_DEINTERLACE_BOB | D3D11_VIDEO_PROCESSOR_PROCESSOR_CAPS_DEINTERLACE_ADAPTIVE | D3D11_VIDEO_PROCESSOR_PROCESSOR_CAPS_DEINTERLACE_MOTION_COMPENSATION;
	mpxp_bool_t is_deinterlace_duplicatedfps;
	HRESULT hr;

	if(!video_surface_infos || !av_inframe || !gcfg)
		return false;
	if(!evr_data->d3d_video_device)
		goto err_out_init;

	dispqt_evr_d3d11_videoprocessor_deinterlace_close(texture_datas);

	if(!mpxplay_dispqt_render_common_deinterlace_condition(av_inframe, gcfg))
		return false;
	is_deinterlace_duplicatedfps = mpxplay_dispqt_render_common_deinterdup_condition(av_inframe, gcfg);

	format_datas = texture_datas->pixel_format_infos;

	pds_memset(&processorDesc, 0, sizeof(processorDesc));
	processorDesc.InputFrameFormat = D3D11_VIDEO_FRAME_FORMAT_INTERLACED_TOP_FIELD_FIRST;
	processorDesc.InputFrameRate.Numerator   = 25000;
	processorDesc.InputFrameRate.Denominator = 1000;
	processorDesc.InputWidth   = texture_datas->texture_width;
	processorDesc.InputHeight  = texture_datas->texture_height;
	processorDesc.OutputWidth  = texture_datas->texture_width;
	processorDesc.OutputHeight = texture_datas->texture_height;
	processorDesc.OutputFrameRate.Numerator = 25000;
	processorDesc.OutputFrameRate.Denominator = 1000;
	processorDesc.Usage = D3D11_VIDEO_USAGE_PLAYBACK_NORMAL;
	if(is_deinterlace_duplicatedfps)
	{
		processorDesc.OutputFrameRate.Numerator *= 2;
		processorDesc.Usage = D3D11_VIDEO_USAGE_OPTIMAL_QUALITY;
	}

	hr = evr_data->d3d_video_device->CreateVideoProcessorEnumerator(&processorDesc, &deint_datas->d3d_deinterlace_enumerator);
	if(FAILED(hr) || !deint_datas->d3d_deinterlace_enumerator)
	{
		if(!is_deinterlace_duplicatedfps)
			goto err_out_init;
		processorDesc.OutputFrameRate.Numerator = 25000;
		processorDesc.Usage = D3D11_VIDEO_USAGE_PLAYBACK_NORMAL;
		hr = evr_data->d3d_video_device->CreateVideoProcessorEnumerator(&processorDesc, &deint_datas->d3d_deinterlace_enumerator);
		if(FAILED(hr) || !deint_datas->d3d_deinterlace_enumerator)
			goto err_out_init;
		is_deinterlace_duplicatedfps = false;
		mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "DEINT deinterlace_init fallback to PLAYBACK_NORMAL");
	}

	hr = deint_datas->d3d_deinterlace_enumerator->CheckVideoProcessorFormat((DXGI_FORMAT)texture_datas->pixel_format_infos->d3d_texture_format, &flags);
	if(FAILED(hr))
		goto err_out_init;
	if(!(flags & D3D11_VIDEO_PROCESSOR_FORMAT_SUPPORT_INPUT) || !(flags & D3D11_VIDEO_PROCESSOR_FORMAT_SUPPORT_OUTPUT))
		goto err_out_init;

	pds_memset(&processorCaps, 0, sizeof(processorCaps));
	hr = deint_datas->d3d_deinterlace_enumerator->GetVideoProcessorCaps(&processorCaps);
	if(FAILED(hr))
		goto err_out_init;

	for(unsigned int i = 0; i < processorCaps.RateConversionCapsCount; i++)
	{
		deint_datas->d3d_deinterlace_enumerator->GetVideoProcessorRateConversionCaps(i, &rateCaps);
		UINT currDeinterlaceSupport = rateCaps.ProcessorCaps & DEINTERLACESUPPORTMASK;
		if(currDeinterlaceSupport > highestDeinterlaceSupport)
		{
			highestDeinterlaceSupport = currDeinterlaceSupport;
			selectedRateConvCapIndex = i;
		}
	}

	if(!funcbit_test(highestDeinterlaceSupport, DEINTERLACESUPPORTMASK)) // there's no any deinterlace support
		goto err_out_init;

	hr = evr_data->d3d_video_device->CreateVideoProcessor(deint_datas->d3d_deinterlace_enumerator, selectedRateConvCapIndex, &deint_datas->d3d_deinterlace_processor);
	if(FAILED(hr) || !deint_datas->d3d_deinterlace_processor)
		goto err_out_init;

	for(int i = 0; i < DISPQT_EVR_DEINTERLACE_REF_FRAMES; i++)
	{
		if(!dispqt_evr_texture2d_create(evr_data->d3d_device, texture_datas->texture_width, texture_datas->texture_height, (DXGI_FORMAT)texture_datas->pixel_format_infos->d3d_texture_format, &deint_datas->d3d_deinterlace_intextures[i], D3D11_USAGE_DEFAULT, D3D11_BIND_RENDER_TARGET))
			goto err_out_init;

		pds_memset(&inDesc, 0, sizeof(inDesc));
		inDesc.ViewDimension = D3D11_VPIV_DIMENSION_TEXTURE2D;
		hr = evr_data->d3d_video_device->CreateVideoProcessorInputView(
				deint_datas->d3d_deinterlace_intextures[i], deint_datas->d3d_deinterlace_enumerator,
				&inDesc, &deint_datas->d3d_deinterlace_inviews[i]);
		if(FAILED(hr) || !deint_datas->d3d_deinterlace_inviews[i])
			goto err_out_init;
	}

	if(!dispqt_evr_input_textures_with_shaderresourceview_alloc(evr_data->d3d_device, format_datas, texture_datas->texture_width, texture_datas->texture_height, &deint_datas->d3d_deinterlace_outtexture[0], &deint_datas->d3d_deinterlace_out_shaderview[0], D3D11_USAGE_DEFAULT, D3D11_BIND_RENDER_TARGET))
		goto err_out_init;

	pds_memset(&outDesc, 0, sizeof(outDesc));
	outDesc.ViewDimension = D3D11_VPOV_DIMENSION_TEXTURE2D;
	hr = evr_data->d3d_video_device->CreateVideoProcessorOutputView(deint_datas->d3d_deinterlace_outtexture[0],
									deint_datas->d3d_deinterlace_enumerator, &outDesc, &deint_datas->d3d_deinterlace_outview);
	if(FAILED(hr) || !deint_datas->d3d_deinterlace_outview)
		goto err_out_init;

	deint_datas->d3d_deinterlace_videoplayer_control = gcfg->selected_videoplayer_control;
	funcbit_enable(mpxplay_video_render_infos.renderer_capabilities, DISPQT_VIDEORENDERER_CAPFLAG_DEINTERLACE);
	if(is_deinterlace_duplicatedfps)
		funcbit_enable(mpxplay_video_render_infos.renderer_capabilities, (DISPQT_VIDEORENDERER_CAPFLAG_DEINTERDUP|DISPQT_VIDEORENDERER_CAPENABLE_DEINTERDUP));

	return true;

err_out_init:
	funcbit_disable(mpxplay_video_render_infos.renderer_capabilities, (DISPQT_VIDEORENDERER_CAPFLAG_DEINTERLACE | DISPQT_VIDEORENDERER_CAPFLAG_DEINTERDUP));
	dispqt_evr_d3d11_videoprocessor_deinterlace_close(texture_datas);
	return false;
}

static bool dispqt_evr_d3d11_videoprocessor_deinterlace_render(struct dispqt_render_evr_data_s *evr_data,
		struct dispqt_evr_input_texture_data_s *texture_datas, struct dispqt_video_surface_info_s *video_surface_infos,
		AVFrame *video_avframe, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_evr_deinterlace_data_s *deint_datas = &texture_datas->deinterlace_datas;
	D3D11_VIDEO_FRAME_FORMAT frameFormat;
	D3D11_VIDEO_PROCESSOR_STREAM stream;
	mpxp_uint32_t curr_pic_index, field_num;
	HRESULT hr;

	if(!mpxplay_dispqt_render_common_deinterlace_condition(video_avframe, gcfg))
	{
		dispqt_evr_d3d11_videoprocessor_deinterlace_reset(texture_datas);
		funcbit_disable(mpxplay_video_render_infos.renderer_capabilities, DISPQT_VIDEORENDERER_CAPENABLE_DEINTERDUP);
		//mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "DEINT deinterlace_render RESET");
		return true;
	}

	if( !deint_datas->d3d_deinterlace_processor || !deint_datas->d3d_deinterlace_intextures[0] || !deint_datas->d3d_deinterlace_inviews[0] || !deint_datas->d3d_deinterlace_out_shaderview[0]
	 || ((deint_datas->d3d_deinterlace_videoplayer_control & MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DUPLICATE_INTERLACED_FRAMES) != (gcfg->selected_videoplayer_control & MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DUPLICATE_INTERLACED_FRAMES))
	){
		dispqt_evr_d3d11_videoprocessor_deinterlace_init(evr_data, texture_datas, video_surface_infos, video_avframe, gcfg);
		if(!deint_datas->d3d_deinterlace_processor || !deint_datas->d3d_deinterlace_intextures[0] || !deint_datas->d3d_deinterlace_inviews[0])
			return false;
	}
	else if(!video_surface_infos->videoout_inputframe_change && !deint_datas->d3d_deinterlace_frdup_pass && deint_datas->d3d_deinterlace_frame_counter)
	{   // input frame is re-used -> deinterlace output also
		pds_memcpy(&texture_datas->d3d_input_shaderview_selector[0], &deint_datas->d3d_deinterlace_out_shaderview[0], sizeof(texture_datas->d3d_input_shaderview_selector));
		return true;
	}

	curr_pic_index = deint_datas->d3d_deinterlace_frame_counter % DISPQT_EVR_DEINTERLACE_REF_FRAMES;
#if MPXPLAY_USE_FFMPEG_V7X
	field_num = (!(video_avframe->flags & AV_FRAME_FLAG_INTERLACED) || (video_avframe->flags & AV_FRAME_FLAG_TOP_FIELD_FIRST))? 0 : 1;
#else
	field_num = (!video_avframe->interlaced_frame || video_avframe->top_field_first)? 0 : 1; // use top_field_first (field_num 0) if not interlaced_frame (forced deinterlace)
#endif
	if(!deint_datas->d3d_deinterlace_frdup_pass)
	{
#ifdef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
		if(texture_datas->av_pixelformat == AV_PIX_FMT_D3D11)
		{
			ID3D11Texture2D *hwdec_texture_array = (ID3D11Texture2D *)video_avframe->data[0];
			const int array_index = (intptr_t)video_avframe->data[1];
			const D3D11_BOX box = {0, 0, 0, (UINT)video_avframe->width, (UINT)video_avframe->height, 1}; // needed to avoid a bottom green line at rendering (due to the float based rectangle)
			evr_data->d3d_device_context->CopySubresourceRegion(deint_datas->d3d_deinterlace_intextures[curr_pic_index], 0, 0,0,0, hwdec_texture_array, array_index, &box);
		}
		else
#endif
		{
			evr_data->d3d_device_context->CopyResource(deint_datas->d3d_deinterlace_intextures[curr_pic_index], texture_datas->d3d_input_texture_2d[0]); // one plane input formats only
		}
	}
	else
	{
		field_num = !field_num;
	}

	frameFormat = (!field_num)? D3D11_VIDEO_FRAME_FORMAT_INTERLACED_TOP_FIELD_FIRST : D3D11_VIDEO_FRAME_FORMAT_INTERLACED_BOTTOM_FIELD_FIRST;

	evr_data->d3d_video_context->VideoProcessorSetStreamFrameFormat(deint_datas->d3d_deinterlace_processor, 0, frameFormat);
	evr_data->d3d_video_context->VideoProcessorSetStreamAutoProcessingMode(deint_datas->d3d_deinterlace_processor, 0, FALSE);

	pds_memset(&stream, 0, sizeof(stream));
	stream.Enable = TRUE;
	stream.InputFrameOrField = field_num;
	stream.OutputIndex = deint_datas->d3d_deinterlace_frdup_pass;
	stream.pInputSurface = deint_datas->d3d_deinterlace_inviews[curr_pic_index];

	if(deint_datas->d3d_deinterlace_frame_counter == 1)
	{
		mpxp_uint32_t prev_pic_index = (deint_datas->d3d_deinterlace_frame_counter - 1) % DISPQT_EVR_DEINTERLACE_REF_FRAMES;
		stream.ppPastSurfaces = &deint_datas->d3d_deinterlace_inviews[prev_pic_index];
		stream.PastFrames = 1;
	}
	else if(deint_datas->d3d_deinterlace_frame_counter >= 2)
	{
		mpxp_uint32_t prev_pic_index = (deint_datas->d3d_deinterlace_frame_counter - 2) % DISPQT_EVR_DEINTERLACE_REF_FRAMES;
		stream.ppPastSurfaces = &deint_datas->d3d_deinterlace_inviews[prev_pic_index];
		stream.PastFrames = 1;
		stream.ppFutureSurfaces = &deint_datas->d3d_deinterlace_inviews[curr_pic_index];
		stream.FutureFrames = 1;
		curr_pic_index = (deint_datas->d3d_deinterlace_frame_counter - 1) % DISPQT_EVR_DEINTERLACE_REF_FRAMES;
		stream.pInputSurface = deint_datas->d3d_deinterlace_inviews[curr_pic_index];
	}

	hr = evr_data->d3d_video_context->VideoProcessorBlt(deint_datas->d3d_deinterlace_processor, deint_datas->d3d_deinterlace_outview, 0, 1, &stream);
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "VideoProcessorBlt failed %8.8X", hr);
		return false;
	}

	pds_memcpy(&texture_datas->d3d_input_shaderview_selector[0], &deint_datas->d3d_deinterlace_out_shaderview[0], sizeof(texture_datas->d3d_input_shaderview_selector));

	if(funcbit_test(mpxplay_video_render_infos.renderer_capabilities, DISPQT_VIDEORENDERER_CAPFLAG_DEINTERDUP))
	{
		if(deint_datas->d3d_deinterlace_frdup_pass)
		{
			deint_datas->d3d_deinterlace_frame_counter++;
			deint_datas->d3d_deinterlace_frdup_pass = 0;
		}
		else
		{
			deint_datas->d3d_deinterlace_frdup_pass++;
		}
		funcbit_enable(mpxplay_video_render_infos.renderer_capabilities, DISPQT_VIDEORENDERER_CAPENABLE_DEINTERDUP);
	}
	else
	{
		deint_datas->d3d_deinterlace_frame_counter++;
	}

	return true;
}

//============================================================================================================
// vertex shader

static void dispqt_d3d_shader_vertex_close(struct dispqt_evr_input_texture_data_s *texture_datas)
{
	SAFE_RELEASE(texture_datas->d3d_vertex_buffer);
	SAFE_RELEASE(texture_datas->d3d_vertex_shader_hnd);
	SAFE_RELEASE(texture_datas->d3d_vertex_input_layout);
}

static bool dispqt_d3d_shader_vertex_build(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas)
{
	const D3D11_INPUT_ELEMENT_DESC vertex_shader_layout[2] =
	{
		{"POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, D3D11_APPEND_ALIGNED_ELEMENT, D3D11_INPUT_PER_VERTEX_DATA, 0},
		{"TEXCOORD", 0, DXGI_FORMAT_R32G32_FLOAT,    0, D3D11_APPEND_ALIGNED_ELEMENT, D3D11_INPUT_PER_VERTEX_DATA, 0}
	};
	ID3DBlob *vertex_shader_blob = NULL;
	D3D11_SUBRESOURCE_DATA vertexInitData, *pInitData;
	D3D11_BUFFER_DESC BufferDesc;
	bool retVal = false;
	HRESULT hr;

	dispqt_d3d_shader_vertex_close(texture_datas);

	// build vertex shader
	vertex_shader_blob = dispqt_render_d3d11_shader_compile(evr_data, d3dVertexShaderFlat, "main", "vs");
	if(!vertex_shader_blob)
		goto err_out_build;

	hr = evr_data->d3d_device->CreateVertexShader(vertex_shader_blob->GetBufferPointer(), vertex_shader_blob->GetBufferSize(), nullptr, &texture_datas->d3d_vertex_shader_hnd);
	if (FAILED(hr))
		goto err_out_build;

	hr = evr_data->d3d_device->CreateInputLayout(vertex_shader_layout, 2, vertex_shader_blob->GetBufferPointer(), vertex_shader_blob->GetBufferSize(), &texture_datas->d3d_vertex_input_layout);
	if (FAILED(hr))
		goto err_out_build;

	// create vertex buffer
	pds_memset(&BufferDesc, 0, sizeof(BufferDesc));
	switch(texture_datas->texture_id)
	{
		case DISPQT_EVR_INPUTTEXUREID_VIDEO:
			BufferDesc.Usage = D3D11_USAGE_DYNAMIC;
			BufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
			pInitData = NULL;
			break;
		default:
			BufferDesc.Usage = D3D11_USAGE_DEFAULT;
			pds_memset(&vertexInitData, 0, sizeof(vertexInitData));
			vertexInitData.pSysMem = &vertices_full_texture_default[0];
			pInitData = &vertexInitData;
			break;
	}
	texture_datas->vertex_buffer_elems = DISPQT_EVR_D3D_NUMVERTICES_DEFAULT;
	BufferDesc.BindFlags = D3D11_BIND_VERTEX_BUFFER;
	BufferDesc.ByteWidth = sizeof(d3d_vertex_t) * texture_datas->vertex_buffer_elems;

	hr = evr_data->d3d_device->CreateBuffer(&BufferDesc, pInitData, &texture_datas->d3d_vertex_buffer);
	if (FAILED(hr))
		goto err_out_build;

	retVal = true;

err_out_build:
	SAFE_RELEASE(vertex_shader_blob);
	return retVal;
}

//------------------------------------------------------------------------------------------------------------
static void dispqt_d3d_shader_pixel_close(struct dispqt_evr_input_texture_data_s *texture_datas)
{
	SAFE_RELEASE(texture_datas->d3d_pixel_shader_hnd);
}

static bool dispqt_d3d_shader_pixel_constant_colortransform_prepare(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas,
		const struct WhitepointCoefficients *custom_whitepoints_src, const struct PrimaryCoefficients *custom_primary_coefs_src)
{
	struct dispqt_evr_format_desc_s *pixform_infos = texture_datas->pixel_format_infos;
	struct dispqt_evr_color_transform_matrixes_s color_transform_data;
	D3D11_MAPPED_SUBRESOURCE constBufMapRes;

	if(pixform_infos->is_rgb_format && (pixform_infos->nb_components == 1)) // we don't need color transform matrixes at simple RGB
		return true;

	mpxplay_dispqt_render_common_shader_pixel_constant_colortransform_calculate(&color_transform_data, pixform_infos, texture_datas->av_colorspace,
			texture_datas->av_colorrange, texture_datas->av_colorprimaries, texture_datas->texture_height, custom_whitepoints_src, custom_primary_coefs_src);

	pds_memset(&constBufMapRes, 0, sizeof(constBufMapRes));
	if(FAILED(evr_data->d3d_device_context->Map((ID3D11Resource *)texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_YUVTORGB], 0, D3D11_MAP_WRITE_DISCARD, 0, &constBufMapRes)))
	   return false;
	pds_memcpy(constBufMapRes.pData, (void *)&color_transform_data, sizeof(struct dispqt_evr_color_transform_matrixes_s));
	evr_data->d3d_device_context->Unmap((ID3D11Resource *)texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_YUVTORGB], 0);

	return true;
}

static void dispqt_d3d11_shader_pixel_assemble(struct dispqt_evr_format_desc_s *pixform_infos, int texture_id,
		struct mmc_dispqt_config_s *gcfg, int av_colorprimaries, int av_colortrc, char pscode_str_body[8192])
{
	char pscode_str_mainfunc[1024];
	HRESULT hr;

	pds_strcpy(pscode_str_body, (char *)dispqt_d3d_pscode_COMMON_datas);
	if(texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
		pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_CONSTANTDATA0_color_mixer);

	pds_strcpy(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_BEGIN);

	if(pixform_infos->is_rgb_format && (pixform_infos->nb_components == 1))
	{
		pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_GET_RGB_from_RGB_packed);
	}
	else
	{
		pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_CONSTANTDATA1_color_transform_YUV2RGB);
		pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_CONSTANTDATA2_runtime_dynamic);

		switch(pixform_infos->av_pixfmt)
		{
			case AV_PIX_FMT_NV12:
			case AV_PIX_FMT_P010LE:
			case AV_PIX_FMT_P016LE:
			case AV_PIX_FMT_NV24:
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_samplecv_NV12_P010);
				break;
			case AV_PIX_FMT_NV21:
			case AV_PIX_FMT_NV42:
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_samplecv_NV21);
				break;
			case AV_PIX_FMT_YUYV422:
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_samplecv_YUY2);
				break;
			default:
				pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_samplecv_GENERAL_3planes_anybits);
		}

		if(pixform_infos->is_rgb_format)
			pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_GET_RGB_from_RGB_planar);
		else
			pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_GET_RGB_from_YUV_planar);

		mpxplay_dispqt_render_common_shader_pixel_assemble_trc(pscode_str_body, pscode_str_mainfunc, gcfg, av_colorprimaries, av_colortrc);
	}

	if(texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
	{
		pds_strcat(pscode_str_body, (char *)dispqt_d3d_pscode_color_mixer);
		pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_CALL_color_mixer);
	}

	pds_strcat(pscode_str_mainfunc, (char *)dispqt_d3d_pscode_mainfunc_END);
	pds_strcat(pscode_str_body, pscode_str_mainfunc);

	mpxplay_debugf(DISPQT_DEBUGOUT_PIXSHADER,"PS: lenall:%d mainlen:%d \n%s", pds_strlen(pscode_str_body), pds_strlen(pscode_str_mainfunc), pscode_str_body);
}

static bool dispqt_d3d_shader_pixel_build(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas, struct mmc_dispqt_config_s *gcfg)
{
	ID3DBlob *pixel_shader_blob = NULL;
	char pscode_str_body[8192];
	bool retVal = false;
	HRESULT hr;

	dispqt_d3d_shader_pixel_close(texture_datas);

	dispqt_d3d11_shader_pixel_assemble(texture_datas->pixel_format_infos, texture_datas->texture_id, gcfg,
			texture_datas->av_colorprimaries, texture_datas->av_colortrc, pscode_str_body);

	// build pixel shader
	pixel_shader_blob = dispqt_render_d3d11_shader_compile(evr_data, (const char *)&pscode_str_body[0], "main", "ps");
	if(!pixel_shader_blob)
	{
		mpxplay_debugf(DISPQT_DEBUG_ERROR,"dispqt_d3d_shader_compile FAILED");
		goto err_out_build;
	}

	hr = evr_data->d3d_device->CreatePixelShader(pixel_shader_blob->GetBufferPointer(), pixel_shader_blob->GetBufferSize(), nullptr, &texture_datas->d3d_pixel_shader_hnd);
	if (FAILED(hr))
		goto err_out_build;

	retVal = true;

err_out_build:
	SAFE_RELEASE(pixel_shader_blob);
	return retVal;
}

//------------------------------------------------------------------------------------------------------------
// update luminance and color matrix by dynamic HDR metadata (if exists int the stream)

static void dispqt_d3d_shader_pixel_dynamic_hdr_update(struct dispqt_render_evr_data_s *evr_data, AVFrame *video_avframe, struct dispqt_video_surface_info_s *video_surface_infos, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	struct dispqt_render_evr_dynamic_hdr_data_s *dynamic_hdr_datas = &evr_data->dynamic_hdr_datas;
	int prev_input_luminance_value = dynamic_hdr_datas->hdr_input_luminance_value, output_luminance_value = 100; // SRGB luminance value
	struct dispqt_evr_dynamic_runtime_data_s dynamic_runtime_datas;
	bool new_custom_primaries_received;
	D3D11_MAPPED_SUBRESOURCE constBufMapRes;

	new_custom_primaries_received = mpxplay_dispqt_render_common_shader_pixel_dynamic_hdr_update(&evr_data->dynamic_hdr_datas, video_avframe, video_surface_infos, gcfg);

	if(dynamic_hdr_datas->hdr_input_luminance_value != prev_input_luminance_value)
	{
		pds_memset(&constBufMapRes, 0, sizeof(constBufMapRes));
		if(FAILED(evr_data->d3d_device_context->Map((ID3D11Resource *)texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_RUNTIME], 0, D3D11_MAP_WRITE_DISCARD, 0, &constBufMapRes)))
		   return;

		pds_memset(&dynamic_runtime_datas, 0, sizeof(dynamic_runtime_datas));
		dynamic_runtime_datas.HDRLuminanceValue = (float)output_luminance_value / (float)dynamic_hdr_datas->hdr_input_luminance_value;

		pds_memcpy(constBufMapRes.pData, (void *)&dynamic_runtime_datas, sizeof(struct dispqt_evr_dynamic_runtime_data_s));
		evr_data->d3d_device_context->Unmap((ID3D11Resource *)texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_RUNTIME], 0);
		mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"Luminance UPDATED: %1.2f", dynamic_runtime_datas.HDRLuminanceValue);
	}

	if(new_custom_primaries_received)
	{
		dispqt_d3d_shader_pixel_constant_colortransform_prepare(evr_data, texture_datas, &dynamic_hdr_datas->custom_wp_coefs, &dynamic_hdr_datas->custom_pri_coefs);
		dynamic_hdr_datas->is_custom_primaries_sent = TRUE;
		mpxplay_debugf(DISPQT_DEBUGOUT_LUMINANCE,"PRIMARIES UPDATED");
	}
}

//------------------------------------------------------------------------------------------------------------
// update color changes for pixel shader
static void dispqt_d3d_shader_pixel_colormixer_configure(struct dispqt_render_evr_data_s *evr_data)
{
	struct dispqt_evr_input_texture_data_s *texture_datas;
	struct dispqt_evr_color_mixer_s color_mixer_data;
	D3D11_MAPPED_SUBRESOURCE constBufMapRes;

	if(!evr_data->update_vmixer_data)
		return;
	evr_data->update_vmixer_data = false;

	mpxplay_dispqt_render_common_shader_pixel_colormixer_configure(&color_mixer_data, evr_data->vmixer_values);

	texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	pds_memset(&constBufMapRes, 0, sizeof(constBufMapRes));
	if(FAILED(evr_data->d3d_device_context->Map((ID3D11Resource *)texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_COLORMIXER], 0, D3D11_MAP_WRITE_DISCARD, 0, &constBufMapRes)))
	   return;
	pds_memcpy(constBufMapRes.pData, (void *)&color_mixer_data, sizeof(struct dispqt_evr_color_mixer_s));
	evr_data->d3d_device_context->Unmap((ID3D11Resource *)texture_datas->d3d_pixelshader_constbuffs[DISPQT_EVR_PIXSHADERCONST_COLORMIXER], 0);
}

//------------------------------------------------------------------------------------------------------------
static void dispqt_d3d_shaders_close(struct dispqt_render_evr_data_s *evr_data, unsigned int texture_id)
{
	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[texture_id];
	dispqt_d3d_shader_vertex_close(texture_datas);
	dispqt_d3d_shader_pixel_close(texture_datas);
	for(int i = 0; i < DISPQT_EVR_PIXSHADERCONST_NUM; i++)
	{
		SAFE_RELEASE(texture_datas->d3d_pixelshader_constbuffs[i]);
	}
}

static void dispqt_evr_shaders_select(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas)
{
	UINT Stride = sizeof(d3d_vertex_t), Offset = 0; // common format for all vertexes

	// set vertex shader
	evr_data->d3d_device_context->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST);
	evr_data->d3d_device_context->VSSetShader(texture_datas->d3d_vertex_shader_hnd, NULL, 0);
	evr_data->d3d_device_context->IASetInputLayout(texture_datas->d3d_vertex_input_layout);
	evr_data->d3d_device_context->IASetVertexBuffers(0, 1, &texture_datas->d3d_vertex_buffer, &Stride, &Offset);

	// set pixel shader
	evr_data->d3d_device_context->PSSetShader(texture_datas->d3d_pixel_shader_hnd, NULL, 0);
	if(texture_datas->d3d_pixelshader_constbuffs[0])
		evr_data->d3d_device_context->PSSetConstantBuffers(0, DISPQT_EVR_PIXSHADERCONST_NUM, &texture_datas->d3d_pixelshader_constbuffs[0]);
	evr_data->d3d_device_context->PSSetShaderResources(0, texture_datas->pixel_format_infos->nb_components, &texture_datas->d3d_input_shaderview_selector[0]);
}

//============================================================================================================

static bool dispqt_evr_input_texture_set_crop_and_rotate(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas)
{
	D3D11_MAPPED_SUBRESOURCE vertexBufMapRes;

	pds_memset(&vertexBufMapRes, 0, sizeof(vertexBufMapRes));
	if(FAILED(evr_data->d3d_device_context->Map((ID3D11Resource *)texture_datas->d3d_vertex_buffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &vertexBufMapRes)))
	   return false;

	mpxplay_dispqt_render_common_vertex_set_input_texture_crop_and_rotate(vertexBufMapRes.pData, &texture_datas->texture_crop,
			texture_datas->input_width, texture_datas->input_height, DISPQT_RENDER_D3D11_VERTEX_POSITION_SCALE);

	evr_data->clear_counter = DISPQT_EVR_BUFFERS_TO_CLEAR;

	evr_data->d3d_device_context->Unmap((ID3D11Resource *)texture_datas->d3d_vertex_buffer, 0);
	return true;
}

static void dispqt_evr_input_texture_set_viewport(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas)
{
	struct dispqt_evr_input_viewport_s *texture_viewport = &texture_datas->texture_viewport;
	D3D11_VIEWPORT viewport;

	viewport.TopLeftX = texture_viewport->pos_x;
	viewport.TopLeftY = texture_viewport->pos_y;
	viewport.Width = texture_viewport->width;
	viewport.Height = texture_viewport->height;
	viewport.MinDepth = 0;
	viewport.MaxDepth = 1;

	evr_data->d3d_device_context->RSSetViewports(1, &viewport);
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_evr_input_texture_render(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas)
{
	struct dispqt_evr_format_desc_s *format_datas = texture_datas->pixel_format_infos;
	dispqt_evr_shaders_select(evr_data, texture_datas);
	dispqt_evr_input_texture_set_viewport(evr_data, texture_datas);
	evr_data->d3d_device_context->Draw(texture_datas->vertex_buffer_elems, 0);
}

//============================================================================================================

// select initial input texture and subtitle format descriptors
static struct dispqt_evr_format_desc_s *dispqt_evr_input_select_default_video_frame_format(unsigned int texture_id)
{
	unsigned int select_format = (texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)? 6 : 3; // NV12 / BGRA
	return (struct dispqt_evr_format_desc_s *)&dispqt_evr_supported_pixel_formats[select_format];
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_evr_input_texture_deinit(struct dispqt_render_evr_data_s *evr_data, unsigned int texture_id)
{
	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[texture_id];
	pds_memset(&texture_datas->d3d_input_shaderview_selector, 0, sizeof(texture_datas->d3d_input_shaderview_selector));
	dispqt_evr_videoframe_download_deinit(evr_data);
	dispqt_evr_d3d11_videoprocessor_deinterlace_close(texture_datas);
	dispqt_evr_input_textures_with_shaderresourceview_clear(&texture_datas->d3d_input_texture_2d[0], &texture_datas->d3d_input_shader_resource_views[0]);
	pds_memset(&texture_datas->texture_crop, 0, sizeof(texture_datas->texture_crop));
}

static bool dispqt_evr_input_texture_init(struct dispqt_render_evr_data_s *evr_data, unsigned int texture_id, int frame_width, int frame_height,
		struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *av_inframe, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[texture_id];
	struct dispqt_evr_format_desc_s *format_datas = texture_datas->pixel_format_infos;
	bool resolution_change = false, pixfmt_change = (format_datas && texture_datas->d3d_pixel_shader_hnd)? false : true;
	bool colorpace_change = false, colormatrix_change = false;
	D3D11_USAGE texture_usage = D3D11_USAGE_DYNAMIC;
	D3D11_TEXTURE2D_DESC desc;

	if(!frame_width || !frame_height)
		return false;

	if(av_inframe && (texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO))
	{
		if(av_inframe->format != texture_datas->av_pixelformat)
			pixfmt_change = true;
		if( (av_inframe->colorspace != texture_datas->av_colorspace) || (av_inframe->color_trc != texture_datas->av_colortrc)
		 || ((evr_data->dynamic_hdr_datas.last_hdr_control_setting & DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV) != (gcfg->video_hdr_control & DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV))
		){
			colorpace_change = true;
		}
		if( (av_inframe->color_range != texture_datas->av_colorrange) || (av_inframe->color_primaries != texture_datas->av_colorprimaries)
		 || ((evr_data->dynamic_hdr_datas.last_hdr_control_setting & DISPQT_CONFIG_VIDEO_HDR_DYN_COLORS) != (gcfg->video_hdr_control & DISPQT_CONFIG_VIDEO_HDR_DYN_COLORS))
		 || (video_surface_infos->videoout_input_change && evr_data->dynamic_hdr_datas.is_custom_primaries_sent)
		){
			colormatrix_change = true;
		}

		if(av_inframe->format == AV_PIX_FMT_D3D11)
		{
			ID3D11Texture2D *d3d_input_texture = (ID3D11Texture2D *)av_inframe->data[0];
			if(!d3d_input_texture)
				return false;
			pds_memset(&desc, 0, sizeof(desc));
			d3d_input_texture->GetDesc(&desc);
			if(!format_datas || (format_datas->d3d_texture_format != desc.Format))
			{
				format_datas = mpxplay_dispqt_render_common_select_dxgi_format(&dispqt_evr_supported_pixel_formats[0], desc.Format);
				if(!format_datas) // ???
					return false;
				pixfmt_change = true;
			}
			texture_usage = D3D11_USAGE_DEFAULT;
		}
		else if(pixfmt_change)
		{
			format_datas = mpxplay_dispqt_render_common_input_select_color_format(&dispqt_evr_supported_pixel_formats[0], &texture_datas->custom_pixel_format_info, av_inframe->format, TRUE);
			if(!format_datas)
				format_datas = dispqt_evr_input_select_default_video_frame_format(texture_id);
		}
	}

	if( (texture_datas->video_width != frame_width) || (texture_datas->video_height != frame_height)
#ifdef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
	 || (!texture_datas->d3d_input_texture_2d[0] && (!av_inframe || (av_inframe->format != AV_PIX_FMT_D3D11)))
#else
	 || !texture_datas->d3d_input_texture_2d[0]
#endif
	){
		texture_datas->texture_width = frame_width & ~format_datas->align_mask_x;
		texture_datas->texture_height = frame_height & ~format_datas->align_mask_y;
		if((texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO) && av_inframe && (av_inframe->format != AV_PIX_FMT_D3D11))
			texture_datas->texture_width = (texture_datas->texture_width + 63) & ~63; // rounding up for SSE image copy
		resolution_change = true;
	}

	if(!resolution_change && !pixfmt_change && !colorpace_change && !colormatrix_change)
		return true;

	texture_datas->pixel_format_infos = format_datas;
	if(av_inframe)
	{
		texture_datas->av_pixelformat = av_inframe->format;
		texture_datas->av_colortrc = av_inframe->color_trc;
		texture_datas->av_colorrange = av_inframe->color_range;
		texture_datas->av_colorspace = av_inframe->colorspace;
		texture_datas->av_colorprimaries = av_inframe->color_primaries;
	}

	if(pixfmt_change || colorpace_change)
	{
		if(!dispqt_d3d_shader_pixel_build(evr_data, texture_datas, gcfg))
		{
			//d3d11_static_datas.d3d_forced_reopen = TRUE; // FIXME: workaround
			mpxplay_debugf(DISPQT_DEBUG_ERROR, "shader_pixel_build FAILED");
			goto err_out_init;
		}
		dispqt_d3d_shader_pixel_constant_colortransform_prepare(evr_data, texture_datas, NULL, NULL);
	}
	else if(colormatrix_change)
	{
		dispqt_d3d_shader_pixel_constant_colortransform_prepare(evr_data, texture_datas, NULL, NULL);
	}

	if(resolution_change || pixfmt_change)
	{
		dispqt_evr_input_texture_deinit(evr_data, texture_id);

		texture_datas->video_width = frame_width;
		texture_datas->video_height = frame_height;

#ifdef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
		if(texture_datas->av_pixelformat == AV_PIX_FMT_D3D11)
		{
			dispqt_evr_input_textures_with_shaderresourceview_clear(&texture_datas->d3d_input_texture_2d[0], &texture_datas->d3d_input_shader_resource_views[0]);
		}
		else
#endif
		if(!dispqt_evr_input_textures_with_shaderresourceview_alloc(evr_data->d3d_device, format_datas, texture_datas->texture_width, texture_datas->texture_height, &texture_datas->d3d_input_texture_2d[0], &texture_datas->d3d_input_shader_resource_views[0], texture_usage, 0))
		{
			goto err_out_init;
		}

		// default input settings belong to the d3d_input_texture_2d
		texture_datas->input_width = texture_datas->texture_width;
		texture_datas->input_height = texture_datas->texture_height;
		pds_memcpy(&texture_datas->d3d_input_shaderview_selector[0], &texture_datas->d3d_input_shader_resource_views[0], sizeof(texture_datas->d3d_input_shaderview_selector));

		if(texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
		{
			dispqt_evr_d3d11_videoprocessor_deinterlace_init(evr_data, texture_datas, video_surface_infos, av_inframe, gcfg);
			evr_data->clear_counter = DISPQT_EVR_BUFFERS_TO_CLEAR;
		}

		mpxplay_debugf(DISPQT_DEBUGOUT_TEXTURE, "input_texture_init OK w:%d h:%d p:%d", texture_datas->texture_width, texture_datas->texture_height, format_datas->av_pixfmt);
	}

	if(texture_id == DISPQT_EVR_INPUTTEXUREID_VIDEO)
		mpxplay_dispqt_render_common_shader_pixel_dynamic_hdr_reset(&evr_data->dynamic_hdr_datas, gcfg);

	return true;

err_out_init:
	dispqt_evr_input_texture_deinit(evr_data, texture_id);
	mpxplay_debugf(DISPQT_DEBUG_ERROR, "input_texture_init FAILED");
	return false;
}

//------------------------------------------------------------------------------------------------------------

static bool dispqt_evr_input_texture_map(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas)
{
	struct dispqt_evr_format_desc_s *format_datas = texture_datas->pixel_format_infos;
	pds_memset(&texture_datas->d3d_texture_mappedResource[0], 0, sizeof(texture_datas->d3d_texture_mappedResource));
	for(int i = 0; i < ((format_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)? format_datas->nb_components : 1); i++)
		if(FAILED(evr_data->d3d_device_context->Map((ID3D11Resource *)texture_datas->d3d_input_texture_2d[i], 0, D3D11_MAP_WRITE_DISCARD, 0, &texture_datas->d3d_texture_mappedResource[i])))
			return false;
	return true;
}

static void dispqt_evr_input_texture_unmap(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas)
{
	struct dispqt_evr_format_desc_s *format_datas = texture_datas->pixel_format_infos;
	for(int i = 0; i < ((format_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)? format_datas->nb_components : 1); i++)
		if(texture_datas->d3d_texture_mappedResource[i].pData)
			evr_data->d3d_device_context->Unmap((ID3D11Resource *)texture_datas->d3d_input_texture_2d[i], 0);
	pds_memset(&texture_datas->d3d_texture_mappedResource[0], 0, sizeof(texture_datas->d3d_texture_mappedResource));
}

static mpxp_bool_t dispqt_evr_input_texture_map_and_fill_video(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas,
		uint8_t *indata_ptrs[AV_NUM_DATA_POINTERS], int indata_linesizes[AV_NUM_DATA_POINTERS], unsigned int indata_lines)
{
	struct dispqt_evr_format_desc_s *format_datas = texture_datas->pixel_format_infos;
	struct dispqt_render_mapped_subresource_s texture_mapres[DISPQT_EVR_NB_MAX_PLANES];
	D3D11_TEXTURE2D_DESC d3dTextureDesc;
	mpxp_bool_t success = FALSE, failure = FALSE;

	if(!dispqt_evr_input_texture_map(evr_data, texture_datas))
		return success;

	pds_memset(&texture_mapres[0], 0, sizeof(texture_mapres));
	for(int i = 0; i < ((format_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)? format_datas->nb_components : 1); i++)
	{
		texture_mapres[i].vidmem_beginptr = texture_datas->d3d_texture_mappedResource[i].pData;
		texture_mapres[i].vidmem_linesize = texture_datas->d3d_texture_mappedResource[i].RowPitch;
		pds_memset(&d3dTextureDesc, 0, sizeof(d3dTextureDesc));
		texture_datas->d3d_input_texture_2d[i]->GetDesc(&d3dTextureDesc);
		if(!d3dTextureDesc.Height)
		{
			failure = TRUE;
			break;
		}
		texture_mapres[i].vidmem_linenum = d3dTextureDesc.Height;
	}
	if(!failure)
	{
		success = mpxplay_dispqt_videorender_common_videoframe_copy(texture_mapres, format_datas, indata_ptrs, indata_linesizes, indata_lines);
	}

	return success;
}

// for subtitle (PACKED)
static mpxp_bool_t dispqt_evr_input_texture_map_and_fill_subtitle(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas, uint8_t *indata_ptr, unsigned int indata_linesize, unsigned int indata_lines)
{
	if(!dispqt_evr_input_texture_map(evr_data, texture_datas))
		return FALSE;
	register const unsigned int linelen_in = indata_linesize;
	register const unsigned int linelen_out = texture_datas->d3d_texture_mappedResource[0].RowPitch;
	register const unsigned int copy_len = min(linelen_in, linelen_out);
	register uint8_t *indataptr = indata_ptr;
	register uint8_t *outdataptr = (uint8_t *)texture_datas->d3d_texture_mappedResource[0].pData;
	register unsigned int linecount = min(indata_lines, texture_datas->texture_height);
	do
	{
		pds_memcpy(outdataptr, indataptr, copy_len);
		outdataptr += linelen_out; indataptr += linelen_in;
	}while(--linecount);
	dispqt_evr_input_texture_unmap(evr_data, texture_datas);
	return TRUE;
}

//============================================================================================================
static void dispqt_evr_input_hwdec_data_clear(struct dispqt_render_evr_data_s *evr_data)
{
	struct dispqt_render_evr_hwdec_data_s *hwdec_datas = &evr_data->hwdec_datas;
	hwdec_datas->hwdec_reference_texture_2d = NULL;
	for(int i = 0; i < DISPQT_EVR_HWDEC_POOL_SIZE_MAX; i++)
		for(int j = 0; j < DISPQT_EVR_NB_MAX_PLANES; j++)
			SAFE_RELEASE(hwdec_datas->hwdec_shader_resource_views[i][j]);
	pds_memset(hwdec_datas, 0, sizeof(hwdec_datas));
}

// prepare shader resource views for the FFmpeg's ID3D11Texture2D array
static bool dispqt_evr_input_hwdec_texture_process(struct dispqt_render_evr_data_s *evr_data, struct dispqt_evr_input_texture_data_s *texture_datas,
		AVFrame *video_avframe, struct dispqt_video_surface_info_s *video_surface_infos, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_render_evr_hwdec_data_s *hwdec_datas = &evr_data->hwdec_datas;
	ID3D11Texture2D *hwdec_texture_array = (ID3D11Texture2D *)video_avframe->data[0];
	int array_index = (intptr_t)video_avframe->data[1];
	bool resource_error = false;
#ifdef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
	bool deinterlace_process_follows = mpxplay_dispqt_render_common_deinterlace_condition(video_avframe, gcfg);
#else
	bool deinterlace_or_autocrop_process_follows = (mpxplay_dispqt_render_common_deinterlace_condition(video_avframe, gcfg) || ((gcfg->video_crop_type == DISPQT_VIDEOSCRCROPTYPE_AUTO) && (video_avframe->height <= gcfg->video_autocrop_limit)));
	bool do_copy = true;
#endif

	if(!hwdec_texture_array)
		return false;

	if( (array_index < DISPQT_EVR_HWDEC_POOL_SIZE_MAX)
#ifndef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
	 && !deinterlace_or_autocrop_process_follows)
#endif
	){ // note: using/adding shader resource view to textures in all cases avoids mem leak on some bogus drivers (at seek with d3d11 deinterlacing)
		D3D11_TEXTURE2D_DESC desc;

		pds_memset(&desc, 0, sizeof(desc));
		hwdec_texture_array->GetDesc(&desc);

		if( video_surface_infos->videoout_input_change || (hwdec_datas->hwdec_reference_texture_2d != hwdec_texture_array) || (hwdec_datas->hwdec_tex_arraysize != desc.ArraySize)
		 || (hwdec_datas->hwdec_tex_width != desc.Width) || (hwdec_datas->hwdec_tex_height != desc.Height) || (hwdec_datas->hwdec_tex_format != desc.Format)
		){
			struct dispqt_evr_format_desc_s *pixel_form = mpxplay_dispqt_render_common_select_dxgi_format(&dispqt_evr_supported_pixel_formats[0], desc.Format);

			dispqt_evr_input_hwdec_data_clear(evr_data);

			if(!pixel_form)
			{
				resource_error = true;
			}
			else
			{
				for(int i = 0; (i < pixel_form->nb_components) && !resource_error; i++)
				{
					for(int j = 0; j < desc.ArraySize; j++)
					{
						if(!dispqt_evr_texture2darray_shaderresourceview_create(evr_data->d3d_device, hwdec_texture_array, (DXGI_FORMAT)pixel_form->shader_view_formats[i], &hwdec_datas->hwdec_shader_resource_views[j][i], j))
						{
							resource_error = true;
							break;
						}
					}
				}
			}

			if(!resource_error)
			{
				hwdec_datas->hwdec_reference_texture_2d = hwdec_texture_array;
				hwdec_datas->hwdec_tex_arraysize = desc.ArraySize;
				hwdec_datas->hwdec_tex_width = desc.Width;
				hwdec_datas->hwdec_tex_height = desc.Height;
				hwdec_datas->hwdec_tex_format = desc.Format;
				evr_data->clear_counter = DISPQT_EVR_BUFFERS_TO_CLEAR;
			}
		}

		if(!resource_error)
		{
			pds_memcpy(&texture_datas->d3d_input_shaderview_selector[0], &hwdec_datas->hwdec_shader_resource_views[array_index][0], sizeof(texture_datas->d3d_input_shaderview_selector));
#ifdef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
			if(!deinterlace_process_follows)
#endif
			{
				texture_datas->input_width = hwdec_datas->hwdec_tex_width;
				texture_datas->input_height = hwdec_datas->hwdec_tex_height;
			}
#ifndef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
			do_copy = false;
#endif
		}
	}

#ifndef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
	if(do_copy)
	{
		const D3D11_BOX box = {0, 0, 0, (UINT)video_avframe->width, (UINT)video_avframe->height, 1}; // needed to avoid a bottom green line at rendering (due to the float based rectangle)
		evr_data->d3d_device_context->CopySubresourceRegion(texture_datas->d3d_input_texture_2d[0], 0, 0,0,0, hwdec_texture_array, array_index, &box);
	}
#endif

	return true;
}

//============================================================================================================
// download and analyze hw decoded or pool buffer video frame (for autocrop)

static void dispqt_evr_videoframe_download_deinit(struct dispqt_render_evr_data_s *evr_data)
{
	struct dispqt_render_evr_textdown_data_s *textdown_datas = &evr_data->textdown_datas;
	SAFE_RELEASE(textdown_datas->d3d_texture_video_download);
	textdown_datas->down_tex_width = textdown_datas->down_tex_height = 0;
	textdown_datas->down_text_format = DXGI_FORMAT_UNKNOWN;
}

static bool dispqt_evr_videoframe_download_init(struct dispqt_render_evr_data_s *evr_data, struct dispqt_video_source_info_s *video_source_infos,
		struct dispqt_evr_input_texture_data_s *texture_datas, struct dispqt_render_evr_textdown_data_s *textdown_datas, DXGI_FORMAT format)
{
	dispqt_evr_videoframe_download_deinit(evr_data);

	if(!dispqt_evr_texture2d_create(evr_data->d3d_device, texture_datas->texture_width, texture_datas->texture_height, format, &textdown_datas->d3d_texture_video_download, D3D11_USAGE_STAGING, 0))
		return false;

	textdown_datas->down_tex_width = texture_datas->texture_width;
	textdown_datas->down_tex_height = texture_datas->texture_height;
	textdown_datas->down_text_format = format;

	return true;
}

static void dispqt_evr_videoframe_autocrop_calculate(struct dispqt_render_evr_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos,
		AVFrame *video_avframe, struct mmc_dispqt_config_s *gcfg)
{
	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	struct dispqt_video_source_info_s *video_source_infos = video_surface_infos->video_source_infos;
	struct dispqt_render_evr_textdown_data_s *textdown_datas = &evr_data->textdown_datas;
	mpxp_uint64_t curr_time_ms;

	if( (gcfg->video_crop_type != DISPQT_VIDEOSCRCROPTYPE_AUTO) || (texture_datas->video_height > gcfg->video_autocrop_limit)
	 || ((video_avframe->format != AV_PIX_FMT_D3D11) && !mpxplay_video_render_infos.render_function_poolbufelem_validity_check(video_avframe))
	){
		return;
	}

	if(video_surface_infos->videoout_input_change || video_surface_infos->videoout_surface_reset)
		textdown_datas->last_process_time_ms = 0;

	curr_time_ms = pds_gettimem();
	if(curr_time_ms >= (textdown_datas->last_process_time_ms + DISPQT_EVR_AUTOCROP_MIN_PROCESS_INTERVAL_MS))
	{
		struct dispqt_evr_format_desc_s *pixform_datas = texture_datas->pixel_format_infos;
		const DXGI_FORMAT format = (pixform_datas->d3d_texture_format == DXGI_FORMAT_UNKNOWN)?  (DXGI_FORMAT)pixform_datas->shader_view_formats[0] : (DXGI_FORMAT)pixform_datas->d3d_texture_format;
		D3D11_MAPPED_SUBRESOURCE mapRes;

		if((textdown_datas->down_tex_width != texture_datas->texture_width) || (textdown_datas->down_tex_height != texture_datas->texture_height) || (textdown_datas->down_text_format != format))
			if(!dispqt_evr_videoframe_download_init(evr_data, video_source_infos, texture_datas, textdown_datas, format))
				return;

#ifdef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
		if(texture_datas->av_pixelformat == AV_PIX_FMT_D3D11)
		{
			ID3D11Texture2D *hwdec_texture_array = (ID3D11Texture2D *)video_avframe->data[0];
			const int array_index = (intptr_t)video_avframe->data[1];
			const D3D11_BOX box = {0, 0, 0, (UINT)video_avframe->width, (UINT)video_avframe->height, 1}; // needed to avoid a bottom green line at rendering (due to the float based rectangle)
			evr_data->d3d_device_context->CopySubresourceRegion((ID3D11Resource *)textdown_datas->d3d_texture_video_download, 0, 0,0,0, hwdec_texture_array, array_index, &box);
		}
		else
#endif
		{
			evr_data->d3d_device_context->CopyResource((ID3D11Resource *)textdown_datas->d3d_texture_video_download, (ID3D11Resource *)texture_datas->d3d_input_texture_2d[0]);
		}
		pds_memset(&mapRes, 0, sizeof(mapRes));
		if(FAILED(evr_data->d3d_device_context->Map((ID3D11Resource *)textdown_datas->d3d_texture_video_download, 0, D3D11_MAP_READ, 0, &mapRes)))
		   return;

		mpxplay_dispqt_videotrans_autocrop_process((mpxp_uint8_t *)mapRes.pData, mapRes.RowPitch, pixform_datas->av_pixfmt,
				texture_datas->video_width, texture_datas->video_height, mapRes.RowPitch / textdown_datas->down_tex_width,
				video_source_infos, &video_surface_infos->videoout_surface_clearpic);

		video_source_infos->sws_autoxcrop_framecounter--; // PROCESS_INTERVAL timing correction

		evr_data->d3d_device_context->Unmap((ID3D11Resource *)textdown_datas->d3d_texture_video_download, 0);
		textdown_datas->last_process_time_ms = curr_time_ms;
		mpxplay_debugf(DISPQT_DEBUGOUT_AUTOCROP, "AUTOCROP do");
	}

	mpxplay_dispqt_videotrans_videooutput_surface_window_calculate(video_source_infos, video_surface_infos, gcfg);
}

//============================================================================================================

static void dispqt_evr_videoframe_input_output_resize(struct dispqt_render_evr_data_s *evr_data, struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	struct dispqt_evr_input_viewport_s *video_input_viewport = &texture_datas->texture_crop;
	struct dispqt_evr_input_viewport_s *video_output_viewport = &texture_datas->texture_viewport;

	if( (video_input_viewport->pos_x != video_surface_infos->sws_crop_x) || (video_input_viewport->pos_y != video_surface_infos->sws_crop_y)
	 || (video_input_viewport->width != video_surface_infos->sws_crop_w) || (video_input_viewport->height != video_surface_infos->sws_crop_h)
	 || (video_input_viewport->text_width != texture_datas->input_width) || (video_input_viewport->text_height != texture_datas->input_height)
	 || (video_input_viewport->rotation_type != video_surface_infos->rotation_type)
	){
		video_input_viewport->pos_x = video_surface_infos->sws_crop_x;
		video_input_viewport->pos_y = video_surface_infos->sws_crop_y;
		video_input_viewport->width = video_surface_infos->sws_crop_w;
		video_input_viewport->height = video_surface_infos->sws_crop_h;
		video_input_viewport->text_width = texture_datas->input_width;
		video_input_viewport->text_height = texture_datas->input_height;
		video_input_viewport->rotation_type = video_surface_infos->rotation_type;

		dispqt_evr_input_texture_set_crop_and_rotate(evr_data, texture_datas);
	}

	if( (video_output_viewport->pos_x != video_surface_infos->sws_dst_pos_x) || (video_output_viewport->pos_y != video_surface_infos->sws_dst_pos_y)
	 || (video_output_viewport->width != video_surface_infos->sws_dst_width) || (video_output_viewport->height != video_surface_infos->sws_dst_height)
	){
		video_output_viewport->pos_x = video_surface_infos->sws_dst_pos_x;
		video_output_viewport->pos_y = video_surface_infos->sws_dst_pos_y;
		video_output_viewport->width = video_surface_infos->sws_dst_width;
		video_output_viewport->height = video_surface_infos->sws_dst_height;

		dispqt_evr_input_texture_set_viewport(evr_data, texture_datas);
	}
}

static mpxp_uint32_t mpxplay_dispqt_evr_get_frame_delay(void *evr_opaque_data)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)evr_opaque_data;
	mpxp_uint32_t render_delay = 0;

	if(!evr_data || !evr_data->d3d_device)
		return render_delay;

	render_delay = evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO].deinterlace_datas.d3d_deinterlace_frame_counter;
	if(render_delay >= DISPQT_EVR_DEINTERLACE_REF_FRAMES)
		render_delay = (DISPQT_EVR_DEINTERLACE_REF_FRAMES - 1);

	return render_delay;
}

static void dispqt_evr_do_swapchain_present(struct dispqt_render_evr_data_s *evr_data, mpxp_uint32_t sync_interval)
{
	DXGI_PRESENT_PARAMETERS presentParams;
	HRESULT hr = -1;

	if(evr_data->swap_chain_sdr1)
	{
		hr = evr_data->swap_chain_sdr1->Present(sync_interval, 0);
	}
	else if(evr_data->swap_chain_sdr2)
	{
		pds_memset(&presentParams, 0, sizeof(presentParams));
		hr = evr_data->swap_chain_sdr2->Present1(sync_interval, 0, &presentParams);
	}
	if(FAILED(hr))
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "dispqt_evr_do_swapchain_present FAILED");
	}
}

//------------------------------------------------------------------------------------------------------------

static void dispqt_evr_deinit_all(struct dispqt_render_evr_data_s *evr_data)
{
	if(!evr_data || !evr_data->d3d_device || !evr_data->d3d_device_context)
		return;
#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
	mpxplay_dispqt_videorender_d3d11_flush(TRUE);
#endif
	evr_data->d3d_device_context->ClearState();
#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
	mpxplay_dispqt_videorender_d3d11_flush(TRUE);
#endif
	mpxplay_dispqt_videorender_common_poolbuf_elems_close(&evr_data->decoder_pool_datas[0]);
	mpxplay_debugf(DISPQT_DEBUG_WARNING, "dispqt_evr_deinit_thread device done");
	dispqt_evr_swapchain_deinit(evr_data);
	dispqt_evr_d3d11_alpha_blend_clear(evr_data);
	dispqt_evr_input_hwdec_data_clear(evr_data);
	for(int i = 0; i < DISPQT_EVR_INPUTTEXUREID_NUM; i++)
	{
		dispqt_evr_input_texture_deinit(evr_data, i);
		dispqt_d3d_shaders_close(evr_data, i);
	}
	mpxplay_debugf(DISPQT_DEBUG_WARNING, "dispqt_evr_deinit_thread textures done");
	SAFE_RELEASE(evr_data->swap_chain_sdr1);
	SAFE_RELEASE(evr_data->swap_chain_sdr2);
	SAFE_RELEASE(evr_data->dxgi_factory1);
	SAFE_RELEASE(evr_data->dxgi_factory2);
	mpxplay_debugf(DISPQT_DEBUG_WARNING, "dispqt_evr_deinit_thread swapchan done");
	SAFE_RELEASE(evr_data->d3d_video_context);
	SAFE_RELEASE(evr_data->d3d_video_device);
	SAFE_RELEASE(evr_data->d3d_device_context);
	SAFE_RELEASE(evr_data->d3d_device);
	mpxplay_debugf(DISPQT_DEBUG_WARNING, "dispqt_evr_deinit_thread device done");
#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
	if(d3d11_static_datas.d3d_multi_thread)
		d3d11_static_datas.d3d_multi_thread->Enter();
#endif
	if(evr_data->d3d_api_funcs.d3dcompiler_dll_hnd)
		FreeLibrary(evr_data->d3d_api_funcs.d3dcompiler_dll_hnd);
	pds_memset(evr_data, 0, sizeof(*evr_data));
#ifdef DISPQT_VIDEO_DECODER_USE_GPU_FLUSH
	if(d3d11_static_datas.d3d_multi_thread)
		d3d11_static_datas.d3d_multi_thread->Leave();
#endif
	mpxplay_debugf(DISPQT_DEBUG_WARNING, "dispqt_evr_deinit_thread all done");
}

//------------------------------------------------------------------------------------------------------------

bool MMCDispQtVideoRendererD3D11::videorenderer_d3d11_textures_load(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
	struct dispqt_evr_input_texture_data_s *video_texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	AVFrame *video_avframe = video_surface_infos->ffmpeg_output_frame;
	int poolbuf_result = 0;

	if(!evr_data->d3d_device_context)
		return false;
#ifndef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
	if(!video_texture_datas->d3d_input_texture_2d[0])
		return false;
#endif
	if(!video_avframe || !video_avframe->data[0] || !video_avframe->height)
		return false;

	// default settings belong to the d3d_input_texture_2d
	video_texture_datas->input_width = video_texture_datas->texture_width;
	video_texture_datas->input_height = video_texture_datas->texture_height;
	pds_memcpy(&video_texture_datas->d3d_input_shaderview_selector[0], &video_texture_datas->d3d_input_shader_resource_views[0], sizeof(video_texture_datas->d3d_input_shaderview_selector));

	if(video_texture_datas->av_pixelformat == AV_PIX_FMT_D3D11) // got a frame in a texture from the GPU
	{
		if(!dispqt_evr_input_hwdec_texture_process(evr_data, video_texture_datas, video_avframe, video_surface_infos, this->main_window->gui_config))
			return false; // got an wrong texture
	}
	else if(
#ifdef MPXPLAY_DISPQT_D3D11_HWDEC_DIRECTCOPY
	        !video_texture_datas->d3d_input_texture_2d[0] ||
#endif
	        (poolbuf_result = dispqt_evr_poolbufelem_check_and_select(evr_data->d3d_device_context, video_texture_datas, video_avframe, video_surface_infos)) < 0)
	{	// got something invalid frame (pool elem, but processed or closed already)
		return false;
	}
	else if(poolbuf_result == 0) // got a frame in normal memory, copy video frame into d3d texture memory
	{
		dispqt_evr_input_texture_map_and_fill_video(evr_data, video_texture_datas, video_avframe->data, video_avframe->linesize, video_avframe->height);
	} // else got a direct rendered (sw decoded) frame from the pool (poolbuf_result = 1)

	// calculate autocrop at zero memcopy (at AV_PIX_FMT_D3D11)
	dispqt_evr_videoframe_autocrop_calculate(evr_data, video_surface_infos, video_avframe, this->main_window->gui_config);

	// apply crop and rotate on texture
	dispqt_evr_videoframe_input_output_resize(evr_data, video_surface_infos);

	// draw subtitle to memory (if there's a subtitle)
	evr_data->subtitle_rendermode = this->renderbase_subtitle_qpaint(video_surface_infos, subtitle_infos, 0);
	if(funcbit_test(evr_data->subtitle_rendermode, (DISPQT_RENDERBASE_SUBPAINT_RETVAL_CLEAR | DISPQT_RENDERBASE_SUBPAINT_RETVAL_NEWRENDER)))
		evr_data->clear_counter = DISPQT_EVR_BUFFERS_TO_CLEAR; // alpha blending requires full clear at subtitle changes

	// close video texture
	dispqt_evr_input_texture_unmap(evr_data, video_texture_datas);

	// load subtitle texture (if there's a subtitle)
	this->videorenderer_d3d11_texture_subtitle_load();

	return true;
}

void MMCDispQtVideoRendererD3D11::videorenderer_d3d11_texture_subtitle_load(void)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;

	if(evr_data->subtitle_rendermode < DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER)
		return;
	if(!dispqt_evr_input_texture_init(evr_data, DISPQT_EVR_INPUTTEXUREID_SUBTITLE, this->get_subtitle_pic_width(), this->get_subtitle_pic_height(), NULL, NULL, NULL))
		return;

	struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_SUBTITLE];

	if(!funcbit_test(evr_data->subtitle_rendermode, DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER))
	{
		if(!dispqt_evr_input_texture_map_and_fill_subtitle(evr_data, texture_datas, this->get_subtitle_image_memory_ptr(), this->get_subtitle_image_memory_linesize(), this->get_subtitle_pic_height()))
			return;
		mpxplay_debugf(DISPQT_DEBUGOUT_SUBTITLE, "SUBTITLE RENDER x:%d y:%d w:%d h:%d ls:%d ptr:%8.8X", this->get_subtitle_pic_globalpos_x(), this->get_subtitle_pic_globalpos_y(),
				this->get_subtitle_pic_width(), this->get_subtitle_pic_height(), this->get_subtitle_image_memory_linesize(), (mpxp_ptrsize_t)this->get_subtitle_image_memory_ptr());
	}

	struct dispqt_evr_input_viewport_s *texture_viewport = &texture_datas->texture_viewport;
	texture_viewport->pos_x = this->get_subtitle_pic_globalpos_x();
	texture_viewport->pos_y = this->get_subtitle_pic_globalpos_y();
	texture_viewport->width = this->get_subtitle_pic_width();
	texture_viewport->height = this->get_subtitle_pic_height();
}

void MMCDispQtVideoRendererD3D11::videorenderer_d3d11_textures_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
	struct dispqt_evr_input_texture_data_s *video_texture_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO];
	AVFrame *video_avframe = video_surface_infos->ffmpeg_output_frame;

	// set dynamic hdr (if there's any info in the frame)
	dispqt_d3d_shader_pixel_dynamic_hdr_update(evr_data, video_avframe, video_surface_infos, this->main_window->gui_config);

	// de-interlace input video texture
	dispqt_evr_d3d11_videoprocessor_deinterlace_render(evr_data, video_texture_datas, video_surface_infos, video_avframe, this->main_window->gui_config);

	// render input video texture to output swapchain (apply pixel and vertex shaders)
	dispqt_evr_input_texture_render(evr_data, video_texture_datas);

	// render subtitle texture (if there's any)
	if(evr_data->subtitle_rendermode >= DISPQT_RENDERBASE_SUBPAINT_RETVAL_OLDRENDER)
	{
		if(!evr_data->d3d_blend_state)
			dispqt_evr_d3d11_alpha_blend_create(evr_data);
		dispqt_evr_input_texture_render(evr_data, &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_SUBTITLE]);
	}
}

void MMCDispQtVideoRendererD3D11::videorenderer_d3d11_textures_present(struct dispqt_video_surface_info_s *video_surface_infos,
		ffmpegvideo_subtitle_info_s *subtitle_infos, bool texture_load_ok)
{
#ifdef MPXPLAY_USE_DEBUGF
	mpxp_uint64_t textures_render_begin_ms = pds_gettimem(), present_time_begin_ms;
	int render_present_gap = 0;
#endif
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;

	if(evr_data->d3d_render_target_view)
		evr_data->d3d_device_context->OMSetRenderTargets(1, &evr_data->d3d_render_target_view, NULL);

	if(evr_data->clear_counter)
	{
		evr_data->clear_counter--;
		dispqt_evr_clear_output_pic(evr_data);
	}

	if(texture_load_ok)
		this->videorenderer_d3d11_textures_render(video_surface_infos, subtitle_infos);

#ifdef MPXPLAY_USE_DEBUGF
	present_time_begin_ms = pds_gettimem();
#endif

	dispqt_evr_do_swapchain_present(evr_data, 0);

#ifdef MPXPLAY_USE_DEBUGF
	if(video_surface_infos->ffmpeg_output_frame)
	{
		mpxp_uint64_t present_time_end_ms = pds_gettimem();
		mpxp_uint64_t render_len = present_time_begin_ms - textures_render_begin_ms;
		mpxp_uint64_t present_len = present_time_end_ms - present_time_begin_ms;
		//if((render_len > 10) || (present_len > 10))
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "PRESENT END rl:%2d pl:%d pg:%d",
					(int)render_len, (int)present_len, render_present_gap);//, maxWait);
		}
	}
#endif
}

// duplicate the previous frame in interlaced mode
bool MMCDispQtVideoRendererD3D11::videorenderer_d3d11_deinterlace_2ndpass(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
	struct dispqt_evr_deinterlace_data_s *deint_datas = &evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO].deinterlace_datas;
	bool rendered2ndpass = false;

	if(deint_datas->d3d_deinterlace_frdup_pass)
	{
//		mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "DEINT2 ps:%d rf:%d vifc:%d vsrc:%d", deint_datas->d3d_deinterlace_frdup_pass,
//				((funcbit_test(video_surface_infos->render_flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_DEINT2NDPASS))? 1 : 0),
//				(int)video_surface_infos->videoout_inputframe_change, (int)video_surface_infos->videoout_surface_recalc);
		if( funcbit_test(video_surface_infos->scheduler_flags, DISPQT_FFMPGVIDEOCALLBACKFLAG_RF_DEINT2NDPASS)
		 && !video_surface_infos->videoout_inputframe_change && !video_surface_infos->videoout_surface_recalc
		){
			this->videorenderer_d3d11_textures_present(video_surface_infos, subtitle_infos, true);
			rendered2ndpass = true;
			mpxplay_debugf(DISPQT_DEBUGOUT_DEINT, "DEINT pass2 END");
		}
		else if(video_surface_infos->videoout_inputframe_change)
		{
			deint_datas->d3d_deinterlace_frame_counter++;
		}
		deint_datas->d3d_deinterlace_frdup_pass = 0;
	}

	return rendered2ndpass;
}

//------------------------------------------------------------------------------------------------------------

MMCDispQtVideoRendererD3D11::MMCDispQtVideoRendererD3D11(MainWindow *mainwindow, QWidget *parent) : MMCDispQtVideoRendererBase(mainwindow, parent)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)pds_calloc(1, sizeof(struct dispqt_render_evr_data_s));
	if(!evr_data)
		return;

	this->private_data = (void *)evr_data;
	this->parent_widget = parent;
}

MMCDispQtVideoRendererD3D11::~MMCDispQtVideoRendererD3D11()
{
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	mpxplay_video_render_infos.d3d_render_handler = NULL;
	if(this->private_data)
	{
		struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
		this->private_data = NULL;
		dispqt_evr_deinit_all(evr_data);
		pds_free(evr_data);
	}
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
}

bool MMCDispQtVideoRendererD3D11::videorenderer_init(struct dispqt_video_surface_info_s *video_surface_infos)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	int mutex_error;
#endif
	if(!evr_data)
		return false;

	mpxplay_debugf(DISPQT_DEBUG_WARNING, "videorenderer_init BEGIN");

#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT / 50);
#endif

	evr_data->parent_window_handler = (HWND)this->parent_widget->winId();

	if(!dispqt_evr_d3d11_init(evr_data))
		goto err_out_init;
	if(!dispqt_evr_d3d11_native_format_support_check(evr_data))
		goto err_out_init;
	if(!dispqt_evr_swapchain_create(evr_data, video_surface_infos))
		goto err_out_init;
	if(!dispqt_evr_d3d11_constant_buffers_init(evr_data))
		goto err_out_init;
	if(!dispqt_evr_d3d11_sampler_state_create(evr_data))
		goto err_out_init;

	for(int i = 0; i < DISPQT_EVR_INPUTTEXUREID_NUM; i++)
	{
		struct dispqt_evr_input_texture_data_s *texture_datas = &evr_data->input_texture_datas[i];
		texture_datas->texture_id = i;
		if(i == DISPQT_EVR_INPUTTEXUREID_SUBTITLE)
			texture_datas->pixel_format_infos = mpxplay_dispqt_render_common_select_best_subtitle_format(&dispqt_evr_supported_pixel_formats[0]);
		if(!texture_datas->pixel_format_infos)
			texture_datas->pixel_format_infos = dispqt_evr_input_select_default_video_frame_format(i);
		if(!dispqt_d3d_shader_vertex_build(evr_data, texture_datas))
			goto err_out_init;
	}

	evr_data->update_vmixer_data = true;
	mpxplay_video_render_infos.d3d_render_handler = (void *)evr_data;
	mpxplay_video_render_infos.hwdevice_avpixfmt = AV_PIX_FMT_D3D11;
	mpxplay_video_render_infos.render_function_check_videodecoder_inputformat = mpxplay_dispqt_videorenderer_d3d11_check_videodecoder_inputformat;
	mpxplay_video_render_infos.render_function_get_frame_delay = mpxplay_dispqt_evr_get_frame_delay;
	dispqt_d3d11_poolbuf_callbacks_assign();
	mpxplay_dispqt_videorender_common_poolbuf_functions_assign();
	mpxplay_video_render_infos.renderer_private_data = (void *)evr_data;

	mpxplay_video_render_infos.renderer_capabilities = DISPQT_VIDEORENDERER_CAPFLAG_DIRECTRENDER | DISPQT_VIDEORENDERER_CAPFLAG_DEINTERLACE
			| DISPQT_VIDEORENDERER_CAPFLAG_ROTATE | DISPQT_VIDEORENDERER_CAPFLAG_COLORMIXER;

#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif

	mpxplay_debugf(DISPQT_DEBUG_WARNING, "videorenderer_init END OK");

	return true;

err_out_init:
	dispqt_evr_deinit_all(evr_data);
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
	mpxplay_debugf(DISPQT_DEBUG_WARNING, "videorenderer_init END FAILED");
	return false;
}

void MMCDispQtVideoRendererD3D11::videorenderer_get_surface_attribs(struct dispqt_video_surface_info_s *video_surface_infos, AVFrame *decoded_frame)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
	struct dispqt_evr_format_desc_s *selected_new_format = NULL;
	struct dispqt_evr_format_desc_s format_info_tmp;

	if(decoded_frame)
	{
		pds_memset(&format_info_tmp, 0, sizeof(format_info_tmp));
		selected_new_format = mpxplay_dispqt_render_common_input_select_color_format(&dispqt_evr_supported_pixel_formats[0], &format_info_tmp, decoded_frame->format, TRUE);
		if(selected_new_format)
			video_surface_infos->videoout_surface_pix_fmt = (AVPixelFormat)selected_new_format->av_pixfmt;
	}

	if(!evr_data || !evr_data->d3d_device)
		return;

	if(!selected_new_format)
		video_surface_infos->videoout_surface_pix_fmt = (AVPixelFormat)evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO].av_pixelformat;
}

void MMCDispQtVideoRendererD3D11::videorenderer_set_mixer(unsigned int u_id, int value)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
	if(!evr_data || (u_id >= DISPQT_VIDEO_VMIXERTYPE_MAX))
		return;
	if(evr_data->vmixer_values[u_id] != value)
	{
		evr_data->vmixer_values[u_id] = value;
		evr_data->update_vmixer_data = true;
	}
	mpxplay_debugf(DISPQT_DEBUGOUT_MIXER,"videorenderer_set_mixer m:%d v:%d", u_id, value);
}

void MMCDispQtVideoRendererD3D11::videorenderer_set_fullscreen(mpxp_bool_t full)
{
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	const int mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	if(mutex_error != MPXPLAY_ERROR_OK)
		return;
#endif
	if(evr_data)
	{
		evr_data->swapchain_width = 2; // to surely update swapchan rectangle
		//mpxplay_dispqt_videorender_d3d11_async_end();
		if(evr_data->swap_chain_sdr1)
			evr_data->swap_chain_sdr1->SetFullscreenState(full, NULL);
		else if(evr_data->swap_chain_sdr2)
			evr_data->swap_chain_sdr2->SetFullscreenState(full, NULL);
		//mpxplay_dispqt_videorender_d3d11_async_end();
	}
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
	PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif
	pds_threads_sleep(0);
}

void MMCDispQtVideoRendererD3D11::videorenderer_videoframe_render(struct dispqt_video_surface_info_s *video_surface_infos, ffmpegvideo_subtitle_info_s *subtitle_infos)
{
	struct dispqt_render_evr_data_s *evr_data = (struct dispqt_render_evr_data_s *)this->private_data;
	AVFrame *video_avframe = video_surface_infos->ffmpeg_output_frame;
	bool texture_load_ok;
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	int mutex_error;
#endif
#ifdef MPXPLAY_USE_DEBUGF
	mpxp_uint64_t render_time_begin_ms = pds_gettimem(), render_time_mutex_lock_end_ms, render_time_texture_init_end_ms;
	mpxp_uint64_t render_time_texture_load_end_ms, render_time_shader_end_ms, render_time_swapchain_end_ms;
	int render_present_gap = 0;
#endif

	if(video_avframe)
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "RENDER START sx:%d sy:%d sw:%d sh:%d dx:%3d dy:%3d dw:%3d dh:%3d pix:%d", video_surface_infos->sws_crop_x, video_surface_infos->sws_crop_y,
			video_surface_infos->sws_crop_w, video_surface_infos->sws_crop_h, video_surface_infos->sws_dst_pos_x, video_surface_infos->sws_dst_pos_y,
			video_surface_infos->sws_dst_width, video_surface_infos->sws_dst_height, ((video_avframe)? video_avframe->format : -1));
	}

	if(!evr_data)
		return;

#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	mutex_error = PDS_THREADS_MUTEX_LOCK(&mpxplay_video_render_infos.d3d_device_mutex, DISPQT_VIDEO_MUTEX_TIMEOUT);
#endif

#ifdef DISPQT_VIDEO_DECODER_DETECT_D3D_FAILURE
	if(mpxplay_dispqt_videorender_d3d11_detect_device_context_failure())
	{
		if(video_avframe && ((video_avframe->format != AV_PIX_FMT_D3D11) || mpxplay_video_render_infos.render_function_poolbufelem_validity_check(video_avframe)))
		{
			if(!funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD))
			{
				funcbit_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD);
				mpxplay_timer_addfunc((void *)mpxplay_dispqt_main_close, NULL, MPXPLAY_TIMERTYPE_SIGNAL, MPXPLAY_SIGNALTYPE_GUIREADY);
			}
		}
		goto err_out_render;
	}
#endif

	if(video_surface_infos->videoout_input_change || video_surface_infos->videoout_surface_reset) // new file or seeking
		dispqt_evr_d3d11_videoprocessor_deinterlace_reset(&evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO]);
	if(video_surface_infos->videoout_surface_clearpic)
		evr_data->clear_counter = DISPQT_EVR_BUFFERS_TO_CLEAR;
	if(!evr_data->clear_counter && !video_avframe) // nothing to present
	{
		mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "videoframe_render NO FRAME");
		goto err_out_render;
	}

#ifdef MPXPLAY_USE_DEBUGF
	render_time_mutex_lock_end_ms = pds_gettimem();
#endif

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "videoframe_render deinterlace_2ndpass");
	if(this->videorenderer_d3d11_deinterlace_2ndpass(video_surface_infos, subtitle_infos))
		goto err_out_render;

	if(video_avframe && dispqt_evr_input_texture_init(evr_data, DISPQT_EVR_INPUTTEXUREID_VIDEO, video_avframe->width, video_avframe->height, video_surface_infos, video_avframe, this->main_window->gui_config)){
		video_surface_infos->videoout_surface_pix_fmt = (AVPixelFormat)evr_data->input_texture_datas[DISPQT_EVR_INPUTTEXUREID_VIDEO].av_pixelformat;
	}else if(!evr_data->clear_counter)
		goto err_out_render;

#ifdef MPXPLAY_USE_DEBUGF
	render_time_texture_init_end_ms = pds_gettimem();
#endif

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "videoframe_render textures_load");
	texture_load_ok = this->videorenderer_d3d11_textures_load(video_surface_infos, subtitle_infos);

#ifdef MPXPLAY_USE_DEBUGF
	render_time_texture_load_end_ms = pds_gettimem();
#endif

	dispqt_d3d_shader_pixel_colormixer_configure(evr_data);

#ifdef MPXPLAY_USE_DEBUGF
	render_time_shader_end_ms = pds_gettimem();
#endif

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "videoframe_render swapchain_init");
	if(!dispqt_evr_swapchain_init(evr_data, video_surface_infos))
	{
		mpxplay_debugf(DISPQT_DEBUG_WARNING, "videoframe_render swapchain_init FAILED");
		goto err_out_render;
	}

#ifdef MPXPLAY_USE_DEBUGF
	render_time_swapchain_end_ms = pds_gettimem();
#endif

	mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "videoframe_render textures_present");
	if(texture_load_ok || evr_data->clear_counter)
		this->videorenderer_d3d11_textures_present(video_surface_infos, subtitle_infos, texture_load_ok);

err_out_render:
	//mpxplay_dispqt_videorender_d3d11_flush(FALSE);
#if defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK) && !defined(MPXPLAY_VIDEO_D3D_DEVICE_USE_GLOBALLOCK)
	if(mutex_error == MPXPLAY_ERROR_OK)
		PDS_THREADS_MUTEX_UNLOCK(&mpxplay_video_render_infos.d3d_device_mutex);
#endif

#ifdef MPXPLAY_USE_DEBUGF
	if(video_avframe)
	{
		mpxp_uint64_t render_time_end_ms = pds_gettimem();
		int render_len = render_time_end_ms - render_time_begin_ms;
		int mutex_lock_len = render_time_mutex_lock_end_ms - render_time_begin_ms;
		int texture_init_len = render_time_texture_init_end_ms - render_time_mutex_lock_end_ms;
		int texture_load_len = render_time_texture_load_end_ms - render_time_texture_init_end_ms;
		int shader_init_len = render_time_shader_end_ms - render_time_texture_load_end_ms;
		int swapchain_init_len = render_time_swapchain_end_ms - render_time_shader_end_ms;
		//if(render_len > 10)
		{
			mpxplay_debugf(DISPQT_DEBUGOUT_RENDER, "RENDER END render:%2d lock:%d ti:%d tl:%d sh:%d sw:%d",
				render_len, mutex_lock_len, texture_init_len, texture_load_len, shader_init_len, swapchain_init_len);
		}
	}
#endif
	return;
}

#endif // MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
