//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2022 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: render common definitions

#ifndef VIDEO_RENDER_COMMON_H
#define VIDEO_RENDER_COMMON_H

#include "moc_config.h"

#ifndef FLOAT
typedef float FLOAT;
#endif

#define DISPQT_RENDER_POOLBUF_SIZE_MAX     DISPQT_FFMPGVIDEO_FRAME_POOL_SIZE_BASE // allocated dynamically till this number
#define DISPQT_EVR_AUTOCROP_MIN_PROCESS_INTERVAL_MS 100

#define DISPQT_EVR_NB_MAX_PLANES       4

#define DISPQT_RENDER_VERTEX_POSITION_BASE_SCALE 1.0

#define DISPQT_EVR_SUPPORTFLAG_INPUT  (1 << 0)
#define DISPQT_EVR_SUPPORTFLAG_OUTPUT (1 << 1)

enum {DISPQT_EVR_INPUTTEXUREID_VIDEO = 0, DISPQT_EVR_INPUTTEXUREID_SUBTITLE, DISPQT_EVR_INPUTTEXUREID_NUM};

#ifndef SAFE_RELEASE
#define SAFE_RELEASE(o) if(o){auto ob = (o); (o) = NULL; ob->Release();}
#endif

// ====================================================================================================================
// color space handling structures

struct dispqt_evr_format_desc_s{
	int           av_pixfmt;           // input video frame pixfmt format (matching with d3d_texture_format)
	mpxp_uint32_t d3d_texture_format;  // input texture DXGI format
	mpxp_uint8_t  bits_per_channel;    // bits per color channel
	mpxp_uint8_t  is_native_support;   // support of d3d_texture_format is detected
	mpxp_uint8_t  is_rgb_format;       // RGB or YUV input format (needs convert to RGB output or not)
	mpxp_uint8_t  align_mask_x;        // x resolution alignment restriction
	mpxp_uint8_t  align_mask_y;        // y resolution alignment restriction
	mpxp_uint8_t  nb_components;       // number of planes/components per pixel (number or shader views also)
	mpxp_uint32_t shader_view_formats[DISPQT_EVR_NB_MAX_PLANES]; // pixel shader view formats (not matching with d3d_texture_format)
	mpxp_uint8_t  data_shift[DISPQT_EVR_NB_MAX_PLANES];          // number of least significant bits that must be shifted away to get the value
	mpxp_uint8_t  plane_pos[DISPQT_EVR_NB_MAX_PLANES];           // plane positions of Y, U, V or R, G, B
};

typedef struct dispqt_evr_input_viewport_s{
	int pos_x, pos_y, width, height, rotation_type; // reference viewport values (last set)
	int text_width, text_height;                    // texture rectangles;
}dispqt_evr_input_viewport_ref_s;

// from libavfilter/colorspace.h to use libavfilter/colorspace.c functions
struct WhitepointCoefficients {
    double xw, yw;
};
struct PrimaryCoefficients {
    double xr, yr, xg, yg, xb, yb;
};

// ====================================================================================================================
// pool buffer structures

#define DISPQT_RENDER_BUFPOOLELEM_STRUCT_ID (((mpxp_uint32_t)'M' << 24) | ((mpxp_uint32_t)'M' << 16) | ((mpxp_uint32_t)'C' << 8) | (mpxp_uint32_t)'B')

enum {DISPQT_RENDER_BUFPOOLELEM_STATUS_FREE = 0, DISPQT_RENDER_BUFPOOLELEM_STATUS_PROCESSING, DISPQT_RENDER_BUFPOOLELEM_STATUS_DONE, DISPQT_RENDER_BUFPOOLELEM_STATUS_INVALIDATED};

struct dispqt_render_mapped_subresource_s {
    void *vidmem_beginptr;        // pointer to the mapped video memory
    unsigned int vidmem_linesize; // bytes per line
    unsigned int vidmem_linenum;  // number of lines (texture_height)
};

typedef struct dispqt_render_poolbufelem_s
{
	mpxp_uint32_t bufpoolelem_struct_id;       // identification elem of evr_poolbufelem_s (DISPQT_RENDER_BUFPOOLELEM_STRUCT_ID)
	mpxp_uint32_t bufpoolelem_status;          // DISPQT_RENDER_BUFPOOLELEM_STATUS_BUSY indicates that this elem is busy (traveling from the decoder to the renderer)
	int elem_id;                               // for debugging only
	int av_hwformat;                           // AV_PIX_FMT_D3D11, AV_PIX_FMT_DXVA2_VLD
	int videoframe_width, videoframe_height;   // width / height of the video frame
	int allocated_width, allocated_height;     // width / height of the allocated texture
	void *poolbufelem_texture_2d[DISPQT_EVR_NB_MAX_PLANES]; // used to store the decoded data (pointer to D3D11/ID3D11Texture2D or D3D9/IDirect3DSurface9)
	void *poolbufelem_shader_resource_views[DISPQT_EVR_NB_MAX_PLANES]; // pointer to ID3D11ShaderResourceView at MPXPLAY_VIDEO_RENDERER_FRAME_POOL_WRITEONLY_FLAG only
	struct dispqt_render_mapped_subresource_s poolbufelem_texture_mappedResource[DISPQT_EVR_NB_MAX_PLANES]; // memory/texture infos from d3d_device_context->Map()
	struct dispqt_evr_format_desc_s bufpoolelem_format_desc;
}dispqt_render_poolbufelem_s;

// ====================================================================================================================
// structures passed to the pixel shader

// vmixer_values to colorchange_values
#define DISPQT_EVR_COLORCHANGE_VALUE_SCALE 100.0

// normalized values of vmixer_values[], shall match with PS_COLOR_MIXER_DATA
typedef struct dispqt_evr_color_mixer_s {
    FLOAT colormixer_values[4];
    FLOAT colormixer_hue_matrix[4][4];
} dispqt_evr_color_mixer_s;

// shall match with PS_COLOR_TRANSFORM_DATA
typedef struct dispqt_evr_color_transform_matrixes_s {
    FLOAT WhitePointMatrix[4*4];
    FLOAT ColorspaceMatrix[4*4];
    FLOAT PrimariesMatrix[4*4];
    FLOAT pixdata_shift_mul[4];
} dispqt_evr_color_transform_matrixes_s;

// shall match with PS_DYNAMIC_RUNTIME_DATA
typedef struct dispqt_evr_dynamic_runtime_data_s {
    FLOAT HDRLuminanceValue; // colorspace dependent value (SRGB: 100, HLG: 1000, SMPTE2084: 10000)
    FLOAT dummy[3];
} dispqt_evr_dynamic_runtime_data_s;

typedef struct dispqt_render_evr_dynamic_hdr_data_s
{
	int hdr_input_luminance_value;
	unsigned int last_hdr_control_setting;
	mpxp_bool_t is_custom_primaries_sent;
	struct WhitepointCoefficients custom_wp_coefs;
	struct PrimaryCoefficients custom_pri_coefs;
}dispqt_render_evr_dynamic_hdr_data_s;

// ====================================================================================================================
// vertex (shader) datas

#define DISPQT_EVR_D3D_NUMVERTICES_DEFAULT  6

typedef struct d3d_vertex_t {
  struct {
    FLOAT x, y, z;
  } position;
  struct {
    FLOAT x, y;
  } texture;
} d3d_vertex_t;

// Vertices for drawing whole texture (2 triangles, 2x3 dots, covers a square full picture, viewport defines the output resolution)
static const d3d_vertex_t vertices_full_texture_default[DISPQT_EVR_D3D_NUMVERTICES_DEFAULT] =
{
  {{-1.0f, -1.0f, 0.0f}, {0.0f, 1.0f}}, {{-1.0f, 1.0f, 0.0f}, {0.0f, 0.0f}}, {{ 1.0f, -1.0f, 0.0f}, {1.0f, 1.0f}},
  {{ 1.0f, -1.0f, 0.0f}, {1.0f, 1.0f}}, {{-1.0f, 1.0f, 0.0f}, {0.0f, 0.0f}}, {{ 1.0f,  1.0f, 0.0f}, {1.0f, 0.0f}}
};

// flat vertex shader code (do nothing, just draw vertices)
static const char *d3dVertexShaderFlat = "\
struct VS_INPUT\n\
{\n\
	float4 Pos : POSITION;\n\
	float2 Tex : TEXCOORD;\n\
};\n\
\n\
struct VS_OUTPUT\n\
{\n\
	float4 Pos : SV_POSITION;\n\
	float2 Tex : TEXCOORD;\n\
};\n\
\n\
VS_OUTPUT main(VS_INPUT input)\n\
{\n\
	return input;\n\
}\n";


// ====================================================================================================================
// pixel shader code blocks for shader model 4.0

// HDR conversion functions

static const char *dispqt_d3d_pscode_hdr_tonemap = "\
inline float4 colorcv_hable(float4 x) {\n\
    const float A = 0.15, B = 0.50, hBC = 0.05, hDE = 0.004, hDF = 0.06, hEpF = (0.02 / 0.30);\n\
    return ((x * (A*x + hBC) + hDE) / (x * (A*x + B) + hDF)) - hEpF;\n\
}\n\n\
inline float4 colorcv_tonemap(float4 rgb) {\n\
    static const float4 HABLE_DIV = colorcv_hable(11.2);\n\
    return (colorcv_hable(rgb * HDRLuminanceValue) / HABLE_DIV);\n\
}\n\n";

static const char *dispqt_d3d_pscode_colorcv_SMPTE_ST2084_to_linear = "\
inline float4 colorcv_SMPTE_ST2084_to_linear(float4 rgb) {\n\
    const float ST2084_m1r = 1.0 / (2610.0 / 4096.0 / 4.0);\n\
    const float ST2084_m2r = 1.0 / (128.0 * 2523.0 / 4096.0);\n\
    const float ST2084_c1 = 3424.0 / 4096.0;\n\
    const float ST2084_c2 = 32.0 * 2413.0 / 4096.0;\n\
    const float ST2084_c3 = 32.0 * 2392.0 / 4096.0;\n\
    rgb = pow(max(rgb, 0.0), ST2084_m2r);\n\
    rgb = max(rgb - ST2084_c1, 0.0) / (ST2084_c2 - ST2084_c3 * rgb);\n\
    rgb = pow(rgb, ST2084_m1r);\n\
    return rgb * 10000;\n\
}\n\n";

static const char *dispqt_d3d_pscode_colorcv_HLG_to_linear = "\
inline float inverse_HLG(float x){\n\
    const float B67_a = 0.17883277;\n\
    const float B67_b = 0.28466892;\n\
    const float B67_c = 0.55991073;\n\
    const float B67_inv_r2 = 4.0;\n\
    if (x <= 0.5)\n\
        x = x * x * B67_inv_r2;\n\
    else\n\
        x = exp((x - B67_c) / B67_a) + B67_b;\n\
    return x;\n\
}\n\n\
inline float4 colorcv_HLG_to_linear(float4 rgb) {\n\
    const float alpha_gain = 2000;\n\
	rgb.r = inverse_HLG(rgb.r);\n\
	rgb.g = inverse_HLG(rgb.g);\n\
	rgb.b = inverse_HLG(rgb.b);\n\
	float3 ootf_2020 = float3(0.2627, 0.6780, 0.0593);\n\
	float ootf_ys = alpha_gain * dot(ootf_2020, rgb);\n\
	return rgb * pow(ootf_ys, 0.200);\n\
}\n\n";

static const char *dispqt_d3d_pscode_colorcv_primaries_transform = "\
inline float4 colorcv_transform_primaries(float4 rgb) {\n\
    return max(mul(rgb, PrimariesMatrix), 0);\n\
}\n\n";

static const char *dispqt_d3d_pscode_colorcv_primaries_empty = "\
inline float4 colorcv_transform_primaries(float4 rgb) {\n\
    return rgb;\n\
}\n\n";

static const char *dispqt_d3d_pscode_colorcv_linear_to_SRGB = "\
inline float4 colorcv_linear_to_srgb(float4 rgb) {\n\
    return pow(rgb, 1.0 / 2.2);\n\
}\n\n";

static const char *dispqt_d3d_pscode_mainfunc_colorcv_SMPTE2084_to_SRGB = "\
    rgba.a = 0;\n\
    rgba = colorcv_SMPTE_ST2084_to_linear(rgba);\n\
    rgba = colorcv_tonemap(rgba);\n\
    rgba = colorcv_transform_primaries(rgba);\n\
    rgba = colorcv_linear_to_srgb(rgba);\n\n";

static const char *dispqt_d3d_pscode_mainfunc_colorcv_HLG_to_SRGB = "\
    rgba.a = 0;\n\
    rgba = colorcv_HLG_to_linear(rgba);\n\
    rgba = colorcv_tonemap(rgba);\n\
    rgba = colorcv_transform_primaries(rgba);\n\
    rgba = colorcv_linear_to_srgb(rgba);\n\n";

static const char *dispqt_d3d_pscode_mainfunc_colorcv_primaries = "\
    rgba = colorcv_transform_primaries(rgba);\n";

static const char *dispqt_d3d_pscode_mainfunc_colorcv_BT709_to_SRGB = "\
    rgba = pow(rgba, (1.0 / 0.45 / 2.20));\n";

static const char *dispqt_d3d_pscode_mainfunc_colorcv_BT470BG_to_SRGB = "\
    rgba = pow(rgba, (2.8 / 2.2));\n";

// ----------------------------------------------------------------------------------------------------
// sample converters

static const char *dispqt_d3d_pscode_samplecv_YUY2 = "\
inline float4 getSampleTexture(SamplerState samplerState, float2 textur) {\n\
    float4 sample;\n\
	sample.x = shaderTexture[0].Sample(samplerState, textur).x;\n\
    sample.y = shaderTexture[0].Sample(samplerState, textur).y;\n\
    sample.z = shaderTexture[0].Sample(samplerState, textur).a;\n\
    sample.a = 1;\n\
    return sample;\n\
}\n\n";

// 2 planes: Y + interleaved U/V (bit depth depends on the DXGI_FORMAT)
static const char *dispqt_d3d_pscode_samplecv_NV12_P010 = "\
inline float4 getSampleTexture(SamplerState samplerState, float2 textur) {\n\
    float4 sample;\n\
    sample.x  = shaderTexture[0].Sample(samplerState, textur).x;\n\
    sample.yz = shaderTexture[1].Sample(samplerState, textur).xy;\n\
    sample.a  = 1;\n\
    return sample;\n\
}\n\n";

// 2 planes: Y + interleaved V/U
static const char *dispqt_d3d_pscode_samplecv_NV21 = "\
inline float4 getSampleTexture(SamplerState samplerState, float2 textur) {\n\
    float4 sample;\n\
    sample.x  = shaderTexture[0].Sample(samplerState, textur).x;\n\
    sample.yz = shaderTexture[1].Sample(samplerState, textur).yx;\n\
    sample.a  = 1;\n\
    return sample;\n\
}\n\n";

static const char *dispqt_d3d_pscode_samplecv_GENERAL_3planes_anybits = "\
inline float4 getSampleTexture(SamplerState samplerState, float2 textur) {\n\
    float4 sample;\n\
    sample.x = shaderTexture[0].Sample(samplerState, textur).x * pixdata_shift_muls.x;\n\
    sample.y = shaderTexture[1].Sample(samplerState, textur).x * pixdata_shift_muls.y;\n\
    sample.z = shaderTexture[2].Sample(samplerState, textur).x * pixdata_shift_muls.z;\n\
    sample.a = 1;\n\
    return sample;\n\
}\n\n";

// ----------------------------------------------------------------------------------------------------
// common functions

// shall match with dispqt_evr_color_mixer_s (FLOAT colormixer_values[DISPQT_VIDEO_VMIXERTYPE_MAX])
static const char *dispqt_d3d_pscode_CONSTANTDATA0_color_mixer = "\
cbuffer PS_COLOR_MIXER_DATA : register(b0)\n\
{\n\
    float ColorMixerBrightness;\n\
    float ColorMixerContrast;\n\
    float ColorMixerSaturation;\n\
    float ColorMixerHueVal;\n\
    float4x4 ColorMixerHueMatrix;\n\
};\n\n";

static const char *dispqt_d3d_pscode_CONSTANTDATA1_color_transform_YUV2RGB = "\
cbuffer PS_COLOR_TRANSFORM_DATA : register(b1)\n\
{\n\
    float4x4 WhitePointMatrix;\n\
    float4x4 ColorSpaceMatrix;\n\
    float4x4 PrimariesMatrix;\n\
    float4 pixdata_shift_muls;\n\
};\n\n";

static const char *dispqt_d3d_pscode_CONSTANTDATA2_runtime_dynamic = "\
cbuffer PS_DYNAMIC_RUNTIME_DATA : register(b2)\n\
{\n\
    float HDRLuminanceValue;\n\
    float dynamicRuntimeDummy1;\n\
    float dynamicRuntimeDummy2;\n\
    float dynamicRuntimeDummy3;\n\
};\n\n";

static const char *dispqt_d3d_pscode_COMMON_datas = "\
SamplerState samplerStateHnd : register(s0);\n\
Texture2D shaderTexture[4];\n\
\n";

static const char *dispqt_d3d_pscode_color_mixer = "\
static float4 mmc_color_mixer(float4 rgba)\n\
{\n\
    if(ColorMixerBrightness != 0.0f)\n\
    {\n\
        rgba = rgba + ColorMixerBrightness;\n\
    }\n\
    if(ColorMixerContrast != 1.0f)\n\
    {\n\
        const float contrast_center = 0.5f;\n\
        rgba = rgba - contrast_center;\n\
        rgba = mul(rgba, ColorMixerContrast);\n\
        rgba = rgba + contrast_center;\n\
    }\n\
    if(ColorMixerSaturation != 1.0f)\n\
    {\n\
        float gray = (0.299 * rgba.r) + (0.587 * rgba.g) + (0.114 * rgba.b);\n\
        rgba.r = (rgba.r - gray) * ColorMixerSaturation + gray;\n\
        rgba.g = (rgba.g - gray) * ColorMixerSaturation + gray;\n\
        rgba.b = (rgba.b - gray) * ColorMixerSaturation + gray;\n\
    }\n\
    if(ColorMixerHueVal != 0.0f)\n\
    {\n\
        rgba = mul(ColorMixerHueMatrix, rgba);\n\
    }\n\
    rgba.a = 1.0;\n\
    return clamp(rgba, 0.0, 1.0);\n\
}\n\n";

static const char *dispqt_d3d_pscode_mainfunc_BEGIN ="\
float4 main(float4 pos : SV_POSITION, float2 Tex : TEXCOORD) : SV_TARGET\n\
{\n";

static const char *dispqt_d3d_pscode_mainfunc_GET_RGB_from_YUV_planar = "\
    float4 sample = getSampleTexture( samplerStateHnd, Tex );\n\
    float4 rgba = max(mul(mul(sample, WhitePointMatrix), ColorSpaceMatrix),0);\n";

static const char *dispqt_d3d_pscode_mainfunc_GET_RGB_from_RGB_packed = "\
    float4 rgba = shaderTexture[0].Sample( samplerStateHnd, Tex );\n";

static const char *dispqt_d3d_pscode_mainfunc_GET_RGB_from_RGB_planar = "\
    float4 rgba = getSampleTexture( samplerStateHnd, Tex );\n";

static const char *dispqt_d3d_pscode_mainfunc_CALL_color_mixer ="\
    rgba = mmc_color_mixer(rgba);\n";

static const char *dispqt_d3d_pscode_mainfunc_END ="\
    return rgba;\n\
}\n";

// ====================================================================================================================

extern void mpxplay_dispqt_render_common_shader_pixel_constant_colortransform_calculate(
		struct dispqt_evr_color_transform_matrixes_s *color_transform_data, const struct dispqt_evr_format_desc_s *pixform_infos,
		AVColorSpace av_colorspace, enum AVColorRange av_colorrange, enum AVColorPrimaries av_colorprimaries, int texture_height,
		const struct WhitepointCoefficients *custom_whitepoints_src, const struct PrimaryCoefficients *custom_primary_coefs_src);
extern void mpxplay_dispqt_render_common_shader_pixel_assemble_trc(char *pscode_str_body, char *pscode_str_mainfunc,
		struct mmc_dispqt_config_s *gcfg, int av_colorprimaries, int av_colortrc);
extern void mpxplay_dispqt_render_common_shader_pixel_dynamic_hdr_reset(struct dispqt_render_evr_dynamic_hdr_data_s *dynamic_hdr_datas,
		struct mmc_dispqt_config_s *gcfg);
extern bool mpxplay_dispqt_render_common_shader_pixel_dynamic_hdr_update(struct dispqt_render_evr_dynamic_hdr_data_s *dynamic_hdr_datas,
		AVFrame *video_avframe, struct dispqt_video_surface_info_s *video_surface_infos, struct mmc_dispqt_config_s *gcfg);
extern void mpxplay_dispqt_render_common_shader_pixel_colormixer_configure(struct dispqt_evr_color_mixer_s *color_mixdata_dst,
		mpxp_int32_t vmixer_values[DISPQT_VIDEO_VMIXERTYPE_MAX]);
extern void mpxplay_dispqt_render_common_vertex_set_input_texture_crop_and_rotate(void *vertex_destptr,
		const struct dispqt_evr_input_viewport_s *crop_datas, int input_texture_width, int input_texture_height, float vertex_position_scale);
extern struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_select_dxgi_format(struct dispqt_evr_format_desc_s *format_descs, int srch_dxgi);
extern struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_input_select_native_color_format(struct dispqt_evr_format_desc_s *format_descs,
		struct dispqt_evr_format_desc_s *format_info_store, int srch_pixfmt, mpxp_bool_t search_next);
extern struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_input_select_color_format(struct dispqt_evr_format_desc_s *format_descs,
		struct dispqt_evr_format_desc_s *format_info_store, int srch_pixfmt, mpxp_bool_t select_intermediate_pixfmt);
extern struct dispqt_evr_format_desc_s *mpxplay_dispqt_render_common_select_best_subtitle_format(struct dispqt_evr_format_desc_s *format_descs);
extern mpxp_bool_t mpxplay_dispqt_render_common_deinterlace_condition(AVFrame *video_avframe, struct mmc_dispqt_config_s *gcfg);
extern mpxp_bool_t mpxplay_dispqt_render_common_deinterdup_condition(AVFrame *video_avframe, struct mmc_dispqt_config_s *gcfg);

extern void mpxplay_dispqt_videorender_common_videoframecopy_PACKED(
		struct dispqt_render_mapped_subresource_s *texture_mapres, int texture_id,
		uint8_t *indata_ptr, unsigned int indata_linesize, unsigned int indata_lines);
extern mpxp_bool_t mpxplay_dispqt_videorender_common_videoframe_copy(
		struct dispqt_render_mapped_subresource_s texture_mapres[DISPQT_EVR_NB_MAX_PLANES],
		struct dispqt_evr_format_desc_s *pix_form,
		uint8_t *indata_ptrs[AV_NUM_DATA_POINTERS], int indata_linesizes[AV_NUM_DATA_POINTERS], unsigned int indata_lines);

extern void mpxplay_dispqt_videorender_common_poolbuf_functions_assign(void);
extern void mpxplay_dispqt_videorender_common_poolbuf_elems_close(struct dispqt_render_poolbufelem_s *bufelem_first);

#endif // VIDEO_RENDER_COMMON_H
