//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: video playing surface event handlers

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT NULL //stdout
#define DISPQT_DEBUGOUT_EVENT NULL //stdout
#define DISPQT_DEBUGOUT_CURSOR stdout

#include <QtGui>
#include <QtWidgets>
#include "moc_video_qt.h"
#include "moc_mainwindow.h"

#define DISPQT_EVENT_EPG_TOOLTIP_INTERVAL    200                // limit refresh interval of EPG tooltip displaying
#define DISPQT_EVENT_EPG_TOOLTIP_SHOWTIME  30000                // show EPG tooltip for 30 secs (if no mouse event happens)
#define DISPQT_EVENT_EPG_TOOLTIP_MAXLINELEN_PV   45             // limit the maximum length of an EPG line on primary video
#define DISPQT_EVENT_EPG_TOOLTIP_MAXLINELEN_VW   40             // limit the maximum length of an EPG line on videowall
#define DISPQT_EVENT_EPG_TOOLTIP_PRIMVIDEOSHOWINDICATORWIDTH 10 // in non-videowall mode, the EPG is displayed, if mouse pointer is in the 0-9 x range

extern "C" {
 extern keyconfig kb[];
 extern unsigned long mpxplay_config_videoplayer_type, mpxplay_config_dvbepg_control_flags;
 extern void mpxplay_inffmp_epg_clear_epginfolist(struct mpxplay_dvb_epgeventlist_t *epg_list);
}

DispQtVideoEventFilter::DispQtVideoEventFilter(MainWindow *main_window, QWidget *parent, QWidget *filtered) : QObject(parent)
{
	this->main_window = main_window;
	this->parent_widget = parent;
	this->filtered_widget = filtered;
	this->video_surface_contextmenu = NULL;
	this->mouse_pressed_buttons = Qt::NoButton;
	this->screen_pos_x0 = this->screen_pos_y0 = 0;
	this->mouse_last_globalpos_x = this->mouse_last_globalpos_y = 0; // not needed
	this->singleclick_event = this->mouse_cursor_forbidden_hide = this->singleclick_fullsrc_start = this->fullsrc_curr = this->is_cursor_overrided = false;

	this->timer_singleclick.setInterval(DISPQT_VIDEOWIDGET_SINGLECLICK_TIMEOUTMS);
	this->timer_singleclick.setSingleShot(true);
	connect(&this->timer_singleclick, SIGNAL(timeout()), this, SLOT(mouse_singleclick_timeouthandler()));
	this->timer_doubleclick.setSingleShot(true);
	this->timer_mouse_hide.setSingleShot(true);
	connect(&this->timer_mouse_hide, SIGNAL(timeout()), this, SLOT(mouse_cursor_hide()));
	this->timer_cmenu_hide.setSingleShot(true);

	// EPG tooltip helper variables
	this->epgtooltip_window = NULL;
	this->epgtooltip_globalpos_x = this->epgtooltip_globalpos_y = -1;
	this->timer_epgtooltip_interval.setInterval(DISPQT_EVENT_EPG_TOOLTIP_INTERVAL);
	this->timer_epgtooltip_interval.setSingleShot(true);
	connect(&this->timer_epgtooltip_interval, SIGNAL(timeout()), this, SLOT(mouse_epgtooltip_presenter_handler()));
	this->timer_epgtooltip_show.setInterval(DISPQT_EVENT_EPG_TOOLTIP_SHOWTIME);
	this->timer_epgtooltip_show.setSingleShot(true);
	connect(&this->timer_epgtooltip_show, SIGNAL(timeout()), this, SLOT(mouse_epgtooltip_close()));
}

bool DispQtVideoEventFilter::is_video_fullscreen(void)
{
	DispQtVideoPlayer *player = (DispQtVideoPlayer *)this->parent_widget;
	return player->video_get_value(DispQtVideoPlayer::VideoCtrlValue_FullScreen);
}

bool DispQtVideoEventFilter::is_video_wall_enabled(void)
{
	DispQtVideoPlayer *player = (DispQtVideoPlayer *)this->parent_widget;
	return player->video_get_value(DispQtVideoPlayer::VideoCtrlValue_VideoWallEnabled);
}

void DispQtVideoEventFilter::videosurface_get_screen_pos0(void)
{
	if((QApplication::desktop()->screenCount() > 1) && this->is_video_fullscreen())
	{
		const int screen_number = this->main_window->video_get_fullscreen_display_number();
		const QRect screenGeometry = QApplication::desktop()->screenGeometry(screen_number);
		this->screen_pos_x0 = screenGeometry.x();
		this->screen_pos_y0 = screenGeometry.y();
	}
	else
	{
		this->screen_pos_x0 = this->screen_pos_y0 = 0;
	}
}

void DispQtVideoEventFilter::videosurface_get_pos(QMouseEvent *me, int *pos_x, int *pos_y)
{
	if(this->is_video_fullscreen())
	{
		*pos_x = me->globalX();
		*pos_y = me->globalY();
	}
	else
	{
		*pos_x = me->x();
		*pos_y = me->y();
	}
}

void DispQtVideoEventFilter::videosurface_handle_singleclick(void)
{
	if(this->is_video_wall_enabled())
	{   // select program by single click
		this->videosurface_get_screen_pos0();
		this->video_callback_call(this->mouse_last_globalpos_x - this->screen_pos_x0, this->mouse_last_globalpos_y - this->screen_pos_y0, MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLCURRSTI, -1, 0);
	}
	else
	{
		this->main_window->mainwin_handle_keypresscode(kb[51].c); // Pause
	}
}

void DispQtVideoEventFilter::videosurface_handle_doubleclick(void)
{
	this->mouse_epgtooltip_close();
	this->fullsrc_curr = this->is_video_fullscreen();
	if(this->is_video_wall_enabled())
	{   // select program by click, exit from videowall
		this->videosurface_get_screen_pos0();
		this->video_callback_call(this->mouse_last_globalpos_x - this->screen_pos_x0, this->mouse_last_globalpos_y - this->screen_pos_y0, MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLCURRSTI_EW, -1, 0);
	}
	else
	{
		if(this->singleclick_event)
			this->videosurface_handle_singleclick(); // start again playing (playing has paused) at slow double-click
		this->fullsrc_curr = !this->fullsrc_curr;
		emit ((DispQtVideoPlayer *)this->parent_widget)->signal_video_switch_fullscreen();
	}
	this->singleclick_event = false;
}

// handle mouse move event on fullscreen video surface, if no button is pressed
void DispQtVideoEventFilter::videosurface_handle_moveevent_nobutton(QMouseEvent *me)
{
	this->mouse_cursor_show_timed(true);
	if(!(me->modifiers() & Qt::ControlModifier))  // feature: if Ctrl button is pressed, main window does not appear
		this->main_window->mainwin_video_check_show(me->globalX(), me->globalY());
	bool videowall_is_enabled = this->is_video_wall_enabled();
	if( (videowall_is_enabled && (this->main_window->gui_config->video_zoom_value == DISPQT_CONFIG_VIDEOZOOM_CENTER))
	 || (!videowall_is_enabled && (me->globalX() < DISPQT_EVENT_EPG_TOOLTIP_PRIMVIDEOSHOWINDICATORWIDTH)) // TODO: we should block better the unuseful EPG request (at non DVB videos)
	){   // show EPG
		this->epgtooltip_globalpos_x = me->globalX();
		this->epgtooltip_globalpos_y = me->globalY();
		if(!this->timer_epgtooltip_interval.isActive())
			this->timer_epgtooltip_interval.start();
	}
	else
	{
		this->mouse_epgtooltip_close();
	}
}

void DispQtVideoEventFilter::mouse_cursor_forbid_hide(bool forbid)
{
	this->mouse_cursor_forbidden_hide = forbid;
	this->mouse_cursor_show_timed(!forbid);
}

void DispQtVideoEventFilter::mouse_cursor_show_timed(bool timed)
{
	bool lock_success = mutex_video.tryLock(DISPQT_VIDEO_MUTEX_TIMEOUT);
	mpxplay_debugf(DISPQT_DEBUGOUT_CURSOR, "mouse_cursor_show_timed b:%d t:%d", this->is_cursor_overrided, timed);
	if(this->is_cursor_overrided){
		QApplication::restoreOverrideCursor();
		QCursor *cursor = QApplication::overrideCursor();
		if(cursor){
			Qt::CursorShape curshape = QApplication::overrideCursor()->shape();
			if((curshape == Qt::BlankCursor) || (curshape == Qt::OpenHandCursor)) // list all modified cursor shape here
				QApplication::setOverrideCursor(Qt::ArrowCursor);
		}
		this->is_cursor_overrided = false;
	}
	if(timed && !this->mouse_cursor_forbidden_hide && this->is_video_fullscreen()){
		this->timer_mouse_hide.setInterval(DISPQT_VIDEO_CURSORHIDE_TIMEOUT);
		this->timer_mouse_hide.start();
	}else
		this->timer_mouse_hide.stop();
	if(lock_success)
		mutex_video.unlock();
}

void DispQtVideoEventFilter::mouse_cursor_hide(void)
{
	bool lock_success = mutex_video.tryLock(DISPQT_VIDEO_MUTEX_TIMEOUT);
	mpxplay_debugf(DISPQT_DEBUGOUT_CURSOR, "mouse_cursor_hide b:%d", this->is_cursor_overrided);
	if(!this->is_cursor_overrided){
		QApplication::setOverrideCursor(Qt::BlankCursor);
		this->is_cursor_overrided = true;
	}
	if(lock_success)
		mutex_video.unlock();
}

void DispQtVideoEventFilter::cmenu_about_hide(void)
{
	this->timer_cmenu_hide.start(DISPQT_VIDEOWIDGET_CMENUHIDE_TIMEOUTMS);
	this->mouse_pressed_buttons = Qt::NoButton;
	this->mouse_cursor_show_timed(true);
}

void DispQtVideoEventFilter::mouse_singleclick_timeouthandler(void)
{
	bool lock_success = mutex_video.tryLock(DISPQT_VIDEO_MUTEX_TIMEOUT);
	mpxplay_debugf(DISPQT_DEBUGOUT_EVENT, "STO: b:%d", this->mouse_pressed_buttons);
	if((this->singleclick_fullsrc_start == this->fullsrc_curr) && !this->timer_doubleclick.isActive())
	{
		if(this->mouse_pressed_buttons == Qt::LeftButton)
		{
			QApplication::setOverrideCursor(Qt::OpenHandCursor);
			this->is_cursor_overrided = true;
		}
		else if(this->mouse_pressed_buttons == Qt::NoButton)
		{
			this->timer_doubleclick.start(DISPQT_VIDEOWIDGET_DOUBLECLICK_TIMEOUTMS);
			this->videosurface_handle_singleclick();
			this->singleclick_event = true;
		}
	}
	if(lock_success)
		mutex_video.unlock();
}

//-----------------------------------------------------------------------------------------------------
DispQtVideoEventEPGTooltip::DispQtVideoEventEPGTooltip(MainWindow *mainwindow, QWidget *parent, QObject *eventfilter, const QString &EPGeventList, bool isVideoWall) : DispQtDialogElemWindow(mainwindow, parent, Qt::Dialog | Qt::FramelessWindowHint | Qt::WindowDoesNotAcceptFocus, DISPQT_DIALOG_WINDOWTYPE_EPGVIDEOINFO)
{
	this->main_window = mainwindow;
	this->parent_filtered_widget = parent;
	this->parent_event_filter = eventfilter;
	this->is_videowall = isVideoWall;
	this->setFocusPolicy(Qt::NoFocus);
	this->setMaximumWidth(200);
	if(this->main_window->mainwin_guilayout_is_dark_background())
	{
		this->setStyleSheet("QDialog { border-style: solid; border-radius: 6px; } QLabel { background: transparent; color: white; font-size:13px;}");
	}
	else
	{
		this->setStyleSheet("QDialog { border-style: solid; border-radius: 6px; } QLabel { font-size:13px;}");
	}
	QGridLayout *layout = new QGridLayout;
	QLabel *text = new QLabel(EPGeventList);
	layout->addWidget(text);
	this->setLayout(layout);
}

bool DispQtVideoEventEPGTooltip::event(QEvent *event)
{
	QMouseEvent *me = static_cast<QMouseEvent*>(event);
	bool close_event = false;
	switch(event->type()){
		case QEvent::Enter: break;
		case QEvent::Leave:
			close_event = true;
			break;
		case QEvent::MouseButtonPress:
		case QEvent::MouseButtonRelease:
			if(this->is_videowall || (me->buttons() & Qt::RightButton))
				this->parent_event_filter->eventFilter(this->parent_filtered_widget, event);
			break;
		case QEvent::Wheel: // TODO: scrollable EPG list
			break;
	}
	if(this->main_window->mainwin_is_on_video())
		emit this->main_window->signal_video_mouse_cursor_show(close_event);
	if(close_event)
		this->close();

	return DispQtDialogElemWindow::event(event);
}

void DispQtVideoEventEPGTooltip::mouseDoubleClickEvent(QMouseEvent *event)
{
	this->parent_event_filter->eventFilter(this->parent_filtered_widget, event);
}

void DispQtVideoEventEPGTooltip::keyPressEvent(QKeyEvent *event)
{
	this->main_window->handle_keypressevent(event);
	if(event->key() == Qt::Key_Escape)
		this->close();
}

//-----------------------------------------------------------------------------------------------------
void DispQtVideoEventFilter::mouse_epgtooltip_close(void)
{
	this->timer_epgtooltip_show.stop();
	if(this->epgtooltip_window) // TODO: check inside close (seems this is ok)
	{
		DispQtVideoEventEPGTooltip *ew = this->epgtooltip_window;
		this->epgtooltip_window = NULL;
		delete ew;
	}
}

void DispQtVideoEventFilter::mouse_epgtooltip_open(int pos_x, int pos_y, int height, char *epg_lst_collected_str, bool is_videwall)
{
	mouse_epgtooltip_close();
	this->epg_event_list = QString::fromUtf8(epg_lst_collected_str);
	if(this->epg_event_list.size() <= 0)
		return;
	this->epgtooltip_window = new DispQtVideoEventEPGTooltip(this->main_window, this->filtered_widget, this, this->epg_event_list, is_videwall);
	this->epgtooltip_window->setMaximumHeight(height);
	this->epgtooltip_window->resize(100, height);
	this->epgtooltip_window->move(pos_x, pos_y);
	this->epgtooltip_window->show();
	this->timer_epgtooltip_show.start();
}

// display EPG list in a dialog window over video surface (single video & videowall too)
void DispQtVideoEventFilter::mouse_epgtooltip_presenter_handler(void)
{
	struct mpxplay_dvb_epgeventlist_t *epg_list = NULL;
	bool is_videowall;

	if(!funcbit_test(mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOSURFACE_SHOW_EPGDIALOG))
	{
		return;
	}

	this->videosurface_get_screen_pos0();

	is_videowall = this->is_video_wall_enabled();
	if(is_videowall) // show EPG of videowall sub-window
	{
		this->video_callback_call(this->epgtooltip_globalpos_x - this->screen_pos_x0, this->epgtooltip_globalpos_y - this->screen_pos_y0, MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLPRGEPGLIST, -1, (mpxp_ptrsize_t)&epg_list);
	}
	else if((this->epgtooltip_globalpos_x - this->screen_pos_x0) < DISPQT_EVENT_EPG_TOOLTIP_PRIMVIDEOSHOWINDICATORWIDTH) // show EPG of primary video
	{
		this->video_callback_call(-1, -1, MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLPRGEPGLIST, MPXPLAY_STREAMTYPEINDEX_VIDEO, (mpxp_ptrsize_t)&epg_list);
	}

	if(epg_list)
	{
		int epg_display_lines = min(epg_list->nb_program_events, ((epg_list->videowall_window_height / 16 - ((is_videowall)? 2 : 5))));
		int epg_window_height = (is_videowall)? (epg_list->videowall_window_height - 16): ((epg_display_lines - 4) * 16);
		int event_limit_y_end = epg_list->videowall_window_pos_y + epg_window_height;
		if(this->epgtooltip_globalpos_y < event_limit_y_end) // exclude some areas from the tooltip event
		{
			int epg_lst_collectlen = pds_strlen(epg_list->program_name) + 2, i; // "program name\r\n"
			char *epg_lst_collected_str, *p;

			for(i = 0; i < epg_display_lines; i++)
			{
				epg_lst_collectlen += pds_strlen(epg_list->program_event_list[i]) + 2; // +2 \r\n
			}

			epg_lst_collected_str = p = (char *)pds_malloc(epg_lst_collectlen + 256);
			if(epg_lst_collected_str)
			{
				p += pds_strcpy(p, epg_list->program_name);
				p += pds_strcpy(p, (char *)"\r\n");
				for(i = 0; i < epg_display_lines; i++)
				{
					char *event_str = epg_list->program_event_list[i];
					if(event_str)
					{
						unsigned int limitlen = (is_videowall)? DISPQT_EVENT_EPG_TOOLTIP_MAXLINELEN_VW : DISPQT_EVENT_EPG_TOOLTIP_MAXLINELEN_PV;
						unsigned int copybytes = pds_utf8_strpos((mpxp_uint8_t *)event_str, limitlen);
						p += pds_strncpy(p, event_str, copybytes);
						p += pds_strcpy(p, (char *)"\r\n");
					}
					if((p - epg_lst_collected_str) > epg_lst_collectlen)
					{
						break;
					}
				}
				p[-2] = 0; // cut last \r\n (empty line)

				this->mouse_epgtooltip_open(this->screen_pos_x0 + epg_list->videowall_window_pos_x, this->screen_pos_y0 + epg_list->videowall_window_pos_y, epg_window_height, epg_lst_collected_str, is_videowall);

				pds_free(epg_lst_collected_str);
			}
		}

		mpxplay_inffmp_epg_clear_epginfolist(epg_list);
	}
	else
	{
		this->mouse_epgtooltip_close();
	}
}

//-----------------------------------------------------------------------------------------------------
void DispQtVideoEventFilter::video_surface_move(int move_x, int move_y)
{
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(mpxplay_config_videoplayer_type == MPXPLAY_VIDEOPLAYERTYPE_FFMPEG)
		emit ((FFMpegVideoWidget *)this->filtered_widget)->signal_ffmpeg_swctx_move(move_x, move_y); // FIXME: hack
#endif
}

void DispQtVideoEventFilter::video_zoom_step(int globalpos_x, int globalpos_y, int direction)
{
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(mpxplay_config_videoplayer_type == MPXPLAY_VIDEOPLAYERTYPE_FFMPEG)
		emit ((FFMpegVideoWidget *)this->filtered_widget)->signal_ffmpeg_swctx_zoom(globalpos_x, globalpos_y, direction); // FIXME: hack
#endif
}

void DispQtVideoEventFilter::video_callback_call(int globalpos_x, int globalpos_y, unsigned long command, mpxp_ptrsize_t arg1, mpxp_ptrsize_t arg2)
{
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	if(mpxplay_config_videoplayer_type == MPXPLAY_VIDEOPLAYERTYPE_FFMPEG)
		((FFMpegVideoWidget *)this->filtered_widget)->videowidget_ffmpeg_callback_call(globalpos_x, globalpos_y, command, arg1, arg2); // FIXME: hack
#endif
}

bool DispQtVideoEventFilter::eventFilter(QObject *watched, QEvent *event)
{
	bool accepted = false;
	if(watched != (QObject *)this->filtered_widget)
		return accepted;
	//mpxplay_debugf(DISPQT_DEBUGOUT_EVENT, "video event: %d", event->type());
	accepted = true;
	QMouseEvent *me = static_cast<QMouseEvent*>(event);
	switch(event->type()){
		case QEvent::Close:
			if(this->main_window->gui_config->video_control & DISPQT_CONFIG_VIDEOCTRL_ENABLE_VIDEO)
				this->main_window->CloseWindowInitiateExternal();
			accepted = false;
			break;
		case QEvent::KeyPress:
			{
				QKeyEvent *ke = static_cast<QKeyEvent*>(event);
				mpxplay_debugf(DISPQT_DEBUGOUT_EVENT, "KEY event DispQtVideoEventFilter");
				this->main_window->handle_keypressevent(ke);
			}
			break;
		case QEvent::MouseButtonPress:
			this->mouse_cursor_show_timed(false);
			this->videosurface_get_pos(me, &this->mouse_last_globalpos_x, &this->mouse_last_globalpos_y);
			this->mouse_pressed_buttons = me->button();
			if(this->mouse_pressed_buttons & Qt::LeftButton){  // start or stop playing with single click, fullscreen switch with double click
				if(!this->timer_doubleclick.isActive() && !this->timer_singleclick.isActive() && !this->timer_cmenu_hide.isActive()) {
					this->timer_singleclick.start(DISPQT_VIDEOWIDGET_SINGLECLICK_TIMEOUTMS);
					this->singleclick_fullsrc_start = this->fullsrc_curr = is_video_fullscreen();
					this->singleclick_event = false;
					//mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "MPE");
				}
			} else if(this->mouse_pressed_buttons & Qt::RightButton){  // context menu open
				//this->menuaction_filename->setText(((DispQtVideoPlayer *)this->parent_widget)->video_media_name());
				this->timer_singleclick.stop();
				this->timer_doubleclick.stop();
				if(this->video_surface_contextmenu){
					disconnect(this->video_surface_contextmenu, SIGNAL(aboutToHide()), this, SLOT(cmenu_about_hide()));
					delete this->video_surface_contextmenu;
				}
				this->video_surface_contextmenu = new DispQtVideoSurfaceContextMenu(this->main_window, this, this->filtered_widget);
				connect(this->video_surface_contextmenu, SIGNAL(aboutToHide()), this, SLOT(cmenu_about_hide()));
				this->video_surface_contextmenu->popup(me->globalPos());
			}
			break;
		case QEvent::MouseButtonRelease:
			if(this->mouse_pressed_buttons & Qt::LeftButton)
				this->mouse_cursor_show_timed(true);
			this->mouse_pressed_buttons = Qt::NoButton;
			break;
		case QEvent::MouseButtonDblClick:
			this->timer_doubleclick.start(DISPQT_VIDEOWIDGET_DOUBLECLICK_TIMEOUTMS);
			this->timer_singleclick.stop();
			this->videosurface_get_pos(me, &this->mouse_last_globalpos_x, &this->mouse_last_globalpos_y);
			this->videosurface_handle_doubleclick();
			this->mouse_pressed_buttons = Qt::NoButton;
			//mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "DC se:%d", (int)this->singleclick_event);
			break;
		case QEvent::MouseMove:
			if(this->mouse_pressed_buttons & Qt::LeftButton){ // drag and move video surface
				int move_x, move_y;
				this->videosurface_get_pos(me, &move_x, &move_y);
				move_x = this->mouse_last_globalpos_x - move_x;
				move_y = this->mouse_last_globalpos_y - move_y;
				if((move_x >= 5) || (move_y >= 5)){  // then it's a move event, disable singleclick event  FIXME: still not enough (sometimes the pause is activated)
					this->timer_singleclick.stop();
					QApplication::setOverrideCursor(Qt::OpenHandCursor);
					this->is_cursor_overrided = true;
				}
				this->video_surface_move(move_x, move_y);
				this->videosurface_get_pos(me, &this->mouse_last_globalpos_x, &this->mouse_last_globalpos_y);
				//mpxplay_debugf(DISPQT_DEBUGOUT_EVENT, "move event x:%d", this->mouse_last_globalpos_x);
			}
			else if((this->mouse_pressed_buttons == Qt::NoButton) && this->is_video_fullscreen())
			{
				this->videosurface_handle_moveevent_nobutton(me);
			}
			accepted = false;
			break;
		case QEvent::Wheel:
			{
				QWheelEvent *we = static_cast<QWheelEvent*>(event);
				//mpxplay_debugf(DISPQT_DEBUGOUT_EVENT, "video wheel event %d", we->delta());
				if(me->modifiers() & Qt::ControlModifier){  // FIXME: even if mainwin is hidden, it catches the wheel control on that surface area, zoom doesn't work (Win7 only?)
					this->video_zoom_step(we->globalX(), we->globalY(), (we->delta() < 0)? -1 : 1);
				}else{
					this->main_window->mainwin_handle_keypresscode(kb[(we->delta() < 0)? 83 : 81].c);  // volume up / down
					if((this->main_window->gui_config->mainwin_control & DISPQT_CONFIG_MAINWINCTRL_SHOW_AVMIXER) || this->is_video_wall_enabled())
						emit this->main_window->signal_video_mainwin_wakeup(true, true);
				}
			}
			break;
		default: accepted = false;
			break;
	}
	if(accepted)
		event->accept();
	else
		accepted = QObject::event(event);
	return accepted;
}
