//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2014 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: display playlist editor / dir browser

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT stdout
#define DISPQT_DEBUGOUT_FUNC NULL
#define DISPQT_DEBUG_KEYE_OUTPUT NULL //stdout

#include <QtWidgets>
#include <QTabWidget>
#include "moc_mainwindow.h"
#include "disp_qt.h"
#include "moc_editor.h"
#include "moc_driveline.h"
#include "mpxplay.h"
#include "display/display.h"
#include "playlist/playlist.h"

#define DISPQT_EDITOR_TAB_HEAD_MAX_TEXTLEN 22

extern "C" {
 extern struct mainvars mvps;
 extern unsigned int refdisp, desktopmode, preloadinfo;
 extern keyconfig kb[];
}

PlaylistEditorWidget::PlaylistEditorWidget(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	struct mmc_dispqt_config_s *gcfg = main_window->gui_config;
	this->main_window = main_window;
	if((this->main_window->mainwin_guibkg_is_transparent()) && (gcfg->editor_flags & DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_INWINDOW))
		funcbit_enable(gcfg->editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK);
	else
		funcbit_disable(gcfg->editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK);
	this->setFocusPolicy(Qt::NoFocus);
	this->action_tabs_refresh = new QAction(this);
	this->act_tabsrefresh_userdataid_tabrefreshcfg = this->action_tabs_refresh->registerUserData();
	connect(this->action_tabs_refresh, SIGNAL(triggered()), this, SLOT(tab_headers_refresh_apply()));
	this->editor_build_all(true);
}

void PlaylistEditorWidget::editor_build_all(bool initial)
{
	struct mmc_dispqt_config_s *gcfg = main_window->gui_config;
	this->last_desktopmode = desktopmode;
	this->last_editorflags = gcfg->editor_flags;

	bool lock_success = this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	if(this->last_editorflags & DISPQT_CONFIG_EDITORFLAG_SHOWPANELS_ALL) {
		this->setUpdatesEnabled(false);
		if(this->last_desktopmode & DTM_EDIT_VERTICAL)
			this->PlaylistEditorLayout = new QHBoxLayout; // to make vertical panels
		else
			this->PlaylistEditorLayout = new QVBoxLayout;
		this->PlaylistEditorLayout->setMargin(1);
		this->PlaylistEditorLayout->setSpacing(1);
	} else
		this->PlaylistEditorLayout = NULL;

	for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++) {
		if(this->last_editorflags & (1 << sidenum)) { // !!! DISPQT_CONFIG_EDITORFLAG_SHOWPANEL_
			struct dispqt_playlisteditor_model_config_s pmc;
			this->editor_side_layout[sidenum] = new QVBoxLayout; // to make horizontal fields on one panel
			if(this->last_editorflags & DISPQT_CONFIG_EDITORFLAG_SHOW_DRIVELINE) {
				this->editor_drive_line[sidenum] = new EditorDriveline(main_window, this, sidenum);
				this->editor_side_layout[sidenum]->addWidget(this->editor_drive_line[sidenum]);
			} else
				this->editor_drive_line[sidenum] = NULL;
			this->editor_tab_widget[sidenum] = new PlaylistEditorTabs(main_window, this, sidenum);
			this->editor_side_layout[sidenum]->addWidget(this->editor_tab_widget[sidenum]);
			this->PlaylistEditorLayout->addLayout(this->editor_side_layout[sidenum]);
			pds_memset(&pmc, 0, sizeof(pmc));
			this->editor_tab_widget[sidenum]->tab_create(&pmc);
		} else {
			this->editor_drive_line[sidenum] = NULL;
			this->editor_tab_widget[sidenum] = NULL;
		}
	}

	if(this->last_editorflags & DISPQT_CONFIG_EDITORFLAG_SHOWPANELS_ALL) {
		this->setLayout(this->PlaylistEditorLayout);
		this->setUpdatesEnabled(true);
		if(!initial && !(this->last_editorflags & (1 << mvps.editorside_selected))){
			editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_SET_SIDE, 0, -1, -1, -1);
		}
	}
	if(lock_success)
		this->mutex_editorwidget.unlock();
	mpxplay_debugf(DISPQT_DEBUGOUT_FUNC, "build_all end");
}

void PlaylistEditorWidget::editor_delete_all(void)
{
	mpxplay_debugf(DISPQT_DEBUGOUT_FUNC, "delete_all begin");
	if(!this->PlaylistEditorLayout)
		return;
	bool lock_success = this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++) {
		if(this->editor_drive_line[sidenum])
			this->editor_drive_line[sidenum]->hide();
		if(this->editor_tab_widget[sidenum])
			this->editor_tab_widget[sidenum]->hide();
	}
	delete this->PlaylistEditorLayout;
	this->PlaylistEditorLayout = NULL;
	for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++) {
		this->editor_drive_line[sidenum] = NULL;
		this->editor_tab_widget[sidenum] = NULL;
	}
	if(lock_success)
		this->mutex_editorwidget.unlock();
}

void PlaylistEditorWidget::editor_config_transparent_back(bool on_video)
{
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;
	if( this->main_window->mainwin_guibkg_is_transparent() &&
	   (   (!on_video && (gcfg->editor_flags & DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_INWINDOW) && this->isActiveWindow())
	   ||  ( on_video && (gcfg->editor_flags & DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_ONVIDEO))
	   ||  (!on_video && (gcfg->editor_flags & DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_INACTIVE) && !this->isActiveWindow()) )
	)
		funcbit_enable(gcfg->editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK);
	else
		funcbit_disable(gcfg->editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK);
}

void PlaylistEditorWidget::editor_config_apply(bool on_video)
{
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;

	this->editor_config_transparent_back(on_video);

	if(  ((desktopmode & DISPQT_CONFIG_DESKTOPMODE_FULLREDRAW) != (this->last_desktopmode & DISPQT_CONFIG_DESKTOPMODE_FULLREDRAW))
	  || ((gcfg->editor_flags & DISPQT_CONFIG_EDITORFLAGS_FULLREDRAW) != (this->last_editorflags & DISPQT_CONFIG_EDITORFLAGS_FULLREDRAW))
	){
		funcbit_smp_disable(refdisp, RDT_EDITOR);
		this->editor_delete_all();
		this->editor_build_all(false);
		funcbit_enable(refdisp, (RDT_EDITOR | RDT_RESET_EDIT));
	} else {
		funcbit_smp_disable(refdisp, RDT_EDITOR);
		bool lock_success = this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
		for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++) {
			if(this->editor_tab_widget[sidenum]) {
				this->editor_tab_widget[sidenum]->editortabs_config_apply();
				this->editor_tab_widget[sidenum]->editortabs_config_style_apply();
			}
		}
		this->last_editorflags = gcfg->editor_flags;
		if(lock_success)
			this->mutex_editorwidget.unlock();
		funcbit_enable(refdisp, RDT_EDITOR);
	}
}

void PlaylistEditorWidget::editor_config_style_apply(bool initial)
{
	funcbit_smp_disable(refdisp, RDT_EDITOR);
	bool lock_success = this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	this->editor_config_transparent_back(this->main_window->mainwin_is_on_video());
	for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++) {
		if(this->editor_drive_line[sidenum])
			this->editor_drive_line[sidenum]->driveline_config_style_apply(initial);
		if(this->editor_tab_widget[sidenum])
			this->editor_tab_widget[sidenum]->editortabs_config_style_apply();
	}
	if(lock_success)
		this->mutex_editorwidget.unlock();
	funcbit_enable(refdisp, RDT_EDITOR);
}

void PlaylistEditorWidget::editor_config_style_onvideo_switch(bool activated_by_event, bool mainwin_is_on_video)
{
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;

	this->editor_config_transparent_back(mainwin_is_on_video);

	if((gcfg->editor_flags & DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK) != (this->last_editorflags & DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK)) {
		funcbit_smp_disable(refdisp, RDT_EDITOR);
		bool lock_success = this->mutex_editorwidget.tryLock((activated_by_event)? 0 : DISPQT_MUTEX_TIMEOUT);
		for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++) {
			if(this->editor_tab_widget[sidenum]) {
				if(activated_by_event) // called from mainwin_event_window_activate (a direct call to editortabs_config_style_apply can crash the program (spending too much time in the mainwindow's event?))
					this->editor_tab_widget[sidenum]->editortabs_config_style_change_request();
				else // called from video_fullscreen_switch (perform a direct call for a faster effect)
					this->editor_tab_widget[sidenum]->editortabs_config_style_apply();
			}
		}
		funcbit_copy(this->last_editorflags, gcfg->editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK);
		if(lock_success)
			this->mutex_editorwidget.unlock();
		funcbit_enable(refdisp, RDT_EDITOR);
	}
}

void PlaylistEditorWidget::set_default_focus(struct playlist_side_info *psi)
{
	if(!psi)
		return;
	unsigned int sidenum = psi->sidenum;
	if(sidenum >= PLAYLIST_MAX_SIDES)
		return;
	PlaylistEditorTabs *peb = this->editor_tab_widget[sidenum];
	if(!peb)
		return;
	QWidget *curwd = peb->currentWidget();
	if(curwd)
		curwd->setFocus();
	else
		peb->setFocus();
}

void PlaylistEditorWidget::keyPressEvent(QKeyEvent *event)
{
	mpxplay_debugf(DISPQT_DEBUG_KEYE_OUTPUT, "KEY event  PlaylistEditorWidget");
	this->main_window->handle_keypressevent(event);
}

struct dispqt_playlisteditor_model_config_s *PlaylistEditorWidget::tab_get_pmc(PlaylistEditorTabs **pet_ptr, int sidenum, int tabnum)
{
	struct dispqt_playlisteditor_model_config_s *pmc = NULL;
	PlaylistEditorTabs *pet = NULL;
	if((sidenum >= 0) && (sidenum < PLAYLIST_MAX_SIDES)){
		pet = this->editor_tab_widget[sidenum];
		if(pet){
			PlaylistEditorTable *table = pet->tabs_get_table(tabnum);
			if(table)
				pmc = table->editortable_get_pmc();
		}
	}
	if(pet_ptr)
		*pet_ptr = pet;
	return pmc;
}

//----------------------------------------------------------------------------------
void mpxplay_dispqt_editor_tab_header_create_info(struct playlist_side_info *psi, struct dispqt_playlisteditor_model_config_s *pmc, bool full)
{
	int namepos = 0;
	struct mainvars *mvp;
	char *fullnamep, strtmp[MAX_PATHNAMELEN], path[MAX_PATHNAMELEN];

	if(!psi || !pmc)
		return;

	pmc->sidenum = psi->sidenum;
	pmc->tabnum = psi->tabnum;
	pmc->editloadtype = psi->editloadtype;
	pmc->editsidetype = psi->editsidetype;
	if(psi->sublistlevel)
		funcbit_disable(pmc->editsidetype, PLT_DIRECTORY);
	pmc->id3ordernum_primary = psi->id3ordernum_primary;
	pmc->filenum_displayed = (psi->editloadtype&PLL_FILTERED)? psi->filtered_files : (psi->lastentry - psi->firstentry + 1);
	pmc->filenum_entry_diff = playlist_editlist_entry_diff(psi, psi->firstentry, psi->editor_from);
	pmc->editorhighline_row = playlist_editlist_entry_diff(psi, psi->editor_from, psi->editorhighline);
	mvp = psi->mvp;
	if(psi == mvp->psie)
		funcbit_enable(pmc->flags, DISPQT_EDITOR_MODELCONFIGFLAG_EDITSIDE);
	if(psi->tabnum == mvp->editorsides_selected_tab[psi->sidenum])
		funcbit_enable(pmc->flags, DISPQT_EDITOR_MODELCONFIGFLAG_EDITTAB);
	if(psi == mvp->psip)
		funcbit_enable(pmc->flags, DISPQT_EDITOR_MODELCONFIGFLAG_PLAYTAB);
	pmc->desktopmode = desktopmode;

	strtmp[0]=0;
	fullnamep = NULL;
	if(pmc->editsidetype & PLT_DIRECTORY){
		char *updir = pds_getfilename_from_fullname(psi->currdir);
		if(updir && *updir)
			pds_strcpy(strtmp, updir);
		else
			pds_strcpy(strtmp, psi->currdir);
		fullnamep = &psi->currdir[0];
	}else if(psi->editloadtype & PLL_JUKEBOX) {
		sprintf(strtmp, "JukeBox%d", (psi->tabnum + 1));
	}else if(psi->savelist_filename[0] || ((psi->savelist_filename[0] || !(psi->editloadtype&PLL_RESTORED)) && psi->sublistnames[psi->sublistlevel][0])){
		char *p = (psi->savelist_filename[0])? &psi->savelist_filename[0]:(&psi->sublistnames[psi->sublistlevel][0]);
		fullnamep = p;
		if(pds_filename_wildcard_chk(p)){
			pds_getpath_nowildcard_from_filename(path, p);
			p = &path[0];
		}
		pds_strcpy(strtmp, pds_getfilename_from_fullname(p));
	}
	if(!strtmp[0])
		sprintf(strtmp, "LIST%d", (psi->tabnum + 1));
	if(!fullnamep)
		fullnamep = &strtmp[0];

	if(psi->editloadtype & PLL_LOCKTAB)
		pmc->head_filename[namepos++] = '#';
	if((psi->editloadtype & PLL_CHG_MANUAL) && !(pmc->editsidetype&PLT_DIRECTORY))
		pmc->head_filename[namepos++] = '*';

	pds_strncpy(&pmc->head_filename[namepos], strtmp, sizeof(pmc->head_filename) - namepos);
	pds_strncpy(pmc->head_fullname, fullnamep, sizeof(pmc->head_fullname));
	pmc->head_filename[sizeof(pmc->head_filename) - 1] = 0;
	pmc->head_fullname[sizeof(pmc->head_fullname) - 1] = 0;

	if(full && funcbit_test(pmc->flags, DISPQT_EDITOR_MODELCONFIGFLAG_EDITTAB) && (psi->editsidetype & PLT_DIRECTORY)){
		int i;
		struct playlist_entry_info *pei;
		struct playlist_side_info psi_drvlist;
		char astr_drv[sizeof(PDS_DIRECTORY_ROOTDIR_STR)];

		memset((void *)&psi_drvlist, 0, sizeof(psi_drvlist));
		psi_drvlist.editsidetype=PLT_DIRECTORY;
		psi_drvlist.mvp = &mvps;

		playlist_loaddir_browser_get_drives(&psi_drvlist, psi->currdrive, psi->currdrive);  // !!! loads only the current drive infos
		if(!psi_drvlist.firstentry)
			return;  // !!!
		pmc->nb_drives = (psi_drvlist.lastentry - psi_drvlist.firstentry + 1);
		pmc->drivenames = (char **)pds_calloc(pmc->nb_drives, sizeof(char *));
		if(pmc->drivenames){
			for(pei = psi_drvlist.firstentry, i = 0; pei <= psi_drvlist.lastentry; pei++, i++) {
				int len1 = pds_strlen(pei->filename), len2 = len1;
				if(len1 <= 2)
					len2 += pds_strlen(pei->id3info[I3I_DFT_STORE]) + 1; // +1 space
				pmc->drivenames[i] = (char *)pds_malloc(len2 + 1);
				if(pmc->drivenames[i]){
					pds_strcpy(pmc->drivenames[i], pei->filename);
					if(len1 <= 2){
						pds_strcpy(&pmc->drivenames[i][len1], (char *)" ");
						pds_strcpy(&pmc->drivenames[i][len1 + 1], pei->id3info[I3I_DFT_STORE]);
					}
				}
			}
		}
		playlist_editlist_tab_clear(&psi_drvlist);

		pds_strncpy(astr_drv, psi->currdir, sizeof(astr_drv));
		astr_drv[sizeof(PDS_DIRECTORY_ROOTDIR_STR) - 1] = 0;
		if(astr_drv[sizeof(PDS_DIRECTORY_ROOTDIR_STR) - 2] != PDS_DIRECTORY_SEPARATOR_CHAR)
			astr_drv[sizeof(PDS_DIRECTORY_ROOTDIR_STR) - 2] = PDS_DIRECTORY_SEPARATOR_CHAR;
		pmc->drive_free_space = 0; pmc->drive_total_space = 0;
		pds_drive_getspace(astr_drv, &pmc->drive_free_space, &pmc->drive_total_space);
	}
}

int mpxplay_dispqt_editor_tab_headers_refresh_create(struct mainvars *mvp, PlaylistEditorWidget *pew)
{
	if(!pew)
		return DISPQT_ERROR_EDITORTABLE_MISSING;
	if(pew->action_tabs_refresh->userData(pew->act_tabsrefresh_userdataid_tabrefreshcfg))
		return DISPQT_ERROR_EDITORTABLE_BUSY;
	struct dispqt_playlisteditor_tabrefresh_config_s *trc = (struct dispqt_playlisteditor_tabrefresh_config_s *)pds_malloc(sizeof(struct dispqt_playlisteditor_tabrefresh_config_s));
	pds_memset(trc, 0 , sizeof(*trc));
	for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++)
	{
		trc->nb_tabs_on_side[sidenum] = mvp->editorsides_num_tabs[sidenum];
		for(int tabnum = 0; tabnum < PLAYLIST_MAX_TABS_PER_SIDE; tabnum++)
		{
			struct playlist_side_info *psi = playlist_editlist_tabpsi_get(mvp, sidenum, tabnum);
			mpxplay_dispqt_editor_tab_header_create_info(psi, &trc->tab_infos[sidenum][tabnum], true);
		}
	}

	pew->action_tabs_refresh->setUserData(pew->act_tabsrefresh_userdataid_tabrefreshcfg, (QObjectUserData *)trc);
	pew->action_tabs_refresh->activate(QAction::Trigger);
	return DISPQT_ERROR_OK;
}

void PlaylistEditorWidget::tab_headers_refresh_apply(void)
{
	struct dispqt_playlisteditor_tabrefresh_config_s *trc = (struct dispqt_playlisteditor_tabrefresh_config_s *)this->action_tabs_refresh->userData(this->act_tabsrefresh_userdataid_tabrefreshcfg);
	if(!trc)
		return;
	this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	this->setUpdatesEnabled(false);

	for(int sidenum = 0; sidenum < PLAYLIST_MAX_SIDES; sidenum++)
	{
		PlaylistEditorTabs *tabs = this->editor_tab_widget[sidenum];
		struct dispqt_playlisteditor_model_config_s *pmc;
		int tabnum;
		if(!tabs) // side is not displayed
			continue;
		pmc = &trc->tab_infos[sidenum][0];
		for(tabnum = 0; tabnum < trc->nb_tabs_on_side[sidenum]; tabnum++, pmc++)
		{
			tabs->tab_create(pmc);
			if(pmc->flags & DISPQT_EDITOR_MODELCONFIGFLAG_EDITTAB){
				tabs->tab_select(tabnum);
				this->tab_changed_handler(pmc);
			}
		}
		for(; tabnum < PLAYLIST_MAX_TABS_PER_SIDE; tabnum++, pmc++)
		{
			pmc->tabnum = tabnum;
			tabs->tab_delete(pmc);
		}
	}

	this->action_tabs_refresh->setUserData(this->act_tabsrefresh_userdataid_tabrefreshcfg, NULL);
	pds_free(trc);
	this->setUpdatesEnabled(true);
	this->mutex_editorwidget.unlock();
}

void PlaylistEditorWidget::tab_changed_handler(struct dispqt_playlisteditor_model_config_s *pmc)
{
	EditorDriveline *edl = this->editor_drive_line[pmc->sidenum];
	if(edl && edl->drive_combobox)
		edl->drive_combobox->drive_list_select_by_tabpath(pmc, QString::fromUtf8(pmc->head_fullname));
}

int PlaylistEditorWidget::table_refresh_create(struct playlist_side_info *psi)
{
	int retcode = DISPQT_ERROR_EDITORTABLE_ARGS;
	if(!psi)
		return retcode;
	unsigned int sidenum = psi->sidenum;
	if(sidenum >= PLAYLIST_MAX_SIDES)
		return retcode;
	bool lock_success = this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	PlaylistEditorTabs *tabs = this->editor_tab_widget[sidenum];
	if(tabs)
		retcode = tabs->table_refresh_create(psi);
	if(lock_success)
		this->mutex_editorwidget.unlock();
	return retcode;
}

void PlaylistEditorWidget::line_select_request(struct playlist_side_info *psi, struct playlist_entry_info *pei)
{
	if(!psi || !pei)
		return;
	unsigned int sidenum = psi->sidenum;
	if(sidenum >= PLAYLIST_MAX_SIDES)
		return;
	bool lock_success = this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	PlaylistEditorTabs *tabs = this->editor_tab_widget[sidenum];
	if(tabs)
		tabs->line_select_request(psi, pei);
	if(lock_success)
		this->mutex_editorwidget.unlock();
}

int PlaylistEditorWidget::editortable_line_count(struct playlist_side_info *psi)
{
	int line_count = 0;
	PlaylistEditorTable *table = NULL;
	if(!psi)
		return line_count;
	unsigned int sidenum = psi->sidenum, tabnum = psi->tabnum;
	if(sidenum >= PLAYLIST_MAX_SIDES)
		return line_count;
	bool lock_success = this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	PlaylistEditorTabs *tabs = this->editor_tab_widget[sidenum];
	if(tabs){
		table = tabs->tabs_get_table(tabnum);
		if(table)
			line_count = table->line_count();
	}
	if(lock_success)
		this->mutex_editorwidget.unlock();
	return line_count;
}

void mpxplay_dispqt_editor_event_callback(void *passdata) // called from Mpxplay's main thread timer
{
	struct dispqt_editor_callbackdata_s *deci = (struct dispqt_editor_callbackdata_s *)passdata;
	struct playlist_side_info *psi;
	struct mainvars *mvp = &mvps;

	if(!deci)
		return;

	switch(deci->cbcmd)
	{
		case DISPQT_EDITOR_CBCMD_SET_SIDE:
			playlist_editlist_tab_select(mvp, deci->sidenum, deci->tabnum);
			goto err_out_event;
		case DISPQT_EDITOR_CBCMD_TAB_ADD:
			playlist_editlist_tab_add(mvp, deci->mode, deci->sidenum, deci->tabnum, deci->dest_sidenum);
			goto err_out_event;
		case DISPQT_EDITOR_CBCMD_TAB_DEL:
			playlist_editlist_tab_del(mvp, deci->sidenum, deci->tabnum);
			goto err_out_event;
	}

	psi = playlist_editlist_tabpsi_get(mvp, deci->sidenum, deci->tabnum);
	if(!psi)
		goto err_out_event;

	deci->sidenum = psi->sidenum; // for -1 arguments
	deci->tabnum = psi->tabnum;   //

	funcbit_enable(deci->flags,DISPQT_EDITOR_FLAG_CONTENTREFRESH);

	switch(deci->cbcmd)
	{
		case DISPQT_EDITOR_CBCMD_TAB_LOCK:
			funcbit_inverse(psi->editloadtype, PLL_LOCKTAB);
			break;
		case DISPQT_EDITOR_CBCMD_SAVELISTAS:
			playlist_savelist_manual_save(psi);
			break;
		case DISPQT_EDITOR_CBCMD_RELOAD:
			playlist_reload_side(mvp, psi);
			break;
		case DISPQT_EDITOR_CBCMD_DIRSWITCH:
			if(psi->editsidetype & PLT_DIRECTORY)
				playlist_loaddir_switch_to_playlist(psi);
			else {
				char *dir = (psi->currdir[0])? (&psi->currdir[0]) : NULL;
				if(!dir) {
					struct playlist_side_info *pss = playlist_editlist_tabpsi_get(mvp, deci->sidenum, 0);
					unsigned int i, num_tabs = mvps.editorsides_num_tabs[deci->sidenum];
					for(i = 0; i < num_tabs; i++, pss++) {
						if(pss->currdir[0]){
							dir = &pss->currdir[0];
							break;
						}
					}
				}
				if(!dir)
					dir = mpxplay_playlist_startdir();
				//loaddir_selectdrive_playlist_to_sublist(psi->psio);
				psi->editsidetype = PLT_DIRECTORY;
				psi->editloadtype = 0;
				playlist_loaddir_initbrowser_tab(psi, dir);
				playlist_loaddir_buildbrowser(psi);
				playlist_id3list_load(psi->mvp, psi);
				playlist_chkfile_start_norm(psi, NULL);
			}
			break;
		case DISPQT_EDITOR_CBCMD_RANDOMIZE:
			playlist_randlist_randomize_side(psi);
			if(preloadinfo & PLI_DISPLOAD)
				display_editorside_reset(psi);
			break;
		case DISPQT_EDITOR_CBCMD_CLEARLIST:
			mpxplay_playlist_sideclear_entries(psi);
			playlist_disable_side_full(psi);
			mpxplay_playlist_sideclear_sidetype(psi);
			psi->currdir[0]=0;
			break;
		case DISPQT_EDITOR_CBCMD_SORT_BY_COL:
			playlist_sortlist_do_sort_by_col(psi, deci->mode);
			break;
		case DISPQT_EDITOR_CBCMD_SORT_BY_KEY:
			playlist_sortlist_do_sort_by_key(psi, deci->mode);
			mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "DISPQT_EDITOR_CBCMD_SORT s:%d t:%d m:%d psi:%8.8X", deci->sidenum, deci->tabnum, deci->mode, (unsigned long)psi);
			break;
		default:
			funcbit_disable(deci->flags,DISPQT_EDITOR_FLAG_CONTENTREFRESH);
			break;
	}

	if(deci->flags & DISPQT_EDITOR_FLAG_CONTENTREFRESH){
		if(deci->tabnum != mvp->editorsides_selected_tab[deci->sidenum])
			funcbit_enable(refdisp, RDT_TABINFOS);
//		else									// FIXME: unneccessary editor refresh (pmc fill rowflags)
			funcbit_enable(refdisp, RDT_EDITOR);
		if(psi && (psi == mvp->psip))
			funcbit_enable(refdisp, RDT_BROWSER);
	}

err_out_event:
	pds_free(deci);
}

void PlaylistEditorWidget::editortabs_editlisteventcb_init(unsigned int cmd, unsigned int mode, int sidenum, int tabnum, int dest_sidenum)
{
	//this->mutex_editorwidget.tryLock(DISPQT_MUTEX_TIMEOUT);
	struct dispqt_editor_callbackdata_s *deci = (struct dispqt_editor_callbackdata_s *)pds_malloc(sizeof(struct dispqt_editor_callbackdata_s));
	deci->cbcmd = cmd;
	deci->flags = 0;
	deci->mode = mode;
	deci->sidenum = sidenum;
	deci->tabnum = tabnum;
	deci->dest_sidenum = dest_sidenum;
	deci->linenum = -1; // don't change linenum
	mpxplay_dispqt_mainthread_callback_init((void *)mpxplay_dispqt_editor_event_callback, (void *)deci, DISPQT_CB_FLAG_MULTYCALL);
	//this->mutex_editorwidget.unlock();
}

//------------------------------------------------------------------------------------------------
// PlaylistEditorTabBar class

PlaylistEditorTabBar::PlaylistEditorTabBar(MainWindow *main_window, QWidget *parent, unsigned int sidenum) : QTabBar(parent)
{
	this->main_window = main_window;
	this->parent_widget = parent;
	this->side_num = sidenum;
	if(this->main_window->mainwin_guilayout_is_nonclassic())
		this->tabbar_config_style_apply(true);
	this->setFocusPolicy(Qt::NoFocus);
	setUsesScrollButtons(true);
}

void PlaylistEditorTabBar::tabbar_config_style_apply(bool initial)
{
	if(this->main_window->mainwin_guilayout_is_nonclassic()) {
		setDocumentMode(true);
		this->setStyleSheet("QTabBar { background: transparent }");
	} else if(!initial){
		setDocumentMode(false);
		this->setStyleSheet(QString());
	}
}

void PlaylistEditorTabBar::mousePressEvent (QMouseEvent *event)
{
	if (event->button() == Qt::LeftButton) {
		int tabCount = count();
		for( int i=0; i< tabCount; i++ ) {
			if (tabRect(i).contains(event->pos())) {
				this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_SET_SIDE, 0, this->side_num, i, -1);
				event->accept();
				break;
			}
		}
	} else
		QTabBar::mousePressEvent(event);
	// emit this->main_window->signal_video_mainwin_wakeup(false, false); // FIXME: ???
}

void PlaylistEditorTabBar::keyPressEvent(QKeyEvent *event)
{
    mpxplay_debugf(DISPQT_DEBUG_KEYE_OUTPUT, "KEY event  PlaylistEditorTabBar");
    this->main_window->handle_keypressevent(event);
}

void PlaylistEditorTabBar::wheelEvent(QWheelEvent* event)
{
	emit this->main_window->signal_video_mainwin_wakeup(true, false);
	event->accept();
}

//------------------------------------------------------------------------------------------------
// PlaylistEditorTabs class

static const unsigned char editortab_sort_elems[DISPQT_EDITOR_CONTEXTMENU_SORT_ELEMS] =
{
 ID3ORDER_PATHFILE, ID3ORDER_ARTIST, ID3ORDER_TITLE, ID3ORDER_TIME, ID3ORDER_FILEEXT, ID3ORDER_FILESIZE, ID3ORDER_FILEDATE
};

static const char *editortab_sort_names[DISPQT_EDITOR_CONTEXTMENU_SORT_ELEMS] =
{
 "Filename", "Artist", "Title", "Duration", "File extension", "File size", "File date",
};

static unsigned int dispqt_editor_orderkey_to_sortelemindex(unsigned int orderkey)
{
	unsigned int sortelemindex = 0;
	for(int i = 0; i < DISPQT_EDITOR_CONTEXTMENU_SORT_ELEMS; i++) {
		if(editortab_sort_elems[i] == orderkey){
			sortelemindex = i;
			break;
		}
	}
	return sortelemindex;
}

PlaylistEditorTabs::PlaylistEditorTabs(MainWindow *main_window, QWidget *parent, unsigned int sidenum) : DispQtDialogElemTabWidget(main_window, parent)
{
	this->main_window = main_window;
	this->parent_widget = parent;
	this->side_num = sidenum;
	this->selected_tab_num = 0;
	this->sort_last_elem_index = 0;
	this->is_config_style_change_request = false;
	this->setFocusPolicy(Qt::NoFocus);

	for(int tabnum = 0; tabnum < PLAYLIST_MAX_TABS_PER_SIDE; tabnum++) {
		this->editor_tab_tables[tabnum] = NULL;
		this->editor_tab_indexes[tabnum] = -1;
	}

	this->action_line_select = new QAction(this);
	this->act_lineselect_userdataid_reqinfo = this->action_line_select->registerUserData();
	connect(this->action_line_select, SIGNAL(triggered()), this, SLOT(line_select_apply()));

	this->tab_head_contextmenu = new DispQtDialogElemQmenu("Tab Menu", this->main_window, this, DISPQT_DIALOG_WINDOWTYPE_QMENU_LIST);
	this->tab_head_menuact_tabname = this->tab_head_contextmenu->addAction("");
	this->tab_head_menuact_tabname->setFont(QFont("", -1, QFont::DemiBold, false));
	this->tab_head_contextmenu->addSeparator();
	this->tab_head_menuact_duplicate = this->tab_head_contextmenu->addAction(QIcon(":/images/page_copy.png"), "Duplicate tab");
	this->tab_head_menuact_duplicate->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_T));
	this->tab_head_menuact_copyToOtherSide = this->tab_head_contextmenu->addAction( "Copy tab to other side");
	this->tab_head_menuact_locktab = this->tab_head_contextmenu->addAction("Lock tab");
	this->tab_head_menuact_newEmptyTab = this->tab_head_contextmenu->addAction(QIcon(":/images/list-add.png"), "New empty tab");
	this->tab_head_menuact_remove = this->tab_head_contextmenu->addAction(QIcon(":/images/fileclose.png"), "Remove tab");
	this->tab_head_menuact_remove->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_W));
	this->tab_head_contextmenu->addSeparator();
	this->tab_head_menuact_loadfiles = this->tab_head_contextmenu->addAction(QIcon(":/images/open.png"), "Add files");
	this->tab_head_menuact_loaddir = this->tab_head_contextmenu->addAction(QIcon(":/images/open.png"), "Add directory");
	this->tab_head_menuact_loadurl = this->tab_head_contextmenu->addAction(QIcon(":/images/open.png"), "Add url");
	this->tab_head_menuact_savelistas = this->tab_head_contextmenu->addAction(QIcon(":/images/save.png"), "Save list As");
	this->tab_head_menuact_savelistas->setShortcut(QKeySequence(Qt::Key_F2));
	this->tab_head_contextmenu->addSeparator();
	this->tab_head_menuact_reload = this->tab_head_contextmenu->addAction(QIcon(":/images/arrow_refresh.png"), "Reload dir/list");
	this->tab_head_menuact_reload->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_R));
	this->tab_head_menuact_randomize = this->tab_head_contextmenu->addAction(QIcon(":/images/arrow_switch.png"), "Randomize list");
	this->tab_head_menuact_randomize->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_N));
	this->tab_head_menuact_dirswitch = this->tab_head_contextmenu->addAction("Dir/List type select");
	this->tab_head_menuact_clearlist = this->tab_head_contextmenu->addAction(QIcon(":/images/new.png"), "Clear list");
	this->tab_head_menuact_clearlist->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_Delete));
	this->tab_head_contextmenu->addSeparator();

	this->tab_head_contexSubSortMenu = this->tab_head_contextmenu->addMenu("Sort by");
	for(int i = 0; i < DISPQT_EDITOR_CONTEXTMENU_SORT_ELEMS; i++) {
		this->tab_head_menuact_sort[i] = this->tab_head_contexSubSortMenu->addAction(QString::fromLatin1(editortab_sort_names[i], pds_strlen((char *)editortab_sort_names[i])));
		connect(this->tab_head_menuact_sort[i], SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_sort()));
	}
	this->tab_head_contextmenu_add_vary_texts();

	connect(this->tab_head_menuact_duplicate, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_duplicate()));
	connect(this->tab_head_menuact_copyToOtherSide, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_copyToOtherSide()));
	connect(this->tab_head_menuact_locktab, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_locktab()));
	connect(this->tab_head_menuact_newEmptyTab, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_newEmptyTab()));
	connect(this->tab_head_menuact_remove, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_remove()));
	connect(this->tab_head_menuact_loadfiles, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_loadfiles()));
	connect(this->tab_head_menuact_loaddir, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_loaddir()));
	connect(this->tab_head_menuact_loadurl, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_loadurl()));
	connect(this->tab_head_menuact_savelistas, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_savelistas()));
	connect(this->tab_head_menuact_reload, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_reload()));
	connect(this->tab_head_menuact_dirswitch, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_dirswitch()));
	connect(this->tab_head_menuact_randomize, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_randomize()));
	connect(this->tab_head_menuact_clearlist, SIGNAL(triggered()),this,SLOT(tab_head_contextmenuaction_clearlist()));

	PlaylistEditorTabBar *petb = new PlaylistEditorTabBar(main_window, this, sidenum);
	this->editor_tab_bar = petb;
	this->setTabBar(petb);
	connect(petb, SIGNAL(customContextMenuRequested(const QPoint &)), SLOT(showContextMenu(const QPoint &)));
	petb->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(this, SIGNAL(signal_tab_head_lockedfuncmsg()), this, SLOT(tab_head_contexmenu_lockedfunc_msg()));
	connect(this, SIGNAL(signal_table_refresh(struct dispqt_playlisteditor_model_config_s *)), this, SLOT(table_refresh_apply(struct dispqt_playlisteditor_model_config_s *)));
}

void PlaylistEditorTabs::editortabs_config_apply(void)
{
	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);
	for(int tabnum = 0; tabnum < PLAYLIST_MAX_TABS_PER_SIDE; tabnum++) {
		if(this->editor_tab_tables[tabnum])
			this->editor_tab_tables[tabnum]->table_config_apply();
	}
	if(lock_success)
		this->mutex_editortabs.unlock();
}

void PlaylistEditorTabs::editortabs_config_style_apply(void)
{
	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);
	this->tabwidget_config_style_apply(false);
	this->editor_tab_bar->tabbar_config_style_apply(false);
	for(int tabnum = 0; tabnum < PLAYLIST_MAX_TABS_PER_SIDE; tabnum++) {
		if(this->editor_tab_tables[tabnum])
			this->editor_tab_tables[tabnum]->table_config_color(false);
	}
	this->is_config_style_change_request = false;
	if(lock_success)
		this->mutex_editortabs.unlock();
}

void PlaylistEditorTabs::editortabs_config_style_change_request(void)
{
	this->is_config_style_change_request = true;
}

void PlaylistEditorTabs::keyPressEvent(QKeyEvent *event)
{
    mpxplay_debugf(DISPQT_DEBUG_KEYE_OUTPUT, "KEY event PlaylistEditorTabs");
    this->main_window->handle_keypressevent(event);
}

//------------------------------------------------------------------------------------------
// tab right click menu actions
void PlaylistEditorTabs::tab_head_contextmenu_add_vary_texts(void)
{
	struct playlist_side_info *psi = playlist_editlist_tabpsi_get(&mvps, this->side_num, this->selected_tab_num); // TODO: not thread safe
	unsigned int primary_key_new = ID3ORDER_DISABLED;

	if(psi) {
		primary_key_new = psi->id3ordernum_primary;
		if(psi->editloadtype & PLL_LOCKTAB) {
			this->tab_head_menuact_locktab->setText("Unlock tab");
			this->tab_head_menuact_locktab->setIcon(QIcon(":/images/lock_open.png"));
			this->tab_head_menuact_remove->setEnabled(false);
			this->tab_head_menuact_loadfiles->setEnabled(false);
			this->tab_head_menuact_loaddir->setEnabled(false);
			this->tab_head_menuact_loadurl->setEnabled(false);
			this->tab_head_menuact_dirswitch->setEnabled(false);
			this->tab_head_menuact_clearlist->setEnabled(false);
		} else {
			this->tab_head_menuact_locktab->setText("Lock tab");
			this->tab_head_menuact_locktab->setIcon(QIcon(":/images/lock.png"));
			this->tab_head_menuact_remove->setEnabled(true);
			this->tab_head_menuact_loadfiles->setEnabled(true);
			this->tab_head_menuact_loaddir->setEnabled(true);
			this->tab_head_menuact_loadurl->setEnabled(true);
			this->tab_head_menuact_dirswitch->setEnabled(true);
			this->tab_head_menuact_clearlist->setEnabled(true);
		}
	}

	this->tab_head_menuact_tabname->setText(this->tabText(this->selected_tab_num));

	this->tab_head_menuact_sort[this->sort_last_elem_index]->setIcon(QIcon(""));
	this->sort_last_elem_index = dispqt_editor_orderkey_to_sortelemindex(primary_key_new);
	this->tab_head_menuact_sort[this->sort_last_elem_index]->setIcon(QIcon(":/images/tick.png"));

	this->tab_head_menuact_sort[0]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F1));
	if((desktopmode&DTM_MASK_COMMANDER) == DTM_MASK_COMMANDER) {
		this->tab_head_menuact_sort[1]->setShortcut(QKeySequence());
		this->tab_head_menuact_sort[2]->setShortcut(QKeySequence());
		this->tab_head_menuact_sort[3]->setShortcut(QKeySequence());
		this->tab_head_menuact_sort[4]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F2));
		this->tab_head_menuact_sort[5]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F3));
		this->tab_head_menuact_sort[6]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F4));
	} else {
		this->tab_head_menuact_sort[1]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F2));
		this->tab_head_menuact_sort[2]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F3));
		this->tab_head_menuact_sort[3]->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F4));
		this->tab_head_menuact_sort[4]->setShortcut(QKeySequence());
		this->tab_head_menuact_sort[5]->setShortcut(QKeySequence());
		this->tab_head_menuact_sort[6]->setShortcut(QKeySequence());
	}
}

void PlaylistEditorTabs::tab_head_contexmenu_lockedfunc_msg(void)
{
	QMessageBox* msg = new QMessageBox(this->parent_widget);
	msg->setWindowTitle("Warning");
	msg->setText("Tab is locked!\n(unlock it)");
	msg->setIconPixmap(QPixmap(":/images/lock.png"));
	msg->show();

	PlaylistEditorTabs *pet;
	struct dispqt_playlisteditor_model_config_s *pmc = this->main_window->PlaylistEditor->tab_get_pmc(&pet, this->side_num, -1);
	if(pmc && pet && pet->editor_tab_tables[pmc->tabnum])
		pet->editor_tab_tables[pmc->tabnum]->setFocus();
}

void PlaylistEditorTabs::tab_head_contextmenuaction_duplicate(void)
{
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_TAB_ADD, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_copyToOtherSide(void)
{
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_TAB_ADD, EDITLIST_ADDTABMODE_TOEND, this->side_num, this->selected_tab_num, !this->side_num);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_locktab(void)
{
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_TAB_LOCK, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_newEmptyTab(void)
{
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_TAB_ADD, (EDITLIST_ADDTABMODE_BLANK | EDITLIST_ADDTABMODE_TOEND), this->side_num, -1, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_remove(void)
{
	struct dispqt_playlisteditor_model_config_s *pmc = this->main_window->PlaylistEditor->tab_get_pmc(NULL, this->side_num, this->selected_tab_num);
	if(!pmc)
		return;
	if(funcbit_test(pmc->editloadtype, PLL_LOCKTAB)) {
		this->tab_head_contexmenu_lockedfunc_msg();
		return;
	}
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_TAB_DEL, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_loadfiles(void)
{
	this->main_window->mainwin_load_files_or_dir(this->side_num, this->selected_tab_num, Mainwindow_SelectLoadFiles);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_loaddir(void)
{
	this->main_window->mainwin_load_files_or_dir(this->side_num, this->selected_tab_num, Mainwindow_SelectLoadDir);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_loadurl(void)
{
	this->main_window->mainwin_load_files_or_dir(this->side_num, this->selected_tab_num, Mainwindow_SelectLoadUrl);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_savelistas(void)
{
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_SAVELISTAS, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_reload(void)
{
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_RELOAD, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_dirswitch(void)
{
	struct dispqt_playlisteditor_model_config_s *pmc = this->main_window->PlaylistEditor->tab_get_pmc(NULL, this->side_num, this->selected_tab_num);
	if(!pmc)
		return;
	if(funcbit_test(pmc->editloadtype, PLL_LOCKTAB)) {
		this->tab_head_contexmenu_lockedfunc_msg();
		return;
	}
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_DIRSWITCH, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_randomize(void)
{
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_RANDOMIZE, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_clearlist(void)
{
	struct dispqt_playlisteditor_model_config_s *pmc = this->main_window->PlaylistEditor->tab_get_pmc(NULL, this->side_num, this->selected_tab_num);
	if(!pmc)
		return;
	if(funcbit_test(pmc->editloadtype, PLL_LOCKTAB)) {
		this->tab_head_contexmenu_lockedfunc_msg();
		return;
	}
	this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_CLEARLIST, 0, this->side_num, this->selected_tab_num, -1);
}

void PlaylistEditorTabs::tab_head_contextmenuaction_sort(void)
{
	QAction *actionThatUserChose = qobject_cast<QAction *>(sender());
	for(int i = 0; i < DISPQT_EDITOR_CONTEXTMENU_SORT_ELEMS; i++) {
		if(actionThatUserChose == this->tab_head_menuact_sort[i]) {
			this->main_window->PlaylistEditor->editortabs_editlisteventcb_init(DISPQT_EDITOR_CBCMD_SORT_BY_KEY, editortab_sort_elems[i], this->side_num, this->selected_tab_num, -1);
			break;
		}
	}
}

void PlaylistEditorTabs::showContextMenu(const QPoint &point)
{
	if (point.isNull())
		return;
	int tabIndex = this->tabBar()->tabAt(point);
	if(tabIndex < 0)
		return;
	if(!this->tabs_get_table(tabIndex))
		return;
	this->main_window->signal_video_mainwin_opacity_disable(true);
	this->selected_tab_num = tabIndex;
	this->tab_head_contextmenu_add_vary_texts();
	this->tab_head_contextmenu->exec(this->tabBar()->mapToGlobal(point));
	this->main_window->signal_video_mainwin_opacity_disable(false);
}

//-----------------------------------------------------------------------------------------
void PlaylistEditorTabs::tab_create(struct dispqt_playlisteditor_model_config_s *pmc)
{
	PlaylistEditorTable *pet;
	int tabnum;

	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);
	tabnum = pmc->tabnum;
	if(tabnum >= PLAYLIST_MAX_TABS_PER_SIDE)
		goto err_out_create;
	pet = this->editor_tab_tables[tabnum];
	if(pet)
		goto err_out_exists;

	pet = new PlaylistEditorTable(this->main_window, this, this->side_num);
	if(!pet)
		goto err_out_create;
	this->editor_tab_tables[tabnum] = pet;

	this->editor_tab_indexes[tabnum] = this->addTab(pet, tr(" "));
	if(this->editor_tab_indexes[tabnum] < 0) {
		delete pet;
		goto err_out_create;
	}

	if(this->main_window->gui_config->editor_flags & DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_BACK)
		pet->table_config_color(false);

err_out_exists:
	pet->pmcdata_update(pmc, false);
	this->tab_header_apply_info(pmc);

err_out_create:
	if(lock_success)
		this->mutex_editortabs.unlock();
}

void PlaylistEditorTabs::tab_delete(struct dispqt_playlisteditor_model_config_s *pmc)
{
	int tabnum = pmc->tabnum;
	if(tabnum >= PLAYLIST_MAX_TABS_PER_SIDE)
		return;
	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);
	if(this->editor_tab_indexes[tabnum] >= 0) {
		this->removeTab(this->editor_tab_indexes[tabnum]);
		this->editor_tab_indexes[tabnum] = -1;
	}
	if(this->editor_tab_tables[tabnum]) {
		delete this->editor_tab_tables[tabnum];
		this->editor_tab_tables[tabnum] = NULL;
	}
	if(lock_success)
		this->mutex_editortabs.unlock();
}

bool PlaylistEditorTabs::tab_select(unsigned int tabnum)
{
	if((tabnum >= PLAYLIST_MAX_TABS_PER_SIDE) || (this->editor_tab_indexes[tabnum] < 0))
		return false;
	if(this->currentIndex() != this->editor_tab_indexes[tabnum])
		this->setCurrentIndex(this->editor_tab_indexes[tabnum]);
	return true;
}

void PlaylistEditorTabs::tab_header_apply_info(struct dispqt_playlisteditor_model_config_s *pmc)
{
	if(!pmc || (this->editor_tab_indexes[pmc->tabnum] < 0))
		return;
	this->setTabText(this->editor_tab_indexes[pmc->tabnum], QString::fromUtf8(pmc->head_filename, DISPQT_EDITOR_TAB_HEAD_MAX_TEXTLEN));
	this->setTabToolTip(this->editor_tab_indexes[pmc->tabnum], QString::fromUtf8(pmc->head_fullname));
	if(funcbit_test(pmc->flags, DISPQT_EDITOR_MODELCONFIGFLAG_PLAYTAB)) {
		this->setTabIcon(this->editor_tab_indexes[pmc->tabnum], QIcon(":/images/player_play.png"));
	} //else if(psi->editloadtype & PLL_LOCKTAB)
		//this->setTabIcon(this->editor_tab_indexes[psi->tabnum], QIcon(":/images/lock.png"));
	else
		this->setTabIcon(this->editor_tab_indexes[pmc->tabnum], QIcon(""));
}

int PlaylistEditorTabs::table_refresh_create(struct playlist_side_info *psi)
{
	struct dispqt_playlisteditor_model_config_s *pmc;
	int retcode = DISPQT_ERROR_EDITORTABLE_ARGS;
	PlaylistEditorTable *table;

	if(!psi)
		return retcode;

	const unsigned int tabnum = psi->tabnum;
	if(tabnum >= PLAYLIST_MAX_TABS_PER_SIDE)
		return retcode;

	if(!this->mutex_editortabs.tryLock(0))
		return DISPQT_ERROR_EDITORTABLE_BUSY;

	table = this->editor_tab_tables[tabnum];
	if(!table){
		retcode = DISPQT_ERROR_EDITORTABLE_MISSING;
		goto err_out_ttrc;
	}
	retcode = table->table_refresh_create(psi, &pmc);

err_out_ttrc:
	this->mutex_editortabs.unlock();
	if(retcode == DISPQT_ERROR_OK){
		mpxplay_debugf(DISPQT_DEBUGOUT_FUNC,"table_refresh_create trigger END t:%3.2f", (float)pds_gettimeh() / 100.0f);
		emit this->signal_table_refresh(pmc);
	}
	return retcode;
}

void PlaylistEditorTabs::table_refresh_apply(struct dispqt_playlisteditor_model_config_s *pmc)
{
	mpxplay_debugf(DISPQT_DEBUGOUT_FUNC,"table_refresh_apply BEGIN t:%3.2f", (float)pds_gettimeh() / 100.0f);
	if(!pmc)
		return;
	if(this->is_config_style_change_request)
		this->editortabs_config_style_apply();
	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);
	PlaylistEditorTable *table = this->editor_tab_tables[pmc->tabnum];
	if(!table) {
		mpxplay_dispqt_editor_table_pmcdata_dealloc(pmc);
		goto err_out_tra;
	}
	table->table_refresh_apply(pmc);
	this->tab_header_apply_info(pmc);

err_out_tra:
	mpxplay_debugf(DISPQT_DEBUGOUT_FUNC,"table_refresh_apply END");
	if(lock_success)
		this->mutex_editortabs.unlock();
}

void PlaylistEditorTabs::line_select_request(struct playlist_side_info *psi, struct playlist_entry_info *pei)
{
	if(!psi)
		return;
	unsigned int tabnum = psi->tabnum;
	if(tabnum >= PLAYLIST_MAX_TABS_PER_SIDE)
		return;
	if(this->action_line_select->userData(this->act_lineselect_userdataid_reqinfo))
		return;
	struct dispqt_editor_lineselect_request_s *lsrq = (struct dispqt_editor_lineselect_request_s *)pds_malloc(sizeof(struct dispqt_editor_lineselect_request_s));
	if(!lsrq)
		return;
	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);
	lsrq->flags = 0;
	if(psi == psi->mvp->psie)
		lsrq->flags |= DISPQT_EDITOR_FLAG_SETFOCUS;
	lsrq->sidenum = psi->sidenum;
	lsrq->tabnum = tabnum;
	lsrq->linenum = playlist_editlist_entry_diff(psi, psi->editor_from, pei);

	this->action_line_select->setUserData(this->act_lineselect_userdataid_reqinfo, (QObjectUserData *)(lsrq));
	if(lock_success)
		this->mutex_editortabs.unlock();
	this->action_line_select->activate(QAction::Trigger);
	//mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "LSR s:%d t:%d i:%2d ef:%8.8X ", psi->sidenum, tabnum, lsrq->linenum, (unsigned long)(psi->editor_from - psi->firstentry));
}

void PlaylistEditorTabs::line_select_apply(void)
{
	struct dispqt_editor_lineselect_request_s *lsrq = (struct dispqt_editor_lineselect_request_s *)this->action_line_select->userData(this->act_lineselect_userdataid_reqinfo);
	PlaylistEditorTable *pet;

	if(!lsrq)
		return;

	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);

	if(!this->tab_select(lsrq->tabnum))
		goto err_out_apply;

	pet = this->editor_tab_tables[lsrq->tabnum];
	if(!pet)
		goto err_out_apply;

	if(lsrq->flags & DISPQT_EDITOR_FLAG_SETFOCUS)
		pet->setFocus();

	pet->table_line_select(lsrq->linenum);

err_out_apply:
	this->action_line_select->setUserData(this->act_lineselect_userdataid_reqinfo, NULL);
	pds_free(lsrq);
	if(lock_success)
		this->mutex_editortabs.unlock();
}

PlaylistEditorTable *PlaylistEditorTabs::tabs_get_table(int tabnum)
{
	PlaylistEditorTable *table;
	if(tabnum < 0)
		tabnum = this->selected_tab_num;
	if(tabnum >= PLAYLIST_MAX_TABS_PER_SIDE)
		return NULL;
	bool lock_success = this->mutex_editortabs.tryLock(DISPQT_MUTEX_TIMEOUT);
	table = this->editor_tab_tables[tabnum];
	if(lock_success)
		this->mutex_editortabs.unlock();
	return table;
}
