//**************************************************************************
//*                     This file is part of the                           *
//*                MMC - Mpxplay Multimedia Commander                      *
//*                   The source code of MMC is                            *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: menu -> configure

//#define MPXPLAY_USE_DEBUGF 1
#define DISPQT_DEBUG_OUTPUT NULL //stdout
#define DISPQT_DEBUG_STREAM NULL // stdout
#define DISPQT_DEBUGOUT_QE stdout

#include <QtGui>
#include <QtWidgets>
#include <QLineEdit>
#include <QLabel>
#include <QFontDialog>
#include "disp_qt.h"
#include "moc_config.h"
#include "moc_mainwindow.h"
#include "moc_video_qt.h"
#include "control/cntfuncs.h"
#include "deparser/ffmpgdec.h"
#include "display/display.h"
#include <libavcodec/version.h>

extern "C" {
 extern unsigned long displaymode, desktopmode, desktopmod_player, desktopmod_commander;
 extern unsigned long playcontrol, playrand, playreplay, su_startupenabled, mpxplay_programcontrol;
 extern unsigned long mpxplay_playlistcontrol, mpxplay_playlistcontrol_mpxpini, loadid3tag, loadid3tag_mpxpini;
 extern unsigned long preloadinfo, preloadinfo_mpxpini, mpxplay_sortcontrol, outmode, outmode_mpxpini;
 extern unsigned long mpxplay_progtimebegin,mpxplay_playtimestart,mpxplay_progtimelen,mpxplay_progtimeexit;
 extern unsigned long intsoundconfig,refdisp;
 extern char mpxplay_extrafiletypes_config[MPXINI_MAX_CHARDATA_LEN];
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
 extern unsigned long mpxplay_config_dvbepg_control_flags, mpxplay_config_dvbepg_pastevents_hours;
 extern char mpxplay_config_video_preffered_audiotypes[MPXINI_MAX_CHARDATA_LEN];
 extern char mpxplay_config_video_preferred_language[MPXPLAY_STREAMTYPEINDEX_PLAYNUM][MPXINI_MAX_CHARDATA_LEN];
 extern unsigned int mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM];
 extern char mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM][MPXINI_MAX_CHARDATA_LEN];
#endif
 extern long mpxplay_card_selectconfig_mpxpini;
 extern struct mpxplay_audioout_info_s au_infos;
 extern struct mainvars mvps;
 extern keyconfig kb[];
}

static int dispqtconfig_content_row_selected;
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
static char mpxplay_config_video_language_select[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM][MPXINI_MAX_CHARDATA_LEN];
static char mpxplay_config_video_language_custom[MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM][MPXINI_MAX_CHARDATA_LEN];

static void dispqt_configpage_streams_comboboxes_init(void);
static void dispqt_configpage_streams_comboboxes_apply(void);

#endif

DispQtConfigWindow::DispQtConfigWindow(MainWindow *main_window, QWidget *parent) : DispQtDialogElemWindow(main_window, parent, Qt::Dialog, DISPQT_DIALOG_WINDOWTYPE_CONFIG)
{
	emit main_window->signal_video_mainwin_opacity_disable(true);

	this->main_window = main_window;
	this->parent_widget = parent;
	this->setMinimumWidth(200);
	this->setMaximumWidth(404);

	dispqt_configpage_streams_comboboxes_init();

	contentsWidget = new QListWidget;
	contentsWidget->setViewMode(QListView::ListMode);
	//contentsWidget->setIconSize(QSize(48, 42));
	contentsWidget->setMovement(QListView::Static);
	contentsWidget->setMaximumWidth(64);

	pagesWidget = new QStackedWidget;
	this->configpage_mainwin = new DispQtConfigpageMainwin(main_window, this);
	pagesWidget->addWidget(this->configpage_mainwin);
	this->configpage_display = new DispQtConfigpageDisplay(main_window, this);
	pagesWidget->addWidget(this->configpage_display);
	this->configpage_color = new DispQtConfigpageColor(main_window, this);
	pagesWidget->addWidget(this->configpage_color);
	this->configpage_font = new DispQtConfigpageFont(main_window, this);
	pagesWidget->addWidget(this->configpage_font);
	this->configpage_opacity = new DispQtConfigpageOpacity(main_window, this);
	pagesWidget->addWidget(this->configpage_opacity);
	this->configpage_playlist = new DispQtConfigpagePlayList(main_window, this);
	pagesWidget->addWidget(this->configpage_playlist);
	this->configpage_playcontrol = new DispQtConfigpagePlayControl(main_window, this);
	pagesWidget->addWidget(this->configpage_playcontrol);
	this->configpage_program = new DispQtConfigpageProgram(main_window, this);
	pagesWidget->addWidget(this->configpage_program);
	this->configpage_audio = new DispQtConfigpageAudio(main_window, this);
	pagesWidget->addWidget(this->configpage_audio);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->configpage_subtitle = new DispQtConfigpageSubtitle(main_window, this);
	pagesWidget->addWidget(this->configpage_subtitle);
#endif
	this->configpage_video = new DispQtConfigpageVideo(main_window, this);
	pagesWidget->addWidget(this->configpage_video);

	QPushButton *button_ok = new QPushButton(tr("OK"));
	QPushButton *button_cancel = new QPushButton(tr("Cancel"));
	QPushButton *button_apply = new QPushButton(tr("Apply"));
	QPushButton *button_reset = new QPushButton(tr("Reset settings"));

	config_create_listlines();
	contentsWidget->setCurrentRow(dispqtconfig_content_row_selected);

	connect(button_ok, SIGNAL(pressed()), this, SLOT(config_ok_close()));
	connect(button_cancel, SIGNAL(pressed()), this, SLOT(close()));
	connect(button_apply, SIGNAL(pressed()), this, SLOT(config_apply()));
	connect(button_reset, SIGNAL(pressed()), this, SLOT(config_reset_close()));

	QHBoxLayout *horizontalLayout = new QHBoxLayout;
	QWidget *horizontal_widget = new QWidget(this);
	horizontalLayout->addWidget(contentsWidget);
	horizontalLayout->addWidget(pagesWidget, 1);
	horizontal_widget->setLayout(horizontalLayout);

	QHBoxLayout *buttonsLayout = new QHBoxLayout;
	QWidget *buttons_widget = new QWidget(this);
	//buttonsLayout->addStretch(1);
	buttonsLayout->addWidget(button_reset, 1, Qt::AlignLeft);
	buttonsLayout->addWidget(button_ok, 0, Qt::AlignRight);
	buttonsLayout->addWidget(button_cancel, 0, Qt::AlignRight);
	buttonsLayout->addWidget(button_apply, 0, Qt::AlignRight);
	buttons_widget->setLayout(buttonsLayout);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(horizontal_widget);
	mainLayout->addWidget(buttons_widget);
	mainLayout->setContentsMargins(0,5,0,0);
	setLayout(mainLayout);

	setWindowTitle(tr("Configure MMC"));
	show();
}

void DispQtConfigWindow::closeEvent(QCloseEvent *e)
{
	emit this->main_window->signal_config_dialogwindow_closed();
	QDialog::closeEvent(e);
}

void DispQtConfigWindow::keyPressEvent(QKeyEvent *event)
{
	if(event->key() == Qt::Key_Escape)
		emit this->main_window->signal_config_dialogwindow_closed();
	QDialog::keyPressEvent(event);
}

static const char *config_menu_list_names[] = {"Layout", "Display", " Color", " Font", " Opacity", "List", "Play", "Program", "Audio",
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
		"Subtitle",
#endif
		"Video"};

#define DISPQT_CONFIGMENU_NB_ELEMS (sizeof(config_menu_list_names) / sizeof(config_menu_list_names[0]))

void DispQtConfigWindow::config_create_listlines()
{
	for(int i = 0; i < DISPQT_CONFIGMENU_NB_ELEMS; i++) {
		QListWidgetItem *button = new QListWidgetItem(contentsWidget);
		//button->setIcon(QIcon(":/images/query.png"));
		button->setText(QString::fromUtf8(config_menu_list_names[i]));
		button->setTextAlignment(Qt::AlignLeft);
		button->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled);
	}

	connect(contentsWidget, SIGNAL(currentItemChanged(QListWidgetItem*,QListWidgetItem*)), this, SLOT(changePage(QListWidgetItem*,QListWidgetItem*)));
}

void DispQtConfigWindow::changePage(QListWidgetItem *current, QListWidgetItem *previous)
{
	if (!current)
		current = previous;
	dispqtconfig_content_row_selected = contentsWidget->row(current);
	pagesWidget->setCurrentIndex(dispqtconfig_content_row_selected);
}

void DispQtConfigWindow::config_apply()
{
	this->configpage_mainwin->configpage_mainwin_applyall();
	this->configpage_display->configpage_display_applyall();
	this->configpage_color->configpage_color_applyall();
	this->configpage_font->configpage_font_applyall();
	this->configpage_opacity->configpage_opacity_applyall();
	this->configpage_playlist->configpage_playlist_applyall();
	this->configpage_playcontrol->configpage_playcontrol_applyall();
	this->configpage_program->configpage_program_applyall();
	this->configpage_audio->configpage_audio_applyall();
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->configpage_subtitle->configpage_subtitle_applyall();
#endif
	this->configpage_video->configpage_video_applyall();
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	dispqt_configpage_streams_comboboxes_apply();
#endif
	emit this->main_window->signal_config_apply(false);
	//mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "fontedit: \"%s\"", this->main_window->fonttext_editor.toLatin1().data());
}

void DispQtConfigWindow::config_ok_close()
{
	emit this->main_window->signal_config_dialogwindow_closed();
	this->config_apply();
	this->close();
}

static void dispqt_config_reset_confirm_apply(void)
{
	mpxplay_control_set_default_config();
	mpxplay_dispqt_configini_reset_configurables();
	mpxplay_dispqt_configini_set_default_fonts();
	mmc_config.selected_videoplayer_control = mpxplay_config_videoplayer_control;
	mpxplay_dispqt_main_gui_config_reset_apply();
	funcbit_enable(refdisp, (RDT_TABINFOS|RDT_EDITOR|RDT_RESET_EDIT));
}

static void dispqt_config_reset_confirm_init(void)
{
	display_textwin_openwindow_confirm(0, (char *)"Reset config", (char *)"Reset config and video surface settings\nto default values?", (void *)&dispqt_config_reset_confirm_apply, NULL);
}

void DispQtConfigWindow::config_reset_close()
{
	mpxplay_dispqt_mainthread_callback_init((void *)&dispqt_config_reset_confirm_init, NULL, 0);
	this->close();
}

//--------------------------------------------------------------------------------------
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

// copy stream settings to config
static void dispqt_configpage_stream_init(unsigned int i)
{
	// note: iso639_1 inputs are handled as 'Custom' here
	if(mpxplay_config_video_preferred_language[i][0] && (mpxplay_ffmplang_search_in_iso639_1to2_convs(&mpxplay_config_video_preferred_language[i][0]) || !mpxplay_ffmplang_getlangname_by_langcode(&mpxplay_config_video_preferred_language[i][0])))
		pds_strcpy(&mpxplay_config_video_language_select[i][0], (char *)"xxx"); // for combobox
	else // standard language code or zero
		pds_strncpy(&mpxplay_config_video_language_select[i][0], &mpxplay_config_video_preferred_language[i][0], MPXINI_MAX_CHARDATA_LEN - 1);
	pds_strncpy(&mpxplay_config_video_language_custom[i][0], &mpxplay_config_video_preferred_language[i][0], MPXINI_MAX_CHARDATA_LEN - 1); // for custom edit
}

// copy stream settings from config
static void dispqt_configpage_stream_apply(unsigned int i)
{
	mpxplay_debugf(DISPQT_DEBUG_STREAM, "configpage_stream_applyall: %d c:%s s:%s p:%s", i, &mpxplay_config_video_language_custom[i][0],
			&mpxplay_config_video_language_select[i][0], &mpxplay_config_video_preferred_language[i][0]);
	pds_str_clean(&mpxplay_config_video_language_custom[i][0]);
	if( (pds_stricmp(&mpxplay_config_video_language_select[i][0], (char *)"xxx") == 0) // 'Custom' selected
	 || ((pds_strcmp(&mpxplay_config_video_language_custom[i][0], &mpxplay_config_video_preferred_language[i][0]) != 0) // language id is edited
		 && (  (pds_strcmp(&mpxplay_config_video_language_select[i][0], &mpxplay_config_video_preferred_language[i][0]) == 0) // combobox selection is not changed
			|| (pds_stricmp(&mpxplay_config_video_language_select[i][0], (char *)"all") == 0) // combobox is selected to 'All'
		    )
	    ) // or changed by custom edit only
	)
		pds_strncpy(&mpxplay_config_video_preferred_language[i][0], &mpxplay_config_video_language_custom[i][0], MPXINI_MAX_CHARDATA_LEN - 1); // store custom lng code
	else if(pds_stricmp(&mpxplay_config_video_language_select[i][0], (char *)"all") == 0) // means deselected language
		mpxplay_config_video_preferred_language[i][0] = 0; // remove language code
	else
		pds_strncpy(&mpxplay_config_video_preferred_language[i][0], &mpxplay_config_video_language_select[i][0], MPXINI_MAX_CHARDATA_LEN - 1); // store combobox selected lng code
}

static const char *config_extstream_loadtype_values[] = {
	"0",  // MPXPLAY_EXTERNALSTREAM_LOADTYPE_NONE
	"1",  // MPXPLAY_EXTERNALSTREAM_LOADTYPE_SEARCH
	"2",  // MPXPLAY_EXTERNALSTREAM_LOADTYPE_USE
	"3",  // MPXPLAY_EXTERNALSTREAM_LOADTYPE_PREFER
};

static const char *config_extstream_loadtype_description[] = {
	"Don't use external streams",
	"Add to stream list only",
	"Use by auto decision",
	"Prefer external streams"
};

static struct configpage_comboboxline_s config_stream_languages_combobox_group[MPXPLAY_STREAMTYPEINDEX_PLAYNUM][2] = {
{
 {"Language name: ", ARG_CHAR, (void *)&mpxplay_config_video_language_select[MPXPLAY_STREAMTYPEINDEX_AUDIO][0], 0, NULL, NULL, "Select preferred audio stream language"},
 {"Language id: ", DISPQT_CONFIG_COMBOTYPE_EDITLINE, &mpxplay_config_video_language_custom[MPXPLAY_STREAMTYPEINDEX_AUDIO][0], MPXINI_MAX_CHARDATA_LEN, NULL, NULL, "You can write (custom) language id here,\nlike eng or hun or any"}
},
{
 {"Language name: ", ARG_CHAR, (void *)&mpxplay_config_video_language_select[MPXPLAY_STREAMTYPEINDEX_VIDEO][0], 0, NULL, NULL, "Select preferred video stream language"},
 {"Language id: ", DISPQT_CONFIG_COMBOTYPE_EDITLINE, &mpxplay_config_video_language_custom[MPXPLAY_STREAMTYPEINDEX_VIDEO][0], MPXINI_MAX_CHARDATA_LEN, NULL, NULL, "You can write (custom) language id here,\nlike eng or hun or any"}
},
{
 {"Language name: ", ARG_CHAR, (void *)&mpxplay_config_video_language_select[MPXPLAY_STREAMTYPEINDEX_SUBTITLE][0], 0, NULL, NULL, "Select preferred subtitle language"},
 {"Language id: ", DISPQT_CONFIG_COMBOTYPE_EDITLINE, &mpxplay_config_video_language_custom[MPXPLAY_STREAMTYPEINDEX_SUBTITLE][0], MPXINI_MAX_CHARDATA_LEN, NULL, NULL, "You can write (custom) language id here,\nlike eng or hun or any"}
}
};

static void dispqt_configpage_streams_comboboxes_init(void)
{
	const char **lang_codes_ptr, **lang_names_ptr;
	int nb_language_codes;

	// fill language combobox pointers (language conversion tables) by ffmplang infos
	nb_language_codes = mpxplay_ffmplang_get_language_fields(&lang_codes_ptr, &lang_names_ptr);

	for(int i = 0; i < MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM; i++)
	{
		config_stream_languages_combobox_group[i][0].nb_values = nb_language_codes;
		config_stream_languages_combobox_group[i][0].data_values = lang_codes_ptr;
		config_stream_languages_combobox_group[i][0].data_description = lang_names_ptr;
		dispqt_configpage_stream_init(i);
	}
}

static void dispqt_configpage_streams_comboboxes_apply(void)
{
	for(int i = 0; i < MPXPLAY_STREAMTYPEINDEX_EXTSTREAMNUM; i++)
		dispqt_configpage_stream_apply(i);
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

//======================================================================================

static struct configpage_chkbox_s mainwin_layout_normal_checkboxes[] = {
	{"Show video window (*)",     &mmc_config.mainwin_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_VIDEO },
	{"Show seekbar",              &mmc_config.mainwin_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_SEEKBAR },
	{"Show toolbar / buttons",    &mmc_config.mainwin_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_TOOLBAR },
	{"Embedd AV mixer",           &mmc_config.mainwin_control, DISPQT_CONFIG_MAINWINCTRL_EMBEDD_AVMIXER },
	{"Show playlist editor (*)",  &mmc_config.mainwin_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_EDITOR },
	{"Show status line",          &mmc_config.mainwin_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_STATUSLINE },
	{"Always on top",             &mmc_config.mainwin_control, DISPQT_CONFIG_MAINWINCTRL_ALWAYS_ON_TOP },
	{" (*) cannot disable both widget, only by video control", NULL, 0 }
};

#define DISPQT_CONFIGPAGE_MAINWIN_NORMAL_CHKBOX_MAX (sizeof(mainwin_layout_normal_checkboxes) / sizeof(struct configpage_chkbox_s))

static struct configpage_chkbox_s mainwin_layout_transparent_checkboxes[] = {
	{"Hide menu",              &mmc_config.transparent_control, DISPQT_CONFIG_MAINWINCTRL_HIDE_MENU },
	{"Show video window",      &mmc_config.transparent_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_VIDEO },
	{"Show seekbar",           &mmc_config.transparent_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_SEEKBAR },
	{"Show toolbar / buttons", &mmc_config.transparent_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_TOOLBAR },
	{"Show playlist editor",   &mmc_config.transparent_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_EDITOR },
	{"Show status line",       &mmc_config.transparent_control, DISPQT_CONFIG_MAINWINCTRL_SHOW_STATUSLINE },
//	{"Frameless window",       &mmc_config.transparent_control, DISPQT_CONFIG_TRANSCTRL_DESKTOPTRANS_FRAMELESS },  // TODO: doesn't work
	{"Maximize mainwin auto",  &mmc_config.transparent_control, DISPQT_CONFIG_TRANSCTRL_DESKTOPTRANS_AUTOMAXI },
	{"From transparency to fullvideo", &mmc_config.transparent_control, DISPQT_CONFIG_TRANSCTRL_EXIT_TO_FULLVIDEO },
	{"Restore at program start",&mmc_config.transparent_control, DISPQT_CONFIG_TRANSCTRL_DESKTOPTRANS_RESTORE }
};

#define DISPQT_CONFIGPAGE_MAINWIN_TRANSPARENT_CHKBOX_MAX (sizeof(mainwin_layout_transparent_checkboxes) / sizeof(struct configpage_chkbox_s))

DispQtConfigpageMainwin::DispQtConfigpageMainwin(MainWindow *main_window, QWidget *parent) : DispQtDialogElemTabWidget(main_window, parent)
{
	QWidget *mainwin_tab_widget_normal = new QWidget(this);
	QWidget *mainwin_tab_widget_transparent = new QWidget(this);
	QVBoxLayout *mainwin_tab_layout_normal = new QVBoxLayout;
	QVBoxLayout *mainwin_tab_layout_transparent = new QVBoxLayout;

	this->groupchkboxes_mainwin_normal_layout = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Main window layout / widgets", mainwin_layout_normal_checkboxes, DISPQT_CONFIGPAGE_MAINWIN_NORMAL_CHKBOX_MAX);
	this->groupchkboxes_mainwin_transparent_layout = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Main window layout in transparent desktop mode", mainwin_layout_transparent_checkboxes, DISPQT_CONFIGPAGE_MAINWIN_TRANSPARENT_CHKBOX_MAX);

	mainwin_tab_layout_normal->setContentsMargins(0,5,0,0);
	mainwin_tab_layout_normal->addWidget(this->groupchkboxes_mainwin_normal_layout);
	mainwin_tab_layout_normal->addStretch(1);
	mainwin_tab_widget_normal->setLayout(mainwin_tab_layout_normal);

	mainwin_tab_layout_transparent->setContentsMargins(0,5,0,0);
	mainwin_tab_layout_transparent->addWidget(this->groupchkboxes_mainwin_transparent_layout);
	mainwin_tab_layout_transparent->addStretch(1);
	mainwin_tab_widget_transparent->setLayout(mainwin_tab_layout_transparent);

	this->addTab(mainwin_tab_widget_normal, "Normal");
	this->addTab(mainwin_tab_widget_transparent, "Transparent");
	this->setDocumentMode(true);
}

void DispQtConfigpageMainwin::configpage_mainwin_applyall(void)
{
	this->groupchkboxes_mainwin_normal_layout->checkboxgroup_applyall();
	this->groupchkboxes_mainwin_transparent_layout->checkboxgroup_applyall();
}

//--------------------------------------------------------------------------------------
static struct configpage_chkbox_s display_editor_checkboxes[] = {
	{"Horizontal layout",     &desktopmode, DTM_EDIT_VERTICAL, "Vertical playlist editor border" },
	{"Show left panel",       &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_SHOWPANEL_LEFT, "Show left/up panel (side 0)" },
	{"Show right panel",      &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_SHOWPANEL_RIGHT, "Show right/down panel (side 1)" },
	{"Show drive control bar",&mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_SHOW_DRIVELINE, "Drive select, drive info, directory navigation buttons" },
	{"Show drives in dirs",   &desktopmode, DTM_EDIT_DRIVES, "Add drive labels and names to the\ntop of directories in the playlist editor\n(switch to another drive like subdir)\n(requires ctrl-'R' or dir change to take effect)" },
	{"Show all files in dirs",&desktopmode, DTM_EDIT_ALLFILES, "Show all files (not only the supported a/v files)\nin the current player/commander mode\n(requires ctrl-'R' or directory change)"},
	{"File infos in tooltip", &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_TOOLTIP_FILEINFOS, "Show file infos in tooltip by mouse cursor\nover playlist editor entries" },
	{"Separate Artist/Title", &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_COLUMNS_SEPARTTITLE, "Separate Artist/Title in editor columns (else merge)" },
	{"Manual column widths",  &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_COLUMNS_MANUALSIZE, "Disable auto width correction for editor columns" },
	{"Disable scrollbar",     &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_DISABLE_SCROLLBAR, "Never show editor list scrollbar" },
	{"Shift entry by mouse",  &desktopmode, DTM_EDIT_MOUSESONGSHIFT, "Drag and move playlist entries by mouse\n(else move complete list)" }
};

#define DISPQT_CONFIGPAGE_DISPLAY_EDITOR_CHKBOX_MAX (sizeof(display_editor_checkboxes) / sizeof(struct configpage_chkbox_s))

#define DISPQT_CONFIGPAGE_DISPLAY_EDITOR_TABCHG_FLAGS (DISPQT_CONFIG_EDITORFLAG_SHOWPANELS_ALL | DISPQT_CONFIG_EDITORFLAG_SHOW_DRIVELINE | DISPQT_CONFIG_EDITORFLAG_COLUMNS_MANUALSIZE | DISPQT_CONFIG_EDITORFLAG_DISABLE_SCROLLBAR)
#define DISPQT_CONFIGPAGE_DISPLAY_EDITOR_CHANGE_FLAGS (DISPQT_CONFIG_EDITORFLAG_TOOLTIP_FILEINFOS | DISPQT_CONFIG_EDITORFLAG_COLUMNS_SEPARTTITLE | DISPQT_CONFIG_EDITORFLAG_COLUMNS_MANUALSIZE)

DispQtConfigpageDisplay::DispQtConfigpageDisplay(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	this->main_window = main_window;
	this->groupchkboxes_editor = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Editor settings", display_editor_checkboxes, DISPQT_CONFIGPAGE_DISPLAY_EDITOR_CHKBOX_MAX);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->setMargin(0);
	mainLayout->addWidget(this->groupchkboxes_editor);
	mainLayout->addStretch(1);
	setLayout(mainLayout);
}

void DispQtConfigpageDisplay::configpage_display_applyall(void)
{
	unsigned long editor_flags_old = this->main_window->gui_config->editor_flags;
	unsigned int desktopmod_old = desktopmode;
	this->groupchkboxes_editor->checkboxgroup_applyall();
	if(displaymode&DISP_DESKTOPMODEC)
		desktopmod_commander=desktopmode;
	else
		desktopmod_player=desktopmode;
	if( ((desktopmode & DTM_EDIT_VERTICAL) != (desktopmod_old & DTM_EDIT_VERTICAL))
	 || ((this->main_window->gui_config->editor_flags &DISPQT_CONFIGPAGE_DISPLAY_EDITOR_TABCHG_FLAGS) != (editor_flags_old & DISPQT_CONFIGPAGE_DISPLAY_EDITOR_TABCHG_FLAGS)) )
		funcbit_enable(refdisp, RDT_TABINFOS);
	if((this->main_window->gui_config->editor_flags & DISPQT_CONFIGPAGE_DISPLAY_EDITOR_CHANGE_FLAGS) != (editor_flags_old & DISPQT_CONFIGPAGE_DISPLAY_EDITOR_CHANGE_FLAGS))
		funcbit_enable(refdisp, (RDT_EDITOR | RDT_RESET_EDIT));
}

//--------------------------------------------------------------------------------------
DispQtConfigpageColorCfgline::DispQtConfigpageColorCfgline(MainWindow *mainwindow, QWidget *parent, QString line_text, char **colortext_ptr) : QWidget(parent)
{
	this->color_text_ptr = colortext_ptr;
	this->label_colorname = new QLabel;
	this->label_colorbar = new QLabel;

	this->label_colorname->setText(line_text);
	this->label_colorbar->setMinimumWidth(60);
	if(mainwindow->mainwin_guibkg_is_transparent())
		this->setStyleSheet("QDialog { background: rgb(0,80,140); color: white}"); // QColorDialog::getColor() background color

	if(*colortext_ptr) {
		QString qs = QString::fromUtf8(*colortext_ptr);
		this->color_selected.setNamedColor(qs);
		this->colorcfgline_setcolor();
	}
	int frameStyle = QFrame::Sunken | QFrame::Panel;
	this->label_colorbar->setFrameStyle(frameStyle);
	this->label_colorbar->setMinimumWidth(100);
	this->label_colorbar->setMaximumHeight(20);
	this->label_colorbar->setAutoFillBackground(true);

	QPushButton *qb = new QPushButton;
	qb->setText(">>");
	qb->setMaximumWidth(25);
	connect(qb, SIGNAL(clicked()), this, SLOT(colorcfgline_getcolor()));

	QHBoxLayout *layout = new QHBoxLayout;
	layout->addWidget(this->label_colorname, 1);
	layout->addWidget(this->label_colorbar, 0);
	layout->addWidget(qb, 0);
	layout->addStretch(0);
	layout->setMargin(0);
	layout->setSpacing(1);
	setLayout(layout);
}

void DispQtConfigpageColorCfgline::colorcfgline_setcolor(void)
{
	if (!this->color_selected.isValid())
		return;

	// configure bar text color (black / white) depending on the bar color (light / dark)
	unsigned long color_hexa = pds_atol16(this->color_selected.name().toUtf8().data());
	unsigned int red = (color_hexa >> 16) & 0xFF;
	unsigned int green = (color_hexa >> 8) & 0xFF;
	unsigned int blue = (color_hexa & 0xFF);
	if((red > 0xF0) || (green > 0x80) || (blue > 0xF0))
		this->label_colorbar->setStyleSheet("QLabel { color: black; }");
	else
		this->label_colorbar->setStyleSheet("QLabel { color: white; }");

	// configure bar background color with color text
	QPalette pal = this->palette();
	pal.setColor(QPalette::Background, this->color_selected);
	this->label_colorbar->setPalette(pal);
	this->label_colorbar->setText(this->color_selected.name());
}

void DispQtConfigpageColorCfgline::colorcfgline_getcolor()
{
	QColor qc;

	if(this->color_selected.isValid())
		qc = this->color_selected;
	else {
		QString qs = QString::fromUtf8(*this->color_text_ptr);
		if(qs.size() > 0)
			qc.setNamedColor(qs);
		else
			qc = Qt::white;
	}

	this->color_selected = QColorDialog::getColor(qc, this);

	this->colorcfgline_setcolor();
}

void DispQtConfigpageColorCfgline::colorcfgline_apply()
{
	if(!this->color_selected.isValid())
		return;
	QString qs = this->color_selected.name();
	QByteArray qba = qs.toUtf8();
	int len = qba.size();
	if(len <= 0)
		return;
	if(*this->color_text_ptr)
		delete *this->color_text_ptr;
	*this->color_text_ptr = new char[len + 4];
	if(*this->color_text_ptr) {
		pds_strncpy(*this->color_text_ptr, qba.data(), len);
		(*this->color_text_ptr)[len] = 0;
	}
}

//---------------------------------------------------------------------------------------------------------------------------------------

static const char *config_general_gui_layout_values[] = {"0", "1", "128"}; // DISPQT_GUILAYOUT_OSDEF_CLASSIC, DISPQT_GUILAYOUT_BLUE_CLASSIC, DISPQT_GUILAYOUT_BLUE_TRANSPARENT
static const char *config_general_gui_layout_description[] = {"Windows Classic", "Classic Blue", "Transparent Blue"};

static const char *config_general_gui_bkgtype_values[] = {"0", "1", "2"}; // DISPQT_GUIBKGTYPE_BLUR, DISPQT_GUIBKGTYPE_TRANSPARENT, DISPQT_GUIBKGTYPE_OPAQUE
static const char *config_general_gui_bkgtype_description[] = {"Blur", "Transparent", "Opaque"};

static const char *config_general_gui_bkgeffect_values[] = {"0", "1", "2"}; // DISPQT_GUIBKGEFCT_NONE, DISPQT_GUIBKGEFCT_COLORIZED, DISPQT_GUIBKGEFCT_GLASS_PNG
static const char *config_general_gui_bkgeffect_description[] = {"None", "Colorized", "Glass"};

static struct configpage_comboboxline_s mainwin_gui_colorization_select[] = {
 {"GUI color type", (DISPQT_CONFIG_COMBOTYPE_COMBOBOX | ARG_NUM), &mmc_config.gui_layout_type, (sizeof(config_general_gui_layout_values) / sizeof(char *)),
  &config_general_gui_layout_values[0], &config_general_gui_layout_description[0], "General (main) GUI colorization"},
 {"Background type", (DISPQT_CONFIG_COMBOTYPE_COMBOBOX | ARG_NUM), &mmc_config.gui_background_type, (sizeof(config_general_gui_bkgtype_values) / sizeof(char *)),
  &config_general_gui_bkgtype_values[0], &config_general_gui_bkgtype_description[0],
  "Blur (translucent) and transparent modes\ndepend on the OS support and settings\n(if a setting is not supported,\nthe next one is selected automatically)"},
 {"Background effect", (DISPQT_CONFIG_COMBOTYPE_COMBOBOX | ARG_NUM), &mmc_config.gui_background_effect, (sizeof(config_general_gui_bkgeffect_values) / sizeof(char *)),
  &config_general_gui_bkgeffect_values[0], &config_general_gui_bkgeffect_description[0],
  "Additional effects for\ntransparent backgrounds"},
 {"Colorization alpha", DISPQT_CONFIG_COMBOTYPE_SPINBOX, &mmc_config.gui_background_color_alpha, DISPQT_CONFIG_SPINBOX_VALPACK_MINMAX(0, DISPQT_CONFIG_MAINWIN_COLOR_ALPHA_MAX),
  NULL, NULL, "Transparency of background color\nif background effect is colorized\n(0 = full transparent, no color\n100 = full color, no transparent)"}
};
#define DISPQT_CONFIGPAGE_GENERAL_GUI_COLORIZATION_COMBOX_MAX (sizeof(mainwin_gui_colorization_select) / sizeof(struct configpage_comboboxline_s))

static const char *colorcfgline_window_names[DISPQT_COLORTEXTID_WINDOW_MAX] = { // !!! must match with dispt_colortext_window_ids
	"Classic background", "Transparent background"
#ifdef DISPQT_COLORTEXTID_CONFIGURE_TEXTCOLORS
	, "Classic text", "Transparent text"
#endif
};

static struct configpage_chkbox_s display_editor_transparent_checkboxes[] = {
	{"Transparent in window",&mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_INWINDOW, "Transparent editor in non-fullscreen\n(if general transparent mode\nis enabled and supported)"},
	{"Transparent on video", &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_ONVIDEO,  "Transparent editor in fullscreen mode\n(over video surface)"},
	{"Transparent inactive", &mmc_config.editor_flags, DISPQT_CONFIG_EDITORFLAG_TRANSPARENT_INACTIVE, "Transparent editor losing focus\n(clicking to an other window)"}
};
#define DISPQT_CONFIGPAGE_DISPLAY_ETCHKBOX_MAX (sizeof(display_editor_transparent_checkboxes) / sizeof(struct configpage_chkbox_s))

//---------------------------------------------------------------------------------------------------

static const char *colorcfgline_editor_names[DISPQT_COLORTEXTID_EDITOR_MAX] = { // !!! must match with dispt_colortext_editor_ids
	"Font base (not playtab)", "Background", "Background 2", "Cursor background", "Mark / selected font",
	"Currsong font (on playtab)", "Nextsong font (on playtab)", "Played font (on playtab)", "Not played font (on playtab)"
};

DispQtConfigpageColor::DispQtConfigpageColor(MainWindow *main_window, QWidget *parent) : DispQtDialogElemTabWidget(main_window, parent)
{
	QWidget *colorctrl_window_widget = new QWidget(this);
	QVBoxLayout *colorctrl_window_layout = new QVBoxLayout;

	this->combobox_gui_general_colorization_control = new DispQtConfigpageComboboxGroupbox(main_window, this, "General GUI colorization", &mainwin_gui_colorization_select[0], DISPQT_CONFIGPAGE_GENERAL_GUI_COLORIZATION_COMBOX_MAX);
	this->groupchkboxes_editor_transparent = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Editor transparent settings", display_editor_transparent_checkboxes, DISPQT_CONFIGPAGE_DISPLAY_ETCHKBOX_MAX);

	QGroupBox *colorGroup_window = new QGroupBox(tr("Window colors"));
	QVBoxLayout *colorLayout_window = new QVBoxLayout;
	colorLayout_window->setMargin(3);
	colorLayout_window->setSpacing(5);
	for(int i = 0; i < DISPQT_COLORTEXTID_WINDOW_MAX; i++){
		this->colorcfgline_window[i] = new DispQtConfigpageColorCfgline(main_window, this, colorcfgline_window_names[i], &main_window->gui_config->color_window_text_name[i]);
		colorLayout_window->addWidget(this->colorcfgline_window[i]);
	}
	colorGroup_window->setLayout(colorLayout_window);

	colorctrl_window_layout->addWidget(this->combobox_gui_general_colorization_control);
	colorctrl_window_layout->addWidget(colorGroup_window);
	colorctrl_window_layout->addWidget(this->groupchkboxes_editor_transparent);
	colorctrl_window_layout->addStretch(1);
	colorctrl_window_widget->setLayout(colorctrl_window_layout);

	//---------------------------------------------------------------------------------------------------
	QWidget *colorctrl_editor_widget = new QWidget(this);
	QVBoxLayout *colorctrl_editor_layout = new QVBoxLayout;
	QGroupBox *colorGroup_editor = new QGroupBox(tr("Editor colors"));
	QVBoxLayout *colorLayout_editor = new QVBoxLayout;
	colorLayout_editor->setMargin(3);
	colorLayout_editor->setSpacing(5);

	for(int i = 0; i < DISPQT_COLORTEXTID_EDITOR_MAX; i++){
		this->colorcfgline_editor[i] = new DispQtConfigpageColorCfgline(main_window, this, colorcfgline_editor_names[i], &main_window->gui_config->color_text_name[i]);
		colorLayout_editor->addWidget(this->colorcfgline_editor[i]);
	}

	colorGroup_editor->setLayout(colorLayout_editor);
	colorctrl_editor_layout->addWidget(colorGroup_editor);
	colorctrl_editor_layout->addStretch(1);
	colorctrl_editor_widget->setLayout(colorctrl_editor_layout);

	//---------------------------------------------------------------------------------------------------
	this->addTab(colorctrl_window_widget, "Window colors");
	this->addTab(colorctrl_editor_widget, "Editor colors");
	this->setDocumentMode(true);
}

void DispQtConfigpageColor::configpage_color_applyall(void)
{
	this->combobox_gui_general_colorization_control->comboboxgroupbox_applyall();
	this->groupchkboxes_editor_transparent->checkboxgroup_applyall();
	for(int i = 0; i < DISPQT_COLORTEXTID_WINDOW_MAX; i++)
		this->colorcfgline_window[i]->colorcfgline_apply();

	for(int i = 0; i < DISPQT_COLORTEXTID_EDITOR_MAX; i++)
		this->colorcfgline_editor[i]->colorcfgline_apply();
}

//-------------------------------------------------------------------------------
DispQtConfigpageFontBox::DispQtConfigpageFontBox(MainWindow *mainwindow, QWidget *parent, QString box_text, char **fonttext_ptr) : QGroupBox(parent)
{
	if(mainwindow->mainwin_guibkg_is_transparent())
		this->setStyleSheet("QDialog { background: rgb(0,80,140); color: white}"); // QFontDialog::getFont() background color
	this->setTitle(box_text);
	this->font_text_ptr = fonttext_ptr;
	QHBoxLayout *fontbox_layout = new QHBoxLayout;
	fontbox_layout->setMargin(2);
	int frameStyle = QFrame::Sunken | QFrame::Panel;
	this->font_label = new QLabel;
	this->font_label->setFrameStyle(frameStyle);
	this->font_label->setMinimumWidth(180);
	this->font_label->setText(*fonttext_ptr);
	this->font_selected.fromString(*fonttext_ptr);
	this->font_label->setFont(this->font_selected);
	QPushButton *button_font = new QPushButton(tr("Change font"));
	fontbox_layout->addWidget(button_font, 0);
	fontbox_layout->addWidget(this->font_label, 1);
	this->setLayout(fontbox_layout);
	connect(button_font, SIGNAL(clicked()), this, SLOT(pagefontbox_getfont()));
}

void DispQtConfigpageFontBox::pagefontbox_getfont()
{
	bool ok = false;
	this->font_selected.fromString(QString::fromUtf8(*this->font_text_ptr));
	QFont font = QFontDialog::getFont(&ok, this->font_selected, this);
	if (!ok)
		return;
	this->font_label->setText(font.toString());
	this->font_label->setFont(font);
	this->font_selected = font;
	//mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "fontbox key     : \"%s\"", qf.key().toLatin1().data());
	//mpxplay_debugf(DISPQT_DEBUG_OUTPUT, "fontbox tostring: \"%s\"", qf.toString().toLatin1().data());
}

void DispQtConfigpageFontBox::pagefontbox_apply()
{
	if(this->font_selected.family().size() <= 0)
		return;
	QString qs = this->font_selected.toString();
	QByteArray qba = qs.toUtf8();
	int len = qba.size();
	if(len <= 0)
		return;
	if(*this->font_text_ptr)
		delete *this->font_text_ptr;
	*this->font_text_ptr = new char[len + 4];
	if(*this->font_text_ptr) {
		pds_strncpy(*this->font_text_ptr, qba.data(), len);
		(*this->font_text_ptr)[len] = 0;
	}
}

static const char *fontbox_names[DISPQT_FONTTEXTID_MAX] = {
	"Playlist font", "Main window font", "Dialog box font", "Video subtitle font" // !!! must match with enum dispt_fonttext_ids
};

DispQtConfigpageFont::DispQtConfigpageFont(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->setMargin(0);
	for(int i = 0; i < DISPQT_FONTTEXTID_MAX; i++){
		this->fontcfgbox[i] = new DispQtConfigpageFontBox(main_window, this, tr(fontbox_names[i]), &main_window->gui_config->font_text_name[i]);
		mainLayout->addWidget(this->fontcfgbox[i]);
	}
	mainLayout->addStretch(1);
	setLayout(mainLayout);
}

void DispQtConfigpageFont::configpage_font_applyall(void)
{
	for(int i = 0; i < DISPQT_FONTTEXTID_MAX; i++)
		this->fontcfgbox[i]->pagefontbox_apply();
}

//--------------------------------------------------------------------------------------
static struct configpage_spinbox_s mainwin_opacity_spinboxes[] = {
	{"Base / General", &mmc_config.mainwin_opacity_general, DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MAX},
	{"Inactive base",  &mmc_config.mainwin_opacity_inactive, DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_GENERAL_MAX},
	{"Show on video",  &mmc_config.mainwin_opacity_onvideo_show, DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_SHOW_MAX},
	{"Hide on video",  &mmc_config.mainwin_opacity_onvideo_hide, DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_HIDE_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_ONVIDEO_HIDE_MAX},
	{"Transparent desktop", &mmc_config.mainwin_opacity_transparent, DISPQT_CONFIG_MAINWIN_OPACITY_TRANSPARENT_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_TRANSPARENT_MAX}
};

static struct configpage_spinbox_s mainwin_opacity_time_spinboxes[] = {
	{"Fade in time", &mmc_config.mainwin_opacity_fadein_time, DISPQT_CONFIG_MAINWIN_OPACITY_FADEIN_TIMEMS_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_FADEIN_TIMEMS_MAX},
	{"Keep showing (by editkeys)", &mmc_config.mainwin_opacity_showkeep_time, DISPQT_CONFIG_MAINWIN_OPACITY_SHOWKEEP_TIMEMS_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_SHOWKEEP_TIMEMS_MAX},
	{"Fade out time", &mmc_config.mainwin_opacity_fadeout_time, DISPQT_CONFIG_MAINWIN_OPACITY_FADEOUT_TIMEMS_MIN, DISPQT_CONFIG_MAINWIN_OPACITY_FADEOUT_TIMEMS_MAX}
};

#define DISPQT_CONFIGPAGE_MAINWIN_OPACITY_SPINBOX_MAX (sizeof(mainwin_opacity_spinboxes) / sizeof(struct configpage_spinbox_s))
#define DISPQT_CONFIGPAGE_MAINWIN_OPACITY_TIME_SPINBOX_MAX (sizeof(mainwin_opacity_time_spinboxes) / sizeof(struct configpage_spinbox_s))

DispQtConfigpageOpacity::DispQtConfigpageOpacity(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	this->main_window = main_window;
	this->groupspinboxes_mainwin_opacity = new DispQtConfigpageSpinboxGroupbox(main_window, this, "Main window opacity / transparency levels (%)", mainwin_opacity_spinboxes, DISPQT_CONFIGPAGE_MAINWIN_OPACITY_SPINBOX_MAX);
	this->groupspinboxes_mainwin_opacity_time = new DispQtConfigpageSpinboxGroupbox(main_window, this, "Main window transparency on video timeout (msec)", mainwin_opacity_time_spinboxes, DISPQT_CONFIGPAGE_MAINWIN_OPACITY_TIME_SPINBOX_MAX);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->setMargin(0);
	mainLayout->addWidget(this->groupspinboxes_mainwin_opacity);
	mainLayout->addWidget(this->groupspinboxes_mainwin_opacity_time);
	mainLayout->addStretch(1);
	setLayout(mainLayout);
}

void DispQtConfigpageOpacity::configpage_opacity_applyall(void)
{
	struct mmc_dispqt_config_s *gcfg = this->main_window->gui_config;
	this->groupspinboxes_mainwin_opacity->spinboxgroup_applyall();
	this->groupspinboxes_mainwin_opacity_time->spinboxgroup_applyall();
	if(gcfg->mainwin_opacity_onvideo_show <= gcfg->mainwin_opacity_onvideo_hide)
		gcfg->mainwin_opacity_onvideo_show = gcfg->mainwin_opacity_onvideo_hide + 1;
}

//----------------------------------------------------------------------------------------------
static struct configpage_chkbox_s playlist_playlistcontrol_checkboxes[] = {
	{"Disable playlist warnings", &mpxplay_playlistcontrol, MPXPLAY_PLAYLISTC_NOLISTWARNINGS, "Disable playlist related warning dialogs" },
	{"Open input songs in its dirs", &mpxplay_playlistcontrol, MPXPLAY_PLAYLISTC_DIRBROWSER2, "Command-line input files (songs,lists) are opened in their directory,\nnot loaded as a playlist (equal to -db2 option)" },
	{"Use local ID3 lists", &mpxplay_playlistcontrol, MPXPLAY_PLAYLISTC_ID3LIST_LOCAL, "Artist:Title infos are loaded from a local ID3 info list too,\nnamed !FILES by default (equal to -il option)" },
	{"Non-filtered ID3 search", &mpxplay_playlistcontrol, MPXPLAY_PLAYLISTC_NONFILT_SEARCH, "Using ID3-search function (alt-letters), non-matching entries\nare keeped in the list (but skiped at up/down arrows)" }
};

static struct configpage_chkbox_s playlist_loadid3tag_checkboxes[] = {
	{"Use cue/extm3u/id3list", &loadid3tag, ID3LOADMODE_LIST, "ID3 tags are loaded from extended (with artist:title info) playlists (too)" },
	{"Load tags from audiofiles", &loadid3tag, ID3LOADMODE_FILE, "ID3 tags are loaded from audio files (too)" },
	{"Create ID3 from filename", &loadid3tag, ID3LOADMODE_CLFN, "ID3 tags are created from proper (lfn) filenames\nand parent directory names (too)\nArtist - Album (Date) \\ NN. [Artist -] Title.mp3"},
	{"Prefer playlist-tags", &loadid3tag, ID3LOADMODE_PREFER_LIST, "Prefer extended list and lfn based tags\nagainst audiofile tags.\nTitle info from http is still updated,\nartist (channel name) is not." },
	{"Load from lowlevel devices", &loadid3tag, ID3LOADMODE_LOWLEVEL, "Load Artist (channel) and Title infos\nfrom the low level devices (http,ice)" }
};

static const char *config_playlist_preloadinfo_data_values[] = {"0","1","2","3","4"};
static const char *config_playlist_preloadinfo_data_description[] = {"Don't preload meta infos (-inl)", "Load at program start", "Load parallel with playing (-ipl)", "Load only displayed entries (-idl)", "Load highlighted (under cursor) (-ihl)"};

static struct configpage_comboboxline_s playlist_preloadinfo_combobox_lines[] = {
{
	"Preload mode: ", ARG_NUM, (void *)&preloadinfo, (sizeof(config_playlist_preloadinfo_data_values) / sizeof(char *)),
	&config_playlist_preloadinfo_data_values[0], &config_playlist_preloadinfo_data_description[0], "Playlist entry informations load timing\n(requires ctrl-'R' after change)"
} };

static struct configpage_chkbox_s playlist_preloadinfo_checkboxes[] = {
	{"Preload from slow devices too (ftp,http)", &loadid3tag, ID3LOADMODE_SLOWACCESS, "By default, ftp and http entries are not pre-checked,\nmedia informations are loaded at playing only"}
};

static struct configpage_chkbox_s playlist_sortcontrol_checkboxes[] = {
	{"Sort dirs/lists too by date or name", &mpxplay_sortcontrol, PLAYLIST_SORTCONTROL_CMD_DIR, "If enabled, directories and playlists are also sorted\nif Artist, Filename, Ext or Filedate order is selected\n(else sorted only by filename in ascending order)" },
	{"Disable auto descending order (len,size,date)", &mpxplay_sortcontrol, PLAYLIST_SORTCONTROL_NOAUTODESC, "By default, song-len, filesize and filedate\nhave initially descending order when selected" },
	{"Don't keep playlists on top of dir (needs ctrl-'R')", &mpxplay_sortcontrol, PLAYLIST_SORTCONTROL_NOLISTTOP, "By default, playlist filenames are\nalways placed at the top of directory" }
};

#define DISPQT_CONFIGPAGE_PLAYLIST_PLAYLISTCONTROL_CHKBOX_MAX (sizeof(playlist_playlistcontrol_checkboxes) / sizeof(struct configpage_chkbox_s))
#define DISPQT_CONFIGPAGE_PLAYLIST_LOADID3TAG_CHKBOX_MAX (sizeof(playlist_loadid3tag_checkboxes) / sizeof(struct configpage_chkbox_s))
#define DISPQT_CONFIGPAGE_PLAYLIST_PRELOADINFO_COMBOX_MAX (sizeof(playlist_preloadinfo_combobox_lines) / sizeof(struct configpage_comboboxline_s))
#define DISPQT_CONFIGPAGE_PLAYLIST_PRELOADINFO_CHKBOX_MAX (sizeof(playlist_preloadinfo_checkboxes) / sizeof(struct configpage_chkbox_s))
#define DISPQT_CONFIGPAGE_PLAYLIST_SORTCONTROL_CHKBOX_MAX (sizeof(playlist_sortcontrol_checkboxes) / sizeof(struct configpage_chkbox_s))

DispQtConfigpagePlayList::DispQtConfigpagePlayList(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	QVBoxLayout *mainLayout = new QVBoxLayout;
	QWidget *widget_control_loadid3 = new QWidget(this);
	QHBoxLayout *layout_control_loadid3 = new QHBoxLayout;
	QGroupBox *groupbox_preloadinfo = new QGroupBox;
	QVBoxLayout *layout_preloadinfo = new QVBoxLayout;
	QWidget *widget_preloadinfo_comboxes = new QWidget(this);
	QWidget *widget_preloadinfo_checkboxes = new QWidget(this);

	this->groupchkboxes_playlistcontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Playlist control", &playlist_playlistcontrol_checkboxes[0], DISPQT_CONFIGPAGE_PLAYLIST_PLAYLISTCONTROL_CHKBOX_MAX);
	this->groupchkboxes_loadid3tag = new DispQtConfigpageCheckboxGroupbox(main_window, this, "ID3 load types", &playlist_loadid3tag_checkboxes[0], DISPQT_CONFIGPAGE_PLAYLIST_LOADID3TAG_CHKBOX_MAX);
	this->groupcomboxes_preloadinfo = new DispQtConfigpageComboboxGroup(main_window, widget_preloadinfo_comboxes, &playlist_preloadinfo_combobox_lines[0], DISPQT_CONFIGPAGE_PLAYLIST_PRELOADINFO_COMBOX_MAX);
	this->groupchkboxes_preloadinfo = new DispQtConfigpageCheckboxGroup(main_window, widget_preloadinfo_checkboxes, &playlist_preloadinfo_checkboxes[0], DISPQT_CONFIGPAGE_PLAYLIST_PRELOADINFO_CHKBOX_MAX);
	this->groupchkboxes_sortcontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Sort control", &playlist_sortcontrol_checkboxes[0], DISPQT_CONFIGPAGE_PLAYLIST_SORTCONTROL_CHKBOX_MAX);

	layout_control_loadid3->setMargin(0);
	layout_control_loadid3->addWidget(this->groupchkboxes_playlistcontrol);
	layout_control_loadid3->addWidget(this->groupchkboxes_loadid3tag);
	layout_control_loadid3->addStretch(1);
	widget_control_loadid3->setLayout(layout_control_loadid3);

	groupbox_preloadinfo->setTitle("ID3 / metadata preload mode");
	layout_preloadinfo->setMargin(0);
	layout_preloadinfo->setSpacing(2);
	layout_preloadinfo->addStretch(1);
	layout_preloadinfo->addWidget(widget_preloadinfo_comboxes);
	layout_preloadinfo->addWidget(widget_preloadinfo_checkboxes);
	groupbox_preloadinfo->setLayout(layout_preloadinfo);

	mainLayout->setMargin(0);
	mainLayout->addWidget(widget_control_loadid3);
	mainLayout->addWidget(groupbox_preloadinfo);
	mainLayout->addWidget(this->groupchkboxes_sortcontrol);
	mainLayout->addStretch(1);
	setLayout(mainLayout);
}

void DispQtConfigpagePlayList::configpage_playlist_applyall(void)
{
	this->groupchkboxes_playlistcontrol->checkboxgroup_applyall();
	mpxplay_playlistcontrol_mpxpini = mpxplay_playlistcontrol;
	this->groupchkboxes_loadid3tag->checkboxgroup_applyall();
	this->groupcomboxes_preloadinfo->comboboxgroup_applyall();
	this->groupchkboxes_preloadinfo->checkboxgroup_applyall();
	loadid3tag_mpxpini = loadid3tag;
	preloadinfo_mpxpini = preloadinfo;
	this->groupchkboxes_sortcontrol->checkboxgroup_applyall();
}

//----------------------------------------------------------------------------------------------
static struct configpage_chkbox_s config_play_control_checkboxes[] = {
	{"Play random/shuffle",    &playrand, 1, "Play media files randomly from playlist\n(skip forward = random, skip back = step on randomly played files backward)" },
	{"Replay/repeat song",     &playreplay, REPLAY_SONG, "Repeat/replay current song only (endlessly)" },
	{"Replay/repeat list",     &playreplay, REPLAY_LIST, "Repeat/replay all playlist entries"},
	{"Exit at end of list",    &playcontrol, PLAYC_EXITENDLIST, "Except if directory has no media files" },
	{"Pause before each song", &playcontrol, PLAYC_PAUSEALL, "Auto pause at each song" },
	{"Pause at next song",     &playcontrol, PLAYC_PAUSENEXT, "Pause playing at the begin of next song"},
	{"Disable auto program skip", &playcontrol, PLAYC_NOAUTOPRGSKIP, "Don't skip (DVB/TS) program automatically\n(if the current/selected program lost)"},
	{"Highlight scan from half",  &playcontrol, PLAYC_HSSTARTPOSHALF, "Start position of the highlight scan is\nthe half/center position of the file.\nThis option is usefull for video files.\n(\"Start timepos\" is ignored in this case)"}
};

#define DISPQT_CONFIGPAGE_PLAY_CONTROL_CHKBOX_MAX (sizeof(config_play_control_checkboxes) / sizeof(struct configpage_chkbox_s))

static struct configpage_chkbox_s config_play_startup_checkboxes[] = {
	{"Save play position",     &su_startupenabled, MPXPLAY_STARTUP_FLAG_SAVE, "Save play position at exit" },
	{"Restore play position",  &su_startupenabled, MPXPLAY_STARTUP_FLAG_LOAD, "Restore play positon at program start" },
	{"Restore all tabs/sides", &su_startupenabled, MPXPLAY_STARTUP_FLAG_2SIDES },
	{"Start playing auto",     &su_startupenabled, MPXPLAY_STARTUP_FLAG_RESDIR, "Start playing audio at program start" },
	{"Save at every new song", &su_startupenabled, MPXPLAY_STARTUP_FLAG_NEWFILE, "Save play position at every track / song change\n(requires program restart to take effect)" }
};

#define DISPQT_CONFIGPAGE_PLAY_STARTUP_CHKBOX_MAX (sizeof(config_play_startup_checkboxes) / sizeof(struct configpage_chkbox_s))

static struct configpage_editline_s config_play_hilitescan_editlines[] = {
	{"Start timepos", &mpxplay_highlitescan_starttime[0], 16, "Song start position of hi-lite scan in MM:SS\n(note: you cannot set zero value for this,\nbut you can use 00:00.01 setting)"},
	{"Duration", &mpxplay_highlitescan_timecount[0], 16, "Duration of hi-lite scan at each song in MM:SS\n(note: you cannot set zero value for this,\nminimum 00:01)"}
};

#define DISPQT_CONFIGPAGE_PLAY_HILITESCAN_EDITLINE_MAX (sizeof(config_play_hilitescan_editlines) / sizeof(struct configpage_editline_s))

DispQtConfigpagePlayControl::DispQtConfigpagePlayControl(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	QWidget *widget_playcontrol = new QWidget(this);
	QWidget *widget_startup_hs = new QWidget(this);
	QVBoxLayout *mainLayout = new QVBoxLayout;
	QHBoxLayout *layout_control = new QHBoxLayout;
	QVBoxLayout *layout_startup_hs = new QVBoxLayout;

	this->groupchkboxes_playcontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Play control", config_play_control_checkboxes, DISPQT_CONFIGPAGE_PLAY_CONTROL_CHKBOX_MAX);
	this->groupchkboxes_startup = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Startup control", config_play_startup_checkboxes, DISPQT_CONFIGPAGE_PLAY_STARTUP_CHKBOX_MAX);
	this->groupeditlines_hilitescan = new DispQtConfigpageEditlineGroupbox(main_window, this, "Highlight scan", config_play_hilitescan_editlines, DISPQT_CONFIGPAGE_PLAY_HILITESCAN_EDITLINE_MAX);

	layout_startup_hs->setMargin(0);
	layout_startup_hs->addWidget(this->groupchkboxes_startup);
	layout_startup_hs->addWidget(this->groupeditlines_hilitescan);
	layout_startup_hs->addStretch(1);
	widget_startup_hs->setLayout(layout_startup_hs);

	layout_control->setMargin(0);
	layout_control->addWidget(this->groupchkboxes_playcontrol);
	layout_control->addWidget(widget_startup_hs);
	layout_control->addStretch(1);
	widget_playcontrol->setLayout(layout_control);

	mainLayout->setMargin(0);
	mainLayout->addWidget(widget_playcontrol);
	mainLayout->addStretch(1);
	setLayout(mainLayout);
}

void DispQtConfigpagePlayControl::configpage_playcontrol_applyall(void)
{
	this->groupchkboxes_playcontrol->checkboxgroup_applyall();
	this->groupchkboxes_startup->checkboxgroup_applyall();
	this->groupeditlines_hilitescan->editlinegroupbox_applyall();
}

//----------------------------------------------------------------------------------------------
static struct configpage_chkbox_s config_program_control_checkboxes[] = {
	{"Confirm exit (at ESC and F10)", &mpxplay_programcontrol, MPXPLAY_PROGRAMC_CONFIRMEXIT, "Show a confirm dialog/window at exit" },
	{"Shutdown system at exit !",     &mpxplay_programcontrol, MPXPLAY_PROGRAMC_SHUTDOWNATX, "Switch off machine at exit\n(except exiting with F10 key)"},
	{"Disable screen saver at play",  &mpxplay_programcontrol, MPXPLAY_PROGRAMC_SCREENSAVEROFF, "Disable OS screen saver while playing\n(screen saver is reactivated at pause/stop/exit)" },
	{"Extended filetype detection",   &mpxplay_programcontrol, MPXPLAY_PROGRAMC_AUTODETECTALWAYS, "By default, file types are detected by extension only.\nThis option helps if the file format doesn't match with the extension." },
	{"Call system app for unknown",   &mpxplay_programcontrol, MPXPLAY_PROGRAMC_SYSTEMAPPOPEN_UNKNOWN, "Call the default system application for the unknown filetypes\n(like doc or pdf files), at pressing the enter key." },
	{"Call system app for all files", &mpxplay_programcontrol, MPXPLAY_PROGRAMC_SYSTEMAPPOPEN_ALLFILES, "Call the default system application for all filetypes at pressing the enter key.\nInside A/V playing still can be controlled by ctrl-enter and skip keys." }
};

static struct configpage_editline_s config_program_extrafiletypes_editline[] = {
    {"File extension list", &mpxplay_extrafiletypes_config[0], sizeof(mpxplay_extrafiletypes_config), "Loaded / displayed extra file types in player mode\nExtension list is separated by ',' ( MP?,EXT2,EXT3,ETC* ) (wildcards are supported)\n(requires ctrl-'R' or dir change and \"Extended filetype detection\" to take effect)"}
};

static char config_program_executetime_strtmp_startpos[MPXINI_MAX_CHARDATA_LEN], config_program_executetime_strtmp_duration[MPXINI_MAX_CHARDATA_LEN], config_program_executetime_strtmp_exittime[MPXINI_MAX_CHARDATA_LEN];

static struct configpage_editline_s config_program_executetime_editlines[] = {
	{"Playstart timepos", &config_program_executetime_strtmp_startpos[0], 16, "Start playing at HH:MM:SS"},
	{"Duration", &config_program_executetime_strtmp_duration[0], 16, "Playing time-len in HH:MM:SS\n(exit from the program after 'start/current time + duration')"},
	{"Exit timepos", &config_program_executetime_strtmp_exittime[0], 16, "Exit from the program at HH:MM:SS\nIf Duration and Exit timepos are also set,\nthen the earlier (less) matters."}
};

#define DISPQT_CONFIGPAGE_PROGRAM_CONTROL_CHKBOX_MAX (sizeof(config_program_control_checkboxes) / sizeof(struct configpage_chkbox_s))
#define DISPQT_CONFIGPAGE_PROGRAM_EXECUTETIME_EDITLINE_MAX (sizeof(config_program_executetime_editlines) / sizeof(struct configpage_editline_s))
#define DISPQT_CONFIGPAGE_EXTRAFILETYPES_EDITLINE_MAX (sizeof(config_program_extrafiletypes_editline) / sizeof(struct configpage_editline_s))

DispQtConfigpageProgram::DispQtConfigpageProgram(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	unsigned long hextime;

	mpxplay_playtimestart+= mpxplay_progtimebegin;
	if(mpxplay_playtimestart > 24*3600)
		mpxplay_playtimestart -= 24*3600;
	hextime = PDS_SECONDS_TO_HEXTIME(mpxplay_playtimestart);
	pds_hextime_to_strtime(hextime, config_program_executetime_strtmp_startpos, sizeof(config_program_executetime_strtmp_startpos));

	if(mpxplay_progtimelen > mpxplay_playtimestart)
		mpxplay_progtimelen -= mpxplay_playtimestart;
	hextime = PDS_SECONDS_TO_HEXTIME(mpxplay_progtimelen);
	pds_hextime_to_strtime(hextime, config_program_executetime_strtmp_duration, sizeof(config_program_executetime_strtmp_duration));

	mpxplay_progtimeexit = mpxplay_progtimebegin + mpxplay_progtimelen;
	if(mpxplay_progtimeexit > 24*3600)
		mpxplay_progtimeexit -= 24*3600;
	hextime = PDS_SECONDS_TO_HEXTIME(mpxplay_progtimeexit);
	pds_hextime_to_strtime(hextime, config_program_executetime_strtmp_exittime, sizeof(config_program_executetime_strtmp_exittime));

	this->groupchkboxes_programcontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Program control", config_program_control_checkboxes, DISPQT_CONFIGPAGE_PROGRAM_CONTROL_CHKBOX_MAX);
	this->groupeditlines_extrafiletypes = new DispQtConfigpageEditlineGroupbox(main_window, this, "Loaded extra filetypes", config_program_extrafiletypes_editline, DISPQT_CONFIGPAGE_EXTRAFILETYPES_EDITLINE_MAX);
	this->groupeditlines_executetime = new DispQtConfigpageEditlineGroupbox(main_window, this, "Program execution times", config_program_executetime_editlines, DISPQT_CONFIGPAGE_PROGRAM_EXECUTETIME_EDITLINE_MAX);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->setMargin(0);
	mainLayout->addWidget(this->groupchkboxes_programcontrol);
	mainLayout->addWidget(this->groupeditlines_extrafiletypes);
	mainLayout->addWidget(this->groupeditlines_executetime);
	mainLayout->addStretch(1);
	setLayout(mainLayout);
}

void DispQtConfigpageProgram::configpage_program_applyall(void)
{
	unsigned long prgcontrol = mpxplay_programcontrol;
	this->groupchkboxes_programcontrol->checkboxgroup_applyall();
	this->groupeditlines_extrafiletypes->editlinegroupbox_applyall();
	this->groupeditlines_executetime->editlinegroupbox_applyall();
	mpxplay_playtimestart=pds_strtime_to_hextime(config_program_executetime_strtmp_startpos,1);
	mpxplay_playtimestart=PDS_HEXTIME_TO_SECONDS(mpxplay_playtimestart);
	mpxplay_progtimelen=pds_strtime_to_hextime(config_program_executetime_strtmp_duration,1);
	mpxplay_progtimelen=PDS_HEXTIME_TO_SECONDS(mpxplay_progtimelen);
	mpxplay_progtimeexit=pds_strtime_to_hextime(config_program_executetime_strtmp_exittime,1);
	mpxplay_progtimeexit=PDS_HEXTIME_TO_SECONDS(mpxplay_progtimeexit);
	mpxplay_control_start_progexectime();
	mpxplay_control_extrafiletypes_slice();
	if((prgcontrol & MPXPLAY_PROGRAMC_SCREENSAVEROFF) != (mpxplay_programcontrol & MPXPLAY_PROGRAMC_SCREENSAVEROFF))
		funcbit_smp_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_PLAYSTATECHG); // FIXME: to wake up mpxplay_dispqt_screensaver_update (not correct signal)
}

//==================================================================================================================
// "Audio" config

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
static struct configpage_comboboxline_s config_stream_audio_extfile_combobox_group[] = {
{
	"External file load: ", ARG_NUM, (void *)&mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_AUDIO], (sizeof(config_extstream_loadtype_values) / sizeof(char *)),
	&config_extstream_loadtype_values[0], &config_extstream_loadtype_description[0],
	"1. Do nothing with external streams (no auto search & load)\n"
	"2. Auto search for external streams, add to the stream list (video context menu)\n"
	"3. Auto search for external streams, use by automatic decision (internal stream, language code)\n"
	"4. Auto search for external streams, always use (prefer external streams over internal ones)"
},
{"File search paths: ", DISPQT_CONFIG_COMBOTYPE_EDITLINE, &mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_AUDIO][0], MPXINI_MAX_CHARDATA_LEN, NULL, NULL,"Configure external audio streams search directories\n(relative to the video file, separated by ';',\nrequires videofile re-open to take effect)"}
};

static struct configpage_comboboxline_s config_video_preferaudio_combobox_group[] = {
 {"Preferred audio types", DISPQT_CONFIG_COMBOTYPE_EDITLINE, &mpxplay_config_video_preffered_audiotypes[0], sizeof(mpxplay_config_video_preffered_audiotypes), NULL, NULL, "Select preferred audio format(s),\nin case of multiply streams in the container,\nseparate by ',' (like AC3,DTS,MP3,AAC,Vorbis)\nNote1: wildcards are supported (like MP?,AAC*)\nNote2: audio language setting has higher priority\n(but combined with this list)"},
 {"Prefer descriptor stream", DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_AUDIOSTREAM_PREF_DESCRIPTOR, NULL, NULL, "Prefer audio stream containing a description of the video content.\nTypically intended for visually-impaired audiences\nor for the cases where the video cannot be seen."}
};

#endif //MPXPLAY_LINK_ORIGINAL_FFMPEG

static unsigned long dispqt_config_audio_control_swapchan, dispqt_config_audio_soundcard_bits, dispqt_config_audio_output_control_selectcfg;

static struct configpage_chkbox_s config_audio_control_checkboxes[] = {
	{"Swap channels ( L <-> R )", &dispqt_config_audio_control_swapchan, 1, "Reverse stereo" },
	{"Enable limiter", &MIXER_controlbits, MIXER_CONTROLBIT_LIMITER, "Fast auto volume reductor\n(to avoid audio distortions)" },
	{"High-res speed control", &MIXER_controlbits, MIXER_CONTROLBIT_SPEED1000, "x10 resolution for audio speed\n(requires program restart to take effect)" }
#if !defined(MPXPLAY_LINK_ORIGINAL_FFMPEG) || !defined(MPXPLAY_ARCH_X64)
	,{"Disable decoder tone (MPX)", &MIXER_controlbits, MIXER_CONTROLBIT_DISABLE_DECODERTONE, "Use mixer tone (high-/low-pass filters) for MPX files too" }
#endif
};

static struct configpage_spinbox_s config_audio_loudness_values_spinboxes[] = {
	{"Volume",   &MIXER_loudness_val_volume,    0, 700},
	{"Surround", &MIXER_loudness_val_surround, 10, 300},
	{"Bass",     &MIXER_loudness_val_bass,      0, 200},
	{"Treble",   &MIXER_loudness_val_treble,    0, 200}
};

#define DISPQT_CONFIGPAGE_AUDIO_CONTROL_CHKBOX_MAX (sizeof(config_audio_control_checkboxes) / sizeof(struct configpage_chkbox_s))
#define DISPQT_CONFIGPAGE_AUDIO_LOUDNESS_SPINBOX_MAX (sizeof(config_audio_loudness_values_spinboxes) / sizeof(struct configpage_spinbox_s))

#define DISPQT_CONFIGPAGE_AUDIO_OUTPUT_FLOAT32_SIGN 33
#define DISPQT_CONFIGPAGE_AUDIO_OUTMODE_CHECKFLAGS (OUTMODE_CONTROL_FILE_FLOATOUT)

static const char *config_audio_output_sndcardname_data_values[]      = {"AUTO","WDS","WIN","NUL"};
static const char *config_audio_output_sndcardname_data_description[] = {"Autodetect","Directsound","Wave mapper","Null output"};

static const char *config_audio_output_sndcardfreq_data_values[]      = {"0","8000","11025","16000","22050","32000","44100","48000","96000","192000"};
static const char *config_audio_output_sndcardfreq_data_description[] = {"As input","8000 Hz","11025 Hz","16000 Hz","22050 Hz","32000 Hz","44100 Hz","48000 Hz","96000 Hz","192000 Hz"};

static const char *config_audio_output_sndcardbits_data_values[]      = {"0","8","16","24","32","33"}; // 33 = DISPQT_CONFIGPAGE_AUDIO_OUTPUT_FLOAT32_SIGN
static const char *config_audio_output_sndcardbits_data_description[] = {"Default (16)", "8 bits", "16 bits", "24 bits", "32 bits", "32-float"};

static const char *config_audio_output_sndcardchans_data_values[]      = {"0","1","2","3","4","5","6","7","8"};
static const char *config_audio_output_sndcardchans_data_description[] = {"Default (2)", "Mono", "Stereo", "2.1 chans", "4.0 chans", "5.0 chans", "5.1 chans", "7.0 chans", "7.1 chans"};

static struct configpage_comboboxline_s config_audio_output_soundcard_combobox_group[] = { {
	"Device      ", ARG_CHAR|ARG_POINTER, (void *)&au_infos.card_selectname, (sizeof(config_audio_output_sndcardname_data_values) / sizeof(char *)),
	&config_audio_output_sndcardname_data_values[0], &config_audio_output_sndcardname_data_description[0],
	"Primary device is Directsound"
},{
	"Frequency   ", ARG_NUM, (void *)&au_infos.freq_set, (sizeof(config_audio_output_sndcardfreq_data_values) / sizeof(char *)),
	&config_audio_output_sndcardfreq_data_values[0], &config_audio_output_sndcardfreq_data_description[0],
	"By default, output frequency equals with the freq of input file\nYou can set a constant output frequency here\n(note: Directsound doesn't accept freqs bellow 44100Hz)"
},{
	"Bits        ", ARG_NUM, (void *)&dispqt_config_audio_soundcard_bits, (sizeof(config_audio_output_sndcardbits_data_values) / sizeof(char *)),
	&config_audio_output_sndcardbits_data_values[0], &config_audio_output_sndcardbits_data_description[0],
	"Output bit (dynamic) resolution"
},{
	"Channels    ", ARG_NUM, (void *)&au_infos.chan_set, (sizeof(config_audio_output_sndcardchans_data_values) / sizeof(char *)),
	&config_audio_output_sndcardchans_data_values[0], &config_audio_output_sndcardchans_data_description[0],
	"Channel mapping of Directsound device.\nNot all devices support all channel variations\n(sound can stop in this case)"
} };

#define DISPQT_AUCARDSCONFIG_LOCHARDWARE 2

static struct configpage_chkbox_s config_audio_output_soundcard_checkboxes[] = {
	{"Double buffer", &outmode, OUTMODE_CONTROL_SNDCARD_DDMA, "Less gaps in playing with higher latency\n(186 -> 372 msec)" },
	{"Hardware mixing", &dispqt_config_audio_output_control_selectcfg, DISPQT_AUCARDSCONFIG_LOCHARDWARE, "Enable DirectSound hardware mixing\n(can give better sound)"},
	{" ",NULL,0,NULL},
	{" ",NULL,0,NULL}
};

#define DISPQT_CONFIGPAGE_AUDIO_OUTPUT_SOUNDCARD_COMBOX_MAX (sizeof(config_audio_output_soundcard_combobox_group) / sizeof(struct configpage_comboboxline_s))
#define DISPQT_CONFIGPAGE_AUDIO_OUTPUT_SOUNDCARD_CHKBOX_MAX (sizeof(config_audio_output_soundcard_checkboxes) / sizeof(struct configpage_chkbox_s))

DispQtConfigpageAudio::DispQtConfigpageAudio(MainWindow *main_window, QWidget *parent) :  DispQtDialogElemTabWidget(main_window, parent)
{
	QWidget *audio_tab_widget_input = new QWidget(this);
	QVBoxLayout *audio_tab_layout_input = new QVBoxLayout;

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->combobox_audio_language_config = new DispQtConfigpageComboboxGroupbox(main_window, this, "Audio language", &config_stream_languages_combobox_group[MPXPLAY_STREAMTYPEINDEX_AUDIO][0], (sizeof(config_stream_languages_combobox_group[MPXPLAY_STREAMTYPEINDEX_AUDIO]) / sizeof(struct configpage_comboboxline_s)));
	this->combobox_audio_extfile_config = new DispQtConfigpageComboboxGroupbox(main_window, this, "External stream", &config_stream_audio_extfile_combobox_group[0], (sizeof(config_stream_audio_extfile_combobox_group) / sizeof(struct configpage_comboboxline_s)));
	this->groupeditlines_audio_format_priority = new DispQtConfigpageComboboxGroupbox(main_window, this, "Audio stream priority", &config_video_preferaudio_combobox_group[0], (sizeof(config_video_preferaudio_combobox_group) / sizeof(struct configpage_comboboxline_s)));
#endif

	audio_tab_layout_input->setContentsMargins(0,5,0,0);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	audio_tab_layout_input->addWidget(this->combobox_audio_language_config);
	audio_tab_layout_input->addWidget(this->combobox_audio_extfile_config);
	audio_tab_layout_input->addWidget(this->groupeditlines_audio_format_priority);
#endif
	audio_tab_layout_input->addStretch(1);
	audio_tab_widget_input->setLayout(audio_tab_layout_input);

	//----------------------------------------------------------------------------------------------------------------
	QWidget *audio_tab_widget_mixer = new QWidget(this);
	QWidget *widget_mixer_control = new QWidget(this);
	QVBoxLayout *audio_tab_layout_mixer = new QVBoxLayout;
	QHBoxLayout *layout_mixer_control = new QHBoxLayout;

	dispqt_config_audio_control_swapchan = MIXER_var_swapchan;

	this->groupchkboxes_audio = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Audio control", config_audio_control_checkboxes, DISPQT_CONFIGPAGE_AUDIO_CONTROL_CHKBOX_MAX);
	this->groupspinboxes_loudness = new DispQtConfigpageSpinboxGroupbox(main_window, this, "Expand / Loudness", config_audio_loudness_values_spinboxes, DISPQT_CONFIGPAGE_AUDIO_LOUDNESS_SPINBOX_MAX);

	layout_mixer_control->setMargin(0);
	layout_mixer_control->addWidget(this->groupchkboxes_audio);
	layout_mixer_control->addWidget(this->groupspinboxes_loudness);
	widget_mixer_control->setLayout(layout_mixer_control);

	audio_tab_layout_mixer->setContentsMargins(0,5,0,0);
	audio_tab_layout_mixer->addWidget(widget_mixer_control);
	audio_tab_layout_mixer->addStretch(1);
	audio_tab_widget_mixer->setLayout(audio_tab_layout_mixer);

	//----------------------------------------------------------------------------------------------------------------
	QWidget *audio_tab_widget_output = new QWidget(this);
	QWidget *widget_output_control = new QWidget(this);
	QVBoxLayout *audio_tab_layout_output = new QVBoxLayout;
	QHBoxLayout *layout_output_control = new QHBoxLayout;

	if(funcbit_test(outmode, OUTMODE_CONTROL_FILE_FLOATOUT))
		dispqt_config_audio_soundcard_bits = DISPQT_CONFIGPAGE_AUDIO_OUTPUT_FLOAT32_SIGN;
	else
		dispqt_config_audio_soundcard_bits = au_infos.bits_set;

	dispqt_config_audio_output_control_selectcfg = (au_infos.card_select_config > 0)? au_infos.card_select_config : 0;

	this->groupcomboxes_soundcard = new DispQtConfigpageComboboxGroupbox(main_window, this, "Audio output config", &config_audio_output_soundcard_combobox_group[0], DISPQT_CONFIGPAGE_AUDIO_OUTPUT_SOUNDCARD_COMBOX_MAX);
	this->groupchkboxes_soundcard = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Audio output control", config_audio_output_soundcard_checkboxes, DISPQT_CONFIGPAGE_AUDIO_OUTPUT_SOUNDCARD_CHKBOX_MAX);

	layout_output_control->setMargin(0);
	layout_output_control->addWidget(this->groupcomboxes_soundcard);
	layout_output_control->addWidget(this->groupchkboxes_soundcard);
	widget_output_control->setLayout(layout_output_control);

	audio_tab_layout_output->setContentsMargins(0,5,0,0);
	audio_tab_layout_output->addWidget(widget_output_control);
	audio_tab_layout_output->addStretch(1);
	audio_tab_widget_output->setLayout(audio_tab_layout_output);

	//----------------------------------------------------------------------------------------------------------------
	this->addTab(audio_tab_widget_input, "Audio input");
	this->addTab(audio_tab_widget_mixer, "Audio mixer");
	this->addTab(audio_tab_widget_output,"Audio output");

	this->setDocumentMode(true);
}

void DispQtConfigpageAudio::configpage_audio_applyall(void)
{
	int freq_old = au_infos.freq_set, bits_old = au_infos.bits_set, chan_old = au_infos.chan_set;
	unsigned long mixer_controlbits_old = MIXER_controlbits, outmode_old = outmode;
	long card_subconfig_old = dispqt_config_audio_output_control_selectcfg;
	char cardname_old[16];

	pds_strncpy(cardname_old, au_infos.card_selectname, sizeof(cardname_old));
	cardname_old[sizeof(cardname_old) - 1] = 0;

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->combobox_audio_language_config->comboboxgroupbox_applyall();
	this->combobox_audio_extfile_config->comboboxgroupbox_applyall();
	this->groupeditlines_audio_format_priority->comboboxgroupbox_applyall();
#endif
	this->groupchkboxes_audio->checkboxgroup_applyall();
	this->groupspinboxes_loudness->spinboxgroup_applyall();
	this->groupcomboxes_soundcard->comboboxgroupbox_applyall();
	this->groupchkboxes_soundcard->checkboxgroup_applyall();

	mpxplay_control_extrafiletypes_slice();

	if(dispqt_config_audio_control_swapchan != MIXER_var_swapchan)
		pds_pushkey(kb[76].c); // !!!

	if(dispqt_config_audio_soundcard_bits == DISPQT_CONFIGPAGE_AUDIO_OUTPUT_FLOAT32_SIGN) {
		funcbit_enable(outmode, OUTMODE_CONTROL_FILE_FLOATOUT);
		au_infos.bits_set = 0;
	} else {
		funcbit_disable(outmode, OUTMODE_CONTROL_FILE_FLOATOUT);
		au_infos.bits_set = dispqt_config_audio_soundcard_bits;
	}

	if( ((outmode_old & OUTMODE_CONTROL_SNDCARD_DDMA) != (outmode & OUTMODE_CONTROL_SNDCARD_DDMA))
	 || (card_subconfig_old != dispqt_config_audio_output_control_selectcfg)
	 || (pds_strlicmp(cardname_old, au_infos.card_selectname) != 0)
	){
		if(playcontrol & PLAYC_RUNNING)
			funcbit_enable(playcontrol, PLAYC_STARTNEXT);
		if(outmode & OUTMODE_CONTROL_SNDCARD_DDMA)
			funcbit_enable(au_infos.card_controlbits, AUINFOS_CARDCNTRLBIT_DOUBLEDMA);
		else
			funcbit_disable(au_infos.card_controlbits, AUINFOS_CARDCNTRLBIT_DOUBLEDMA);
		au_infos.card_select_config = (dispqt_config_audio_output_control_selectcfg & DISPQT_AUCARDSCONFIG_LOCHARDWARE)? DISPQT_AUCARDSCONFIG_LOCHARDWARE : -1;
		AU_close(&au_infos);
		AU_init(&au_infos);
		funcbit_enable(au_infos.card_controlbits, AUINFOS_CARDCNTRLBIT_UPDATEFREQ);
		mvps.adone = ADONE_REOPEN;
	}

	if((freq_old != au_infos.freq_set) || (bits_old != au_infos.bits_set) || (chan_old != au_infos.chan_set) || ((outmode_old & DISPQT_CONFIGPAGE_AUDIO_OUTMODE_CHECKFLAGS) != (outmode & DISPQT_CONFIGPAGE_AUDIO_OUTMODE_CHECKFLAGS))){
		funcbit_enable(au_infos.card_controlbits, AUINFOS_CARDCNTRLBIT_UPDATEFREQ);
		mvps.adone = ADONE_REOPEN;
	}

	if(mixer_controlbits_old != MIXER_controlbits)
		mvps.adone = ADONE_REOPEN;

	outmode_mpxpini = outmode;
	mpxplay_card_selectconfig_mpxpini = au_infos.card_select_config;
}

//============================================================================================================================
// "Subtitle" config

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

static struct configpage_comboboxline_s config_stream_subtitle_extfile_combobox_group[] = {
 {
  "External file load: ", ARG_NUM, (void *)&mpxplay_config_video_extstream_loadtype[MPXPLAY_STREAMTYPEINDEX_SUBTITLE], (sizeof(config_extstream_loadtype_values) / sizeof(char *)),
  &config_extstream_loadtype_values[0], &config_extstream_loadtype_description[0],
  "Program can search and load various subtitle files automatically.\n"
  "(matching the subtitle filename with the video name,\n"
  "optionally extended with language code, like: videoname.lng.srt)\n"
  "Load mode variants are the same as at the external audio files."
 },
 {"File search paths: ", DISPQT_CONFIG_COMBOTYPE_EDITLINE, &mpxplay_config_video_extstream_subdirs[MPXPLAY_STREAMTYPEINDEX_SUBTITLE][0], MPXINI_MAX_CHARDATA_LEN, NULL, NULL, "Configure subtitle files search directories\n(relative to the video file, separated by ';',\nrequires videofile re-open to take effect)"}
};

static struct configpage_chkbox_s config_stream_subtitle_checkbox_group[] = {
 {"Disable forced && default subtitles", &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_STREAMS_FORCED_DISABLE, "Disable automatic use of 'Forced'\nand 'Default' embedded subtitles"},
 {"Disable all embedded subtitles", &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_SUBTITLES_EMBEDDED_DISABLE, "Don't use automatically embedded (internal) subtitle streams\n(just put them in the stream list for manual selection)"},
 {"Enable UTF-8 text decoding", &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_SUBTITLES_UTF8DEC_ENABLE, "Enable UTF-8 text decoding on subtitle files without UTF header\n(if a text file has no 3-bytes UTF header, local codepage is used by default)\n(change has effect after file reopen only)"}
};

static struct configpage_chkbox_s config_stream_other_swithes[] = {
{"Display short language codes", &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_STREAMLIST_SHORT_LANGNAMES, "Display short language codes\nin Video context menu -> Streams\n([lng] instead of [Language])"}};

DispQtConfigpageSubtitle::DispQtConfigpageSubtitle(MainWindow *main_window, QWidget *parent) : QWidget(parent)
{
	this->combobox_subtitle_language_config = new DispQtConfigpageComboboxGroupbox(main_window, this, "Subtitle language", &config_stream_languages_combobox_group[MPXPLAY_STREAMTYPEINDEX_SUBTITLE][0], (sizeof(config_stream_languages_combobox_group[MPXPLAY_STREAMTYPEINDEX_SUBTITLE]) / sizeof(struct configpage_comboboxline_s)));
	this->combobox_subtitle_extfile_config = new DispQtConfigpageComboboxGroupbox(main_window, this, "External subtitle", &config_stream_subtitle_extfile_combobox_group[0], (sizeof(config_stream_subtitle_extfile_combobox_group) / sizeof(struct configpage_comboboxline_s)));
	this->checkbox_subtitle_switches = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Subtitle settings", &config_stream_subtitle_checkbox_group[0], (sizeof(config_stream_subtitle_checkbox_group) / sizeof(struct configpage_chkbox_s)));
	this->checkbox_other_switches = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Other stream settings", &config_stream_other_swithes[0], (sizeof(config_stream_other_swithes) / sizeof(struct configpage_chkbox_s)));

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->setMargin(0);
	mainLayout->addWidget(this->combobox_subtitle_language_config);
	mainLayout->addWidget(this->combobox_subtitle_extfile_config);
	mainLayout->addWidget(this->checkbox_subtitle_switches);
	mainLayout->addWidget(this->checkbox_other_switches);
	mainLayout->addStretch(1);
	setLayout(mainLayout);
}

void DispQtConfigpageSubtitle::configpage_subtitle_applyall(void)
{
	this->combobox_subtitle_language_config->comboboxgroupbox_applyall();
	this->combobox_subtitle_extfile_config->comboboxgroupbox_applyall();
	this->checkbox_subtitle_switches->checkboxgroup_applyall();
	this->checkbox_other_switches->checkboxgroup_applyall();
}

#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

//============================================================================================================================
// "Video" config

static const char *config_video_decodertype_data_values[] = {
	"0",  // MPXPLAY_VIDEOPLAYERTYPE_NONE
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	"1",  // MPXPLAY_VIDEOPLAYERTYPE_FFMPEG
#endif
#ifdef MPXPLAY_LINK_QTMEDIA
	"5",  // MPXPLAY_VIDEOPLAYERTYPE_QTMEDIA
#endif
};

static const char *config_video_decodertype_description[] = {
	"None (audio only)",
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	"FFmpeg (internal)",
#endif
#ifdef MPXPLAY_LINK_QTMEDIA
	"QtMedia (DShow)",
#endif
};

//-------
static const char *config_video_renderertype_data_values[] = {
#if MPXPLAY_DISPQT_ENABLE_RENDER_QPAINT
	"0"  // DISPQT_VIDEORENDERTYPE_QPAINT
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
	,"1"  // DISPQT_VIDEORENDERTYPE_D3D11
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
	,"2"  // DISPQT_VIDEORENDERTYPE_D3D9
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_VULKAN
	,"3"  // DISPQT_VIDEORENDERTYPE_VULKAN
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_GDI
	,"4"  // DISPQT_VIDEORENDERTYPE_GDI
#endif
};

static const char *config_video_renderertype_description[] = {
#if MPXPLAY_DISPQT_ENABLE_RENDER_QPAINT
	"QWidget / QPaint"
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
	,"Direct3D 11"
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
	,"Direct3D 9"
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_VULKAN
	,"Vulkan"
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_GDI
	,"Windows GDI"
#endif
};

//-------
static const char *config_video_windowedartype_data_values[] = {
	"0",  // DISPQT_VIDEOWINDOWEDARTYPE_EQFULLSCREEN
	"1",  // DISPQT_VIDEOWINDOWEDARTYPE_VIDEO
	"2"   // DISPQT_VIDEOWINDOWEDARTYPE_FILL
};

static const char *config_video_windowedartype_description[] = {
	"Equal to fullscreen setting",
	"Use the video aspect ratio",
	"Fill window (no aspect ratio)"
};

//-------
static const char *config_video_autocrop_limit_data_values[] = {
	"999999",
	"576",
	"720",
	"1080",
	"1440",
	"2160",
};

static const char *config_video_autocrop_limit_description[] = {
	"None",
	"576p",
	"720p",
	"1080p",
	"1440p",
	"2160p"
};

//-------
static struct configpage_comboboxline_s config_video_decoder_config_combobox_group[] = {
{
	"Decoder type: ", ARG_NUM, (void *)&mmc_config.selected_videoplayer_type, (sizeof(config_video_decodertype_data_values) / sizeof(char *)),
	&config_video_decodertype_data_values[0], &config_video_decodertype_description[0],
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	"FFmpeg means built-in decoders (no external codec required)"
#endif
#ifdef MPXPLAY_LINK_QTMEDIA
	"\n\nQtMedia requires Windows DirectShow filters and decoders\n(like Haali Media splitter and FFDShow)"
#endif
#if defined(MPXPLAY_LINK_ORIGINAL_FFMPEG) && defined(MPXPLAY_LINK_QTMEDIA)
	"\n\nWarning: changing this setting, the main window re-opens,\nthe playing restarts, the recording stops!"
#endif
},
{
	"Renderer type: ", ARG_NUM, (void *)&mmc_config.video_renderer_type, (sizeof(config_video_renderertype_data_values) / sizeof(char *)),
	&config_video_renderertype_data_values[0], &config_video_renderertype_description[0],
	"Rendering outputs for FFmpeg decoder:"
#if MPXPLAY_DISPQT_ENABLE_RENDER_QPAINT
	"\n\nWidget video output uses a simple QWidget\nwith sw based colorspace and transformation functions."
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D11
	"\n\nDirect3D11 output uses GPU accelerated functions\nto buffer, process and display video frames,\nincluding D3D11VA decoding, HDR support,\ncrop/zoom and color corrections."
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_D3D9
	"\n\nDirect3D9 output uses GPU accelerated functions\nto buffer, process and display video frames,\nincluding DXVA decoding, crop/zoom and color corrections.\n(needs DirectX runtime libs for complete functionality)"
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_VULKAN
	"\n\nVulkan video output uses Vulkan video API\nto render and display video frames.\n(with sw based colorspace and transformation functions)"
#endif
#if MPXPLAY_DISPQT_ENABLE_RENDER_GDI
	"\nWindows GDI video renderer is an sw-synchronized video output, using MS GDI interface"
#endif
	"\n\nWarning: changing this setting, the main window re-opens,\nthe playing restarts, the recording stops!"
},
{
	"Windowed aspect ratio: ", ARG_NUM, (void *)&mmc_config.video_windowed_ar_type, (sizeof(config_video_windowedartype_data_values) / sizeof(char *)),
	&config_video_windowedartype_data_values[0], &config_video_windowedartype_description[0],
	"Ascpect ratio in Windowed (non-fullscreen) mode\n"
	"can follow the AR setting of fullscreen mode\n"
	"(configured in the video surface context menu)\n"
	"or uses the video AR, or fully fills the window"
},
{
	"Autocrop limit: ", ARG_NUM, (void *)&mmc_config.video_autocrop_limit, (sizeof(config_video_autocrop_limit_data_values) / sizeof(char *)),
	&config_video_autocrop_limit_data_values[0], &config_video_autocrop_limit_description[0],
	"Process (limit) autocrop by resolution\n"
	"(because autocrop is a slow function and\n"
	"usually 4k/8k videos don't require crop)"
}
};

#define DISPQT_CONFIGPAGE_VIDEO_DECODER_SELECT_COMBOX_MAX (sizeof(config_video_decoder_config_combobox_group) / sizeof(struct configpage_comboboxline_s))

//----------------------------------------------------------------------------------------------------------------------------------------------------------
static struct configpage_chkbox_s config_video_control_mainwindow_checkboxes[] = {
 {"Show main window on video",     &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_SHOW_MAINWIN_ON_VIDEO, "Show main MMC window over the video surface\n(popup by mouse pointer or editkeys)" },
 {"Auto hide mainwindow on video", &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_AUTO_HIDE_MAINWIN, "Hide main window over video with fade-out\nafter the configured time-out" },
 {"Popup mainwindow by editkeys",  &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_POPUP_MAINWIN_BY_KEYS, "Show main window over video,\nif an editor key is pressed (like up/down arrows)" }
};

static struct configpage_chkbox_s config_video_control_fullscreen_checkboxes[] = {
 {"Fullscreen video at play-start",  &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_PLAYSTART_FULLSCREEN, "Switch to fullscreen video mode\nat the start of (first) playing\n(playing starts from stop status)" },
 {"Fullscreen video at enter key",   &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_ENTERKEY_FULLSCREEN, "Switch to fullscreen video mode\nat starting new file with enter key\nor with double mouse click" },
 {"Exit fullscreen at play-stop",    &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_EXIT_FULLSCREEN_AT_STOP, "Exit from fullscreen video\nwhen playing has finished\n(end of the playlist or stop key)" },
 {"Pause at ESC from fullscreen",    &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_FULLSCREEN_ESC_PAUSE, "Pause the playing at pressing ESC key\nin fullscreen video mode\n(else ESC only exits from fullscreen)"}
};

static struct configpage_comboboxline_s config_video_control_bottomcontrolbar_group[] = {
 {"Enable bar  ", DISPQT_CONFIG_COMBOTYPE_CHKBOX,  &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_ENABLE_CONTROLBAR, NULL, NULL, "Popup a bottom video surface\ncontrol bar by mouse pointer" },
 {"Bar length  ", DISPQT_CONFIG_COMBOTYPE_SPINBOX, &mmc_config.videocontrolbar_size_percent, DISPQT_CONFIG_SPINBOX_VALPACK_MINMAX(DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_MIN, DISPQT_CONFIG_VIDEO_CONTROLBAR_SIZE_MAX), NULL, NULL, "Bottom control bar length in\npercent of display width (35 - 100)" },
 {"Use fadeout ", DISPQT_CONFIG_COMBOTYPE_CHKBOX,  &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_CONTROLBAR_FADEOUT, NULL, NULL, "Enable fade-out at hiding\ncontrol bar and seek preview" },
 {"Headerless  ", DISPQT_CONFIG_COMBOTYPE_CHKBOX,  &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_CONTROLBAR_HEADERLESS, NULL, NULL, "Use headerless transparent control bar\n(else use Aero/Blur dialog-like bar)\n(option takes effect properly at program restart)" }
};

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
static const char *config_video_control_seekpreview_ar_values[] = {"0","1","3","4","5","6","7","8"};
static const char *config_video_control_seekpreview_ar_description[] = {"None", "Video", "4 : 3", "16 : 9", "2.35:1", "1 : 1", "Custom", "Pixel"};

static struct configpage_comboboxline_s config_video_control_seekpreview_group[] = {
 {"Enable      ",  DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mmc_config.video_control, DISPQT_CONFIG_VIDEOCTRL_SEEKPREVIEW_ENABLE, NULL, NULL, "Show seek preview thumbnail window\nover bottom control bar" },
 {"Window size ",  DISPQT_CONFIG_COMBOTYPE_SPINBOX,  &mmc_config.videoseekpreview_size_percent, DISPQT_CONFIG_SPINBOX_VALPACK_MINMAX(DISPQT_CONFIG_VIDEO_SEEKPREVIEW_SIZE_MIN, DISPQT_CONFIG_VIDEO_SEEKPREVIEW_SIZE_MAX), NULL, NULL, "Seek preview window size\nin percent of display height (5 - 90)"},
 {"Aspect ratio", (DISPQT_CONFIG_COMBOTYPE_COMBOBOX | ARG_NUM), &mmc_config.videoseekpreview_aspect_ratio_type, (sizeof(config_video_control_seekpreview_ar_values) / sizeof(char *)), &config_video_control_seekpreview_ar_values[0], &config_video_control_seekpreview_ar_description[0], "Seek preview individual window aspect ratio\n(None means: ar equals with fullscreen setting)"},
 {"Refresh time",  DISPQT_CONFIG_COMBOTYPE_SPINBOX,  &mmc_config.videoseekpreview_refresh_interval_ms, DISPQT_CONFIG_SPINBOX_VALPACK_MINMAX(DISPQT_CONFIG_VIDEO_SEEKPREVIEW_INTERVAL_MIN, DISPQT_CONFIG_VIDEO_SEEKPREVIEW_INTERVAL_MAX), NULL, NULL, "Seek preview refresh interval in msecs (0 - 1000)\n(zero means the fastest seek preview)"},
 {"",  DISPQT_CONFIG_COMBOTYPE_CHKBOX,  NULL, 0, NULL, NULL, ""}
};
#define DISPQT_CONFIGPAGE_VIDEO_CONTROL_SEEKPREVIEW_COMBOX_MAX (sizeof(config_video_control_seekpreview_group) / sizeof(struct configpage_comboboxline_s))
#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

#define DISPQT_CONFIGPAGE_VIDEO_CONTROL_MW_CHKBOX_MAX (sizeof(config_video_control_mainwindow_checkboxes) / sizeof(struct configpage_chkbox_s))
#define DISPQT_CONFIGPAGE_VIDEO_CONTROL_FS_CHKBOX_MAX (sizeof(config_video_control_fullscreen_checkboxes) / sizeof(struct configpage_chkbox_s))
#define DISPQT_CONFIGPAGE_VIDEO_CONTROL_BOTTOMCONTROLBAR_COMBOX_MAX (sizeof(config_video_control_bottomcontrolbar_group) / sizeof(struct configpage_comboboxline_s))

//----------------------------------------------------------------------------------------------------------------------------------------------------------
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
static struct configpage_chkbox_s config_video_player_control_checkboxes[] = {
 {"Enable HW (GPU) decoding",    &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_USEHWDECODING,  "Enable GPU (DXVA/D3D11VA) decoding for AV1, H264, HEVC, MPEG2, VC1, VP9, WMV\nat selecting D3D9 or D3D11 video output\n(note: this setting is hardware and video driver dependent)\nWarning: changing this setting, the playing restarts, the recording stops!" },
 {"Disable HW decoding for SD",  &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESDHWDEC, "Disable HW decoding for SD content\n(960x576 or lower resolution)" },
 {"Texture buf for SW decoding", &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_TEXBUFFORSWDECODED, "Put SW decoded video frames in HW (d3d9/d3d11) textures\nfor a faster rendering and more precise frame scheduling\nby the cost of extra resources (cpu & memory)\n(this option can be useful at 4k/60fps)" },
 {"Duplicate interlaced frames", &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DUPLICATE_INTERLACED_FRAMES, "Duplicate frame rate (fps) of interlaced video\ngenerating intermediate pictures\n(works at D3D11 output only)\n(setting deinterlace mode to 'Enable',\nprogressive frames are also duplicated up to 4k/33fps)" },
 {"Disable auto drop frame",     &mmc_config.selected_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESKIPFRAME, "Disable automatic video frame drop\n(at slow decoding -> large video delay)\n(SW: drop frames at decoding; HW: drop frames at output)" }
};
#define DISPQT_CONFIGPAGE_VIDEO_PLAYERCONTROL_CHKBOX_MAX (sizeof(config_video_player_control_checkboxes) / sizeof(struct configpage_chkbox_s))

static struct configpage_chkbox_s config_video_hdr_control_checkboxes[] = {
 {"HDR to SDR conversion", &mmc_config.video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_TO_SDR_CONV,   "Enable HDR to SDR conversion (tonemap)\n(works only with D3D11 output)" },
 {"HDR dynamic luminance", &mmc_config.video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_DYN_LUMINANCE, "Enable dynamic (custom) HDR luminance runtime update\n(option has effect only if video stream contains dynamic HDR infos)" },
 {"HDR dynamic colors",    &mmc_config.video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_DYN_COLORS,    "Enable dynamic (custom) color primaries runtime update\n(option has effect only if video stream contains dynamic HDR infos)" },
 {"HDR delayed luminance", &mmc_config.video_hdr_control, DISPQT_CONFIG_VIDEO_HDR_DYN_LUM_SLOW,  "Use delayed (average) dynamic luminance\n(to avoid flashing)" },
};
#define DISPQT_CONFIGPAGE_VIDEO_HDRCONTROL_CHKBOX_MAX (sizeof(config_video_hdr_control_checkboxes) / sizeof(struct configpage_chkbox_s))

static struct configpage_editline_s config_video_config_custom_editlines[] = {
 {"Aspect ratio", &mmc_config.video_ar_custom[0],   sizeof(mmc_config.video_ar_custom), "Custom video aspect ratio in N:D format\n(integer numbers only)"},
 {"Source crop",  &mmc_config.video_crop_custom[0], sizeof(mmc_config.video_crop_custom), "Custom source video crop ratio in N:D format\n(integer numbers only)"},
 {"Zoom ratio",   &mmc_config.video_zoom_custom[0], sizeof(mmc_config.video_zoom_custom), "Custom video zoom ratio in N:D format\n(integer numbers only)"}
};
#define DISPQT_CONFIGPAGE_VIDEO_CONFIG_CUSTOM_EDITLINE_MAX (sizeof(config_video_config_custom_editlines) / sizeof(struct configpage_editline_s))
#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

//----------------------------------------------------------------------------------------------------------------------------------------------------------
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG

static struct configpage_chkbox_s config_video_dvbepg_videowall_group[] = {
 {"Hide main window", &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_HIDEMAINWIN, "Hide main window over fullscreen videowall surface\n(don't wake up by move or click event, only by wheel event)\n(has effect only if 'Auto hide' is enabled)" },
 {"Keyframes only",   &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_KEYFRAMES_ONLY, "Decode only keyframes of non-primary videowall windows\n(some formats may have no KeyFrames,\ncausing decoding delay initially)" },
 {"Deinterlace",      &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_FILTER_WINDOW, "Add base video filtering (deinterlace)\nto seek-preview and all videowall windows\n(requires file re-open to change)" },
 {"Restore",          &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOWALL_RESTORE, "Restore (open in) video wall mode\nafter channel re-open or program restart" },
 {"Video surface EPG",&mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_VIDEOSURFACE_SHOW_EPGDIALOG, "Show EPG list over fullscreen video surface\n(single DVB video and videwall too)" }
};
#define DISPQT_CONFIGPAGE_VIDEO_DVBEPG_VIDEOWALL_CHKBOX_MAX (sizeof(config_video_dvbepg_videowall_group) / sizeof(struct configpage_chkbox_s))

static unsigned int dvbepg_sort_control;
static const char *dvbepg_sort_values[] = {"0", "1"};
static const char *dvbepg_sort_description[] = {"Program ID", "Frequency"};

static struct configpage_comboboxline_s video_dvbepg_programlistcontrol_group[] = {
 {"Sort channels by", (DISPQT_CONFIG_COMBOTYPE_COMBOBOX | ARG_NUM), &dvbepg_sort_control, (sizeof(dvbepg_sort_values) / sizeof(char *)), &dvbepg_sort_values[0], &dvbepg_sort_description[0], "Program list sorting mode\nin EPG window and DTV directory\n(needs DTV directory refresh)"},
 {"Ignore encrypted",  DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_FILTER_ENCRYPTED, NULL, NULL, "Ignore (don't show) encrypted programs\n(needs dtv direcory refresh)" },
 {"Ignore non-media",  DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_FILTER_NONMEDIA, NULL, NULL, "Ignore (don't show) invalid (non-media) program entries\nwhich has no name or valid channel id\n(needs DTV directory refresh)" },
 {"ToolTip items",     DISPQT_CONFIG_COMBOTYPE_CHKBOX, &mpxplay_config_dvbepg_control_flags, MPXPLAY_CONFIG_DVBEPGCTRL_EPGWINDOW_ITEMS_TOOLTIP, NULL, NULL, "Show EPG item titles in ToolTips over EPG window" },
 {"Past events hours", DISPQT_CONFIG_COMBOTYPE_SPINBOX, &mpxplay_config_dvbepg_pastevents_hours, DISPQT_CONFIG_SPINBOX_VALPACK_MINMAX(DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MIN, DISPQT_CONFIG_DVBEPG_PASTEVENTS_HOURS_MAX), NULL, NULL, "Show past events in the EPG dialog\ntill the configured hours"}
};
#define DISPQT_CONFIGPAGE_VIDEO_DVBEPG_PROGRAMLISTCONTROL_COMBOX_MAX (sizeof(video_dvbepg_programlistcontrol_group) / sizeof(struct configpage_comboboxline_s))
#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

DispQtConfigpageVideo::DispQtConfigpageVideo(MainWindow *main_window, QWidget *parent) : DispQtDialogElemTabWidget(main_window, parent)
{
	this->main_window = main_window;

	QWidget *video_tab_videocontrol_mainwidget = new QWidget(this);
	QHBoxLayout *video_tab_videocontrol_mainlayout = new QHBoxLayout;

	QWidget *video_tab_config_mainwidget = new QWidget(this);
	QVBoxLayout *video_tab_config_mainlayout = new QVBoxLayout;

#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	QWidget *video_tab_dvbepg_mainwidget = new QWidget(this);
	QVBoxLayout *video_dvbepg_config_mainlayout = new QVBoxLayout;
#endif

	//---------------------------------------------------------------------------------------------------
	this->groupchkboxes_videocontrol_mainwindowcontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Mainwindow control", config_video_control_mainwindow_checkboxes, DISPQT_CONFIGPAGE_VIDEO_CONTROL_MW_CHKBOX_MAX);
	this->groupchkboxes_videocontrol_fullscreencontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Fullscreen control", config_video_control_fullscreen_checkboxes, DISPQT_CONFIGPAGE_VIDEO_CONTROL_FS_CHKBOX_MAX);
	this->combobox_videocontrol_bottomcontrolbar = new DispQtConfigpageComboboxGroupbox(main_window, this, "Control bar", &config_video_control_bottomcontrolbar_group[0], DISPQT_CONFIGPAGE_VIDEO_CONTROL_BOTTOMCONTROLBAR_COMBOX_MAX);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->combobox_videocontrol_seekpreview = new DispQtConfigpageComboboxGroupbox(main_window, this, "Seek preview", &config_video_control_seekpreview_group[0], DISPQT_CONFIGPAGE_VIDEO_CONTROL_SEEKPREVIEW_COMBOX_MAX);
	this->combobox_videocontrol_videostreamlanguage = new DispQtConfigpageComboboxGroupbox(main_window, this, "Video stream language", &config_stream_languages_combobox_group[MPXPLAY_STREAMTYPEINDEX_VIDEO][0], (sizeof(config_stream_languages_combobox_group[MPXPLAY_STREAMTYPEINDEX_VIDEO]) / sizeof(struct configpage_comboboxline_s)));
#endif

	QWidget *videoctrl_surfacectrl_widget = new QWidget(this);
	QVBoxLayout *videoctrl_surfacectrl_layout = new QVBoxLayout;
	QWidget *videoctrl_controlbar_widget = new QWidget(this);
	QVBoxLayout *videoctrl_controlbar_layout = new QVBoxLayout;

	videoctrl_surfacectrl_layout->setMargin(0);
	videoctrl_surfacectrl_layout->addWidget(this->groupchkboxes_videocontrol_mainwindowcontrol);
	videoctrl_surfacectrl_layout->addWidget(this->groupchkboxes_videocontrol_fullscreencontrol);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	videoctrl_surfacectrl_layout->addWidget(this->combobox_videocontrol_videostreamlanguage);
#endif
	videoctrl_surfacectrl_layout->addStretch(1);
	videoctrl_surfacectrl_widget->setLayout(videoctrl_surfacectrl_layout);

	videoctrl_controlbar_layout->setMargin(0);
	videoctrl_controlbar_layout->addWidget(this->combobox_videocontrol_bottomcontrolbar);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	videoctrl_controlbar_layout->addWidget(this->combobox_videocontrol_seekpreview);
#endif
	videoctrl_controlbar_layout->addStretch(1);
	videoctrl_controlbar_widget->setLayout(videoctrl_controlbar_layout);

	video_tab_videocontrol_mainlayout->setMargin(0);
	video_tab_videocontrol_mainlayout->addWidget(videoctrl_surfacectrl_widget);
	video_tab_videocontrol_mainlayout->addWidget(videoctrl_controlbar_widget);
	video_tab_videocontrol_mainlayout->addStretch(1);
	video_tab_videocontrol_mainwidget->setLayout(video_tab_videocontrol_mainlayout);

	//---------------------------------------------------------------------------------------------------
	this->groupcomboxes_videoconfig_videoconfig = new DispQtConfigpageComboboxGroupbox(main_window, this, "Video config", &config_video_decoder_config_combobox_group[0], DISPQT_CONFIGPAGE_VIDEO_DECODER_SELECT_COMBOX_MAX);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->groupchkboxes_videoconfig_videoplayercontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Video decoder control", config_video_player_control_checkboxes, DISPQT_CONFIGPAGE_VIDEO_PLAYERCONTROL_CHKBOX_MAX);
	this->groupcomboxes_videoconfig_hdrcontrol = new DispQtConfigpageCheckboxGroupbox(main_window, this, "HDR control", config_video_hdr_control_checkboxes, DISPQT_CONFIGPAGE_VIDEO_HDRCONTROL_CHKBOX_MAX);

	this->groupeditlines_videoconfig_custompicturesettings = new DispQtConfigpageEditlineGroupbox(main_window, this, "Custom video settings", config_video_config_custom_editlines, DISPQT_CONFIGPAGE_VIDEO_CONFIG_CUSTOM_EDITLINE_MAX);
#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

	video_tab_config_mainlayout->setContentsMargins(0,5,0,0);
	video_tab_config_mainlayout->addWidget(this->groupcomboxes_videoconfig_videoconfig);
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	QWidget *videoDecCtrlHDRCustomCommonWidget = new QWidget(this);
	QHBoxLayout *videoDecCtrlHDRCustomCommonLayout = new QHBoxLayout;

	QWidget *playerHDRControlCustomWidget = new QWidget(videoDecCtrlHDRCustomCommonWidget);
	QVBoxLayout *playerHDRControlCustomLayout = new QVBoxLayout;
	playerHDRControlCustomLayout->setMargin(0);
	playerHDRControlCustomLayout->addWidget(this->groupcomboxes_videoconfig_hdrcontrol);
	playerHDRControlCustomLayout->addWidget(this->groupeditlines_videoconfig_custompicturesettings);
	playerHDRControlCustomWidget->setLayout(playerHDRControlCustomLayout);

	videoDecCtrlHDRCustomCommonLayout->setMargin(0);
	videoDecCtrlHDRCustomCommonLayout->addWidget(this->groupchkboxes_videoconfig_videoplayercontrol);
	videoDecCtrlHDRCustomCommonLayout->addWidget(playerHDRControlCustomWidget);
	videoDecCtrlHDRCustomCommonWidget->setLayout(videoDecCtrlHDRCustomCommonLayout);

	video_tab_config_mainlayout->addWidget(videoDecCtrlHDRCustomCommonWidget);
#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG
	video_tab_config_mainlayout->addStretch(1);
	video_tab_config_mainwidget->setLayout(video_tab_config_mainlayout);

	//---------------------------------------------------------------------------------------------------
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	dvbepg_sort_control = mpxplay_config_dvbepg_control_flags & MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORTTYPE_MASK;

	this->groupchkboxes_dvbepg_videowall = new DispQtConfigpageCheckboxGroupbox(main_window, this, "Video wall", &config_video_dvbepg_videowall_group[0], DISPQT_CONFIGPAGE_VIDEO_DVBEPG_VIDEOWALL_CHKBOX_MAX);
	this->groupcomboxes_dvbepg_programlistcontrol = new DispQtConfigpageComboboxGroupbox(main_window, this, "Program list control", &video_dvbepg_programlistcontrol_group[0], DISPQT_CONFIGPAGE_VIDEO_DVBEPG_PROGRAMLISTCONTROL_COMBOX_MAX);

	QWidget *video_dvbepg_videowalllistcontrol_widget = new QWidget(this);
	QHBoxLayout *video_dvbepg_videowalllistcontrol_layout = new QHBoxLayout;
	video_dvbepg_videowalllistcontrol_layout->setMargin(0);
	video_dvbepg_videowalllistcontrol_layout->addWidget(this->groupchkboxes_dvbepg_videowall);
	video_dvbepg_videowalllistcontrol_layout->addWidget(this->groupcomboxes_dvbepg_programlistcontrol);
	video_dvbepg_videowalllistcontrol_widget->setLayout(video_dvbepg_videowalllistcontrol_layout);

	video_dvbepg_config_mainlayout->setMargin(0);
	video_dvbepg_config_mainlayout->addWidget(video_dvbepg_videowalllistcontrol_widget);
	video_dvbepg_config_mainlayout->addStretch(1);
	video_tab_dvbepg_mainwidget->setLayout(video_dvbepg_config_mainlayout);
#endif // MPXPLAY_LINK_ORIGINAL_FFMPEG

	//---------------------------------------------------------------------------------------------------
	this->addTab(video_tab_videocontrol_mainwidget, "Video control");
	this->addTab(video_tab_config_mainwidget, "Video config");
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->addTab(video_tab_dvbepg_mainwidget, " DVB / EPG ");
#endif
	this->setDocumentMode(true);
}

void DispQtConfigpageVideo::configpage_video_applyall(void)
{
	unsigned long v_ar_type = mmc_config.video_windowed_ar_type;
	unsigned long v_render_type = mmc_config.video_renderer_type;
	unsigned long v_player_type = mmc_config.selected_videoplayer_type;
	unsigned long v_hdr_control = mmc_config.video_hdr_control;
	unsigned long v_autocrop_limit = mmc_config.video_autocrop_limit;
//	unsigned long video_control = mmc_config.video_control;
	this->groupchkboxes_videocontrol_mainwindowcontrol->checkboxgroup_applyall();
	this->groupchkboxes_videocontrol_fullscreencontrol->checkboxgroup_applyall();
	this->combobox_videocontrol_bottomcontrolbar->comboboxgroupbox_applyall();
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->combobox_videocontrol_seekpreview->comboboxgroupbox_applyall();
	this->combobox_videocontrol_videostreamlanguage->comboboxgroupbox_applyall();
#endif

	//---------------------------------------------------------------------------------------------------
	this->groupcomboxes_videoconfig_videoconfig->comboboxgroupbox_applyall();
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->groupchkboxes_videoconfig_videoplayercontrol->checkboxgroup_applyall();
	this->groupcomboxes_videoconfig_hdrcontrol->checkboxgroup_applyall();
	this->groupeditlines_videoconfig_custompicturesettings->editlinegroupbox_applyall();  // TODO: changes have no immediate effect (if custom is already selected)
#endif
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
//	if((video_control & DISPQT_CONFIG_VIDEOCTRL_VIDEOWALL_FILTER_WINDOW) != (mmc_config.video_control & DISPQT_CONFIG_VIDEOCTRL_VIDEOWALL_FILTER_WINDOW)) // TODO: this can be wrong if file recording is running
//		mvps.adone = ADONE_REOPEN;
	if((mmc_config.selected_videoplayer_type != v_player_type) && (v_player_type == MPXPLAY_VIDEOPLAYERTYPE_FFMPEG))
		funcbit_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD);
	if(this->main_window->qt_video_player) {
		if((mmc_config.video_windowed_ar_type != v_ar_type) || (mmc_config.video_autocrop_limit != v_autocrop_limit))
			emit this->main_window->qt_video_player->signal_video_surface_resize_refresh();
		else if(mmc_config.video_hdr_control != v_hdr_control)
			emit this->main_window->qt_video_player->signal_video_surface_filtered_eq_refresh();
		if(mmc_config.video_renderer_type != v_render_type)
			funcbit_enable(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_GUIREBUILD);
	}
#endif

	//---------------------------------------------------------------------------------------------------
#ifdef MPXPLAY_LINK_ORIGINAL_FFMPEG
	this->groupchkboxes_dvbepg_videowall->checkboxgroup_applyall();     // TODO: changes have no immediate effect
	this->groupcomboxes_dvbepg_programlistcontrol->comboboxgroupbox_applyall();
	funcbit_copy(mpxplay_config_dvbepg_control_flags, dvbepg_sort_control, MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORTTYPE_MASK);
#endif
}
