//**************************************************************************
//*                     This file is part of the                           *
//*                    Mpxplay-MMC - video player.                         *
//*                The source code of Mpxplay-MMC is                       *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: digital TV (DVB) frequency scanning

//#define MPXPLAY_USE_DEBUGF
#define MPXPLAY_DEBUG_OUTPUT     stdout

#include "mpxplay.h"

#if defined(MPXPLAY_LINK_INFILE_FF_MPEG) && defined(MPXPLAY_WIN32)

#include "dtv_drv.h"
#include "control/cntfuncs.h"
#include "display/display.h"

#define DRVDTV_SCAN_PROCESS_INTERVAL_MSEC   200
#define DRVDTV_SCAN_TUNE_TIMEOUT_MSEC      5000
#define DRVDTV_SCAN_ANALYZE_TIMEOUT_MSEC  15000
#define DRVDTV_SCAN_EPGLOAD_MIN_TIMEOUT_MSEC   4000
#define DRVDTV_SCAN_EPGLOAD_MAX_TIMEOUT_MSEC  60000 // FIXME: if the events are wrong, then the epg-load of one freq runs till this timeout

extern unsigned long mpxplay_signal_events, mpxplay_config_dvbepg_control_flags;
extern struct mainvars mvps;

#ifdef MPXPLAY_GUI_QT
extern mpxp_bool_t mpxplay_dispqt_epgdialog_send_scaninfos(void *protocol_data); // TODO: this should be a callback config option
#endif

static void drvdtv_scan_send_scaninfos(dvb_device_t *d, mpxp_bool_t finish_it)
{
#ifdef MPXPLAY_GUI_QT
	struct mpxplay_dvb_epg_protocol_data_s *prot_data = NULL;

	if(!mpxplay_dispqt_epgdialog_send_scaninfos((void *)prot_data)) // dialog is closed (scan has started from dtv-drive)
		return;

	if(finish_it) // send a full protocol data with all freq infos
	{
		mpxplay_dtvdrv_database_frequencies_sort_by_type(d->protocol_number_id, (mpxplay_config_dvbepg_control_flags & MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORTTYPE_MASK));
		prot_data = mpxplay_dtvdrv_database_dup_protocol_info(d->protocol_number_id, mpxplay_config_dvbepg_control_flags);
		if(!prot_data)
			return;
		funcbit_enable(prot_data->prot_flags, (MPXPLAY_DVBEPG_PROTOCOL_FLAG_FULLDATA | MPXPLAY_DVBEPG_PROTOCOL_FLAG_SCANFINISH));
	}
	else
	{
		prot_data = mpxplay_drvdtv_database_protocol_frequency_info(d->protocol_number_id, d->scan_infos.scan_freq_curr_khz * 1000, mpxplay_config_dvbepg_control_flags);
		if(!prot_data)
			return;
	}

	if(!mpxplay_dispqt_epgdialog_send_scaninfos((void *)prot_data))
		mpxplay_dtvdrv_database_clear_protocols(prot_data); // dialog is closed (should not happen here)

#else
	if(finish_it)
		mpxplay_dtvdrv_database_frequencies_sort_by_type(d->protocol_number_id, (mpxplay_config_dvbepg_control_flags & MPXPLAY_CONFIG_DVBEPGCTRL_CHANNELS_SORTTYPE_MASK));
#endif
}

static int drvdtv_start_tune(dvb_device_t *d)
{
	d->scan_infos.status_starttime_ms = pds_gettimem();

	return dtv_bda_device_start(d);
}

static void drvdtv_stop_tune(dvb_device_t *d)
{
	dtv_bda_device_stop(d);
}

static void drvdtv_scan_process(void *dtv_device)
{
	dvb_device_t *d = (dvb_device_t *)dtv_device;
	struct dtv_scan_info_s *scan_infos;
	char sout[256];

	if(!d || !d->data_buffer || !d->dtv_resource || funcbit_test(d->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE))
		return;
	if(funcbit_test(mpxplay_signal_events, MPXPLAY_SIGNALTYPE_DISKDRIVTERM))
		goto err_out_scan;

	scan_infos = &d->scan_infos;

	switch(scan_infos->status)
	{
		case DTVBDA_SCAN_STATUS_WAITINGSIGNAL:
			if(dtv_bda_device_is_signal(d))
			{
				scan_infos->found_channels++;
				scan_infos->status = DTVBDA_SCAN_STATUS_ANALYZE;
				scan_infos->status_starttime_ms = pds_gettimem();
				break;
			}
			else if(pds_gettimem() < (scan_infos->status_starttime_ms + DRVDTV_SCAN_TUNE_TIMEOUT_MSEC))
			{
				break;
			}
			// @suppress("No break at end of case")
		case DTVBDA_SCAN_STATUS_TUNING:
			drvdtv_stop_tune(d);
			drvdtv_scan_send_scaninfos(d, FALSE);
			if(d->scan_do_epgload && scan_infos->scan_epgload_freq_chain) // EPG load only
			{
				scan_infos->scan_epgload_freq_chain = scan_infos->scan_epgload_freq_chain->next_frequency;
				if(!scan_infos->scan_epgload_freq_chain)
				{
					scan_infos->status = DTVBDA_SCAN_STATUS_FINISH;
					return;
				}
				scan_infos->scan_freq_curr_khz = scan_infos->scan_epgload_freq_chain->frequency / 1000;
			}
			else // scan freq
			{
				scan_infos->scan_freq_curr_khz += d->scan_freq_bandwidth;
				if(scan_infos->scan_freq_curr_khz > d->scan_freq_end)
				{
					scan_infos->status = DTVBDA_SCAN_STATUS_FINISH;
					return;
				}
			}
			d->frequency = scan_infos->scan_freq_curr_khz * 1000;
			if(drvdtv_start_tune(d) != MPXPLAY_ERROR_DISKDRIV_OK)
				goto err_out_scan;
			scan_infos->status = DTVBDA_SCAN_STATUS_WAITINGSIGNAL;
			break;
		case DTVBDA_SCAN_STATUS_ANALYZE:
			{
				mpxplay_dvb_epg_frequency_data_s *freq_data = mpxplay_dtvdrv_database_frequency_search(d->protocol_number_id, scan_infos->scan_freq_curr_khz * 1000);
				mpxp_bool_t success = FALSE;
				if(d->scan_do_epgload)
				{
					if(pds_gettimem() > (scan_infos->status_starttime_ms + DRVDTV_SCAN_EPGLOAD_MIN_TIMEOUT_MSEC))
						success = mpxplay_drvdtv_database_frequency_programs_events_check(freq_data);
				}
				else
				{
					success = mpxplay_drvdtv_database_frequency_programs_check(freq_data);
				}
				if(success)
				{
					freq_data->freq_signal_quality = dtv_bda_device_get_signal_strength(d);
					scan_infos->status = DTVBDA_SCAN_STATUS_TUNING; // skip to next freq
				}
				else
				{
					int timeout = ((d->scan_do_epgload)? DRVDTV_SCAN_EPGLOAD_MAX_TIMEOUT_MSEC : DRVDTV_SCAN_ANALYZE_TIMEOUT_MSEC);
					if(pds_gettimem() > (scan_infos->status_starttime_ms + timeout))
					{
						scan_infos->status = DTVBDA_SCAN_STATUS_TUNING;
						mpxplay_debugf(MPXPLAY_DEBUG_OUTPUT, "ANALYZE timeout (%d) at %dMhz, signal:%d frqd:%8.8X",
								timeout, (scan_infos->scan_freq_curr_khz / 1000), dtv_bda_device_get_signal_strength(d),
								(mpxp_ptrsize_t)freq_data);
					}
				}
			}
			break;
		case DTVBDA_SCAN_STATUS_FINISH:
			goto err_out_scan;
	}

	{
		int scan_range = d->scan_freq_end - d->scan_freq_begin;
		if(scan_range <= 0)
			scan_range = 1000;
		snprintf(sout, sizeof(sout), "DTV: scan %d Mhz (%d%%), found channels: %d", scan_infos->scan_freq_curr_khz / 1000, 100 * (scan_infos->scan_freq_curr_khz - d->scan_freq_begin) / scan_range, scan_infos->found_channels);
		display_timed_message(sout);
	}
	return;

err_out_scan:
	mpxplay_drvdtv_scan_stop(d);
}

int mpxplay_drvdtv_scan_start(dvb_device_t *d)
{
	int retval = MPXPLAY_ERROR_DISKDRIV_ERROR;
	struct dtv_scan_info_s *scan_infos;
	char sout[256];

	if(!d || (d->scan_freq_begin < 10000) || (d->scan_freq_begin > 10000000) || (d->scan_freq_end < 10000) || (d->scan_freq_end > 10000000) || (d->scan_freq_bandwidth <= 1000))
	{
		snprintf(sout, sizeof(sout), "DTV: scan begin failed with invalid arguments!");
		display_timed_message(sout);
		return retval;
	}

	mpxplay_drvdtv_database_load_from_file();

	funcbit_enable(d->flags, MPXPLAY_DRVDTV_FLAG_OPENMODE_SCAN);

	scan_infos = &d->scan_infos;
	scan_infos->scan_freq_curr_khz = d->scan_freq_begin;
	scan_infos->found_channels = 0;
	scan_infos->status = DTVBDA_SCAN_STATUS_TUNING;

	if(d->scan_do_epgload)
	{
		struct mpxplay_dvb_epg_protocol_data_s *prot_data = mpxplay_dtvdrv_database_protocol_add_or_search(d->protocol_number_id, TRUE);
		if(prot_data && prot_data->frequency_data_chain && funcbit_test(prot_data->frequency_data_chain->freq_flags, MPXPLAY_DVBEPG_FREQ_FLAG_LOADED_SDT))
		{
			scan_infos->scan_epgload_freq_chain = prot_data->frequency_data_chain;
			scan_infos->scan_freq_curr_khz = prot_data->frequency_data_chain->frequency / 1000;
		}
	}

	d->frequency = scan_infos->scan_freq_curr_khz * 1000;

	snprintf(sout, sizeof(sout), "DTV: scan begin at %d Mhz", scan_infos->scan_freq_curr_khz / 1000);
	display_timed_message(sout);

	retval = dtv_bda_resource_assign(d);
	if(retval != MPXPLAY_ERROR_DISKDRIV_OK)
		return retval;

	retval = drvdtv_start_tune(d);
	if(retval != MPXPLAY_ERROR_DISKDRIV_OK)
		return retval;

	scan_infos->status = DTVBDA_SCAN_STATUS_WAITINGSIGNAL;

	mpxplay_timer_addfunc(&drvdtv_scan_process, (void *)d, MPXPLAY_TIMERTYPE_REPEAT, mpxplay_timer_msecs_to_counternum(DRVDTV_SCAN_PROCESS_INTERVAL_MSEC));

	return retval;
}

void mpxplay_drvdtv_scan_stop(dvb_device_t *d)
{
	if(d && d->scan_freq_begin)
	{
		funcbit_enable(d->flags, MPXPLAY_DRVDTV_FLAG_TERMINATE);
		mpxplay_timer_deletefunc(&drvdtv_scan_process, (void *)d);
		if(dtv_bda_resource_remove(d) == MPXPLAY_ERROR_DISKDRIV_OK)        // resource is not deleted by a manual program start
			if(funcbit_test(d->flags, MPXPLAY_DRVDTV_FLAG_PLAYTERMBYSCAN)) // scan terminated a program play
				mvps.adone = ADONE_REOPEN;                                 // reopen the current program to play // FIXME: it reopens non-DVB files too
		drvdtv_scan_send_scaninfos(d, TRUE);
		display_clear_timed_message();
	}
}

#endif // defined(MPXPLAY_LINK_INFILE_FF_MPEG) && defined(MPXPLAY_WIN32)
