//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2019 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFMPEG packet and frame queue handling (used outside of in_ffmpg too)

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

int mpxplay_ffmpgdec_packetqueue_get(mpxp_ptrsize_t demuxer_data, mpxplay_packetqueue_t *packet_queue, mpxplay_packetlist_t **pktlist_elem_dest, int wait_ms)
{
	int retcode = -1, lock_result;

	if(!packet_queue || (packet_queue->nb_packets <= 0))
		return retcode;

	lock_result = PDS_THREADS_MUTEX_LOCK(&packet_queue->mutexhnd_queue, INFFMPG_MUTEX_QUEUEGET_TIMEOUT);
	if((lock_result != MPXPLAY_ERROR_INFILE_OK) && (packet_queue->streamtype_index != MPXPLAY_STREAMTYPEINDEX_TEMP))
		return retcode;

	do{
		mpxplay_packetlist_t *first_pktlist_elem = packet_queue->first_pkt;
		if (first_pktlist_elem){
			unsigned int pop_elem = 1;
			switch(first_pktlist_elem->packet_type)
			{
				case MPXPLAY_PACKETTYPE_AVCODECCTX:
					if((!packet_queue->last_pkt || (packet_queue->last_pkt == first_pktlist_elem)) && funcbit_test(first_pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_RESET))
					{
						pop_elem = 1;
					}
					else
					{
						if(packet_queue->last_pkt)
							first_pktlist_elem = packet_queue->last_pkt; // we always give back the last pushed codec_ctx
						pop_elem = 0;  // we always keep the top elem(s) of codec queue
					}
					retcode = MPXPLAY_ERROR_INFILE_OK;
					if(packet_queue->streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO)
					{
						mpxplay_debugf(MPXPLAY_DEBUG_QUEUEGET,"packetqueue_get AVCODECCTX fpe:%8.8X", (mpxp_ptrsize_t)first_pktlist_elem);
					}
					break;
				case MPXPLAY_PACKETTYPE_AVPACKET:
					{
						AVPacket *avpkt = (AVPacket *)(first_pktlist_elem->frame_pkt);
						struct ffmpg_demuxer_data_s *ffmpi;
						if(packet_queue->streamtype_index >= MPXPLAY_STREAMTYPEINDEX_MAX)
							break;
						retcode = MPXPLAY_ERROR_INFILE_OK;
						if(!avpkt)
							break;
						if((packet_queue->streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) || (packet_queue->streamtype_index >= MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW))
							break;
						ffmpi = (struct ffmpg_demuxer_data_s *)demuxer_data;
						if(!ffmpi)
							break;
						if(first_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)
						{
							ffmpi->stream_last_pts[packet_queue->streamtype_index] = INFFMPG_INVALID_PTS;
							ffmpi->external_file_infos[packet_queue->streamtype_index].avtimestamp_us = INFFMPG_INVALID_PTS;
							ffmpi->external_file_infos[packet_queue->streamtype_index].avclocktime_us = INFFMPG_INVALID_PTS;
						}
						first_pktlist_elem->timestamp_us = in_ffmpgdec_get_avpacket_timestamp(ffmpi, packet_queue->streamtype_index, avpkt);
						if(first_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)
						{
							mpxplay_debugf(MPXPLAY_DEBUG_QUEUEGET,"packetqueue_get AVPACKET %s i:%d nb:%d us:%llu diff:%lld dts:%llu pts:%llu",
								((first_pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)? "CLEAR" : ""),
								packet_queue->streamtype_index, packet_queue->nb_packets, first_pktlist_elem->timestamp_us,
								(first_pktlist_elem->timestamp_us - ffmpi->stream_last_pts[packet_queue->streamtype_index]),
								avpkt->dts, avpkt->pts);
						}
						//ffmpi->stream_last_pts[packet_queue->streamtype_index] = first_pktlist_elem->timestamp_us; // FIXME: ??? (audio and video use avframe pts, subtitle doesn't need this)
					}
					break;
				case MPXPLAY_PACKETTYPE_AVFRAME_AUDIO:
				case MPXPLAY_PACKETTYPE_AVFRAME_VIDEO:
					{
						AVFrame *avframe = (AVFrame *)(first_pktlist_elem->frame_pkt);
						if(packet_queue->streamtype_index >= MPXPLAY_STREAMTYPEINDEX_NUM)
							break;
						retcode = MPXPLAY_ERROR_INFILE_OK;
						if(!avframe)
							break;
						first_pktlist_elem->timestamp_us = in_ffmpgdec_select_avframe_pts(avframe);
					}
					break;
			}
			if(pop_elem){
				packet_queue->first_pkt = first_pktlist_elem->next;
				if(!packet_queue->first_pkt)
					packet_queue->last_pkt = NULL;
				packet_queue->nb_packets--;
			}
			*pktlist_elem_dest = first_pktlist_elem;
			break;
		}else{
			packet_queue->nb_packets = 0;
		}
		if(wait_ms <= 0) {
			retcode = -1;
			break;
		}
	    //pds_threads_cond_timedwait(&packet_queue->condhnd_queue, &packet_queue->mutexhnd_queue, wait_ms);
		wait_ms = 0;
	}while(1);

	if(lock_result == MPXPLAY_ERROR_INFILE_OK)
	{
		PDS_THREADS_MUTEX_UNLOCK(&packet_queue->mutexhnd_queue);
	}

	return retcode;
}

int mpxplay_ffmpgdec_packetqueue_put(mpxplay_packetqueue_t *packet_queue, unsigned int packet_type, unsigned int packet_flags,
		void *codec_ctx, unsigned int codecctx_counter, void *avstream, void *pkt_new)
{
	mpxplay_packetlist_t *pktlist_new;
	int lock_result;

	if(!packet_queue || (packet_type >= MPXPLAY_PACKETTYPE_MAX))
		return -1;

	pktlist_new = mpxplay_ffmpgdec_queuelistelem_prepare(packet_type, packet_flags, codec_ctx, codecctx_counter, avstream, pkt_new);
	if(!pktlist_new)
		return -1;

	lock_result = PDS_THREADS_MUTEX_LOCK(&packet_queue->mutexhnd_queue, INFFMPG_MUTEX_TIMEOUT);

	if (packet_queue->last_pkt)
		packet_queue->last_pkt->next = pktlist_new;
	else
		packet_queue->first_pkt = pktlist_new;
	packet_queue->last_pkt = pktlist_new;
	packet_queue->nb_packets++;

	//pds_threads_cond_signal(&packet_queue->condhnd_queue);

	if(lock_result == MPXPLAY_ERROR_INFILE_OK)
	{
		PDS_THREADS_MUTEX_UNLOCK(&packet_queue->mutexhnd_queue);
	}

	if(packet_queue->streamtype_index == MPXPLAY_STREAMTYPEINDEX_AUDIO){
		mpxplay_debugf(MPXPLAY_DEBUG_QUEUEPUT,"mpxplay_ffmpgdec_packetqueue_put END i:%d t:%d nb:%d",
				packet_queue->streamtype_index, packet_type, packet_queue->nb_packets);
	}

	return 0;
}

mpxplay_packetlist_t *mpxplay_ffmpgdec_queuelistelem_prepare(unsigned int packet_type, unsigned int packet_flags,
		void *codec_ctx, unsigned int codecctx_counter, void *avstream, void *pkt_new)
{
	mpxplay_packetlist_t *pktlist_new;

	if(packet_type >= MPXPLAY_PACKETTYPE_MAX)
		return NULL;

	pktlist_new = av_mallocz(sizeof(mpxplay_packetlist_t));
	if(!pktlist_new)
		return NULL;

	pktlist_new->packet_type = packet_type;
	pktlist_new->flags = packet_flags;
	pktlist_new->codec_ctx = codec_ctx;
	pktlist_new->avstream = avstream;
	pktlist_new->codecctx_counter = codecctx_counter;

	switch(packet_type)
	{
		case MPXPLAY_PACKETTYPE_AVCODECCTX:
			break;
		case MPXPLAY_PACKETTYPE_AVPACKET:
			if(pkt_new){
				AVPacket *pkt_clone;
				if(((AVPacket *)pkt_new)->size <= 0)
					goto err_out_pkt_put;
				pkt_clone = av_packet_clone((AVPacket *)pkt_new);
				if(!pkt_clone)
					goto err_out_pkt_put;
				pktlist_new->frame_pkt = (void *)pkt_clone;
			} // else we put NULL avpackets too (for clear buffer flag)
			break;
		case MPXPLAY_PACKETTYPE_AVFRAME_AUDIO:
		case MPXPLAY_PACKETTYPE_AVFRAME_VIDEO:
			if(pkt_new){
				pktlist_new->timestamp_us = in_ffmpgdec_select_avframe_pts((AVFrame *)pkt_new);
				pktlist_new->frame_pkt = pkt_new;
			}
			break;
	}

	return pktlist_new;

err_out_pkt_put:
	av_free(pktlist_new);
	return NULL;
}

void mpxplay_ffmpgdec_queuelistelem_clear(mpxplay_packetlist_t **list_elem_p)
{
	mpxplay_packetlist_t *list_elem;
	AVCodecContext *avctx;
	void *framepkt;

	if(!list_elem_p)
		return;
	list_elem = *list_elem_p;
	if(!list_elem)
		return;
	*list_elem_p = NULL;

	list_elem->avstream = NULL;
	list_elem->codecctx_counter = 0;
	avctx = (AVCodecContext *)(list_elem->codec_ctx);
	list_elem->codec_ctx = NULL;
	framepkt = list_elem->frame_pkt;
	list_elem->frame_pkt = NULL;

	switch(list_elem->packet_type)
	{
		case MPXPLAY_PACKETTYPE_AVCODECCTX:
			if(mpxplay_ffmpstrm_is_avctx_valid(avctx, MPXPLAY_STREAMTYPEINDEX_VIDEO))
			{
				in_ffmpgdec_hwdec_deinit(avctx);
				mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "mpxplay_ffmpgdec_queuelistelem_clear avcodec_free_context BEGIN ctx:%8.8X tht:%8.8X", (mpxp_ptrsize_t)avctx, avctx->active_thread_type);
				avcodec_free_context(&avctx);
				//mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "mpxplay_ffmpgdec_queuelistelem_clear avcodec_free_context END");
			}
			break;
		case MPXPLAY_PACKETTYPE_AVPACKET:
			if(framepkt)
				av_packet_free((AVPacket **)&framepkt);
			break;
		case MPXPLAY_PACKETTYPE_AVFRAME_AUDIO:
		case MPXPLAY_PACKETTYPE_AVFRAME_VIDEO:
			if(framepkt)
				av_frame_free((AVFrame **)&framepkt);
			break;
	}
	pds_memset(list_elem, 0, sizeof(*list_elem));
	av_free(list_elem);
}

unsigned int mpxplay_ffmpgdec_packetqueue_clear(mpxplay_packetqueue_t *packet_queue, int flag_test)
{
	mpxplay_packetlist_t *list_elem;
	unsigned int found_next, found_flag = 0, retry = INFFMPG_QUEUE_PACKETNUM_MAX, sc;
	int lock_result;

	if(!packet_queue)
		return found_flag;

	lock_result = PDS_THREADS_MUTEX_LOCK(&packet_queue->mutexhnd_queue, INFFMPG_MUTEX_TIMEOUT);

	do{
		found_next = 0;
		if(flag_test){ // clear all elems till the last flagged elem
			list_elem = packet_queue->first_pkt;
			sc = 0;
			do{
				if(!list_elem)
					break;
				if(list_elem->flags & flag_test){
					found_flag = 1;
					if(list_elem != packet_queue->first_pkt)
					{
						found_next = 1;
						mpxplay_debugf(MPXPLAY_DEBUG_QUEUECLEAR,"mpxplay_ffmpgdec_packetqueue_clear SEARCH i:%d nb:%d sc:%d",
							packet_queue->streamtype_index, packet_queue->nb_packets, sc);
						break;
					}
				}
				list_elem = list_elem->next;
				sc++;
			}while(1);
			if(!found_next)
				break;
		}

		sc = 0;
		do{
			list_elem = packet_queue->first_pkt;
			if(!list_elem){
				packet_queue->last_pkt = NULL;
				packet_queue->nb_packets = 0;
				break;
			}
			if(list_elem->flags & flag_test){
				if(!found_next){
					found_flag = 1;
					mpxplay_debugf(MPXPLAY_DEBUG_QUEUECLEAR,"mpxplay_ffmpgdec_packetqueue_clear CLEAR i:%d nb:%d sc:%d",
							packet_queue->streamtype_index, packet_queue->nb_packets, sc);
					break;
				}
			}
			found_next = 0;
			packet_queue->first_pkt = list_elem->next;
			packet_queue->nb_packets--;
			mpxplay_ffmpgdec_queuelistelem_clear(&list_elem);
			sc++;
		}while(1);
	}while(flag_test && (--retry));

	if(flag_test && !found_flag && packet_queue->first_pkt){
		mpxplay_debugf(MPXPLAY_DEBUG_QUEUECLEAR,"mpxplay_ffmpgdec_packetqueue_clear END sti:%d pt:%d nb:%d",
				packet_queue->streamtype_index, packet_queue->first_pkt->packet_type, packet_queue->nb_packets);
	}

	if(lock_result == MPXPLAY_ERROR_INFILE_OK)
	{
		PDS_THREADS_MUTEX_UNLOCK(&packet_queue->mutexhnd_queue);
	}

	return found_flag;
}

// search for ctx (depend on packet type), clear list elems before it (made for codec_ctx consoldidation)
unsigned int mpxplay_ffmpgdec_queue_clear_to_elemctx(mpxplay_packetqueue_t *packet_queue, void *ctx)
{
	mpxplay_packetlist_t *list_elem;
	unsigned int found = 0;

	if(!packet_queue)
		return found;

	do{
		list_elem = packet_queue->first_pkt;
		if(!list_elem){
			packet_queue->last_pkt = NULL;
			packet_queue->nb_packets = 0;
			break;
		}
		if(ctx){
			switch(list_elem->packet_type){
				case MPXPLAY_PACKETTYPE_AVCODECCTX:
					if((list_elem->codec_ctx == ctx) || (packet_queue->nb_packets <= 1))  // we always keep the last elem in the codec_ctx queue
						found = 1;
					break;
				default:
					if(list_elem->frame_pkt == ctx)
						found = 1;
					break;
			}
		}
		if(found)
			break;
		packet_queue->first_pkt = list_elem->next;
		packet_queue->nb_packets--;
		mpxplay_ffmpgdec_queuelistelem_clear(&list_elem);
	}while(!found);

	return found;
}

void in_ffmpgdec_packet_queues_clear(struct ffmpg_demuxer_data_s *ffmpi, mpxplay_packetqueue_t *packet_queue)
{
	int i;
	for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_NUM; i++, packet_queue++)
		mpxplay_ffmpgdec_packetqueue_clear(packet_queue, 0);
}

void mpxplay_ffmpgdec_packetqueue_init(mpxplay_packetqueue_t *packet_queue, unsigned int streamtype_index)
{
	if(!packet_queue)
		return;
	pds_memset(packet_queue, 0, sizeof(mpxplay_packetqueue_t));
	packet_queue->streamtype_index = streamtype_index;
	if(streamtype_index != MPXPLAY_STREAMTYPEINDEX_TEMP)
		pds_threads_mutex_new(&packet_queue->mutexhnd_queue);
	//pds_threads_cond_new(&packet_queue->condhnd_queue);
}

mpxp_bool_t mpxplay_ffmpgdec_packetqueue_check(mpxplay_packetqueue_t *packet_queue)
{
	if(!packet_queue || !packet_queue->mutexhnd_queue)
		return FALSE;
	return TRUE;
}

void mpxplay_ffmpgdec_packetqueue_close(mpxplay_packetqueue_t *packet_queue)
{
	if(!packet_queue)
		return;
	mpxplay_ffmpgdec_packetqueue_clear(packet_queue, 0);
	pds_threads_mutex_del(&packet_queue->mutexhnd_queue);
	//pds_threads_cond_del(&packet_queue->condhnd_queue);
}

void in_ffmpgdec_packet_queues_init(mpxplay_packetqueue_t *packet_queue)
{
	int i;
	for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_NUM; i++, packet_queue++)
		mpxplay_ffmpgdec_packetqueue_init(packet_queue, i);
}

void in_ffmpgdec_packet_queues_close(struct ffmpg_demuxer_data_s *ffmpi, mpxplay_packetqueue_t *packet_queue)
{
	int i;
	for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_NUM; i++, packet_queue++)
		mpxplay_ffmpgdec_packetqueue_close(packet_queue);
}

#endif //MPXPLAY_LINK_INFILE_FF_MPEG
