//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFMPEG decoder common functions and callback

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

#include "mpxplay.h"
#include "tagging.h"
#include <guiddef.h>

#ifdef MPXPLAY_GUI_QT

#include <libavutil/pixdesc.h>

#ifdef __cplusplus
extern "C" {
#endif
 extern mpxp_bool_t mpxplay_dispqt_videowidget_videorender_static_init(void);
 extern void mpxplay_dispqt_videowidget_videorender_mutex_lock(void);
 extern void mpxplay_dispqt_videowidget_videorender_mutex_unlock(void);
 extern mpxp_bool_t mpxplay_dispqt_videowidget_videorender_refer_d3ddevice(struct mpxplay_video_d3d_device_ref_s *d3d_device);
#ifdef __cplusplus
}
#endif

extern unsigned long mpxplay_config_videoplayer_control;
struct mpxplay_video_renderer_info_s mpxplay_video_render_infos;

#endif // MPXPLAY_GUI_QT

//------------------------------------------------------------------------------------------------------------------
void in_ffmpgdec_avsync_assign(struct ffmpg_demuxer_data_s *ffmpi, mpxp_int64_t pts)
{
	ffmpi->avsync_timestamp = pts;
	ffmpi->avsync_clocktime = pds_gettimeu();
	mpxplay_debugf(MPXPLAY_DEBUG_TIMESTAMP, "ASSIGN TS: %llu clock: %llu", ffmpi->avsync_timestamp, ffmpi->avsync_clocktime);
}

static mpxp_int64_t in_ffmpgdec_streamtime_to_avtime(AVStream *stream, mpxp_int64_t pkt_ts)
{
	if(stream && stream->time_base.num && stream->time_base.den)
		pkt_ts = pkt_ts * AV_TIME_BASE * stream->time_base.num / stream->time_base.den;
	else
		pkt_ts = pkt_ts * AV_TIME_BASE / 90000.0; // !!! MPEG like pts to usecs

	pkt_ts += INFFMPG_TIMESTAMP_SHIFT;  // first timestamp can be 0 (-audio_delay gives negative sync)

	return pkt_ts; // in AV_TIME_BASE
}

// convert stream time based dts/pts to AV_TIME_BASEd
mpxp_int64_t in_ffmpgdec_get_avpacket_timestamp_fctx(AVFormatContext *fctx, int primary_ff_stream_idx, mpxp_int64_t pkt_ts)
{
	AVStream *stream = (fctx && (primary_ff_stream_idx >= 0) && (primary_ff_stream_idx < fctx->nb_streams))? fctx->streams[primary_ff_stream_idx] : NULL;
	return in_ffmpgdec_streamtime_to_avtime(stream, pkt_ts);
}

mpxp_int64_t in_ffmpgdec_get_avpacket_ts_timestamp(struct ffmpg_demuxer_data_s *ffmpi, unsigned int streamtype_index, mpxp_int64_t pkt_ts)
{
	AVStream *stream = (ffmpi && (streamtype_index < MPXPLAY_STREAMTYPEINDEX_MAX))? ffmpi->selected_avstream[streamtype_index] : NULL;
	return in_ffmpgdec_streamtime_to_avtime(stream, pkt_ts);
}

// get AV_TIME_BASE based timestamp for an AVPacket
mpxp_int64_t in_ffmpgdec_get_avpacket_timestamp(struct ffmpg_demuxer_data_s *ffmpi, unsigned int streamtype_index, AVPacket *pkt)
{
	mpxp_int64_t pts = ((pkt->pts >= 0) && ((pkt->pts >= pkt->dts) || (pkt->dts < 0)))? pkt->pts : pkt->dts;
	return in_ffmpgdec_get_avpacket_ts_timestamp(ffmpi, streamtype_index, pts);
}

// select pts/dts (which seems to be the better)
mpxp_int64_t in_ffmpgdec_select_avframe_pts(AVFrame *avframe)
{
	mpxp_int64_t pts = (avframe->pts >= 0)? avframe->pts : ((avframe->pkt_dts >= 0)? avframe->pkt_dts : 0);
	return pts;
}

// get AV_TIME_BASE based timestamp for an AVFrame
mpxp_int64_t in_ffmpgdec_get_avframe_timestamp(struct ffmpg_demuxer_data_s *ffmpi, unsigned int streamtype_index, AVFrame *avframe)
{
	mpxp_int64_t pts = in_ffmpgdec_select_avframe_pts(avframe);
	if(pts < 0)
		return 0;
	return in_ffmpgdec_get_avpacket_ts_timestamp(ffmpi, streamtype_index, pts);
}

static mpxp_int64_t inffmpg_get_avframe_pts_corrected(struct ffmpg_demuxer_data_s *ffmpi, unsigned int streamtype_index, AVFrame *avframe)
{
	mpxp_int64_t dts = in_ffmpgdec_get_avframe_timestamp(ffmpi, streamtype_index, avframe);
#ifdef INFFMPG_USE_VIDEOPTS_CORRECTON
	mpxp_int64_t dts_video_last = ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_VIDEO];

	if((streamtype_index == MPXPLAY_STREAMTYPEINDEX_VIDEO) && (dts_video_last > INFFMPG_TIMESTAMP_SHIFT))  // TODO: wide test for non-streams too
	{
		AVStream *st = ffmpi->selected_avstream[streamtype_index];
		mpxp_int64_t dts_file = dts;
		mpxp_int64_t avduration = av_frame_get_pkt_duration(avframe);
		mpxp_int64_t duration_us = (st && st->time_base.num && st->time_base.den)? (avduration * AV_TIME_BASE * st->time_base.num / st->time_base.den) : (avduration * AV_TIME_BASE);

		if(((duration_us <= 1000) || (duration_us >= (AV_TIME_BASE / 2))) && st && st->avg_frame_rate.num && st->avg_frame_rate.den)  // TODO: max 999fps / 2fps is allowed
			duration_us = AV_TIME_BASE * st->avg_frame_rate.den / st->avg_frame_rate.num;
		if(duration_us <= 1000) // TODO: max 999 fps (because some MKVs contain invalid 1000 fps value)
			duration_us = 40000; // TODO: fake 25 fps

		if((dts <= INFFMPG_TIMESTAMP_SHIFT) || (dts <= dts_video_last) || (dts > (dts_video_last + duration_us * 2))) // TODO: more tests for this workaround if video pts is not sequential
		{
			mpxp_int64_t dts_audio = ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO];
			unsigned int seq_corr = 1;

			if( (dts > INFFMPG_TIMESTAMP_SHIFT)
			 && ((dts < (dts_video_last - INFFMPG_TIMESTAMP_MAX_AVDIFFERENCE)) || (dts > (dts_video_last + INFFMPG_TIMESTAMP_MAX_AVDIFFERENCE)))
			 && (dts_audio > INFFMPG_TIMESTAMP_SHIFT)
			){
				mpxp_int64_t dts_adiff = dts_video_last - dts_audio;

				if(dts_adiff < 0)
					dts_adiff = -dts_adiff;

				if(dts_adiff > INFFMPG_TIMESTAMP_MAX_AVDIFFERENCE) // TODO: more test for this workaround if video dts/pts is completely false
				{
					dts = dts_audio;
					seq_corr = 0;
					mpxplay_debugf(MPXPLAY_DEBUG_QUEUEGET,"get_avpacket_dts_corrected AVSYNC dts:%lld diff:%lld adi:%llu cd:%llu cp:%llu dur:%llu",
						dts, (dts - dts_video_last), dts_adiff, (avframe->pkt_dts), (avframe->pkt_pts), duration_us);
				}
				else
				{
//					mpxplay_debugf(MPXPLAY_DEBUG_QUEUEGET,"get_avpacket_dts_corrected NODIFF dts:%llu diff:%lld vdts:%llu vpts:%llu",
//							dts, (dts - dts_video_last),
//							((avpkt)? avpkt->dts : avframe->pkt_dts), ((avpkt)? avpkt->pts : avframe->pkt_pts));
				}
			}

			if(seq_corr){
				mpxp_int64_t correction = duration_us >> 4; // a smaller correction to file/frame dts from calculated
				mpxp_int64_t dvl = dts_video_last + duration_us;
				dts = dvl + ((dts <= dts_video_last)? -correction : correction);
			}
			mpxplay_debugf(MPXPLAY_DEBUG_QUEUEGET,"BADSEQ dr:%llu dts:%lld dtsf:%lld last:%lld diff:%lld df:%lld da:%lld dtsa:%lld pdts:%lld ppts:%lld",
				duration_us, dts, dts_file, dts_video_last, (dts - dts_video_last),
				(dts_file - dts_video_last), (dts_video_last - dts_audio), dts_audio, (avframe->pkt_dts), (avframe->pkt_pts));
		}
	}
#endif
	return dts; // in AV_TIME_BASE (usec)
}

//------------------------------------------------------------------------------------------------------------------
#ifdef MPXPLAY_GUI_QT

#define INFFMPG_STREAMINFO_MAX_LEN 128

static struct mpxplay_programinfo_t *inffmpg_programinfolist_fill(struct ffmpg_demuxer_data_s *ffmpi)
{
	struct mpxplay_programinfo_t *program_list;
	int i, pc;

	if(!in_ffmpgdec_content_has_programs(ffmpi))
		return NULL;

	program_list = pds_calloc(1, sizeof(*program_list));
	if(!program_list)
		return program_list;

	for(i = 0; i < ffmpi->fctx->nb_programs; i++)
	{
		AVProgram *prg = ffmpi->fctx->programs[i];
		if(prg)
		{
			AVDictionaryEntry *t = av_dict_get((const AVDictionary *)prg->metadata, "service_name", NULL, 0); // TODO: check, program elements without service names are ignored!
			if(t && t->value)
				program_list->nb_programs++;  // number of valid programs
		}
	}

	if(program_list->nb_programs <= 1)
		goto err_out_ipf;

	program_list->program_names = pds_calloc(program_list->nb_programs, sizeof(program_list->program_names[0]));
	program_list->program_ids = pds_calloc(program_list->nb_programs, sizeof(program_list->program_ids[0]));
	if(!program_list->program_names || !program_list->program_ids)
		goto err_out_ipf;

	for(i = 0, pc = 0; (i < ffmpi->fctx->nb_programs) && (pc < program_list->nb_programs); i++)
	{
		AVProgram *prg = ffmpi->fctx->programs[i];
		AVDictionaryEntry *t;

		if(!prg)
			continue;

		t = av_dict_get((const AVDictionary *)prg->metadata, "service_name", NULL, 0);
		if(!t || !t->value)
			continue;

		if(prg->id == ffmpi->program_number_current)
			program_list->selected_prognum = pc;
		program_list->program_ids[pc] = prg->id;
		program_list->program_names[pc] = malloc(pds_strlen(t->value) + 1);
		if(program_list->program_names[pc])
			pds_strcpy(program_list->program_names[pc], t->value);
		pc++;
	}

	return program_list;

err_out_ipf:
	if(program_list){
		pds_free(program_list->program_names);
		pds_free(program_list->program_ids);
		pds_free(program_list);
	}
	return NULL;
}

static struct mpxplay_streaminfo_t *inffmpg_streaminfolist_fill(struct ffmpg_demuxer_data_s *ffmpi)
{
	struct mpxplay_streaminfo_t *stream_list = pds_calloc(1, sizeof(*stream_list));
	struct ffmpeg_external_filelist_elem_s *files_chain;
	int i, j;

	if(!stream_list)
		return stream_list;

	stream_list->is_record_running = in_ffmpfile_file_output_running(ffmpi);
	stream_list->is_videowall_enabled = ffmpi->videowall_mode_enabled;
	files_chain = ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO].external_files_chain;
	if(files_chain)
		pds_strcpy(stream_list->primary_media_filename, files_chain->external_filename); // FIXME: bullshit

	for(i = 0; i <= MPXPLAY_STREAMTYPEINDEX_SUBTITLE; i++)
	{
		unsigned int nb_in_streams, nb_ext_streams, nb_all_streams;
		int stream_number = 0, stream_ff_idx = INFFMPG_STREAM_INDEX_DISABLE;
		in_ffmpstrm_streams_count(ffmpi, i, &stream_number, &stream_ff_idx, NULL);
		nb_in_streams = ffmpi->nb_streams[i];
		nb_ext_streams = (i != MPXPLAY_STREAMTYPEINDEX_VIDEO)? ffmpi->external_file_infos[i].nb_external_files : 0; // FIXME: incorrect extra video stream list
		nb_all_streams = nb_in_streams + nb_ext_streams;
		mpxplay_debugf(MPXPLAY_DEBUG_STRMLIST, "inffmpg_streams %d ni:%d ne:%d", i, nb_in_streams, nb_ext_streams);
		if(nb_all_streams)
		{
			struct ffmpeg_external_filelist_elem_s *external_files_chain;
			if(i == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
			{
				nb_in_streams++; nb_all_streams++; // "Disable" selection
			}
			stream_list->stream_names[i] = pds_calloc(nb_all_streams, sizeof(stream_list->stream_names[i]));
			stream_list->stream_idx[i] = pds_calloc(nb_all_streams, sizeof(stream_list->stream_idx[i]));
			if(stream_list->stream_names[i] && stream_list->stream_idx[i])
			{
				int j = 0;
				stream_list->nb_entries[i] = nb_all_streams;
				stream_list->selected_stream_index[i] = (ffmpi->selected_stream_number[i] < INFFMPG_STREAM_INDEX_DISABLE)? INFFMPG_STREAM_INDEX_DISABLE : ffmpi->selected_stream_number[i];

				if(i == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
				{
					stream_list->stream_names[i][0] = pds_malloc(INFFMPG_STREAMINFO_MAX_LEN);
					pds_strcpy(stream_list->stream_names[i][0], "S:  Disable");
					stream_list->stream_idx[i][0] = INFFMPG_STREAM_INDEX_DISABLE;
					j++;
				}

				// internal stream list (from the primary file)
				stream_number = 0;
				for( ; (j < nb_in_streams); j++, stream_number++)
				{
					int sn_save;
					AVStream *st;
					AVDictionaryEntry *dict_lang, *dict_title;
					mpxp_bool_t is_und_lang = TRUE;
					char streamtitle_str[48] = "", language_str[32] = "", color_str[16] = "", fps_str[16] = "", bit_rate_str[16] = "", channel_str[16];

					sn_save = stream_number;
					stream_ff_idx = INFFMPG_STREAM_INDEX_DISABLE;
					if(in_ffmpstrm_streams_count(ffmpi, i, &stream_number, &stream_ff_idx, NULL) <= 0)
					{
						mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "inffmpg_streams_count FAILED %d", i);
						break;
					}
					if((sn_save > stream_number) || (stream_ff_idx < 0) || (stream_ff_idx >= ffmpi->fctx->nb_streams))
					{
						mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "inffmpg_streams_count stream_ff_idx FAILED i:%d sn:%d idx:%d", i, stream_number, stream_ff_idx);
						break;
					}

					st = ffmpi->fctx->streams[stream_ff_idx];
					if(!st)
						continue;

					//mpxplay_debugf(MPXPLAY_DEBUG_STRMLIST, "inffmpg_streaminfolist_fill i:%d ffi:%d", i, stream_ff_idx);

					stream_list->stream_names[i][j] = pds_malloc(INFFMPG_STREAMINFO_MAX_LEN);
					if(!stream_list->stream_names[i][j])
						break;

					dict_lang = av_dict_get((const AVDictionary *)st->metadata, "language", NULL, 0);
					if(dict_lang)
					{
						if(!dict_lang->value || !dict_lang->value[0])
							dict_lang = NULL;
						else
							is_und_lang = (pds_stricmp(dict_lang->value,"und") == 0)? TRUE : FALSE;
					}

					dict_title = av_dict_get((const AVDictionary *)st->metadata, "title", NULL, 0);
					if(dict_title && (!dict_title->value || !dict_title->value[0]))
						dict_title = NULL;

					if(dict_lang && ((i == MPXPLAY_STREAMTYPEINDEX_SUBTITLE) || !is_und_lang)) // we put [und] flag to subtitle streams (only)
					{
						const char *lang_code_ptr = dict_lang->value, *lang_name_ptr = NULL;
						if((i != MPXPLAY_STREAMTYPEINDEX_SUBTITLE) && !is_und_lang) // stream title is not displayed to audio and video streams if they have valid language code
							dict_title = NULL;
						else if(dict_title && pds_strstri(dict_title->value, (char *)lang_code_ptr)) // language code is not displayed, if stream title contains it
							lang_code_ptr = NULL;
						if(lang_code_ptr && (dict_title || !funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_STREAMLIST_SHORT_LANGNAMES)))
						{
							lang_name_ptr = mpxplay_ffmplang_getlangname_by_langcode(lang_code_ptr);
							if(dict_title && pds_strstri(dict_title->value, (char *)lang_name_ptr)) // language name is not displayed, if stream title contains it
								lang_code_ptr = lang_name_ptr = NULL;
						}
						if(lang_code_ptr)
							snprintf(language_str, sizeof(language_str), "[%s] ", ((lang_name_ptr)? lang_name_ptr : lang_code_ptr));
						snprintf(streamtitle_str, sizeof(streamtitle_str), " %s%s%s", language_str, ((dict_title)? dict_title->value : ""),
							(((i != MPXPLAY_STREAMTYPEINDEX_SUBTITLE) || dict_title || !(st->disposition & (AV_DISPOSITION_DEFAULT|AV_DISPOSITION_FORCED)))? "" :
							((st->disposition & AV_DISPOSITION_FORCED)? "Forced" : "Default") ) ); // we display "Forced" and "Default" for subtitle streams only, if no stream title
					}
					else if(i == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
					{
						if(dict_title) // display title (only) if there's no valid lang code
							snprintf(streamtitle_str, sizeof(streamtitle_str), " %s ", dict_title->value);
						else  // display stream id, if no lang and title
							snprintf(streamtitle_str, sizeof(streamtitle_str), " %2.2X ", st->id);
					}

					if(st->codecpar)
					{
						int bit_rate = st->codecpar->bit_rate / 1000, channel_num;
						if(bit_rate > 0)
							snprintf(bit_rate_str, sizeof(bit_rate_str), ", %d kb/s", bit_rate);
						switch(i)
						{
							case MPXPLAY_STREAMTYPEINDEX_AUDIO:
								channel_num = st->codecpar->ch_layout.nb_channels;
								switch(channel_num){
									case 1: pds_strcpy(channel_str, "Mono"); break;
									case 2: pds_strcpy(channel_str, "Stereo"); break;
									case 6: pds_strcpy(channel_str, "5.1 chans"); break;
									default: snprintf(channel_str, sizeof(channel_str), "%d chans", channel_num); break;
								}
								snprintf(stream_list->stream_names[i][j], INFFMPG_STREAMINFO_MAX_LEN, "A: %s%s, %d Hz, %s%s",
									streamtitle_str, avcodec_get_name(st->codecpar->codec_id), st->codecpar->sample_rate, channel_str, bit_rate_str);
								break;
							case MPXPLAY_STREAMTYPEINDEX_VIDEO:
								{
									const AVPixFmtDescriptor *desc = av_pix_fmt_desc_get((enum AVPixelFormat)st->codecpar->format);
									if(desc && (desc->comp[0].depth > 8))
										snprintf(color_str, sizeof(color_str), ", %d-bit%s", desc->comp[0].depth,
												((st->codecpar->color_trc == AVCOL_TRC_SMPTE2084) || (st->codecpar->color_trc == AVCOL_TRC_ARIB_STD_B67))? " HDR": "");
								}
								if(st->avg_frame_rate.num && st->avg_frame_rate.den)
									snprintf(fps_str, sizeof(fps_str), ", %2.2f fps", (float)st->avg_frame_rate.num / (float)st->avg_frame_rate.den);
								snprintf(stream_list->stream_names[i][j], INFFMPG_STREAMINFO_MAX_LEN, "V: %s%s, %dx%d%s%s%s",
									streamtitle_str, avcodec_get_name(st->codecpar->codec_id),
									st->codecpar->width, st->codecpar->height,
									color_str, fps_str, bit_rate_str);
								break;
							case MPXPLAY_STREAMTYPEINDEX_SUBTITLE:
								snprintf(stream_list->stream_names[i][j], INFFMPG_STREAMINFO_MAX_LEN, "S: %s", streamtitle_str);
								break;
						}
					}
					else
					{
						char *shorttypename = (i == MPXPLAY_STREAMTYPEINDEX_AUDIO)? "A:" : ((i == MPXPLAY_STREAMTYPEINDEX_VIDEO)? "V:" : "S:");
						snprintf(stream_list->stream_names[i][j], INFFMPG_STREAMINFO_MAX_LEN, "%s Track %d%s", shorttypename, (j + 1), streamtitle_str); // TODO: better stream names
					}
					stream_list->stream_idx[i][j] = stream_number;
					mpxplay_debugf(MPXPLAY_DEBUG_STRMLIST, "MPXPLAY_INFILE_CBKCTRL_GET_STREAMLIST %d/%d %d/%d %s", i, j, stream_number, stream_ff_idx, stream_list->stream_names[i][j]);
				}

				// external stream list (from extra audio/subtitle stream files)
				external_files_chain = ffmpi->external_file_infos[i].external_files_chain;
				for( ; (j < nb_all_streams) && external_files_chain; j++)
				{
					char *shorttypename, language_str[32] = "", bit_rate_str[32] = "", channel_str[32] ="", samplerate_str[32] = "";
					int bit_rate;

					stream_list->stream_names[i][j] = pds_malloc(INFFMPG_STREAMINFO_MAX_LEN);
					if(!stream_list->stream_names[i][j])
						break;

					if(external_files_chain->language_name[0] || external_files_chain->language_code[0])
						snprintf(language_str, sizeof(language_str), " [%s] ", ((!funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_STREAMLIST_SHORT_LANGNAMES) && external_files_chain->language_name[0])? external_files_chain->language_name : external_files_chain->language_code));

					bit_rate = external_files_chain->bitrate / 1000;
					if(bit_rate > 0)
						snprintf(bit_rate_str, sizeof(bit_rate_str), ", %d kb/s", bit_rate);

					if(external_files_chain->channels > 0)
					{
						switch(external_files_chain->channels){
							case 1: pds_strcpy(channel_str, ", Mono"); break;
							case 2: pds_strcpy(channel_str, ", Stereo"); break;
							case 6: pds_strcpy(channel_str, ", 5.1 chans"); break;
							default: snprintf(channel_str, sizeof(channel_str), ", %d chans", external_files_chain->channels); break;
						}
					}

					if(external_files_chain->samplerate > 0)
						snprintf(samplerate_str, sizeof(samplerate_str), ", %d Hz", external_files_chain->samplerate);

					shorttypename = (i == MPXPLAY_STREAMTYPEINDEX_AUDIO)? "A:" : ((i == MPXPLAY_STREAMTYPEINDEX_VIDEO)? "V:" : "S:");
					snprintf(stream_list->stream_names[i][j], INFFMPG_STREAMINFO_MAX_LEN, "%s %s%s%s%s%s", shorttypename, language_str,
							((external_files_chain->format_name[0])? external_files_chain->format_name : ((external_files_chain->codec_id > AV_CODEC_ID_NONE)? avcodec_get_name(external_files_chain->codec_id) : pds_getfilename_from_fullname(external_files_chain->external_filename))),
							samplerate_str, channel_str, bit_rate_str);

					stream_list->stream_idx[i][j] = external_files_chain->stream_index;

					external_files_chain = external_files_chain->next_external_file;
				}

				if(j <= ((i == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)? 1 : 0))  // number of streams has changed or no valid stream found
				{
					stream_list->nb_entries[i] = 0;
					stream_list->selected_stream_index[i] = INFFMPG_STREAM_INDEX_DISABLE;
					for(j = 0; j < nb_all_streams; j++)
						if(stream_list->stream_names[i][j])
							pds_free(stream_list->stream_names[i][j]);
					pds_free(stream_list->stream_names[i]);
					pds_free(stream_list->stream_idx[i]);
					stream_list->stream_names[i] = NULL;
					stream_list->stream_idx[i] = NULL;
				}
			}
		}
	}
	return stream_list;
}

static void inffmpg_pictureinfo_metadata_update(struct ffmpg_demuxer_data_s *ffmpi, AVDictionary *metadata)
{
	if((ffmpi->flags & INFFMPG_FLAG_STILLPICTURE) && ffmpi->miis)
	{
		AVDictionaryEntry *e_make = av_dict_get(metadata, "Make", NULL, 0);
		AVDictionaryEntry *e_model = av_dict_get(metadata, "Model", NULL, 0);
		AVDictionaryEntry *e_fnum = av_dict_get(metadata, "FNumber", NULL, 0);
		AVDictionaryEntry *e_etime = av_dict_get(metadata, "ExposureTime", NULL, 0);
		AVDictionaryEntry *e_iso = av_dict_get(metadata, "ISOSpeedRatings", NULL, 0);
		int len = -1, val1, val2;
		float f_fnum = 0.0f, f_etime = 0.0f;
		char str[256];

		if(((!e_make || !e_make->value) && (!e_model || !e_model->value)) || !e_fnum || !e_etime || !e_fnum->value || !e_etime->value)
			return;

		if(e_fnum){
			val1 = val2 = 1;
			sscanf(e_fnum->value, "%d:%d", &val1, &val2);
			if(val2 > 0)
				f_fnum = (float) val1 / (float)val2;
		}
		if(e_etime){
			val1 = val2 = 1;
			sscanf(e_etime->value, "%d:%d", &val1, &val2);
			if(val2 > 0)
				f_etime = 1.0f / ((float) val1 / (float)val2);
		}
		snprintf(str, sizeof(str), "%s %s, F%.1f, 1/%.0f, ISO%d", ((e_make)? e_make->value : ""), ((e_model)? e_model->value : ""), f_fnum, f_etime, ((e_iso)? pds_atol(e_iso->value) : 0));
		ffmpi->miis->control_cb(ffmpi->fbds,MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_PACKFUNCTTI3I(MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_PUT,MPXPLAY_TEXTCONV_TYPE_UTF8,I3I_COMMENT),str,&len);
	}
}

#endif // MPXPLAY_GUI_QT

//--------------------------------------------------------------------------------------------------------------------------------------
// HW based video decoding (DXVA and D3D11VA hardware accelerators are enabled in FFmpeg lib)
#if defined(MPXPLAY_GUI_QT) && defined(MPXPLAY_LINK_DXVA)

#include <libavutil/hwcontext_d3d11va.h>
#include <libavutil/hwcontext_dxva2.h>
#include <libavcodec/version.h>

extern AVCodec ff_libdav1d_decoder;
extern AVCodec ff_av1_decoder;
extern const GUID ff_DXVA2_ModeAV1_VLD_Profile0;

static void in_ffmpgdec_hwdec_dealloc(AVCodecContext *codec_ctx);

#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
static void in_ffmpgdec_hwdec_lock_d3d_device(void *mutex_handler)
{
	mpxplay_dispqt_videowidget_videorender_mutex_lock();
}

static void in_ffmpgdec_hwdec_unlock_d3d_device(void *mutex_handler)
{
	mpxplay_dispqt_videowidget_videorender_mutex_unlock();
}
#endif // MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK

static int in_ffmpgdec_hwdec_get_format_callback(AVCodecContext *avctx, const enum AVPixelFormat *pix_fmts)
{
	struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;

	while(*pix_fmts != AV_PIX_FMT_NONE)
	{
		if(*pix_fmts == vri_datas->hwdevice_avpixfmt)
		{
			if( avctx->hw_device_ctx
			 && (avcodec_get_hw_frames_parameters(avctx, avctx->hw_device_ctx, (enum AVPixelFormat)vri_datas->hwdevice_avpixfmt, &avctx->hw_frames_ctx) == 0)
			 && avctx->hw_frames_ctx
			){
				struct AVHWFramesContext *av_hwframes_ctx = (struct AVHWFramesContext *)avctx->hw_frames_ctx->data;

				av_hwframes_ctx->initial_pool_size += MPXPLAY_VIDEO_RENDERER_FRAME_HWDECTEXTURES_SAVED_MAX;
				if(av_hwframes_ctx->initial_pool_size < MPXPLAY_VIDEO_RENDERER_FRAME_HWDECTEXTURES_POOL_MIN) // should not happen, because initial_pool_size is set in libavcodec/dxva.c
					av_hwframes_ctx->initial_pool_size = MPXPLAY_VIDEO_RENDERER_FRAME_HWDECTEXTURES_POOL_MIN;

				switch(vri_datas->hwdevice_avpixfmt)
				{
					case AV_PIX_FMT_D3D11:
					{
						struct AVD3D11VAFramesContext *d3d11_hwframes_ctx = (struct AVD3D11VAFramesContext*)av_hwframes_ctx->hwctx;
						d3d11_hwframes_ctx->BindFlags = D3D11_BIND_DECODER;
						if(vri_datas->d3d_render_handler)
							funcbit_enable(d3d11_hwframes_ctx->BindFlags, D3D11_BIND_SHADER_RESOURCE);
						break;
					}
					// TODO: other hw accelerators
				}
				if(av_hwframe_ctx_init(avctx->hw_frames_ctx) == 0)
				{
					// successful hw decoding initialization
					break;
				}
			}
		}
		else
		{
			const AVPixFmtDescriptor *desc = av_pix_fmt_desc_get(*pix_fmts);

			if(!(desc->flags & AV_PIX_FMT_FLAG_HWACCEL))
			{   // fall back to sw decoding, if hw decoding cannot be initialized (format / resolution is not supported)
				break;
			}
		}
		pix_fmts++;
	}

	return *pix_fmts;
}

#endif

void mpxplay_in_ffmpgdec_hwdec_override_avcodec(AVCodecParameters *codecpar, AVCodec **codec_av_new_p)
{
	if(!funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_USEHWDECODING))
		return;
	if(funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESDHWDEC) && (codecpar->width <= 960) && (codecpar->height <= 576))
		return;
	if((*codec_av_new_p == &ff_libdav1d_decoder) && mpxplay_video_render_infos.render_function_check_videodecoder_inputformat(ff_DXVA2_ModeAV1_VLD_Profile0, codecpar->format, codecpar->width, codecpar->height))
		*codec_av_new_p = &ff_av1_decoder;
}

#ifdef MPXPLAY_GUI_QT

#if defined(MPXPLAY_LINK_DXVA)
static void in_ffmpgdec_hwdec_dealloc(AVCodecContext *codec_ctx)
{
	mpxplay_debugf(MPXPLAY_DEBUG_HWDEC, "in_ffmpgdec_hwdec_dealloc START ctx:%8.8X", (mpxp_ptrsize_t)codec_ctx);
	if(codec_ctx->hw_device_ctx)
	{
		AVBufferRef *avctx_hwd_ctx = codec_ctx->hw_device_ctx;
		AVBufferRef *avctx_hwf_ctx = codec_ctx->hw_frames_ctx;

		codec_ctx->hw_device_ctx = NULL;
		codec_ctx->hw_frames_ctx = NULL;
		codec_ctx->get_format = NULL;

		av_buffer_unref(&avctx_hwf_ctx);
		av_buffer_unref(&avctx_hwd_ctx);
	}
	mpxplay_debugf(MPXPLAY_DEBUG_HWDEC, "in_ffmpgdec_hwdec_dealloc END ctx:%8.8X", (mpxp_ptrsize_t)codec_ctx);
}
#endif // MPXPLAY_LINK_DXVA

static int in_ffmpgdec_hwdec_init(struct ffmpg_demuxer_data_s *ffmpi, AVCodecContext *codec_ctx_new)
{
	int dxva_result = -1;
	mpxplay_debugf(MPXPLAY_DEBUG_HWDEC, "in_ffmpgdec_hwdec_init START ctx:%8.8X", (mpxp_ptrsize_t)codec_ctx_new);
#ifdef MPXPLAY_LINK_DXVA
	if(funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_USEHWDECODING))
	{
		if(codec_ctx_new->coded_width <= 0)
			codec_ctx_new->coded_width = codec_ctx_new->width;
		if(codec_ctx_new->coded_height <= 0)
			codec_ctx_new->coded_height = codec_ctx_new->height;
		if((codec_ctx_new->coded_width > 0) && (codec_ctx_new->coded_height > 0) && codec_ctx_new->codec
		 && ( !funcbit_test(mpxplay_config_videoplayer_control, MPXPLAY_CONFIG_VIDEOPLAYERCONTROL_DISABLESDHWDEC)
			  || (codec_ctx_new->coded_width > 960) || (codec_ctx_new->coded_height > 576))
		){
			struct mpxplay_video_renderer_info_s *vri_datas = &mpxplay_video_render_infos;
			enum AVHWDeviceType hw_dev_type = AV_HWDEVICE_TYPE_NONE;
			int i = 0;

			if(!mpxplay_dispqt_videowidget_videorender_static_init())
				return dxva_result;
			if(!vri_datas->d3d_video_device || (vri_datas->hwdevice_avpixfmt == AV_PIX_FMT_NONE))
				return dxva_result;

			do{
				const AVCodecHWConfig *config = avcodec_get_hw_config(codec_ctx_new->codec, i++);
				if (!config)
					return dxva_result;
				if( (config->methods & (AV_CODEC_HW_CONFIG_METHOD_HW_DEVICE_CTX | AV_CODEC_HW_CONFIG_METHOD_HW_FRAMES_CTX))
				 && (config->pix_fmt == vri_datas->hwdevice_avpixfmt)
				 && (config->device_type > AV_HWDEVICE_TYPE_NONE)
				){
					hw_dev_type = config->device_type;
					break;
				}
			}while(TRUE);

			codec_ctx_new->hw_device_ctx = av_hwdevice_ctx_alloc(hw_dev_type);
			if(!codec_ctx_new->hw_device_ctx)
				goto err_out_dxva;

			switch(vri_datas->hwdevice_avpixfmt)
			{
				case AV_PIX_FMT_D3D11:
				{
					struct AVD3D11VADeviceContext *hwctx = (struct AVD3D11VADeviceContext*)((struct AVHWDeviceContext *)codec_ctx_new->hw_device_ctx->data)->hwctx;
					struct mpxplay_video_d3d_device_ref_s d3d_device_refs = {0};
					if(!hwctx)
						goto err_out_dxva;
					if(!mpxplay_dispqt_videowidget_videorender_refer_d3ddevice(&d3d_device_refs))
						goto err_out_dxva;
					if(!d3d_device_refs.d3d_device_ptr || !d3d_device_refs.d3d_device_ctx || !d3d_device_refs.d3d_video_ctx)
						goto err_out_dxva;
					hwctx->device = d3d_device_refs.d3d_device_ptr;
					hwctx->device_context = d3d_device_refs.d3d_device_ctx;
					hwctx->video_device = d3d_device_refs.d3d_video_device;
					hwctx->video_context = d3d_device_refs.d3d_video_ctx;
#ifdef MPXPLAY_VIDEO_D3D_DEVICE_USE_LOCK
					hwctx->lock = in_ffmpgdec_hwdec_lock_d3d_device;
					hwctx->unlock = in_ffmpgdec_hwdec_unlock_d3d_device;
					hwctx->lock_ctx = vri_datas->d3d_device_mutex;
#endif
					break;
				}
				case AV_PIX_FMT_DXVA2_VLD:
				{
					struct AVDXVA2DeviceContext *hwctx = (struct AVDXVA2DeviceContext*)((struct AVHWDeviceContext *)codec_ctx_new->hw_device_ctx->data)->hwctx;
					if(!hwctx)
						goto err_out_dxva;
					hwctx->devmgr = vri_datas->d3d_video_device; // IDirect3DDeviceManager9 stored on d3d_video_device ptr
					break;
				}
			}

			dxva_result = av_hwdevice_ctx_init(codec_ctx_new->hw_device_ctx);
			if(dxva_result < 0)
				goto err_out_dxva;

			switch(vri_datas->hwdevice_avpixfmt)
			{
				case AV_PIX_FMT_D3D11:
					codec_ctx_new->get_format = in_ffmpgdec_hwdec_get_format_callback;
					break;
			}
		}
	}
#endif // MPXPLAY_LINK_DXVA
	mpxplay_debugf(MPXPLAY_DEBUG_HWDEC, "in_ffmpgdec_hwdec_init END ctx:%8.8X", (mpxp_ptrsize_t)codec_ctx_new);
	return dxva_result;

#ifdef MPXPLAY_LINK_DXVA
err_out_dxva:
	mpxplay_debugf(MPXPLAY_DEBUG_ERROR, "in_ffmpgdec_hwdec_init ERR_OUT_DXVA res:%d ctx:%8.8X", dxva_result, (mpxp_ptrsize_t)codec_ctx_new);
	in_ffmpgdec_hwdec_dealloc(codec_ctx_new);
	return -1;
#endif
}

#endif // MPXPLAY_GUI_QT

void in_ffmpgdec_hwdec_deinit(AVCodecContext *codec_ctx)
{
#if defined(MPXPLAY_GUI_QT)
	mpxplay_debugf(MPXPLAY_DEBUG_HWDEC, "in_ffmpgdec_hwdec_deinit START ctx:%8.8X", (mpxp_ptrsize_t)codec_ctx);
	// at a normal close, avcodec_free_context deallocs the hw decoding related data (in_ffmpgdec_hwdec_dealloc is not called here)
	mpxplay_video_render_infos.render_function_frame_buffer_consolidate();
	mpxplay_debugf(MPXPLAY_DEBUG_HWDEC, "in_ffmpgdec_hwdec_deinit END ctx:%8.8X", (mpxp_ptrsize_t)codec_ctx);
#endif
}

//--------------------------------------------------------------------------------------------------------------------------------------
// number of software (not HW) based FFmpeg decoder threads
static int inffmpgdec_get_nb_sw_video_decoder_threads(struct ffmpg_demuxer_data_s *ffmpi)
{
	int thread_count = 1;
	if(!(ffmpi->flags & INFFMPG_FLAG_STILLPICTURE))
	{
		thread_count = pds_threads_get_number_of_processors();
		if(thread_count < 2)
			thread_count = 2;
	}
	return thread_count;
}

static int inffmpgdec_get_avtimesync_us(struct ffmpg_demuxer_data_s *ffmpi, unsigned int stream_type, mpxp_int64_t *frame_timestamp_us)
{
	int retcode = MPXPLAY_ERROR_INFILE_NODATA;

	if(stream_type == MPXPLAY_STREAMTYPEINDEX_VIDEO)
	{
		ffmpi->avsync_videotimestamp = *frame_timestamp_us;
		ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_VIDEO] = *frame_timestamp_us; // TODO: unused (remove above?)
	}

	if((ffmpi->avsync_timestamp < 0) && (stream_type < MPXPLAY_STREAMTYPEINDEX_NUM))
	{
		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_INDEPENDENTSTREAMSYNC))
		{
			struct ffmpeg_external_files_info_s *extfilehandler =  &ffmpi->external_file_infos[stream_type];
			if((extfilehandler->avclocktime_us < 0) || (extfilehandler->avtimestamp_us < 0))
			{
				extfilehandler->avclocktime_us = pds_gettimeu();
				extfilehandler->avtimestamp_us = *frame_timestamp_us; // !!! frame timestamp must be passed
			}
			*frame_timestamp_us = pds_gettimeu() - extfilehandler->avclocktime_us + extfilehandler->avtimestamp_us;
			retcode = MPXPLAY_ERROR_INFILE_OK;
		}
		else if(!ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_AUDIO] || (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets >= INFFMPG_QUEUE_PACKETNUM_MAX)
//		     || funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC) // FIXME:
		){
			in_ffmpgdec_avsync_assign(ffmpi, *frame_timestamp_us);
			mpxplay_debugf(MPXPLAY_DEBUG_TIMESTAMP, "in_ffmpgdec_infile_callback set TIMESYNC: %llu", ffmpi->avsync_timestamp);
		}
		else
		{
			*frame_timestamp_us = 0;
		}
	}

	if(ffmpi->avsync_timestamp >= 0)
	{
#ifdef MPXPLAY_USE_DEBUGF
		mpxp_int64_t av_diff = ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] - ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_VIDEO];
#endif
		*frame_timestamp_us = pds_gettimeu() - ffmpi->avsync_clocktime + ffmpi->avsync_timestamp;
#ifdef MPXPLAY_USE_DEBUGF
		if( (ffmpi->selected_stream_number[MPXPLAY_STREAMTYPEINDEX_VIDEO] >= 0)
		 && ((av_diff < -INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE) || (av_diff > INFFMPG_TIMESTAMP_LIVE_MAX_AVDIFFERENCE))
		){
			mpxplay_debugf(MPXPLAY_DEBUG_TIMESTAMP, "TIMESTAMP: st:%d sf:%d ts:%llu ats:%llu vts:%llu avdiff:%lld",
				stream_type, (funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC)? 1 : 0),
				*frame_timestamp_us, ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO], ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_VIDEO],
				av_diff);
		}
#endif
		retcode = MPXPLAY_ERROR_INFILE_OK;
	}

	return retcode;
}

static int inffmpgdec_codecctx_init(struct ffmpg_demuxer_data_s *ffmpi, unsigned int stream_type, mpxplay_packetlist_t **packet_dest)
{
	int retcode = MPXPLAY_ERROR_INFILE_NODATA;
	mpxplay_packetlist_t *pktlist_elem;

	if(stream_type >= MPXPLAY_STREAMTYPEINDEX_MAX)
		return retcode;

	//mpxplay_debugf(MPXPLAY_DEBUG_CALLBACK, "MPXPLAY_INFILE_CBKCTRL_GET_CODECCTXINIT_FFMPEG %d", stream_type);
	pktlist_elem = NULL;
	mpxplay_ffmpgdec_packetqueue_get((mpxp_ptrsize_t)ffmpi, &ffmpi->codecctx_queues[stream_type], &pktlist_elem, 0);
	if(pktlist_elem)
	{
		if((pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_INIT) && pktlist_elem->codec_ctx)
		{
			AVCodecContext *codec_ctx_new = (AVCodecContext *)pktlist_elem->codec_ctx;
			AVStream *codec_av_stream = (AVStream *)pktlist_elem->avstream;
			int ffret;

			if(!mpxplay_ffmpstrm_is_avstream_parsed(codec_av_stream, stream_type))
				return MPXPLAY_ERROR_INFILE_SYNC_IN;
			if(avcodec_parameters_to_context(codec_ctx_new, codec_av_stream->codecpar) < 0)
				goto err_out_codecctx_init;

			codec_ctx_new->opaque = ffmpi;
#if defined(MPXPLAY_GUI_QT)
			if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE))
			{
				int dxva_result = -1;
				if(stream_type == MPXPLAY_STREAMTYPEINDEX_VIDEO)
					dxva_result = in_ffmpgdec_hwdec_init(ffmpi, codec_ctx_new);
				if(dxva_result != 0)
				{
					codec_ctx_new->thread_count = inffmpgdec_get_nb_sw_video_decoder_threads(ffmpi);
					if(codec_ctx_new->codec == &ff_libdav1d_decoder)
					{
						funcbit_enable(codec_ctx_new->flags, AV_CODEC_FLAG_LOW_DELAY); // less 'blocking' threads, less memory usage, minimal cpu loss
					}
					if(stream_type >= MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
					{
						codec_ctx_new->thread_type = FF_THREAD_SLICE;
						if(stream_type == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
							funcbit_enable(codec_ctx_new->flags, AV_CODEC_FLAG_LOW_DELAY);
					}
				}
				mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "in_ffmpgdec_infile_callback avcodec_open2 BEGIN ffmpi:%8.8X ctx:%8.8X cav:%8.8X i:%d dxva:%d w:%d h:%d",
						(mpxp_ptrsize_t)ffmpi, (mpxp_ptrsize_t)codec_ctx_new, (mpxp_ptrsize_t)codec_ctx_new->codec, stream_type, dxva_result, codec_ctx_new->coded_width, codec_ctx_new->coded_height);
			}
#endif
			ffret = avcodec_open2(codec_ctx_new, codec_ctx_new->codec, NULL);
			if(ffret < 0)
			{
				mpxplay_debugf(MPXPLAY_DEBUG_ERROR, "in_ffmpgdec_infile_callback avcodec_open2 FAILED! ret:%d ffmpi:%8.8X ctx:%8.8X cdc:%8.8X sti:%d",
						ffret, (mpxp_ptrsize_t)ffmpi, (mpxp_ptrsize_t)codec_ctx_new, (mpxp_ptrsize_t)codec_ctx_new->codec, stream_type);
				goto err_out_codecctx_init;
			}
			mpxplay_debugf(MPXPLAY_DEBUG_OPENCLSDTL, "in_ffmpgdec_infile_callback avcodec_open2 DONE ffmpi:%8.8X ctx:%8.8X cav:%8.8X i:%d w:%d h:%d",
					(mpxp_ptrsize_t)ffmpi, (mpxp_ptrsize_t)codec_ctx_new, (mpxp_ptrsize_t)codec_ctx_new->codec, stream_type, codec_ctx_new->coded_width, codec_ctx_new->coded_height);
		}
		if(stream_type == MPXPLAY_STREAMTYPEINDEX_VIDEO)
		{
			if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_STILLPICTURE))
				funcbit_enable(pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_STILLPICT);
			if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM))
				funcbit_enable(pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_LIVESTREAM);
		}
		funcbit_disable(pktlist_elem->flags, MPXPLAY_PACKETLISTFLAG_INIT);
		*packet_dest = pktlist_elem;
		retcode = MPXPLAY_ERROR_INFILE_OK;
	}

err_out_codecctx_init:
	return retcode;
}

int in_ffmpgdec_infile_callback(mpxp_ptrsize_t demuxer_data, unsigned int control, mpxp_ptrsize_t arg1, mpxp_ptrsize_t arg2, int timeout_ms)
{
	struct ffmpg_demuxer_data_s *ffmpi = (struct ffmpg_demuxer_data_s *)demuxer_data;
	int retcode = MPXPLAY_ERROR_INFILE_NODATA;
	mpxplay_packetlist_t *pktlist_elem;

	if(!ffmpi || (ffmpi->struct_id != MPXPLAY_FFMPEGDEMUX_STRUCT_ID))
		return retcode;

	if(timeout_ms != INFFMPG_MUTEXTIME_NOLOCK)
	{
		retcode = PDS_THREADS_MUTEX_LOCK(&ffmpi->mutexhnd_ffmpeg, timeout_ms);
		if(retcode != MPXPLAY_ERROR_OK){
			if(timeout_ms > MPXPLAY_MUTEXTIME_SHORT){
				mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "in_ffmpgdec_infile_callback pds_threads_mutex_lock FAILED cmd:%8.8X pd:%8.8X ret:%d tms:%d", control, (int)demuxer_data, retcode, timeout_ms);
			}
			return retcode;
		}
	}

	switch(control)
	{
#ifdef MPXPLAY_GUI_QT
		case MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLMODE:
			if(arg1){
				ffmpi->videowall_mode_enabled = in_ffmpstrm_all_programs_videostreams_initialize(ffmpi);
			}else{
				in_ffmpstrm_all_programs_videostreams_terminate(ffmpi);
				ffmpi->videowall_mode_enabled = arg1;
			}
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_SELECT_PROGRAM:
			ffmpi->select_new_program_number = arg1;
			retcode = MPXPLAY_ERROR_INFILE_OK;
			mpxplay_debugf(MPXPLAY_DEBUG_PRGCHG, "in_ffmpgdec_infile_callback SELECT_PROGRAM %d", ffmpi->select_new_program_number);
			break;
#endif
		case MPXPLAY_INFILE_CBKCTRL_SET_SELECT_STREAM:
			//mpxplay_debugf(MPXPLAY_DEBUG_CALLBACK, "MPXPLAY_INFILE_CBKCTRL_SET_SELECT_STREAM %d -> %d", arg1_num, *arg2);
			if(arg1 < MPXPLAY_STREAMTYPEINDEX_PLAYNUM){
				ffmpi->select_new_stream_number[arg1] = arg2;
				retcode = MPXPLAY_ERROR_INFILE_OK;
			}
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_PACKETQUEUE_CLEAR:
			if(arg1 < MPXPLAY_STREAMTYPEINDEX_MAX)
#ifdef MPXPLAY_GUI_QT
				if(!funcbit_test(arg2, MPXPLAY_PACKETLISTFLAG_KEYFRAME) || !funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVSYNC)) // we don't skip keyframes if A/V sync is not finished
#endif
					retcode = mpxplay_ffmpgdec_packetqueue_clear(&ffmpi->packet_queues[arg1], arg2);
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_CODECCTX_CLEAR:
			if(arg1 < MPXPLAY_STREAMTYPEINDEX_MAX)
				retcode = mpxplay_ffmpgdec_packetqueue_clear(&ffmpi->codecctx_queues[arg1], arg2);
			break;
#ifdef MPXPLAY_GUI_QT
		case MPXPLAY_INFILE_CBKCTRL_SET_RECORDSTREAMSSTOP:
			in_ffmpfile_file_output_close(ffmpi);
			break;
#endif
		default:
			if(!arg1) // used as a pointer bellow
				goto err_out_cbk;
	}

	switch(control){
#ifdef MPXPLAY_GUI_QT
		case MPXPLAY_INFILE_CBKCTRL_SET_ADDSTREAMFILESTOVIDEO:
			retcode = in_ffmpfile_filelist_files_add_lfcbds(ffmpi, (struct mpxplay_loadfile_callbackdata_s *)arg1);
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_RECORDSTREAMSTOFILE:
			in_ffmpfile_file_output_init(ffmpi, (struct mpxplay_loadfile_callbackdata_s *)arg1);
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_PROGRAMLIST:
			*((struct mpxplay_programinfo_t **)arg1) = inffmpg_programinfolist_fill(ffmpi);
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_STREAMLIST:
			*((struct mpxplay_streaminfo_t **)arg1) = inffmpg_streaminfolist_fill(ffmpi);
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLISAVAILABLE:
			if(in_ffmpgdec_content_has_programs(ffmpi))
				*((mpxp_int32_t *)arg1) = TRUE;
			else
				*((mpxp_int32_t *)arg1) = FALSE;
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLISENABLED:
			*((mpxp_int32_t *)arg1) = ffmpi->videowall_mode_enabled;
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLCURRSTI:
			*((mpxp_int32_t *)arg1) = ffmpi->primary_video_stream_index;
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLCURRSTI:
			in_ffmpstrm_all_programs_videostreams_program_select(ffmpi, (int)arg1);
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_VIDEOWALLCURRSTI_EW:
			in_ffmpstrm_all_programs_videostreams_program_select(ffmpi, (int)arg1);
			in_ffmpstrm_all_programs_videostreams_terminate(ffmpi);
			ffmpi->videowall_mode_enabled = 0;
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_TIMESYNC:
			retcode = inffmpgdec_get_avtimesync_us(ffmpi, arg2, (mpxp_int64_t *)arg1);
			//if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && (ffmpi->avsync_videotimestamp > 0) && ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_AUDIO] && (ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_AUDIO] > (ffmpi->avsync_videotimestamp + 8 * AV_TIME_BASE)))
			//	funcbit_enable(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVRESYNC); // FIXME: bad side effects
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_VIDFRAME_METADATA:
			inffmpg_pictureinfo_metadata_update(ffmpi, (AVDictionary *)arg1);
			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
#endif // MPXPLAY_GUI_QT
		case MPXPLAY_INFILE_CBKCTRL_GET_CODECCTXINIT_FFMPEG:
			retcode = inffmpgdec_codecctx_init(ffmpi, arg2, (mpxplay_packetlist_t **)arg1);
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_CODECCTX_FFMPEG:
			if(arg2 < MPXPLAY_STREAMTYPEINDEX_MAX){
				mpxplay_debugf(MPXPLAY_DEBUG_CALLBACK, "in_ffmpgdec_infile_callback MPXPLAY_INFILE_CBKCTRL_GET_CODECCTX_FFMPEG ffmpi:%8.8X sti:%d", (mpxp_ptrsize_t)ffmpi, arg2);
				pktlist_elem = NULL;
				mpxplay_ffmpgdec_packetqueue_get((mpxp_ptrsize_t)ffmpi, &ffmpi->codecctx_queues[arg2], &pktlist_elem, 0);
				if(pktlist_elem && pktlist_elem->codec_ctx){
					*((mpxplay_packetlist_t **)arg1) = pktlist_elem;
					retcode = MPXPLAY_ERROR_INFILE_OK;
				}
			}
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_PACKETLIST_ELEM:
			if(arg2 < MPXPLAY_STREAMTYPEINDEX_MAX){
				pktlist_elem = NULL;
				retcode = mpxplay_ffmpgdec_packetqueue_get((mpxp_ptrsize_t)ffmpi, &ffmpi->packet_queues[arg2], &pktlist_elem, 0);
				if((arg2 != MPXPLAY_STREAMTYPEINDEX_VIDEO) && pktlist_elem && (pktlist_elem->flags & MPXPLAY_PACKETLISTFLAG_CLEARBUF)){
					AVCodecContext *packet_ctx = (AVCodecContext *)pktlist_elem->codec_ctx;
					if(mpxplay_ffmpstrm_is_avctx_valid(packet_ctx, arg2))
						avcodec_flush_buffers(packet_ctx);  // TODO: right place?
				}
				*((mpxplay_packetlist_t **)arg1) = pktlist_elem;
				retcode = MPXPLAY_ERROR_INFILE_OK;
			}
			break;
		case MPXPLAY_INFILE_CBKCTRL_SET_CODECCTX_CONSOLIDATE:
			if(arg2 < MPXPLAY_STREAMTYPEINDEX_MAX)
				retcode = mpxplay_ffmpgdec_queue_clear_to_elemctx(&ffmpi->codecctx_queues[arg2], (void *)arg1);
			break;
		default:
			if(!arg2)  // other controls need 2 pointer arguments
				goto err_out_cbk;
	}

#ifdef MPXPLAY_GUI_QT
	switch(control)
	{
		case MPXPLAY_INFILE_CBKCTRL_GET_CORRECTEDVTIMESTAMP:
			*((mpxp_int64_t *)arg2) = inffmpg_get_avframe_pts_corrected(ffmpi, MPXPLAY_STREAMTYPEINDEX_VIDEO, ((AVFrame *)arg1));

//			mpxplay_debugf(MPXPLAY_DEBUG_TIMESTAMP,"CBKCTRL_GET_CORRECTEDVTIMESTAMP vs:%llu diff:%lld pts:%lld ppts:%lld pdts:%lld cpn:%d ts:%lld",
//				*((mpxp_int64_t *)arg2), (*((mpxp_int64_t *)arg2) - ffmpi->stream_last_pts[MPXPLAY_STREAMTYPEINDEX_VIDEO]),
//				((AVFrame *)arg1)->pts, ((AVFrame *)arg1)->pkt_pts, ((AVFrame *)arg1)->pkt_dts, ((AVFrame *)arg1)->coded_picture_number,
//				(pds_gettimeu() - ffmpi->avsync_clocktime + ffmpi->avsync_timestamp));

			retcode = MPXPLAY_ERROR_INFILE_OK;
			break;
		case MPXPLAY_INFILE_CBKCTRL_GET_VIDEOWALLPRGEPGLIST:
			*((struct mpxplay_dvb_epgeventlist_t **)arg2) = mpxplay_inffmp_epg_prepare_epginfolist(ffmpi, (int)arg1);
			if(*((struct mpxplay_dvb_epgeventlist_t **)arg2))
				retcode = MPXPLAY_ERROR_INFILE_OK;
			else
				retcode = MPXPLAY_ERROR_INFILE_NODATA;
			break;
	}
#endif // MPXPLAY_GUI_QT

err_out_cbk:
	if(timeout_ms != INFFMPG_MUTEXTIME_NOLOCK)
	{
		PDS_THREADS_MUTEX_UNLOCK(&ffmpi->mutexhnd_ffmpeg);
	}

	return retcode;
}

#endif //MPXPLAY_LINK_INFILE_FF_MPEG
