//**************************************************************************
//*                     This file is part of the                           *
//*                 Mpxplay/MMC - multimedia player.                       *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2020 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: FFMPEG demultiplexer functions

#include "ffmpgdec.h"

#ifdef MPXPLAY_LINK_INFILE_FF_MPEG

#include "mpxinbuf.h"
#ifdef MPXPLAY_GUI_QT
#include <disp_qt/disp_qt.h>
#endif

#ifdef MPXPLAY_GUI_QT
mpxp_int64_t mpxplay_infmpeg_ffmpdmux_seekpreview_seek;
mpxp_bool_t mpxplay_infmpeg_ffmpdmux_seekpreview_reset;
#endif
extern unsigned int playcontrol;

void in_ffmpdmux_seek_process_all(struct ffmpg_demuxer_data_s *ffmpi)
{
	mpxp_int64_t seek_pos = ffmpi->seek_request_pos;
	unsigned int seek_flags = ffmpi->seek_request_flags;

#ifdef INFFMP_USE_DEMUXER_SEEK
	ffmpi->seek_request_pos = INFFMPG_SEEKREQUEST_INVALID_POS;
#endif

	mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"in_ffmpdmux_seek_process_all BEGIN ffmpi:%8.8X fl:%8.8X pos:%lld dur:%lld fs:%d",
			(mpxp_ptrsize_t)ffmpi, seek_flags, seek_pos, ffmpi->fctx->duration, (long)ffmpi->fbfs->filelength(ffmpi->fbds));

	ffmpi->seek_request_retcode = in_ffmpfile_file_seek(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO], seek_pos, seek_flags);
	in_ffmpfile_file_seek(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_AUDIO], seek_pos, seek_flags | AVSEEK_FLAG_ANY);
	in_ffmpfile_file_seek(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_SUBTITLE], seek_pos, seek_flags | AVSEEK_FLAG_ANY);

	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK) && (ffmpi->seek_request_retcode == MPXPLAY_ERROR_OK))
	{
		funcbit_enable(ffmpi->flags, INFFMPG_FLAG_SEEK_CLEAR);
		if(funcbit_test(seek_flags, AVSEEK_FLAG_PAUSE))
			mpxplay_ffmpgdec_packetqueue_clear(&ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO], 0);
	}
	mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"in_ffmpdmux_seek_process_all END ff:%8.8X fl:%8.8X pos:%lld ret:%d", (mpxp_ptrsize_t)ffmpi, seek_flags, seek_pos, ffmpi->seek_request_retcode);
}

#ifdef MPXPLAY_GUI_QT
static void in_ffmpdmux_seekpreview_seek(struct ffmpg_demuxer_data_s *ffmpi)
{
	if(mpxplay_infmpeg_ffmpdmux_seekpreview_seek > 0)
	{
		struct ffmpeg_external_files_info_s *extfilehandler_main = &ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_VIDEO];
		struct ffmpeg_external_files_info_s *extfilehandler_seek = &ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW];
		long seekpreview_seek = mpxplay_infmpeg_ffmpdmux_seekpreview_seek;
		mpxplay_infmpeg_ffmpdmux_seekpreview_seek = INFFMPG_SEEKREQUEST_INVALID_POS;
		if(!extfilehandler_seek->external_file_avctx && extfilehandler_main->external_files_chain && !funcbit_test(extfilehandler_main->control_flags, INFFMPG_FLAG_IS_LIVESTREAM) && (ffmpi->streamtype_limit >= MPXPLAY_STREAMTYPEINDEX_VIDEO))
		{
			int stream_index = in_ffmpfile_filelist_file_add(ffmpi, extfilehandler_main->external_files_chain->external_filename, MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW);
			if(in_ffmpstrm_initialize_one_stream(ffmpi, MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW, stream_index) <= 0)
			{
				in_ffmpfile_filelist_close(extfilehandler_seek);
			}
		}
		if(extfilehandler_seek->external_file_avctx)
		{
			struct mpxpframe_s *frp = (struct mpxpframe_s *)extfilehandler_main->fbds; // dirty hack to match seekframe seekpos with the primary file seekpos
			mpxp_int64_t seek_timepos = (ffmpi->miis->allframes)? (mpxp_int64_t)((float)ffmpi->miis->timemsec * (float)seekpreview_seek / (float)ffmpi->miis->allframes) * (AV_TIME_BASE/MPXPLAY_TIME_BASE) : 0;
			unsigned int seek_flags = (seekpreview_seek <= frp->frameNum)? AVSEEK_FLAG_BACKWARD : 0;

			in_ffmpfile_file_seek(extfilehandler_seek, seek_timepos, seek_flags);

			avio_flush(extfilehandler_seek->external_file_avctx->pb);
			avformat_flush(extfilehandler_seek->external_file_avctx);
			funcbit_enable(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX);
		}
		ffmpi->select_new_stream_number[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW] = INFFMPG_INVALID_STREAM_INDEX;
	}
}

static void in_ffmpdmux_seekpreview_read_packet(struct ffmpg_demuxer_data_s *ffmpi)
{
	struct ffmpeg_external_files_info_s *extfile_info = &ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW];
	int read_retry = INFFMPG_QUEUE_PACKETNUM_FILL_MIN * 16;

	while(extfile_info->external_file_avctx && (--read_retry)
	 && ((ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW].nb_packets < INFFMPG_QUEUE_PACKETNUM_MIN) || funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX)))
	{
		int retval = in_ffmpfile_file_packet_read(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], ffmpi->pkt);
		if(retval < 0)
		{
			mpxplay_debugf(MPXPLAY_DEBUG_DEMUXSEEKP,"in_ffmpdmux_read_packets SEEKPREVIEW r:%d nbs:%3d", retval, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW].nb_packets);
			break;
		}
		if(ffmpi->pkt->stream_index == ffmpi->selected_ff_stream_index[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW])
		{
			unsigned int packet_flags = 0;
			if(funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX))
			{
				funcbit_enable(packet_flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
				funcbit_disable(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX);
				if(!mpxplay_infmpeg_ffmpdmux_seekpreview_reset && (extfile_info->last_seek_dts == ffmpi->pkt->dts))
					break;
				mpxplay_infmpeg_ffmpdmux_seekpreview_reset = FALSE;
				if(ffmpi->pkt->dts >= 0)
					extfile_info->last_seek_dts = ffmpi->pkt->dts;
			}
			funcbit_copy(packet_flags, ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW].control_flags, INFFMPG_FLAGS_DIRECTCOPY_TO_PACKETTYPE);
			mpxplay_ffmpgdec_packetqueue_put(&ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], MPXPLAY_PACKETTYPE_AVPACKET, packet_flags, ffmpi->codec_ctx[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW],
					ffmpi->codecctx_seq_counters[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], ffmpi->selected_avstream[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW], ffmpi->pkt);
			mpxplay_debugf(MPXPLAY_DEBUG_DEMUXSEEKP,"in_ffmpdmux_read_packets SEEKPREVIEW nbs:%3d nbv:%3d st:%d dts:%lld pkt:%8.8X pfl:%8.8X",
									ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets,
									ffmpi->pkt->stream_index, ffmpi->pkt->dts, (mpxp_ptrsize_t)ffmpi->pkt, packet_flags);
			if(funcbit_test(packet_flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF))
				funcbit_enable(ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW].last_pkt->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
		}
		av_packet_unref(ffmpi->pkt);
		//mpxplay_debugf(MPXPLAY_DEBUG_DEMUXSEEKP,"in_ffmpdmux_read_packets SEEKPREVIEW r:%d nbs:%3d", retval, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW].nb_packets);
	}
}
#endif

static mpxp_bool_t inffmp_demux_stream_reconfig(struct ffmpg_demuxer_data_s *ffmpi)
{
	mpxp_bool_t is_video_stream_change;
	if(ffmpi->select_new_program_number >= 0)
	{
		ffmpi->program_number_current = ffmpi->select_new_program_number;
		ffmpi->select_new_program_number = INFFMPG_INVALID_STREAM_INDEX;
		in_ffmpstrm_primary_streams_select(ffmpi);
		//if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_NONREWINDPRGCHG))
		//	funcbit_disable(ffmpi->flags, INFFMPG_FLAG_NONREWINDPRGCHG); // TODO: this causes slower program change, buffer rewind keeped
		//else
			funcbit_enable(ffmpi->flags, INFFMPG_FLAG_DEMUX_BUFREWIND);
		in_ffmpfile_filelist_close(&ffmpi->external_file_infos[MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW]);
		if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_NOPRGIDCHGSEND)) // context menu based or auto program change
		{
			funcbit_enable(ffmpi->flags, INFFMPG_FLAG_PROGIDCHGSEND);
		}
		else // playlist (dtv drive) based program change
		{
			funcbit_disable(ffmpi->flags, INFFMPG_FLAG_NOPRGIDCHGSEND); // we send the context menu or auto program changes after the playlist (dtv drive) based one
		}
		mpxplay_debugf(MPXPLAY_DEBUG_STRMCHG, "select_new_program_number %d", ffmpi->program_number_current);
	}
	else
	{
		in_ffmpstrm_primary_streams_check(ffmpi);
	}

	is_video_stream_change = in_ffmpstrm_primary_streams_initialize(ffmpi, FALSE);

//	AVStream *video_stream = ffmpi->selected_avstream[MPXPLAY_STREAMTYPEINDEX_VIDEO];
//	if(video_stream && (!video_stream->avg_frame_rate.num || !video_stream->avg_frame_rate.den))
//	{
//		video_stream->codec_info_nb_frames++;
//		video_stream->avg_frame_rate = av_guess_frame_rate(ffmpi->fctx, video_stream, NULL);
//	}

	return is_video_stream_change;
}

static int inffmp_demux_read_frame(struct ffmpg_demuxer_data_s *ffmpi)
{
	int retcode;
#ifdef INFFMPG_USE_MPXPLAY_IOCTX
	struct mpxpframe_s *frp = (struct mpxpframe_s *)ffmpi->fbds;
	unsigned int buffertype_save;
	long pbbf_save;
	AVIOContext *pb;

	if(!frp || !ffmpi->fctx)
		return AVERROR(EAGAIN);
	pb = ffmpi->fctx->pb;
	if(!pb)
		return AVERROR(EAGAIN);

	buffertype_save = frp->buffertype;
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD)) // file and DTV (else http audio)
	{
		funcbit_disable(frp->buffertype, PREBUFTYPE_INT);     // allow file read, out of prebuffer
	}
	else if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && funcbit_test(frp->buffertype, PREBUFTYPE_INT) && (frp->prebufferbytes_forward <= frp->prebufferblocksize)) // TODO: check
	{
		return AVERROR(EAGAIN);
	}

	pbbf_save = frp->prebufferbytes_forward;
#endif

	av_packet_unref(ffmpi->pkt);
	//mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"av_read_frame BEGIN frp:%8.8X br:%d bf:%d", frp, frp->prebufferbytes_rewind, frp->prebufferbytes_forward);
	retcode = av_read_frame(ffmpi->fctx, ffmpi->pkt);

#ifdef INFFMPG_USE_MPXPLAY_IOCTX
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD))
	{
		funcbit_smp_copy(frp->buffertype, buffertype_save, PREBUFTYPE_INT);
	}

	if(retcode < 0)
	{
		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && (((pbbf_save >= (frp->prebufferblocksize * 8)) && (pbbf_save == frp->prebufferbytes_forward)) || (frp->prebufferbytes_forward >= (frp->prebuffersize - frp->prebufferblocksize)))) // TODO: check
		{
			if(++ffmpi->errorcount_demux > (INFFMPG_MAX_FRAME_MISMACTHES * 4))
			{
				if(!frp->mvp->adone)
				{
					frp->mvp->adone = ADONE_REOPEN; // FIXME: ffmpeg (mkv) pipe error cannot be recovered on other way?
					mpxplay_debugf(MPXPLAY_DEBUG_ERROR,"inffmp_demux_read_frame REOPEN! ro:%d", frp->mvp->adone);
				}
			}
		}
		else
		{
			if(ffmpi->errorcount_demux > (INFFMPG_MAX_FRAME_MISMACTHES * 4))
			{
				frp->mvp->adone = 0;
			}
			ffmpi->errorcount_demux = 0;
		}
		mpxplay_debugf(MPXPLAY_DEBUG_WARNING,"av_read_frame r:%d s:%4d id:%2d br:%7d bf:%7d bs:%d bls:%d nba:%2d nbv:%d", retcode,
					((ffmpi->pkt)? ffmpi->pkt->size : -1), ((ffmpi->pkt)? ffmpi->pkt->stream_index : -1),
					frp->prebufferbytes_rewind, frp->prebufferbytes_forward, frp->prebuffersize, frp->prebufferblocksize,
					ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets);
	}
#endif // INFFMPG_USE_MPXPLAY_IOCTX
	return retcode;
}

void in_ffmpdmux_read_packets(struct ffmpg_demuxer_data_s *ffmpi)
{
	int i, retry, max_retry, first, fill_max, stt_limit, stremtype_demux_select, demux_retcode = AVERROR(EAGAIN), nb_queue_packets_start[MPXPLAY_STREAMTYPEINDEX_PLAYNUM];
	mpxp_bool_t is_video_stream_change;
	struct mpxpframe_s *frp;
#if defined(INFFMP_USE_DEMUXER_THREAD) && defined(MPXINBUF_USE_FILLBUF_MUTEX)
	int lock_fillbuf_result = -1;
#endif
#if INFFMP_USE_DEMUXER_MUTEX
	int lock_result = -1;
#endif

	if(!ffmpi || (ffmpi->struct_id != MPXPLAY_FFMPEGDEMUX_STRUCT_ID))
		return;

	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
	{
		demux_retcode = AVERROR(ENOFILE);
		goto err_out_packetsread;
	}

#ifdef MPXPLAY_GUI_QT
	in_ffmpdmux_seekpreview_seek(ffmpi);
	in_ffmpdmux_seekpreview_read_packet(ffmpi);
#endif

	if(!ffmpi->fctx || !ffmpi->fctx->pb)
	{
		demux_retcode = AVERROR(EBADF);
		goto err_out_packetsread;
	}
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
	{
		demux_retcode = AVERROR(ENOFILE);
		goto err_out_packetsread;
	}

	frp = (struct mpxpframe_s *)ffmpi->fbds;
	if(!frp)
		return;

#if defined(INFFMP_USE_DEMUXER_THREAD) && defined(MPXINBUF_USE_FILLBUF_MUTEX)
	mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"in_ffmpdmux_read_packets START %8.8X pbr:%7d pbf:%7d nba:%2d nbv:%d dt:%d is:%d", (unsigned int)frp,
				frp->prebufferbytes_rewind, frp->prebufferbytes_forward,
				ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets,
				((funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD))? 1 : 0), ((funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK))? 1 : 0));
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD) && !funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK))
	{
		lock_fillbuf_result = PDS_THREADS_MUTEX_LOCK(&frp->mutexhnd_fillbuffer, INFFMPG_MUTEX_TIMEOUT);
		mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"in_ffmpdmux_read_packets mutexhnd_fillbuffer LOCK end frp:%8.8X mtx:%8.8X res:%d", (unsigned int)frp, frp->mutexhnd_fillbuffer, lock_fillbuf_result);
	}
#endif // defined(INFFMP_USE_DEMUXER_THREAD) && defined(MPXINBUF_USE_FILLBUF_MUTEX)

#ifdef INFFMP_USE_DEMUXER_MUTEX
	mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"in_ffmpdmux_read_packets LOCK begin");
	lock_result = PDS_THREADS_MUTEX_LOCK(&ffmpi->mutexhnd_demuxer, INFFMPG_MUTEX_TIMEOUT);
	mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"in_ffmpdmux_read_packets LOCK end res:%d", lock_result);
#endif

#ifdef INFFMP_USE_DEMUXER_SEEK
	if(ffmpi->seek_request_pos != INFFMPG_SEEKREQUEST_INVALID_POS)
	{
		in_ffmpdmux_seek_process_all(ffmpi);
	}
#endif

	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_SEEK_CLEAR))
	{
		for(i = 0; i <= ffmpi->streamtype_limit; i++)
		{
			struct ffmpeg_external_files_info_s *extfilehandler = &ffmpi->external_file_infos[i];
			if(!funcbit_test(extfilehandler->control_flags, INFFMPG_FLAG_IS_LIVESTREAM))//) || !funcbit_test(ffmpi->flags, INFFMPG_FLAG_INDEPENDENTSTREAMSYNC))
			{
				AVFormatContext *fctx = ffmpi->external_file_infos[i].external_file_avctx;
				if(fctx)
				{
					avio_flush(fctx->pb);
					avformat_flush(fctx);
				}
				funcbit_enable(ffmpi->stream_flags[i], (INFFMPG_STREAMFLAG_CLEARBUF_DEMUX|INFFMPG_STREAMFLAG_SEEK));
			}
		}
		for(i = MPXPLAY_STREAMTYPEINDEX_VIDEOWALL; i < MPXPLAY_STREAMTYPEINDEX_MAX; i++)
		{
			funcbit_enable(ffmpi->stream_flags[i], (INFFMPG_STREAMFLAG_CLEARBUF_DEMUX|INFFMPG_STREAMFLAG_SEEK));
		}
		stremtype_demux_select = MPXPLAY_STREAMTYPEINDEX_SUBTITLE; // to read first external streams after seeking
		funcbit_disable(ffmpi->flags, INFFMPG_FLAG_SEEK_CLEAR);
	}

	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD))
		if(PDS_THREADS_MUTEX_LOCK(&ffmpi->mutexhnd_ffmpeg, 100) != MPXPLAY_ERROR_OK)
			goto err_out_packetsread;

	is_video_stream_change = inffmp_demux_stream_reconfig(ffmpi);

	if(funcbit_test(playcontrol, PLAYC_RUNNING) && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets >= (INFFMPG_QUEUE_PACKETNUM_MAX - INFFMPG_QUEUE_PACKETNUM_FILL_MIN)))
	{
		if( (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets > 0) && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets < INFFMPG_QUEUE_PACKETNUM_FILL_MIN)
		 && !funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], (INFFMPG_STREAMFLAG_AVSYNC | INFFMPG_STREAMFLAG_AVRESYNC))
		 && ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].last_pkt
		){
			mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC,"DEMUX BUF FULL a:%d v:%d afl:%8.8X", ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets,
					ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets, ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO]);
			funcbit_enable(ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].last_pkt->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
			funcbit_enable(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_AVRESYNC);
		}
	}

	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD))
	{
		PDS_THREADS_MUTEX_UNLOCK(&ffmpi->mutexhnd_ffmpeg);
#ifdef MPXPLAY_GUI_QT
		if(is_video_stream_change && funcbit_test(ffmpi->flags, INFFMPG_FLAG_CALLBACKVIDEOCFG))
		{
			mpxplay_dispqt_ffmpegvideo_config_callback(MPXPLAY_INFILE_CBKCFG_SRVFFMV_REFRESH_CALLBACK, (mpxp_ptrsize_t)in_ffmpgdec_infile_callback, (mpxp_ptrsize_t)ffmpi);
		}
#endif
	}

	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM) && funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_BUFREWIND) && infmpeg_is_fileformat_video_stream(ffmpi->fctx) && ffmpi->fbfs && ffmpi->fbfs->fseek)
	{
		long seek_back_size = min(frp->prebufferbytes_rewind, (frp->prebuffersize /  4)); // TODO: seek back size? (and hack of frp)
		avio_flush(ffmpi->fctx->pb);
		avformat_flush(ffmpi->fctx);
		if((seek_back_size > 0) && (frp->prebufferbytes_forward < (frp->prebuffersize / 2)))
		{
			if(ffmpi->fbfs->fseek((void *)frp, -seek_back_size, SEEK_CUR) < 0)
			{
				mpxplay_debugf(MPXPLAY_DEBUG_ERROR, "in_ffmpdmux_read_packets SEEK BACK FAILED br:%d bf:%d bs:%d back:%d", frp->prebufferbytes_rewind, frp->prebufferbytes_forward, frp->prebuffersize, seek_back_size);
			}
			else
			{
				mpxplay_debugf(MPXPLAY_DEBUG_PRGCHG,"in_ffmpdmux_read_packets SEEK BACK br:%d bf:%d bs:%d", frp->prebufferbytes_rewind, frp->prebufferbytes_forward, frp->prebuffersize);
			}
		}
		else
		{
			mpxplay_debugf(MPXPLAY_DEBUG_PRGCHG,"in_ffmpdmux_read_packets SEEK BACK NONE br:%d bf:%d bs:%d", frp->prebufferbytes_rewind, frp->prebufferbytes_forward, frp->prebuffersize);
		}
	}
	funcbit_disable(ffmpi->flags, INFFMPG_FLAG_DEMUX_BUFREWIND);

#ifdef INFFMP_USE_DEMUXER_BUFILL
	if(!funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK))
		frp->mvp->fdone = mpxplay_mpxinbuf_buffer_check(frp, FALSE);  // FIXME: hack
#endif

	nb_queue_packets_start[MPXPLAY_STREAMTYPEINDEX_AUDIO] = ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets;
	nb_queue_packets_start[MPXPLAY_STREAMTYPEINDEX_VIDEO] = ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets;

	stt_limit = min(ffmpi->streamtype_limit, MPXPLAY_STREAMTYPEINDEX_VIDEO);
	fill_max = (funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD))? ((funcbit_test(ffmpi->flags,INFFMPG_FLAG_IS_LIVESTREAM) || !funcbit_test(playcontrol, PLAYC_RUNNING))? INFFMPG_QUEUE_PACKETNUM_FILL_MIN : INFFMPG_QUEUE_PACKETNUM_FILL_MAX) : INFFMPG_QUEUE_PACKETNUM_MIN;
	max_retry = (ffmpi->fctx->nb_streams + 2) * INFFMPG_QUEUE_PACKETNUM_FILL_MIN * 3;
	stremtype_demux_select = 0;
	retry = first = 1;
	demux_retcode = 0;

	while(retry && (--max_retry)
		&& (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets < INFFMPG_QUEUE_PACKETNUM_MAX)
		&& (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets < INFFMPG_QUEUE_PACKETNUM_MAX)
	){
		mpxp_bool_t have_codex_ctxs = FALSE;
		int ff_stream_index;
		AVFormatContext *avformctx;

		if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_TERMINATE))
		{
			demux_retcode = AVERROR(ENOFILE);
			goto err_out_packetsread;
		}

		retry = 0;
		for(i = 0; i <= stt_limit; i++)
		{
			if(ffmpi->codec_ctx[i] && avcodec_is_open(ffmpi->codec_ctx[i]))
			{
				if(funcbit_test(ffmpi->stream_flags[i], INFFMPG_STREAMFLAG_SEEK) ||
					(   (ffmpi->packet_queues[i].nb_packets < fill_max)
					 && ((ffmpi->packet_queues[i].nb_packets < INFFMPG_QUEUE_PACKETNUM_FILL_MIN) || ((ffmpi->packet_queues[i].nb_packets - nb_queue_packets_start[i]) < INFFMPG_QUEUE_PACKETNUM_FILL_STEP))
					)
				){
					retry++;
				}
				have_codex_ctxs = TRUE;
			}
		}
		if(!retry && have_codex_ctxs)
			break;

		demux_retcode = -1;

		//for(stremtype_demux_select = MPXPLAY_STREAMTYPEINDEX_AUDIO; stremtype_demux_select < MPXPLAY_STREAMTYPEINDEX_PLAYNUM; stremtype_demux_select++)
		{
			if(stremtype_demux_select != MPXPLAY_STREAMTYPEINDEX_VIDEO) // TODO: move primary video file demux (inffmp_demux_read_frame) into common structure
			{
				int fill_num_ext = ((stremtype_demux_select == MPXPLAY_STREAMTYPEINDEX_SUBTITLE) && !funcbit_test(ffmpi->stream_flags[stremtype_demux_select], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX))? INFFMPG_QUEUE_PACKETNUM_MIN : fill_max;
				if( ffmpi->external_file_infos[stremtype_demux_select].external_file_avctx
					&& (ffmpi->selected_stream_number[stremtype_demux_select] >= INFFMPG_STREAM_INDEX_EXTERNALFILE_BEGIN)
					&& (ffmpi->packet_queues[stremtype_demux_select].nb_packets < fill_num_ext)
				){
					demux_retcode = in_ffmpfile_file_packet_read(&ffmpi->external_file_infos[stremtype_demux_select], ffmpi->pkt);
					if(demux_retcode == 0)
					{
						avformctx = ffmpi->external_file_infos[stremtype_demux_select].external_file_avctx;
						//break;
					}
				}
			}
		}
		if(++stremtype_demux_select >= MPXPLAY_STREAMTYPEINDEX_PLAYNUM)
			stremtype_demux_select = 0;

		if(demux_retcode != 0)
		{
			demux_retcode = inffmp_demux_read_frame(ffmpi);
			avformctx = ffmpi->fctx;
		}

		if((demux_retcode != 0) || !ffmpi->pkt)
		{
			mpxplay_debugf(MPXPLAY_DEBUG_WARNING, "in_ffmpdmux_read_packets ret:%d", demux_retcode);
			if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK))
				continue;
			break;
		}

		ff_stream_index = ffmpi->pkt->stream_index;
		if(ffmpi->pkt->data && (ffmpi->pkt->size > 0) && (ff_stream_index >= 0) && (ff_stream_index < avformctx->nb_streams))
		{
			AVStream *avstrm = avformctx->streams[ff_stream_index];
			enum AVMediaType pkt_mediatype = (avstrm && avstrm->codecpar)? avstrm->codecpar->codec_type : AVMEDIA_TYPE_UNKNOWN;
			int streamindex_max = (ffmpi->videowall_mode_enabled)? (MPXPLAY_STREAMTYPEINDEX_MAX - 1) : ffmpi->streamtype_limit;

			in_ffmpfile_file_output_writeframe(ffmpi, avformctx, ffmpi->pkt);

			for(i = 0; i <= streamindex_max; i++){
				if(i == MPXPLAY_STREAMTYPEINDEX_SEEKPREVIEW)
					continue;
				if(ffmpi->selected_stream_number[i] < INFFMPG_STREAM_INDEX_VALID)
					continue;
				if(pkt_mediatype != in_ffmpstrm_streamtypeindex_to_avmediatype(i))
					continue;
				if((avformctx != ffmpi->fctx) && ((i >= MPXPLAY_STREAMTYPEINDEX_PLAYNUM) || (avformctx != ffmpi->external_file_infos[i].external_file_avctx)))
					continue;
				//if(!avcodec_is_open(ffmpi->codec_ctx[i])) // FIXME: !!!
				//	continue;
				if(ff_stream_index == ffmpi->selected_ff_stream_index[i]){
					unsigned int packet_flags = (ffmpi->pkt->flags & AV_PKT_FLAG_KEY)? MPXPLAY_PACKETLISTFLAG_KEYFRAME : 0;
					if(funcbit_test(ffmpi->stream_flags[i], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX)){
						funcbit_enable(packet_flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF);
						if(ffmpi->codecctx_queues[i].last_pkt
						 && ((i == MPXPLAY_STREAMTYPEINDEX_AUDIO)
							|| ((i == MPXPLAY_STREAMTYPEINDEX_VIDEO)
							   && (!funcbit_test(playcontrol, PLAYC_RUNNING)
							      || (ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].last_pkt
							      && !funcbit_test(ffmpi->codecctx_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].last_pkt->flags, packet_flags)
							      && !funcbit_test(ffmpi->stream_flags[MPXPLAY_STREAMTYPEINDEX_AUDIO], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX)
							)  )  )  )
						){
							funcbit_enable(ffmpi->codecctx_queues[i].last_pkt->flags, packet_flags);
						}
						funcbit_disable(ffmpi->stream_flags[i], (INFFMPG_STREAMFLAG_CLEARBUF_DEMUX|INFFMPG_STREAMFLAG_SEEK));
					}
					if((i < MPXPLAY_STREAMTYPEINDEX_PLAYNUM))
					{
						funcbit_copy(packet_flags, ((ffmpi->selected_stream_number[i] >= INFFMPG_STREAM_INDEX_EXTERNALFILE_BEGIN)? ffmpi->external_file_infos[i].control_flags : ffmpi->flags), INFFMPG_FLAGS_DIRECTCOPY_TO_PACKETTYPE);
						ffmpi->nb_frame_mismatches[i] = 0;
					}
					else
					{
						funcbit_copy(packet_flags, ffmpi->flags, INFFMPG_FLAGS_DIRECTCOPY_TO_PACKETTYPE);
					}
					mpxplay_ffmpgdec_packetqueue_put(&ffmpi->packet_queues[i], MPXPLAY_PACKETTYPE_AVPACKET, packet_flags, ffmpi->codec_ctx[i], ffmpi->codecctx_seq_counters[i], ffmpi->selected_avstream[i], ffmpi->pkt);
#ifdef MPXPLAY_USE_DEBUGF
					if(i == MPXPLAY_STREAMTYPEINDEX_SUBTITLE)
					{
						mpxplay_debugf(MPXPLAY_DEBUG_DEMUX, "in_ffmpdmux_read_packets pf:%8.8X pts:%lld \"%s\"", packet_flags, ffmpi->pkt->dts, ffmpi->pkt->data);
					}
					//if((i >= MPXPLAY_STREAMTYPEINDEX_VIDEO))// && (ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets < fill_max))
					{
						struct mpxpframe_s *frp = (struct mpxpframe_s *)ffmpi->fbds;
						mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"in_ffmpdmux_read_packets PUTPACK i:%d frp:%8.8X nba:%3d nbv:%3d fm:%d br:%6d bf:%5d pts:%lld f:%8.8X", i, (mpxp_ptrsize_t)frp,
								 ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets,
								 fill_max, frp->prebufferbytes_rewind, frp->prebufferbytes_forward, ffmpi->pkt->pts, ffmpi->pkt->flags);
					}
#endif
					if(!ffmpi->videowall_mode_enabled || (i != MPXPLAY_STREAMTYPEINDEX_VIDEO)) // primary packet have to be duplicated in videowall mode
					{
						break;
					}
				}
				else if( (i < MPXPLAY_STREAMTYPEINDEX_PLAYNUM)
					  && funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_STREAM)
					  && ((++ffmpi->nb_frame_mismatches[i]) > ((avformctx->nb_programs + 1) * ((funcbit_test(ffmpi->flags, INFFMPG_FLAG_IS_LIVESTREAM)? (INFFMPG_MAX_FRAME_MISMACTHES * 2) : INFFMPG_MAX_FRAME_MISMACTHES)))) // this seems to be better for single TS with stream changes
//				else if(!(ffmpi->flags & INFFMPG_FLAG_INITIAL_SEEK) && (ffmpi->flags & INFFMPG_FLAG_IS_STREAM) && ((++ffmpi->nb_frame_mismatches[i]) > (INFFMPG_MAX_FRAME_MISMACTHES * (avformctx->nb_programs + 1)))
					  && (ffmpi->select_new_stream_number[i] != INFFMPG_STREAM_INDEX_SEARCH)
					  && (ffmpi->selected_stream_number[i] < INFFMPG_STREAM_INDEX_EXTERNALFILE_BEGIN) // we change/drop automatically only internal streams
				){
					ffmpi->select_new_stream_number[i] = INFFMPG_STREAM_INDEX_LOST; // change stream if the current lost
					ffmpi->nb_frame_mismatches[i] = 0;
					retry = 0;
					if(i == MPXPLAY_STREAMTYPEINDEX_AUDIO)
					{
						mpxplay_debugf(MPXPLAY_DEBUG_WARNING,"in_ffmpdmux_read_packets stream lost!");
					}
				}
			}
			// change program if the current lost
			if( (i < MPXPLAY_STREAMTYPEINDEX_PLAYNUM)
			 && ((ffmpi->selected_stream_number[MPXPLAY_STREAMTYPEINDEX_AUDIO] < INFFMPG_STREAM_INDEX_VALID) || (ffmpi->select_new_stream_number[MPXPLAY_STREAMTYPEINDEX_AUDIO] == INFFMPG_STREAM_INDEX_LOST))
			 && ((ffmpi->selected_stream_number[MPXPLAY_STREAMTYPEINDEX_VIDEO] < INFFMPG_STREAM_INDEX_VALID) || (ffmpi->select_new_stream_number[MPXPLAY_STREAMTYPEINDEX_VIDEO] == INFFMPG_STREAM_INDEX_LOST))
			){
				in_ffmpstrm_skip_program_auto(ffmpi, ffmpi->program_number_current);
				retry = 0;
			}
		}
		av_packet_unref(ffmpi->pkt);
		first = 0;
	}
#ifdef MPXPLAY_USE_DEBUGF
	if(!first && (demux_retcode != 0 || !max_retry))
	{
		mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"in_ffmpdmux_read_packets %8.8X rd:%d mr:%d nba:%d nbv:%d da:%d dv:%d",
			(mpxp_ptrsize_t)ffmpi, demux_retcode, max_retry,
			ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets, ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets,
			ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_AUDIO].nb_packets - nb_queue_packets_start[MPXPLAY_STREAMTYPEINDEX_AUDIO],
			ffmpi->packet_queues[MPXPLAY_STREAMTYPEINDEX_VIDEO].nb_packets - nb_queue_packets_start[MPXPLAY_STREAMTYPEINDEX_VIDEO]);
	}
#endif

	// send stream metadata infos to the upper layer
	for(i = 0; i <= stt_limit; i++)
	{
		if(ffmpi->selected_avstream[i] && funcbit_test(ffmpi->selected_avstream[i]->event_flags, AVSTREAM_EVENT_FLAG_METADATA_UPDATED))
		{
			inffmpg_tags_update(ffmpi->miis, ffmpi->fbds, ffmpi->selected_avstream[i]->metadata, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_UPDATE);
			funcbit_disable(ffmpi->selected_avstream[i]->event_flags, AVSTREAM_EVENT_FLAG_METADATA_UPDATED);
		}
	}
	if(funcbit_test(ffmpi->fctx->event_flags, AVFMT_EVENT_FLAG_METADATA_UPDATED))
	{
		inffmpg_tags_update(ffmpi->miis, ffmpi->fbds, ffmpi->fctx->metadata, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_TAG_UPDATE);
		funcbit_disable(ffmpi->fctx->event_flags, AVFMT_EVENT_FLAG_METADATA_UPDATED);
	}

	// send (DVB) program change to the upper layer
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_PROGIDCHGSEND) && !funcbit_test(ffmpi->flags, INFFMPG_FLAG_INITIAL_SEEK))
	{
		if(ffmpi->miis && ffmpi->miis->control_cb)
		{
			mpxp_int32_t prog_id = ffmpi->program_number_current;
			ffmpi->miis->control_cb(ffmpi->fbds, MPXPLAY_CFGFUNCNUM_INFILE_ENTRY_PROGID_SET, (void *)&prog_id, NULL); // set the program change on the dtv drive too
		}
		funcbit_disable(ffmpi->flags, INFFMPG_FLAG_PROGIDCHGSEND);
	}

	// send an empty packet with (clear) flags to the decoder, if it's not sent above (no incoming packet read at demuxing after seek for a stream)
	for(i = 0; i <= ffmpi->streamtype_limit; i++)
	{
		if(funcbit_test(ffmpi->stream_flags[i],INFFMPG_STREAMFLAG_SEEK) && funcbit_test(ffmpi->stream_flags[i], INFFMPG_STREAMFLAG_CLEARBUF_DEMUX))
		{
			unsigned int packet_flags = MPXPLAY_PACKETLISTFLAG_CLEARBUF;
			funcbit_copy(packet_flags, ((ffmpi->selected_stream_number[i] >= INFFMPG_STREAM_INDEX_EXTERNALFILE_BEGIN)? ffmpi->external_file_infos[i].control_flags : ffmpi->flags), INFFMPG_FLAGS_DIRECTCOPY_TO_PACKETTYPE);
			mpxplay_ffmpgdec_packetqueue_put(&ffmpi->packet_queues[i], MPXPLAY_PACKETTYPE_AVPACKET, packet_flags, ffmpi->codec_ctx[i], ffmpi->codecctx_seq_counters[i], ffmpi->selected_avstream[i], NULL);
			if((i == MPXPLAY_STREAMTYPEINDEX_AUDIO) && ffmpi->codecctx_queues[i].last_pkt){
				funcbit_enable(ffmpi->codecctx_queues[i].last_pkt->flags, MPXPLAY_PACKETLISTFLAG_CLEARBUF); // FIXME: not thread safe
				mpxplay_debugf(MPXPLAY_DEBUG_AVRESYNC, "SEND CLEARBUF packet %d f:%d", i, ffmpi->packet_queues[i].last_pkt->flags);
			}
		}
	}

	// clear seek flags
	for(i = 0; i < MPXPLAY_STREAMTYPEINDEX_MAX; i++)
	{
		funcbit_disable(ffmpi->stream_flags[i], INFFMPG_STREAMFLAG_SEEK);
	}

err_out_packetsread:
	ffmpi->retcode_demux = demux_retcode;
	if(funcbit_test(ffmpi->flags, INFFMPG_FLAG_DEMUX_THREAD) && funcbit_test(playcontrol, PLAYC_RUNNING))
	{
		ffmpi->fbds = NULL;
	}
#if defined(INFFMP_USE_DEMUXER_THREAD) && defined(MPXINBUF_USE_FILLBUF_MUTEX)
	if(lock_fillbuf_result == MPXPLAY_ERROR_OK)
	{
		PDS_THREADS_MUTEX_UNLOCK(&frp->mutexhnd_fillbuffer);
	}
	mpxplay_debugf(MPXPLAY_DEBUG_DEMUX,"in_ffmpdmux_read_packets mutexhnd_fillbuffer UNLOCK %8.8X res:%d pl:%d", (unsigned int)frp, lock_fillbuf_result, ((playcontrol&PLAYC_RUNNING)? 1 : 0));
#endif
#ifdef INFFMP_USE_DEMUXER_MUTEX
	if(lock_result == MPXPLAY_ERROR_OK)
	{
		mpxplay_debugf(MPXPLAY_DEBUG_SEEK,"in_ffmpdmux_read_packets UNLOCK");
		PDS_THREADS_MUTEX_UNLOCK(&ffmpi->mutexhnd_demuxer);
	}
#endif
}

#endif //MPXPLAY_LINK_INFILE_FF_MPEG
