; E100PKT, packet driver for DOS
; Copyright (C) 2018, Seth Simon (sethsimon@sdf.org)
;
; This program is free software: you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation, either version 3 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program.  If not, see
; <https://www.gnu.org/licenses/>.

;==========================
;=======            =======
;       GET_E100_INT
;=======            =======
;==========================
get_e100_int:
    ; Returns AL = int_no, or ZF=0 if not found.
    mov     ax, 0x3560
.loop:
    mov     cx, tsr.sig_end - tsr.sig
    push    es
    int     0x21            ; ES:BX := handler
    mov     si, tsr.sig
    mov     di, si
    repe    cmpsb
    pop     es

    jz      .done
    inc     al
    jns     .loop
.done:
    ret

;========================
;=======          =======
;       INT_IN_USE
;=======          =======
;========================
int_in_use:
    ; AL = interrupt #
    ; Returns ZF=1 if the vector is free, ZF=0 if not free
    push    es

    mov     ah, 0x35
    int     0x21
    mov     ax, es
    add     ax, bx

    pop     es
    ret

;========================
;=======          =======
;       SET_VECTOR
;=======          =======
;========================
set_vector:
    ; AL = int_no
    ; DS:DX = entry point of new vector
    ; Returns DX:AX = address of old vector
    ; Does not modify CX
    push    es

    mov     ah, 0x35        ; Get interrupt vector
    int     0x21            ; ES:BX := old vector

    mov     ah, 0x25        ; Set vector AL to DS:DX
    int     0x21

    mov     dx, es
    xchg    ax, bx
    pop     es
    ret

;=============================
;=======               =======
;       GET_IRQ_OR_QUIT
;=======               =======
;=============================
get_irq_or_quit:
    mov     bl, 0x3c    ; Interrupt Line Register
    call    get_pci_var
    mov     dx, .irq_is
    call    print_str_byte

    cmp     al, 0xf
    mov     dx, .bad_irq_str
    jnbe    print_message_and_quit
    mov     di, pic_port

    ; if(irq < 8) { intvec = irq +   8 = irq - 0xf8, pic_port = 0x21 }
    ; else        { intvec = irq + 104 = irq - 0x98, pic_port = 0xa1 }
    mov     cx, 0xa1
    cmp     al, 8       ; lt 8 => CF=1      gte 8 => CF=0
    sbb     bx, bx      ; lt 8 => -1        gte 8 => 0
    and     bx, 0x80a0
    sub     cl, bh
    sub     bl, 0x98    ; -0x98 == +0x68
    add     bl, al

    xchg    ax, cx              ; AX := pic_port, CL := irq
    stosw
    xchg    ax, bx              ; AL := irq_int, BX := pic_port
    stosb

    mov     dx, .aka_int
    call    print_str_byte

    ; Calculate a mask for the PIC == 1 << (irq & 7)
    and     cl, 7
    mov     al, 1
    shl     al, cl
    stosb
    xchg    ax, cx  ; CL := irq mask

    ; If the IRQ is unmasked, abort since I don't support IRQ sharing.
    mov     dx, bx          ; PIC_PORT
    in      al, dx
    and     al, cl
    mov     dx, .not_masked_str
    jz      print_message_and_quit
    jmp     crlf

.irq_is: db "The BIOS configured the adapter to use IRQ $"
.bad_irq_str: db `.\r\n\r\nError! Invalid IRQ.$`
.aka_int: db ", which is INT $"
.not_masked_str: db `\r\n\r\nError! The IRQ is unmasked, but I `
                 db `don't support interrupt sharing.$`

