////////////////////////////////////////////////////////////////////////////////
//
//	File Management System - Core file
//
//	(c) Copyright 2003,2004 Point Mad, Lukas Lipka. All rights reserved.
//
////////////////////////////////////////////////////////////////////////////////
#include "kernel.h"
#include "fms.h"
#include "registry.h"
#include "iodlg.h"

l_ulong AppVersion = ULONG_ID(0,0,0,1);
char    AppName[] = "File Management System";
l_uid	nUID = "fms";
l_uid NeededLibs[] = { "menu", "" };

PList FileManagers;
PFileMan Default;

p_bitmap ICONDir32;
p_bitmap ICONDir16;
p_bitmap ICONDrv32;
p_bitmap ICONDrv16;
p_bitmap ICONFile32;
p_bitmap ICONFile16;
p_bitmap ICONApp32;
p_bitmap ICONApp16;
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
PFileInfo LinkGetLinkedFile ( PFileInfo f ) {
	l_text filename = LoadData(TYPE_TEXT,f->VPathName);
	PFileInfo o = NULL;
	if ( filename ) {
		o = FileGetInfo(filename);
		free(filename);
	}
	return o;
}
////////////////////////////////////////////////////////////////////////////////
l_bool LinkRun ( PFileInfo f, l_text ex ) {
	PFileInfo o = LinkGetLinkedFile(f);
	l_bool r = false;
	if ( o ) {
		r = FileRunEx(o,ex);
		FreeFileInfo(o);
	}
	return r;
}
////////////////////////////////////////////////////////////////////////////////
p_bitmap LinkGetIcon ( PFileInfo f, l_ulong Size, l_ulong *ToFree ) {
	PFileInfo o = LinkGetLinkedFile(f);
	p_bitmap i = NULL;
	if ( o ) {
		i = FileGetIconEx(o,Size,ToFree);
		FreeFileInfo(o);
	}
	return i;
}
////////////////////////////////////////////////////////////////////////////////
void LinkGetAllThings ( PFileInfo f, l_text name, l_text*n, p_bitmap*i16, p_bitmap*i32, p_bitmap*i48, l_ulong *fi ) {
	PFileInfo o = LinkGetLinkedFile(f);
	if ( n ) {
		if ( TextRChr(f->Name,'.') )
			*n = TextNDup(f->Name,TextRChr(f->Name,'.')-f->Name);
		else
			*n = TextDup(f->Name);
	}
	if ( o ) {
		FileGetAllThingsEx(o, NULL, i16, i32, i48, fi);
		FreeFileInfo(o);
	}
}
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

p_bitmap __FileGetIconEx ( PFileInfo f, l_ulong Size, l_ulong *ToFree ) {

	if ( f->Type == FT_HDD ) {
		if ( Size == 16 ) return ICONDrv16;
		if ( Size == 32 ) return ICONDrv32;
	} else if ( f->Type == FT_FOLDER ) {
		if ( Size == 16 ) return ICONDir16;
		if ( Size == 32 ) return ICONDir32;
	} else {
		if ( Size == 16 ) return ICONFile16;
		if ( Size == 32 ) return ICONFile32;
	}

	return NULL;
}
////////////////////////////////////////////////////////////////////////////////
PFileMan _FileGetFileman ( l_text file ) {
	l_text ext = TextRChr(file,'.');
	if ( ext ) {
		PFileMan m = ListKeyCase(FileManagers,ext+1);
		if ( !m ) m = Default;
		return m;
	} else
		return Default;
}
////////////////////////////////////////////////////////////////////////////////
PFileMan FileGetFileman ( PFileInfo f ) {
	
	if ( f->FMS ) return (PFileMan)f->FMS;
	if ( f->Attributes & FA_DIREC ) return Default;
	return _FileGetFileman(f->VPathName);
}
////////////////////////////////////////////////////////////////////////////////
p_bitmap DefaultGetIcon ( PFileInfo f, l_ulong Size, l_ulong *ToFree ) {
	return __FileGetIconEx(f,Size,ToFree);
}
////////////////////////////////////////////////////////////////////////////////
p_bitmap FileGetIconExEx ( PFileMan m, PFileInfo f, l_ulong Size, l_ulong *ToFree ) {
	p_bitmap r;
	if ( m->GetIcon ) if ( r = m->GetIcon(f,Size,ToFree) ) return r;
	if ( Size == 16 ) {
		if ( r = m->Icon16 ) return r;
	} else if ( Size == 32 ) {
		if ( r = m->Icon32 ) return r;
	} else if ( Size == 48 ) {
		if ( r = m->Icon48 ) return r;
	}
	return DefaultGetIcon(f,Size,ToFree);
}
////////////////////////////////////////////////////////////////////////////////
p_bitmap FileGetIconEx ( PFileInfo f, l_ulong Size, l_ulong *ToFree ) {
	PFileMan m = FileGetFileman(f);
	if ( ToFree ) *ToFree = 0;
	if ( m ) {
		return FileGetIconExEx(m,f,Size,ToFree);
	}
	return DefaultGetIcon(f,Size,ToFree);
}
////////////////////////////////////////////////////////////////////////////////
l_text FileGetNiceNameEx ( PFileInfo f ) {
	PFileMan m = FileGetFileman(f);
	if ( m ) {
		if ( m->GetNiceName ) return m->GetNiceName(f,f->Name);
	}
	return TextDup(f->Name);
}
////////////////////////////////////////////////////////////////////////////////
void FileGetAllThingsEx ( PFileInfo f, l_text*n, p_bitmap*i16, p_bitmap*i32, p_bitmap*i48, l_ulong *fi ) {
	PFileMan m = FileGetFileman(f);
	if ( fi ) *fi = 0;
	if ( m ) {
		if ( m->GetAllThings )
			m->GetAllThings(f,f->Name,n,i16,i32,i48,fi);
		else {
			if ( i16 ) *i16 = FileGetIconExEx(m,f,16,fi);
			if ( i32 ) *i32 = FileGetIconExEx(m,f,32,fi);
			if ( i48 ) *i48 = FileGetIconExEx(m,f,48,fi);
			if ( n ) 		*n = TextDup(f->Name);
		}
	}
}
////////////////////////////////////////////////////////////////////////////////
void FileGetAllThings ( l_text file, l_text name, l_text*n, p_bitmap*i16, p_bitmap*i32, p_bitmap*i48, l_ulong *fi ) {
	PFileInfo f = FileGetInfo(file);
	if ( f ) {
		FileGetAllThingsEx(f,n,i16,i32,i48,fi);
		FreeFileInfo(f);
	}
}
////////////////////////////////////////////////////////////////////////////////
l_text FileGetNiceName ( l_text file, l_text name ) {
	PFileInfo f = FileGetInfo(file);
	if ( f ) {
		l_text r = FileGetNiceNameEx(f);
		FreeFileInfo(f);
		return r;
	}
	return NULL;
}
////////////////////////////////////////////////////////////////////////////////
p_bitmap 	FileGetIcon ( l_text file, l_ulong Size, l_ulong *ToFree ) {
	PFileInfo f = FileGetInfo(file);
	if ( f ) {
		p_bitmap r = FileGetIconEx(f,Size,ToFree);
		FreeFileInfo(f);
		return r;
	}
	return NULL;
}
////////////////////////////////////////////////////////////////////////////////
l_bool FileRunEx ( PFileInfo f, l_text ex ) {
	PFileMan m = FileGetFileman(f);
	if ( m ) {
		if ( m->Run ) 		return m->Run(f,ex);
		if ( m->Runner ) 	return DynLdRun(m->Runner,f->VPathName) ? true : false;
	} else
		DebugMessage("Unable to run %s",f->VPathName);
	return false;
}
////////////////////////////////////////////////////////////////////////////////
l_bool FileRun ( l_text file, l_text ex ) {
	PFileInfo f = FileGetInfo(file);
	if ( f ) {
		l_bool r = FileRunEx(f,ex);
		FreeFileInfo(f);
		return r;
	}
	return false;
}
////////////////////////////////////////////////////////////////////////////////
PMenu FileGetMenuEx ( PFileInfo f, l_ulong Flags ) {
	PFileMan o = FileGetFileman(f);
	PMenu    m = NewMenu(NULL);
	m->Items = NewMenuItem("Propreties",NULL,MSG_IOMENU_PROPERTIES,0,NULL,m->Items);
	m->Items = NewMenuItemSeparator(m->Items);
	
	if ( f->Attributes & FA_DIREC ) 
	m->Items = NewMenuItem("Paste",NULL,MSG_IOMENU_PASTE,0,NULL,m->Items);
	
	m->Items = NewMenuItem("Cut",NULL,MSG_IOMENU_CUT,0,NULL,m->Items);
	m->Items = NewMenuItem("Copy",NULL,MSG_IOMENU_COPY,0,NULL,m->Items);

	m->Items = NewMenuItemSeparator(m->Items);
	
	if ( f->Attributes & FA_DIREC && Flags ) 
		m->Items = NewMenuItem("New directory",NULL,MSG_IODLG_NEWDIR,0,NULL,m->Items);
	if ( ! Flags ) 
		m->Items = NewMenuItem("Rename",NULL,MSG_IOMENU_RENAME,0,NULL,m->Items);
	if ( o ) {
		if ( o->GenerateSubMenuItems )
			m->Items = o->GenerateSubMenuItems(o,f,m->Items);
	}
	m->Items = NewMenuItem("Delete",NULL,MSG_IOMENU_DELETE,0,NULL,m->Items);
	if ( !Flags ) m->Items = NewMenuItem("Open",NULL,MSG_IOMENU_OPEN,0,NULL,m->Items);
	return m;
}
////////////////////////////////////////////////////////////////////////////////
PMenu FileGetMenu ( l_text file ) {
	PFileInfo f = FileGetInfo(file);
	if ( f ) {
		PMenu r = FileGetMenuEx(f,0);
		FreeFileInfo(f);
		return r;
	}
	return NULL;
}
////////////////////////////////////////////////////////////////////////////////
l_bool FileHandleMenuMsgEx ( PFileInfo f, l_ulong msg ) {
	PFileMan o = FileGetFileman(f);
	if ( o ) {
		if ( o->HandleMenuMsg )
			return o->HandleMenuMsg(o,f,msg);

	}

	switch ( msg ){
		case MSG_IOMENU_PROPERTIES:
			DynLdRun("xlib/fprop.dl",f->VPathName);
			return true;
		break;

		case MSG_IOMENU_DELETE:
			MessageBox(&Me, "Help", "delete !", 0);
			return true;
		break;
		
	}

	return false;
}
////////////////////////////////////////////////////////////////////////////////
l_bool FileHandleMenuMsg ( l_text file, l_ulong msg ) {
	PFileInfo f = FileGetInfo(file);
	if ( f ) {
		l_bool r = FileHandleMenuMsgEx(f,msg);
		FreeFileInfo(f);
		return r;
	}
	return false;	
}
////////////////////////////////////////////////////////////////////////////////
PFileMan NewFileManager ( l_text Name, l_text Runner, l_text Extention, l_bool (*Run) ( PFileInfo, l_text ),
													p_bitmap Icon16, p_bitmap Icon32, p_bitmap Icon48,
													p_bitmap (*GetIcon) ( PFileInfo, l_ulong, l_ulong* ), l_text (*GetNiceName) ( PFileInfo, l_text ),
													void   		(*GetAllThings) ( PFileInfo, l_text, l_text*, p_bitmap*, p_bitmap*, p_bitmap*, l_ulong * ) ) {
	PFileMan o = malloc(sizeof(TFileMan));
	if ( !o ) return NULL;
	memset(o,0,sizeof(TFileMan));

	o->Name = TextDup(Name);
	o->Runner = TextDup(Runner);
	o->Extention = TextDup(Extention);
	o->Run = Run;
	o->Icon16 = Icon16;
	o->Icon32 = Icon32;
	o->Icon48 = Icon48;
	o->GetIcon = GetIcon;
	o->GetNiceName = GetNiceName;
	o->GetAllThings = GetAllThings;

	ListAdd(FileManagers,Extention,o,&free);

	DebugMessage("FMS :: New manager - %s (EXT: %s | RUN: %s)", Name, Extention, Runner);

	return o;
}
////////////////////////////////////////////////////////////////////////////////
l_bool ExecutableRun ( PFileInfo f, l_text ex ) {
	return DynLdRun(f->VPathName,ex) ? true : false;
}
////////////////////////////////////////////////////////////////////////////////
l_bool FMSRegistryAdd ( char* Key ){
	NewFileManager(KeyGetText(TextArgs("/SYSTEM/FMS/%s/type", Key), ""),KeyGetText(TextArgs("/SYSTEM/FMS/%s/run", Key), ""),Key,NULL,LoadImage(KeyGetText(TextArgs("/SYSTEM/FMS/%s/icon16", Key), "")),LoadImage(KeyGetText(TextArgs("/SYSTEM/FMS/%s/icon32", Key), "")),NULL,NULL,NULL,NULL);
	return true;
}
////////////////////////////////////////////////////////////////////////////////
l_bool FMSRegistryLoad ( void ){
	PRegKey o =	ResolveKey("/SYSTEM/FMS");
	if ( !o ) return false;

	if ( o->Last ){
		PRegKey a = o->Last->Next;
		PRegKey b = a;

		do
		{
			if ( a->Type == RKT_NODATA )
				FMSRegistryAdd((l_text)a->Name);

			a = a->Next;
		}
		while ( a != b );
	}

	return true;
}

////////////////////////////////////////////////////////////////////////////////
l_bool LibMain ( l_text Args )
{

	if ( Me.Ressource ) {
	
	ICONDir32 = ListKey(Me.Ressource,"FOLDER32");//LoadImage("SYSTEM/ICONS/folder32.bmp");
	ICONDir16 = ListKey(Me.Ressource,"FOLDER16");//LoadImage("SYSTEM/ICONS/folder16.bmp");
	ICONDrv32 = ListKey(Me.Ressource,"DRIVE32");//LoadImage("SYSTEM/ICONS/drive32.bmp");
	ICONDrv16 = ListKey(Me.Ressource,"DRIVE16");//LoadImage("SYSTEM/ICONS/drive16.bmp");
	ICONFile32 = ListKey(Me.Ressource,"FILE32");//LoadImage("SYSTEM/ICONS/file32.bmp");
	ICONFile16 = ListKey(Me.Ressource,"FILE16");//LoadImage("SYSTEM/ICONS/file16.bmp");
	ICONApp32 = ListKey(Me.Ressource,"APP32");//LoadImage("SYSTEM/ICONS/app32.bmp");
	ICONApp16 = ListKey(Me.Ressource,"APP16");//LoadImage("SYSTEM/ICONS/app16.bmp");
	}
	
	APPEXPORT(ICONDir32);
	APPEXPORT(ICONDir16);
	APPEXPORT(FileGetFileman);
	APPEXPORT(FileGetIcon);
	APPEXPORT(FileGetNiceName);
	APPEXPORT(NewFileManager);
	APPEXPORT(FileGetAllThings);
	APPEXPORT(FileRun);
	APPEXPORT(FileGetMenu);
	APPEXPORT(FileHandleMenuMsg);
	APPEXPORT(FileGetMenuEx);
	APPEXPORT(FileHandleMenuMsgEx);

	APPEXPORT(FileGetIconEx);
	APPEXPORT(FileGetNiceNameEx);
	APPEXPORT(FileGetAllThingsEx);
	APPEXPORT(FileRunEx);
	FileManagers = NewList();

	Default = NewFileManager("File",NULL,"*",NULL,ICONFile16,ICONFile32,NULL,&__FileGetIconEx,NULL,NULL);

	NewFileManager("DynLd Application",NULL,"app",&ExecutableRun,ICONApp16,ICONApp32,NULL,NULL,NULL,NULL);
	NewFileManager("DynLd Library",NULL,"dl",&ExecutableRun,ICONApp16,ICONApp32,NULL,NULL,NULL,NULL);
	
	NewFileManager("Link",NULL,"ln",&LinkRun,ICONApp16,ICONApp32,NULL,&LinkGetIcon,NULL,&LinkGetAllThings);
	
	if ( ResolveSymbol(&Me,"system") ) {
	
		NewFileManager("dosrun","xapps/rundos.app","exe",NULL,ICONApp16,ICONApp32,NULL,NULL,NULL,NULL);
		NewFileManager("dosrun","xapps/rundos.app","com",NULL,ICONApp16,ICONApp32,NULL,NULL,NULL,NULL);

	}
	
	FMSRegistryLoad();

	/*NewFileManager("Text file",NULL,"txt",&ExecutableRun,LoadImage("SYSTEM/ICONS/txt16.bmp"),LoadImage("SYSTEM/ICONS/txt32.bmp"),NULL,NULL,NULL,NULL);

	NewFileManager("BMP file",NULL,"bmp",&ExecutableRun,LoadImage("SYSTEM/ICONS/img16.bmp"),LoadImage("SYSTEM/ICONS/img32.bmp"),NULL,NULL,NULL,NULL);
	NewFileManager("PNG file",NULL,"png",&ExecutableRun,LoadImage("SYSTEM/ICONS/img16.bmp"),LoadImage("SYSTEM/ICONS/img32.bmp"),NULL,NULL,NULL,NULL);
	NewFileManager("JPEG file",NULL,"jpg",&ExecutableRun,LoadImage("SYSTEM/ICONS/img16.bmp"),LoadImage("SYSTEM/ICONS/img32.bmp"),NULL,NULL,NULL,NULL);

	NewFileManager("C file",NULL,"c",&ExecutableRun,LoadImage("SYSTEM/ICONS/c16.bmp"),LoadImage("SYSTEM/ICONS/c32.bmp"),NULL,NULL,NULL,NULL);
	NewFileManager("C++ file",NULL,"cpp",&ExecutableRun,LoadImage("SYSTEM/ICONS/cpp16.bmp"),LoadImage("SYSTEM/ICONS/cpp32.bmp"),NULL,NULL,NULL,NULL);
	NewFileManager("C++ file",NULL,"cx",&ExecutableRun,LoadImage("SYSTEM/ICONS/cpp16.bmp"),LoadImage("SYSTEM/ICONS/cpp32.bmp"),NULL,NULL,NULL,NULL);
	NewFileManager("C++ file",NULL,"cc",&ExecutableRun,LoadImage("SYSTEM/ICONS/cpp16.bmp"),LoadImage("SYSTEM/ICONS/cpp32.bmp"),NULL,NULL,NULL,NULL);
	NewFileManager("H file",NULL,"h",&ExecutableRun,LoadImage("SYSTEM/ICONS/h16.bmp"),LoadImage("SYSTEM/ICONS/h32.bmp"),NULL,NULL,NULL,NULL);
	NewFileManager("PYTHON file",NULL,"py",&ExecutableRun,LoadImage("SYSTEM/ICONS/python16.bmp"),LoadImage("SYSTEM/ICONS/python32.bmp"),NULL,NULL,NULL,NULL);*/

	return true;
}
////////////////////////////////////////////////////////////////////////////////
void Close (void)
{
	/*ICONDir32 = LoadImage("SYSTEM/ICONS/folder32.bmp");
	ICONDir16 = LoadImage("SYSTEM/ICONS/folder16.bmp");
	ICONDrv32 = LoadImage("SYSTEM/ICONS/drive32.bmp");
	ICONDrv16 = LoadImage("SYSTEM/ICONS/drive16.bmp");
	ICONFile32 = LoadImage("SYSTEM/ICONS/file32.bmp");
	ICONFile16 = LoadImage("SYSTEM/ICONS/file16.bmp");*/

}
