/*
    play.c
    Functions to select a move and play the game.
    copyright (C) 2020 Jim Hall <jhall@freedos.org>

    This is free software under the GNU GPL (only version 2).
    See the file COPYING for the full text of the GNU GPL.
*/

#include <stdio.h> /* sprintf */
#include <conio.h>
#include <graph.h>
#include "board.h"
#include "play.h"
#include "win.h"

int
select_move(int *board, int player, int move)
{
/* Let the user select a piece to move. Try different methods.

   Returns 0 to 29 for the piece (square) the user has selected to move.
   Returns -1 if the user wants to quit (or an error)
   (return values of 30 and greater should not happen)
*/

    int sq;
    int key, key2;
    int suggest;
    char msg[80];

    /* display a prompt */

    sprintf(msg, "Player %s: Select a piece to move [Space or Enter to confirm] [ESC to quit]", player_ch(player));

    print_message(msg);

    /* what's the first square with the player's piece? */

    for (sq = 0; (sq < 30) && (board[sq] != player); sq++); /* empty loop */

    if (sq == 30) {
	return -1; /* did not find the player? */
    }

    draw_square(sq, board[sq], 1); /* active */

    /* select a square with a piece to move */
    /* THERE'S A LOT OF LOGIC IN HERE ABOUT ARROW KEYS, TABS, ETC */
    /* THIS IS ALL TO MAKE IT EASIER ON THE USER TO SELECT, BUT IT
	CAN BE A LITTLE CONFUSING TO READ THIS CODE */

    do {
	key = getch();

	switch (key) {
	/* DID THE USER ASK TO QUIT THE GAME? */

	case 27: /* esc */
	case 'Q':
	    print_message("Do you want to quit the game? (Y/N)");

	    key2 = getch();
	    if ((key2 == 'Y') || (key2 == 'y')) {
		return -1;
	    }

	    print_message("");
	    break;

	/* CHECK IF THE USER HAS SELECTED A SQUARE */

	case ' ': /* space */
	case 13:  /* enter */
	    if (board[sq] == player) {
		if (is_valid_move(board, player, move, sq)) {
		    return sq; /* valid move */
		}
		else {
		    suggest = any_valid_move(board, player, move);
		    sprintf(msg, "You cannot make that move [try square %d]", suggest);
		    print_error(msg);
		}
	    }
	    else if (board[sq] != player) {
		print_error("That is not your piece");
	    }
	    else { /* empty */
		print_error("Cannot select an empty square");
	    }
	    break;
	} /* switch (key) */

	/* DIFFERENT METHODS TO SELECT A SQUARE */
	/* these all involve changing the selected square */

	/* set the current square to inactive */ 
	draw_square(sq, board[sq], 0); /* not active */

	switch (key) { /* still using same key as above */
	/* simplest method: + and - for next and prev square */

	case '+': /* next square */
	case '=':
	    sq++;
	    if (sq == 30) {
		sq = 29;
	    }
	    break;
	case '-': /* prev square */
	case '_':
	    sq--;
	    if (sq == -1) {
		sq = 0;
	    }
	    break;

	/* Tab to select next avail square (loop to beginning) */

	case 9: /* tab */
	    /* what's the next square with the player's piece? */
	    for (++sq; (sq < 30) && (board[sq] != player); sq++); /* empty loop */

	    if (sq == 30) {
		/* no next piece found ... loop from beginning */
		for (sq = 0; (sq < 30) && (board[sq] != player); sq++); /* empty loop */
	    }

	    if (sq == 30) {
		return -1; /* did not find the player? */
	    }
	    break;

	/* Up/Down/Left/Right arrow keys (VERY WEIRD MATH HERE) */

	case 0: /* extended key (read getch again for ext key code */
	    key2 = getch();

	    switch (key2) {
	    case 75: /* left */
		if ((sq >= 10) && (sq <= 19)) {
		    /* from middle row */
		    sq++;

		    if (sq == 20) {
			sq = 19;
		    }
		}
		else {
		    /* from top or bottom row */
		    sq--;

		    if (sq == -1) {
			sq = 0;
		    }
		    else if (sq == 19) {
			sq = 20;
		    }
		}
		break;
	    case 77: /* right */
		if ((sq >= 10) && (sq <= 19)) {
		    /* from middle row */
		    sq--;

		    if (sq == 0) {
			sq = 10;
		    }
		}
		else {
		    /* from top or bottom row */
		    sq++;

		    if (sq == 30) {
			sq = 29;
		    }
		    else if (sq == 10) {
			sq = 9;
		    }
		}
		break;
	    case 80: /* down */
		if (sq <= 9) {
		    /* from first row */
		    sq = 19 - sq;
		}
		else if ((sq >= 10) && (sq <= 19)) {
		    /* from middle row */
		    sq = 39 - sq;
		}
		/* else { don't move down if bottom row already } */
		break;
	    case 72: /* up */
		if (sq >= 20) {
		    /* from bottom row */
		    sq = 39 - sq;
		}
		else if ((sq >= 10) && (sq <= 19)) {
		    /* from middle row */
		    sq = 19 - sq;
		}
		/* else { don't move up if top row already } */
	    } /* switch (key2) */
	} /* switch (key) */

	/* set the current square to active */ 
	draw_square(sq, board[sq], 1); /* active */

    } while (1); /* loop forever (returns from inside the loop) */
}

int
is_valid_move(int *board, int player, int move, int sq)
{
/* Determine if this is a valid move for this player from this square.
   THIS IS WHERE MOST OF THE GAME RULES LIVE.

   Returns a positive value (True) if this is a valid move forward.
   Returns a negative value (True) if this is a valid capture.
   Returns zero (False) if this is not a valid move.
*/

/* The board looks like this: (counting from zero)
   0  1  2  3  4  5  6  7  8  9
   19 18 17 16 15 14 13 12 11 10
   20 21 22 23 24 25 26 27 28 29
*/

    /* ALMOST ALL THE GAME RULES ARE HERE */

    /* it would be dumb to allow move=0 */

    if (move == 0) {
	return 0;
    }

    /* check special squares */

    switch (sq) {
    case 29: /* "1" square */
	/* only allow move off if move=1 */
	if (move == 1) {
	    return 1;
	}
	else {
	    return 0;
	}
	break;
    case 28: /* "2" square */
	/* only allow move=2 */
	if (move == 2) {
	    return 1;
	}
	else {
	    return 0;
	}
	break;
    case 27: /* "3" square */
	/* only allow move=3 */
	if (move == 3) {
	    return 1;
	}
	else {
	    return 0;
	}
	break;

    /* there is no case for sq=26 because that's the trap square */

    case 25: /* the "stop here" square */
	/* cannot move from 25 to 26 (trap) if someone is on sq 14 */
	/* and cannot capture in the last few squares */
	if (move == 1) {
	    if (board[14]) {
		return 0;
	    }
	    else {
		return 1;
	    }
	}
	else {
	    if (board[sq+move]) {
		return 0;
	    }
	    else {
		return 1;
	    }
	}
	break;

    /* squares before the "stop here" square must stop on sq=25 exactly */
    case 24:
    case 23:
    case 22:
    case 21:
	if ((sq+move) > 25) {
	    return 0;
	}
    default:
	/* I'm being tricky here because this doesn't have a 'break' on the
	    line above. case 21 actually continues here if the move does not
	    go past sq 25. check for capture or move fwd. */
	/* this is meant to apply for all other squares on the board */

	if (board[sq+move] == board[sq]) {
	    return 0; /* already occupied by me */
	}
	else if (board[sq+move] == 0) {
	    return 1; /* move fwd */
	}
	else {
	    return -1; /* capture */
	}
    }

    return 1;
}

int
any_valid_move(int *board, int player, int move)
{
    /* walk through the entire board, find the first playable move */

    int sq;

    for (sq = 0; sq < 30; sq++) {
	if ((board[sq] == player) && (is_valid_move(board, player, move, sq))) {
	    /* return the square number (counting from 1) [True] */
	    return (sq+1);
	}
    }

    /* if none found, return 0 [False] */

    return 0;
}