/*
 * low-level stuff for bcc
 * Copyright (c) 2017,2019 Andreas K. Foerster <info@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <bios.h>
#include "bcc-syst.h"

#if !defined(__BCC__) || !defined(__AS386_16__)
#error "Just for Bruce's C compiler (bcc), 16-bit"
#endif

#ifdef MDA			/* MDA or Hercules */
#define VIDEO_SEGMENT 0xB000
#else /* CGA or higher */
#define VIDEO_SEGMENT 0xB800
#endif

static void hide_cursor ();

static unsigned int bcc_x, bcc_y;
static unsigned char bcc_attribute;


void
bcc_position (x, y)
     int x, y;
{
  bcc_x = (unsigned int) x - 1;
  bcc_y = (unsigned int) y - 1;
}


void
bcc_character (c)
     int c;
{
  switch (c)
    {
    case '\r':
      bcc_x = 0;
      break;

    case '\n':
    case '\v':
      ++bcc_y;
      bcc_x = 0;
      break;

    case '\b':
      if (bcc_x > 0)
	--bcc_x;
      break;

    case '\t':
      bcc_x += 8;
      break;

    case '\a':
      bcc_beep ();
      break;

    default:
      if (bcc_x < 80 && bcc_y < 25)
	{
	  unsigned int es = __get_es ();
	  __set_es (VIDEO_SEGMENT);
	  __doke_es (((bcc_y * 80) + bcc_x) * 2, c | (bcc_attribute << 8));
	  __set_es (es);

	  ++bcc_x;
	}
      break;
    }
}


void
bcc_print (s)
     const char *s;
{
  while (*s)
    bcc_character (*s++);
}


void
bcc_clear_screen ()
{
  unsigned int es, i, v;

  hide_cursor ();

  es = __get_es ();
  __set_es (VIDEO_SEGMENT);

  v = ' ' | (bcc_attribute << 8);
  for (i = 0; i < (80 * 25 * 2); i += 2)
    __doke_es (i, v);

  __set_es (es);

  bcc_x = bcc_y = 0;
}


void
bcc_clear_rest ()
{
  unsigned int es, i, v, o;

  es = __get_es ();
  __set_es (VIDEO_SEGMENT);

  v = ' ' | (bcc_attribute << 8);
  o = ((bcc_y * 80) + bcc_x) * 2;

  for (i = 0; i < (80 - bcc_x); ++i)
    __doke_es (o + (i * 2), v);

  __set_es (es);
}


void
bcc_set_attribute (attribute)
     int attribute;
{
  bcc_attribute = (unsigned char) attribute;
}


int
bcc_get_attribute ()
{
  return (int) bcc_attribute;
}


/* *INDENT-OFF* */


/* micoseconds, unsigned long, max. 4s */
void
bcc_usleep (us)
{
#asm
  mov bx, sp
  mov dx, [bx+2]
  mov cx, [bx+4]
  mov ax, #$8600
  int $15
#endasm
}


void
bcc_beep ()
{
#asm
  xor bx, bx
  mov ax, #$0E07
  int $10
#endasm
}


void
bcc_idle ()
{
#asm
  mov ax, #$1680
  int $2F
#endasm
}


static void
hide_cursor ()
{
#asm
  xor bx, bx
  mov dx, #$FFFF
  mov ax, #$0200
  int $10
#endasm
}
