/*
 * EW Snake - Versione 0.5
 * Copyright (C) 2001 di Federico Marverti (yfede)
 *
 * Questo programma e` distribuito secondo i termini della Licenza
 * Pubblica Generica GNU (GNU GPL: GNU General Public License).
 * Per maggiori informazioni e riguardo leggere il file README.TXT
 * allegato alla distribuzione del presente codice sorgente.
 */

/* --- Inizio di game.c --- */

#include "defs.h"
#include <allegro.h>

void init_game( struct st_game *game ); /* Inizializza le variabili di gioco con valori di default */
int play( struct st_config *cfg );  /* Gioca usando la configurazione specificata */
void move_snake( struct st_game *game );    /* Sposta il serpente in base a quanto indicato nella struttura */
int check_snake( struct st_game *game );    /* Controlla lo stato del serpente */

extern void reset_screen( void );   /* Cancella lo schermo e colora lo sfondo */
extern void draw_wall( struct st_config *cfg, struct st_game *game );   /* Disegna il muro */
extern void draw_goal( struct st_config *cfg, struct st_game *game );   /* Disegna l'obiettivo */
extern void draw_snake( struct st_config *cfg, struct st_game *game );  /* Disegna il serpente */
extern void redraw_snake( struct st_config *cfg, struct st_game *game );    /* Ridisegna il serpente */

void init_game( struct st_game *game )
{
    int i,j;

    game->direction = DE_DIRECTION; /* Imposta le variabili di gioco a dei valori di default */
    game->size = DE_SIZE;
    game->goal_x = DE_STARTX + 3;
    game->goal_y = DE_STARTY + 1;
    game->score = 0;

    game->to_grow = 0;          /* Ancora non deve crescere */

    for( i = 0; i < DE_SIZE; i++ )  /* Mette i primi DE_SIZE blocchi del serpente dietro alla testa */
    {
        (game->x)[i] = DE_STARTX - i;
        (game->y)[i] = DE_STARTY;
    }

    (game->x)[game->size] = OUTOFSCREEN;    /* Uguaglia a -1, -1, le coordinate del pezzo da cancellare */
    (game->x)[game->size] = OUTOFSCREEN;

    for( i = 0; i < VCELLS; i++ )   /* Vuota il muro */
        for( j = 0; j < HCELLS; j++ )
            (game->wall)[j][i] = W_EMPTY;

    for( i = 0; i < VCELLS; i++ )   /* Mette il muro ai confini dello schermo */
    {
        (game->wall)[0][i] = W_WALL;
        (game->wall)[HCELLS-1][i] = W_WALL;
    }

    for( i = 0; i < HCELLS; i++ )
    {
        (game->wall)[i][0] = W_WALL;
        (game->wall)[i][VCELLS-1] = W_WALL;
    }
}

int play( struct st_config *cfg )
{
    struct st_game game;        /* Struttura contenente i dati di gioco */
    int scancode = KEY_NONE;    /* Scancode per il tasto premuto */
    long wait;                  /* Tempo di attesa (millisecondi) */
    int status = ST_NORMAL;     /* Stato iniziale : Normale */

    init_game( &game );         /* Inizializza i dati di gioco */

    reset_screen();             /* Cancella lo schermo */
    draw_wall( cfg, &game );    /* Disegna il muro */
    draw_goal( cfg, &game );    /* Disegna l'obiettivo */
    draw_snake( cfg, &game );   /* Disegna il serpente */

    text_mode( CL_BACKGROUND ); /* Imposta il colore di sfondo per scrivere il punteggio */
    textprintf_centre( screen, font, (cfg->resx)/2, (cfg->resy)-(cfg->cellsize)/2-DE_FONTH/2, CL_TEXT,
                       " Livello: %d, Punteggio: %d/%d ", cfg->level, game.score, cfg->record );

    wait = (long)(FIX_DELAY+(MAX_LEVEL-(cfg->level))*LEV_DELAY);    /* Calcola il tempo di attesa */
    clear_keybuf();             /* Cancella il buffer della tastiera */

    for(;;)                     /* Ciclo infinito */
    {
        rest( wait );           /* Attende il tempo specificato */

        if( keypressed() == TRUE )  /* Se e` stato premuto un tasto */
            scancode = readkey() >> 8;  /* Legge lo scancode */
        else                        /* Altrimenti */
            scancode = KEY_NONE;    /* Scegnala che non e` stato premuto niente */

        switch( scancode )      /* In base al tasto premuto imposta le variabili di gioco */
        {
            case KEY_UP:        /* La direzione non puo` essere invertita direttamente */
                if( game.direction != D_DOWN ) game.direction = D_UP;
            break;
            
            case KEY_DOWN:
                if( game.direction != D_UP ) game.direction = D_DOWN;
            break;

            case KEY_LEFT:
                if( game.direction != D_RIGHT ) game.direction = D_LEFT;
            break;

            case KEY_RIGHT:
                if( game.direction != D_LEFT ) game.direction = D_RIGHT;
            break;

            case KEY_SPACE:     /* Spazio: Pausa */
                readkey();      /* Attende la pressione di un tasto */
            break;
        }
        if( scancode == KEY_ESC ) break;    /* Se e` stato premuto esc esce dal ciclo */

        move_snake( &game );    /* Muove il serpente */

        status = check_snake( &game );  /* Controlla lo stato */

        if( status == ST_SNAKE || status == ST_WALL ) break;    /* Se a preso il muro o se stesso esce */

        redraw_snake( cfg, &game ); /* Lo ridisegna */

        if( status == ST_GOAL ) /* Se ha preso l'obiettivo */
        {
            game.score += (cfg->level) + 1; /* Incrementa il punteggio */
            for( ;; )           /* Ciclo infinito interrotto con break */
            {
                game.goal_x = rand() % HCELLS;  /* Imposta un nuovo punto */
                game.goal_y = rand() % VCELLS;

                if( game.wall[game.goal_x][game.goal_y] != W_WALL ) /* Se non e` sul muro */
                {
                    int i;      /* Indice per i blocchi del serpente */
                    for( i = 0; i < game.size; i++ )    /* Controlla che il goal non cada sul serpente */
                        if( game.x[i] == game.goal_x && game.y[i] == game.goal_y ) break;
                    if( i == game.size ) break; /* Se ha controllato tutti i blocchi positivamente esce dal ciclo */
                }
            }
            draw_goal( cfg, &game );    /* Disegna il nuovo goal */
            game.to_grow = DE_GROW; /* Imposta la crescita del serpente */
            text_mode( CL_BACKGROUND ); /* Imposta il colore di sfondo per scrivere il punteggio */
            textprintf_centre( screen, font, (cfg->resx)/2, (cfg->resy)-(cfg->cellsize)/2-DE_FONTH/2, CL_TEXT,
                               " Livello: %d, Punteggio: %d/%d ", cfg->level, game.score, cfg->record );
        }
    }
    return game.score;          /* Quando termina ritorna il punteggio ottenuto */
}

void move_snake( struct st_game *game )
{
    int i;

    for( i = (game->size); i > 0; i-- ) /* Sposta i blocchi del serpente */
    {
        (game->x)[i] = (game->x)[i-1];
        (game->y)[i] = (game->y)[i-1];
    }

    if( (game->to_grow) > 0 )   /* Se sta crescendo */
    {
        (game->size)++;         /* Incrementa la dimensione */
        (game->to_grow)--;      /* Diminuisce la dimensione di cui deve crescere */
        (game->x)[game->size] = OUTOFSCREEN;    /* Mette la casella da cancellare fuori dallo schermo */
        (game->y)[game->size] = OUTOFSCREEN;
    }

    switch( game->direction )   /* In base alla direzione calcola la posizione della testa */
    {                           /* E` implementato il warp */
        case D_UP:
            if( (game->y)[0] == 0 ) (game->y)[0] = VCELLS - 1;
            else (game->y)[0]--;
        break;

        case D_DOWN:
            if( (game->y)[0] == VCELLS-1 ) (game->y)[0] = 0;
            else (game->y)[0]++;
        break;

        case D_LEFT:
            if( (game->x)[0] == 0 ) (game->x)[0] = HCELLS - 1;
            else (game->x)[0]--;
        break;

        case D_RIGHT:
            if( (game->x)[0] == HCELLS-1 ) (game->x)[0] = 0;
            else (game->x)[0]++;
        break;
    }
}

int check_snake( struct st_game *game )
{
    int i;

    if( (game->wall)[(game->x)[0]][(game->y)[0]] == W_WALL )    /* Se ha preso il muro */
        return ST_WALL;         /* Ritorna lo stato relativo */

    for( i = 1; i < (game->size); i++ ) /* Per ogni blocco del serpente tranne la testa */
        if( (game->x)[0] == (game->x)[i] && (game->y)[0] == (game->y)[i] )  /* Se si e` mangiato */
            return ST_SNAKE;    /* Ritorna lo stato relativo */

    if( (game->x)[0] == (game->goal_x) && (game->y)[0] == (game->goal_y) )  /* Se ha preso l'obiettivo */
        return ST_GOAL;         /* Ritorna lo stato relativo */

    return ST_NORMAL;           /* Altrimenti ritorna lo stato normale */            
}
/* --- Fine di game.c --- */