;************************;
; SESSION Display Output ;
;     By Eric Tauck      ;
;************************;

TABSIZE EQU     8               ;spaces per tabstop

mode    DW      TEXT8025        ;session video mode
color   DW      0               ;(0=default, 1=color, 2=b/w)

;--- window data structure

w_atr   EQU     0       ;attribute
w_col   EQU     1       ;cursor column
w_row   EQU     2       ;cursor row
w_col1  EQU     3       ;left edge
w_row1  EQU     4       ;top edge
w_col2  EQU     5       ;right edge
w_row2  EQU     6       ;bottom edge
w_loc   EQU     7       ;saved cursor location pointer
w_sav   EQU     9       ;saved cursor locations (room for 5)
W_SIZE  EQU     19      ;bytes in data structure

;========================================
; Initialize video.

Display_Init    PROC    NEAR

;--- select video mode

        mov     al, BYTE mode   ;load mode
        call    VidOpn          ;intialize video
        jnc     disini1         ;jump if okay
        mov     al, TEXT8025    ;use default mode
        call    VidOpn          ;initialize video, assume success

;--- save attribute

disini1 call    AtrGet          ;get attribute
        mov     sattr, al       ;save it

        call    Display_Setup   ;set up display data
        ret
        ENDP

;========================================
; Set up display data.

Display_SetUp   PROC    NEAR
        push    di
        push    si

;--- set colors

        mov     ax, color               ;load color flag
        dec     ax                      ;check if color = 1
        jz      disset2
        dec     ax                      ;check if color = 2
        jz      disset1
        call    VidCol                  ;get default
        jc      disset2

disset1 mov     si, OFFSET attr_bw
        jmps    disset3
disset2 mov     si, OFFSET attr_col
disset3 mov     di, OFFSET attr_start
        mov     cx, ATTR_COUNT
        cld
        rep
        movsb                           ;copy attributes

;--- save number of rows and columns

        call    ModDim          ;get screen dimensions
        mov     cols, al        ;save columns
        mov     rows, ah        ;save rows

;--- window tops and bottoms

        mov     bx, OFFSET window1      ;terminal window
        mov     di, OFFSET window2      ;local window
        mov     si, OFFSET window3      ;remote window

        mov     al, rows                ;number of rows
        mov     ah, al
        dec     al                      ;last row

        mov     BYTE [bx + w_row1], 0   ;terminal top
        mov     [bx + w_row2], al       ;terminal bottom

        mov     BYTE [si + w_row1], 0   ;remote top
        shr     ah
        dec     ah
        mov     [si + w_row2], ah       ;remote bottom

        inc     ah
        inc     ah
        mov     [di + w_row1], ah       ;local top
        mov     [di + w_row2], al       ;local bottom

;--- window edges

        mov     BYTE [bx + w_col1], 0   ;terminal left
        mov     BYTE [di + w_col1], 0   ;local left
        mov     BYTE [si + w_col1], 0   ;remote left
        mov     al, cols                ;total columns
        dec     al                      ;last column
        mov     [bx + w_col2], al       ;terminal right
        mov     [di + w_col2], al       ;local right
        mov     [si + w_col2], al       ;remote right

;--- window colors

        mov     al, attr_term                   ;
        mov     BYTE [bx + w_atr], al           ;terminal
        mov     al, attr_loc                    ;
        mov     BYTE [di + w_atr], al           ;local
        mov     al, attr_rem                    ;
        mov     BYTE [si + w_atr], al           ;remote

;--- cursor save locations

        lea     ax, [bx + w_sav]        ;start of save locations
        mov     [bx + w_loc], ax
        lea     ax, [di + w_sav]        ;start of save locations
        mov     [di + w_loc], ax
        lea     ax, [si + w_sav]        ;start of save locations
        mov     [si + w_loc], ax

;--- fix cursor if in proper video mode

        call    ModGet                  ;get mode
        cmp     al, TEXT8025            ;check if standard mode
        jne     disset4                 ;exit if not
        call    CurFix                  ;cursor fix

disset4 pop     si
        pop     di
        ret
        ENDP

;========================================
; Restore the video state.

Display_Done    PROC    NEAR

;--- clear screen

        mov     al, sattr       ;saved attribute
        call    AtrSet          ;set attribute
        call    ScrCls          ;clear screen

;--- home cursor

        sub     ax, ax          ;row and column zero
        call    CurMov          ;move to upper left
        mov     al, 1
        call    CurSet          ;turn cursor on

;--- restore video mode

        call    VidClo          ;close video
        ret
        ENDP

;========================================
; Set color attribute.
;
; In: AL= attribute (0=off, 1=bold,
;     5=blink); DI= window pointer.

Window_Attr PROC NEAR
        mov     dl, [di+w_atr]  ;load attribute

        mov     ah, dl
        and     ah, NOT (BLINK OR BOLD)
        or      al, al
        jz      winatr1

        mov     ah, dl
        or      ah, BOLD
        cmp     al, 1
        je      winatr1

        mov     ah, dl
        or      ah, BLINK
        cmp     al, 5
        je      winatr1

winatr1 mov     [di+w_atr], ah  ;save attribute
        ret
        ENDP

;========================================
; Set foreground color.
;
; In: AL= color; DI= window pointer.

Window_Fore PROC NEAR
        mov     ah, [di+w_atr]  ;load current attribute
        and     ah, 0F8H        ;clear bits to set
        or      al, ah          ;combine foreground and background
        mov     [di+w_atr], al  ;save attribute
        ret
        ENDP

;========================================
; Set background color.
;
; In: AL= color; DI= window pointer.

Window_Back PROC NEAR
        mov     cl, 4           ;bits to shift
        shl     al, cl          ;put color in background nibble
        and     al, 70H         ;clear bold bit on new color
        mov     ah, [di+w_atr]  ;load attribute
        and     ah, 8FH         ;clear bits to set
        or      al, ah          ;combine foreground and background
        mov     [di+w_atr], al  ;save attribute
        ret
        ENDP

;========================================
; Update the window cursor.
;
; In: DI= window pointer.

Window_Update   PROC    NEAR
        mov     ax, [di + w_col]        ;cursor location
        cmp     al, [di + w_col2]       ;check if too far right
        jbe     winupd1                 ;jump if not
        mov     al, [di + w_col2]       ;last column
winupd1 push    ax
        call    CurPos                  ;get current position
        pop     dx
        cmp     ax, dx                  ;check if already at position
        je      winupd2                 ;exit if so
        mov     ax, dx
        call    CurMov                  ;position cursor
winupd2 ret
        ENDP

;========================================
; Simulate newline if past end of right
; edge.

Window_Check PROC NEAR
        mov     al, [di + w_col]        ;load column
        cmp     al, [di + w_col2]       ;check if past end
        jbe     winchk1
        call    Window_Line             ;newline
        mov     al, [di + w_col1]
        mov     [di + w_col], al        ;set to first column
winchk1 ret
        ENDP

;========================================
; Clear a window.
;
; In: DI= window pointer.

Window_Clear    PROC    NEAR
        mov     al, [di + w_atr]        ;attribute
        call    AtrSet                  ;set color
        mov     bx, [di + w_col1]       ;upper left
        mov     cx, [di + w_col2]       ;lower right
        call    ScrClr                  ;clear screen
        mov     ax, [di + w_col1]       ;home location
        mov     [di + w_col], ax        ;new cursor location
        call    Window_Update           ;position cursor
        ret
        ENDP

;========================================
; Clear from cursor to end of line

Window_EOL PROC NEAR
        call    Window_Check            ;fix cursor
        call    Window_Update           ;move cursor

        mov     al, [di + w_atr]        ;attribute
        call    AtrSet                  ;set color
        mov     cl, [di+w_col2]         ;end column
        sub     cl, [di+w_col]          ;minus current column
        inc     cl                      ;at least one
        mov     al, ' '                 ;space
        call    WrtChrs                 ;display
        ret
        ENDP

;========================================
; Clear from cursor to end of screen.

Window_EOS PROC NEAR
        call    Window_EOL              ;clear to end of line

        mov     al, [di + w_atr]        ;attribute
        call    AtrSet                  ;set color
        mov     bh, [di + w_row]        ;current row
        cmp     bh, [di + w_row2]       ;check if bottom
        je      wineos1
        inc     bh
        sub     bl, bl                  ;column zero
        mov     cx, [di + w_col2]       ;lower right
        call    ScrClr                  ;clear screen
wineos1 ret
        ENDP

;========================================
; Return the current cursor position.
;
; In: DI= window pointer.
;
; Out: AH,AL= row/column.

Window_Position PROC NEAR
        mov     ax, [di+w_col]          ;physical location
        sub     ax, [di+w_col1]         ;adjust for window
        ret
        ENDP

;========================================
; Move cursor
;
; In: AH,AL= row/column; DI= window
;     pointer.

Window_Move PROC NEAR
        add     ax, [di+w_col1]         ;add upper left offset
        mov     [di+w_col], ax          ;save location
        call    Window_Update           ;move cursor
        ret
        ENDP

;========================================
; Move the cursor up.
;
; In: AL= rows to move up; DI= window
;     pointer.

Window_Up PROC  NEAR
        mov     ah, [di+w_row]          ;current location
        sub     ah, [di+w_row1]         ;to edge of window
        cmp     al, ah                  ;check maximum move
        jbe     winup1                  ;jump if okay
        mov     al, ah                  ;set to max
winup1  sub     [di+w_row], al          ;adjust row
        call    Window_Update           ;move cursor
        ret
        ENDP

;========================================
; Move the cursor down.
;
; In: AL= rows to move down; DI= window
;     pointer.

Window_Down PROC NEAR
        mov     ah, [di+w_row2]         ;current location
        sub     ah, [di+w_row]          ;to edge of window
        cmp     al, ah                  ;check maximum move
        jbe     windwn1                 ;jump if okay
        mov     al, ah                  ;set to max
windwn1 add     [di+w_row], al          ;adjust row
        call    Window_Update           ;move cursor
        ret
        ENDP

;========================================
; Move the cursor left.
;
; In: AL= columns to move left; DI=
;     window pointer.

Window_Left PROC  NEAR
        mov     ah, [di+w_col]          ;current location
        sub     ah, [di+w_col1]         ;to edge of window
        jnc     winlef1                 ;jump if okay
        sub     ah, ah                  ;past edge, pretend at edge
winlef1 cmp     al, ah                  ;check maximum move
        jbe     winlef2                 ;jump if okay
        mov     al, ah                  ;set to max
winlef2 sub     [di+w_col], al          ;adjust column
        call    Window_Update           ;move cursor
        ret
        ENDP

;========================================
; Move the cursor right.
;
; In: AL= columns to move right; DI=
;     window pointer.

Window_Right PROC NEAR
        mov     ah, [di+w_col2]         ;current location
        sub     ah, [di+w_col]          ;to edge of window
        cmp     al, ah                  ;check maximum move
        jbe     winrig1                 ;jump if okay
        mov     al, ah                  ;set to max
winrig1 add     [di+w_col], al          ;adjust column
        call    Window_Update           ;move cursor
        ret
        ENDP

;========================================
; Save cursor location.
;
; In: DI= window pointer.

Window_Save PROC NEAR
        mov     bx, [di+w_loc]          ;load save pointer
        push    WORD [di+w_col]
        pop     WORD [bx]               ;copy current location
        inc     bx                      ;
        inc     bx                      ;point to next location
        mov     [di+w_loc], bx          ;save for next save
        ret
        ENDP

;========================================
; Restore cursor location.
;
; In: DI= window pointer.

Window_Rest PROC NEAR
        mov     bx, [di+w_loc]          ;load save pointer
        dec     bx                      ;
        dec     bx                      ;point to last location
        mov     [di+w_loc], bx          ;save for next restore
        mov     ax, [bx]                ;load saved location
        mov     [di+w_col], ax          ;set location
        call    Window_Update           ;move cursor
        ret
        ENDP

;========================================
; Display a character.
;
; In: DI= window pointer; AL= character.

Window_Type     PROC    NEAR

;--- check if beep

        cmp     al, BEL         ;beep
        jne     wintyp1         ;jump if not
        call    Beep_Std        ;beep
        ret

;--- check if backspace

wintyp1 cmp     al, BS          ;backspace
        jne     wintyp2         ;jump if not
        call    Window_Bkspc    ;backspace
        ret

;--- check if tab

wintyp2 cmp     al, TAB         ;tab
        jne     wintyp3         ;jump if not
        call    Window_Tab      ;tab
        ret

;--- check if linefeed

wintyp3 cmp     al, LF                  ;check if linefeed
        jne     wintyp4                 ;jump if not
        call    Window_Line             ;do linefeed
        ret

;--- check if carriage return

wintyp4 cmp     al, CR                  ;check if carriage return
        jne     wintyp5                 ;jump if not
        mov     al, [di + w_col1]       ;first column
        mov     [di + w_col], al        ;move cursor there
        call    Window_Update           ;update cursor
        ret

;--- check if formfeed

wintyp5 cmp     al, FF                  ;check if formfeed
        jne     wintyp6                 ;jump if not
        call    Window_Clear            ;update cursor
        ret

;--- must be character

wintyp6 push    ax
        mov     al, [di + w_atr]        ;color
        call    AtrSet                  ;set color
        call    Window_Check            ;fix cursor
        call    Window_Update           ;position cursor
        pop     ax
        call    WrtChr                  ;display character

;--- advance cursor

        inc     BYTE [di + w_col]       ;increment column
        call    Window_Update           ;update cursor
        ret
        ENDP

;========================================
; Backspace.
;
; In: DI= window pointer.

Window_BkSpc    PROC    NEAR
        call    Window_Check            ;fix cursor

;--- check if at edge

        mov     ax, [di + w_col]        ;current location
        cmp     al, [di + w_col1]       ;check if column zero
        jz      winbks3                 ;jump if so
        dec     al                      ;previous column

;--- do a backspace

winbks1 mov     [di + w_col], ax        ;save location
        call    CurMov                  ;position cursor
        mov     al, [di + w_atr]        ;color
        call    AtrSet                  ;set color
        mov     al, ' '                 ;blank
        call    WrtChr                  ;display character
        call    Window_Update           ;update cursor
winbks2 ret

;--- at left edge

winbks3 cmp     ah, [di + w_row1]       ;check if at top row
        je      winbks2                 ;jump if so
        dec     ah                      ;previous row
        mov     al, [di + w_col2]       ;load left column
        jmps    winbks1
        ENDP

;========================================
; Tab.
;
; In: DI= window pointer.

Window_Tab      PROC    NEAR
        call    Window_Check            ;fix cursor
        mov     al, [di + w_col]        ;current column
        sub     al, [di + w_col1]       ;column number
        sub     ah, ah
        mov     dl, TABSIZE             ;spaces per tab
        div     dl
        sub     dl, ah                  ;spaces to advance
        add     [di + w_col], dl        ;advance cursor
        call    Window_Update           ;update cursor
        ret
        ENDP

;========================================
; Display a newline.
;
; In: DI= window pointer.

Window_Line     PROC    NEAR

;--- check if scroll

        mov     ax, [di + w_col]        ;current location
        inc     ah                      ;next row

        cmp     ah, [di + w_row2]       ;check if bottom of screen
        jbe     winlin1

        mov     ah, [di + w_row2]       ;use last row
        push    ax
        mov     ax, [di + w_col1]       ;upper left
        mov     bx, ax
        inc     bh
        mov     cx, [di + w_col2]       ;lower right
        call    ScrCpy                  ;scroll screen
        mov     al, [di + w_atr]        ;attribute
        call    AtrSet                  ;set it
        mov     al, [di + w_col1]       ;left
        mov     ah, [di + w_row2]       ;bottom
        call    CurMov                  ;position cursor
        mov     cl, [di + w_col2]       ;right edge
        sub     cl, [di + w_col1]       ;minus left edge
        inc     cl                      ;adjust
        mov     al, ' '                 ;space
        call    WrtChrs                 ;blank line
        pop     ax

;--- save cursor location

winlin1 mov     [di + w_col], ax        ;save cursor location
        call    Window_Update           ;update cursor
        ret
        ENDP

;========================================
; Display an ASCIIZ string using a
; specific attribute.
;
; In: DI= window pointer; AX= string
;     address; DL= attribute.

Window_String PROC      NEAR
        push    si
        mov     si, ax          ;put address in SI
        mov     al, [di+w_atr]  ;save attribute
        mov     [di+w_atr], dl  ;new attribute
        push    ax
        jmps    winstr2

winstr1 call    Window_Type     ;display character
winstr2 cld
        lodsb                   ;load byte
        or      al, al          ;check if end of string
        jnz     winstr1         ;loop back if not end

        pop     ax
        mov     [di+w_atr], al
        pop     si
        ret
        ENDP

;========================================
; Clear the remote input window.

Remote_Clear    PROC    NEAR
        push    di
        mov     di, rem_win     ;load remote window pointer
        call    Window_Clear    ;clear window
        pop     di
        ret
        ENDP

;========================================
; Type a character to the remote input
; window.
;
; In: AL= character.

Remote_Type     PROC    NEAR
        push    di
        mov     di, rem_win     ;load remote window pointer
        call    Window_Type     ;type character
        pop     di
        ret
        ENDP

;========================================
; Clear the local input window.

Local_Clear     PROC    NEAR
        push    di
        mov     di, loc_win     ;load local window pointer
        cmp     di, rem_win     ;check if same window
        je      locclr1         ;skip clear if so
        call    Window_Clear    ;clear window
locclr1 pop     di
        ret
        ENDP

;========================================
; Type a character to the local input
; window.
;
; In: AL= character.

Local_Type      PROC    NEAR
        push    di
        mov     di, loc_win     ;load local window pointer
        call    Window_Type     ;type character
        pop     di
        ret
        ENDP

;========================================
; Display a system message.

System_String   PROC    NEAR
        push    di
        mov     di, rem_win     ;remote window pointer
        mov     dl, attr_mess   ;attribute
        call    Window_String   ;type string
        pop     di
        ret
        ENDP

;========================================
; Display a debug message.

Debug_String    PROC    NEAR
        push    di
        mov     di, loc_win     ;local window pointer
        mov     dl, attr_debug  ;attribute
        call    Window_String   ;type string
        pop     di
        ret
        ENDP

;========================================
; Clear both windows.

Clear_Both      PROC    NEAR
        call    Remote_Clear    ;clear remote window
        call    Local_Clear     ;clear local window
        ret
        ENDP

;========================================
; Merge the screen (i.e. single window
; mode).

Screen_Merge    PROC    NEAR

;--- copy remote attribute to main attribute

        mov     al, window3 + w_atr     ;
        mov     window1+ w_atr, al      ;copy attribute

;--- merge

        mov     ax, OFFSET window1      ;local window
        mov     loc_win, ax             ;save it
        mov     rem_win, ax             ;save it
        call    Remote_Clear            ;clear screen
        ret
        ENDP

;========================================
; Split the screen.

Screen_Split    PROC    NEAR

;--- copy main attribute to remote attribute

        mov     al, window1 + w_atr     ;
        mov     window3 + w_atr, al     ;copy attribute

;--- set up windows

        mov     ax, OFFSET window2      ;local window
        mov     loc_win, ax             ;save it
        mov     ax, OFFSET window3      ;remote window
        mov     rem_win, ax             ;save it

;--- draw center line

        mov     bx, rem_win             ;remote window
        mov     ah, [bx + w_row2]       ;bottom row
        inc     ah                      ;next row
        sub     al, al                  ;column zero
        call    CurMov                  ;move cursor
        mov     al, attr_line           ;line color
        call    AtrSet                  ;set color
        mov     al, 205                 ;double line
        mov     cl, cols                ;number of columns
        call    WrtChrs                 ;display line

        call    Clear_Both              ;clear both windows
        ret
        ENDP

;========================================
; Save the current screen.

Screen_Save     PROC    NEAR

;--- save cursor position

        call    CurPos          ;cursor position
        mov     WORD s_col, ax  ;save it

;--- save screen

        sub     bx, bx          ;upper left
        mov     cx, WORD cols   ;load rows and columns
        dec     ch              ;end row
        dec     cl              ;end column
        push    bx
        push    cx
        call    ScrSiz          ;get bytes needed
        call    MemAll          ;allocate memory
        pop     cx
        pop     bx
        jc      scrsav1         ;jump if error
        mov     s_seg, ax       ;save segment
        mov     dx, ax
        sub     ax, ax
        call    ScrGet          ;copy screen
        clc
scrsav1 ret
        ENDP

;========================================
; Save the current screen.

Screen_Restore  PROC    NEAR

;--- restore screen

        sub     bx, bx          ;upper left
        mov     cx, WORD cols   ;load rows and columns
        dec     ch              ;end row
        dec     cl              ;end column
        mov     dx, s_seg       ;segment
        sub     ax, ax          ;offset
        call    ScrPut          ;restore screen
        mov     ax, s_seg       ;segment
        call    MemRel          ;release memory

;--- restore cursor

        mov     ax, WORD s_col  ;load row and column
        call    CurMov          ;position cursor
        mov     al, 1           ;non-zero
        call    CurSet          ;turn cursor on
        ret
        ENDP

;========================================
; Display a screen of data.

Screen_Display  PROC    NEAR
        mov     si, ax

        sub     al, al
        call    CurSet          ;cursor off
        call    Screen_Save     ;save screen

        mov     al, attr_help   ;help attribute
        call    AtrSet          ;set color

        call    ScrCls          ;clear screen
        sub     ax, ax
        call    CurMov          ;home cursor
        jmps    scrdis2         ;enter loop

scrdis1 call    WrtChr          ;display character
        call    CurAdv          ;advance cursor
scrdis2 cld
        lodsb                   ;load next character
        or      al, al          ;check if end
        jz      scrdis3
        cmp     al, 10          ;check if new line
        jne     scrdis1

        call    CurPos          ;get position
        inc     ah              ;next row
        sub     al, al          ;column zero
        call    CurMov          ;move cursor
        jmps    scrdis2         ;loop back

scrdis3 call    KeyWai          ;wait for key
        call    Screen_Restore  ;restore screen
        ret
        ENDP
