/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: _LOCKING.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int _locking(int fd, int cmd, off_t offset, long length)
	unsigned _lockRetryCount = __LOCK_RETRY_COUNT;
	unsigned _lockDelay = __LOCK_DELAY;

	Lock/unlock a region of the file at offset of the length.
	Under DOS share must be loaded to have locks on local drives.

	_lockRetryCount defines how often the _locking() mechanism should
	wait, if the requested function fails and no-block mode was
	not requested. (0 == 65536!)

	_lockDelay defines how many seconds the _locking() mechanism
	should wait between to attempts to get the lock. Range: 0..4020.
	If _lockDelay == 0, there is no delay between the attempts.

	Input:
		length > 0

	Return:
		-1 on failure
		0 on success
		1 if the region is already locked

	Note:
		<none>

	Conforms to:
		<none>

	See also:
		lock, unlock, locking

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.2  1997/10/27 03:53:28  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <dos.h>
#include <errno.h>			/* ESHARE */
#include <io.h>
#include <process.h>		/* sleep() */
#include <string.h>			/* memcpy() */
#include <sys/locking.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: _LOCKING.C 1.2 1997/10/27 03:53:28 ska Exp $";
#endif

/* Note: _locking() is the only function that uses these variables,
	therefore it's absolutely acceptable to define them here.
												- 1997/10/20 ska*/
_CLibVar int _lockRetryCount = __LOCK_RETRY_COUNT;
_CLibVar int _lockDelay = __LOCK_DELAY;


_CLibFunc int
_locking(const int fd
        ,const int cmd
        ,const off_t offset
        ,const long length)
{	struct REGPACK r, r1;
	REG int retVal, cnt;

	assert(length > 0);

			/* LK_UNLOCK includes LK_NOBLOCK! */
	r.r_ax = cmd == LK_UNLOCK? 0x5c01: 0x5c00;
	r.r_bx = fd;
	r.r_cx = HIWORD(offset);
	r.r_dx = LOWORD(offset);
	r.r_si = HIWORD(length);
	r.r_di = LOWORD(length);

	memcpy(&r1, &r, sizeof(r));		/* save the values for retry */

	if((retVal = _callDOS(&r)) == ESHARE && !(cmd & LK_NOBLOCK)) {
		/* lock violation, but no immediate return requested
			--> Retry! */
		cnt = _lockRetryCount;
		do {
			sleep(_lockDelay);
			memcpy(&r, &r1, sizeof(r));
		} while((retVal = _callDOS(&r)) == ESHARE && --cnt);
	}

	return retVal == ESHARE? 1: retVal? -1: 0;
}
