/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: _FOPEN.C $
   $Locker:  $	$Name:  $	$State: Exp $

	FILE *__fopen(char *_fname, int _fd, FILE *_fp, char *mode)
	FILE *fopen(char *fname, char *mode)
	FILE *fdopen(int fd, char *mode)
	FILE *freopen(char *fname, char *mode, FILE *fp)

	(Re-)Opens a streams.

	fopen(), fdopen(),& freopen() are preprocessor macros of __fopen().

	fopen() opens or creates a new stream.
	fdopen() associates a stream with an already opened file descriptor.
	freopen() associates an already opened stream with another file.

	mode is a combination of the following characters:
		'r':	if the file does not exist, fail; open for reading otherwise
		'w':	if the file exist, overwrite; create a new file, otherwise
		'a':	if the file exists, append; create a new file, otherwise
		'+':	grant read & write permission
		'b':	open in binary mode (no character translation)
		't':	open in text mode

	One of 'r', 'w' or 'a' is required.
	If neither 'b' nor 't' is specified, fopen() opens the stream in
	text mode, if the stream is associated with a TTY.

	Except in freopen(), on success a buffer of BUFSIZ bytes (specified
	at compile-time of the C Library) is associated with the stream.
	Both the buffer and the FILE structure itself are allocated via
	malloc().

	The default buffer mode is full buffering. If the stream is
	connected to a TTY, it's line buffering.

	If there is not enough space left to allocate the stream buffer,
	but the FILE structure has been allocated, the stream is opened
	in unbuffered mode.

	If _fp == NULL, a new file pointer is created; otherwise, the
	file pointer is closed and re-used.
	If _fname == NULL, _fd is assumed to be valid & opened file
	descriptor; otherwise, the file _fname is opened.
	If _fname == NULL && _fd < 0, __fopen() acts as fclose().

	Input:
		fname != NULL
		mode != NULL
		fp != NULL; a valid FILE pointer, may be closed
		fd an opened file descriptor

	Return:
		NULL: on failure
		else: opened FILE pointer (fp if freopen() succeeded)

	Note:
		<none>

	Conforms to:
		fopen, freopen: ANSI
		fdopen: POSIX

	See also:
		fclose, open

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"
#include <alloc.h>
#include <fcntl.h>
#include <io.h>				/* open(), isatty() */

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: _FOPEN.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc FILE *
__fopen(const char *fname, int fd, FILE *fp, const char *mode)
{
   int   open_mode = 0;
#if __MODE_IOTRAN
   int	 do_iosense = 1;
#endif
   int   fopen_mode = 0;
   FILE *nfp = NULL;

	assert(mode != NULL);

   /* If we've got an fp close the old one (freopen) */
   if (fp) {
      /* Careful, don't de-allocate it */
      fopen_mode |= (fp->mode & (__MODE_BUF | __MODE_FREEFIL | __MODE_FREEBUF));
      fp->mode &= ~(__MODE_FREEFIL | __MODE_FREEBUF);
      fclose(fp);
   }

   /* decode the new open mode */
   while (*mode)
      switch (*mode++) {
      case 'r':
		 fopen_mode |= __MODE_READ;
		 break;
      case 'w':
		 fopen_mode |= __MODE_WRITE;
		 open_mode = (O_CREAT | O_TRUNC);
		 break;
      case 'a':
		 fopen_mode |= __MODE_WRITE;
		 open_mode = (O_CREAT | O_APPEND);
		 break;
      case '+':
		 fopen_mode |= __MODE_RDWR;
		 break;
#if __MODE_IOTRAN
      case 'b':		/* Binary */
		 fopen_mode &= ~__MODE_IOTRAN;
		 do_iosense=0;
		 break;
      case 't':		/* Text */
		 fopen_mode |= __MODE_IOTRAN;
		 do_iosense=0;
		 break;
#endif
      }

   /* Add in the read/write options to mode for open() */
   switch (fopen_mode & (__MODE_READ | __MODE_WRITE)) {
   case 0:
      return NULL;
   case __MODE_READ:
      open_mode |= O_RDONLY;
      break;
   case __MODE_WRITE:
      open_mode |= O_WRONLY;
      break;
   default:
      open_mode |= O_RDWR;
      break;
   }

   /* Allocate the (FILE) before we do anything irreversable */
   if (!fp) {
      nfp = malloc(sizeof(FILE));
      if (!nfp)	return NULL;
   }

   /* Open the file itself */
   if (fname)
      fd = open(fname, open_mode, 0666);
   if (fd < 0) {			/* Grrrr */
      if (nfp)
		 free(nfp);
      return NULL;
   }

   /* If this isn't freopen create a (FILE) and buffer for it */
   if (fp == NULL) {
      fp = nfp;
      fp->next = __IO_list;
      __IO_list = fp;

      fp->mode = __MODE_FREEFIL;
      if( isatty(fd) ) {
		 fp->mode |= _IOLBF;
#if __MODE_IOTRAN
		 if( do_iosense ) fopen_mode |= __MODE_IOTRAN;
#endif
      }
      else
		 fp->mode |= _IOFBF;

      fp->bufstart = malloc(BUFSIZ);
      if (fp->bufstart == NULL) {	/* Oops, no mem */
      	/* Humm, full buffering with a two(!) byte * buffer. */
		 fp->bufstart = fp->unbuf;
		 fp->bufend = fp->unbuf + sizeof(fp->unbuf);
      }
      else {
		 fp->bufend = fp->bufstart + BUFSIZ;
		 fp->mode |= __MODE_FREEBUF;
      }
   }

   /* Ok, file's ready clear the buffer and save important bits */
   fp->bufpos = fp->bufread = fp->bufwrite = fp->bufstart;
   fp->mode |= fopen_mode;
   fp->fd = fd;

   return fp;
}
