/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: _CALLDOS.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int _callDOS(struct REGPACK far *rp)

	_callDOS() invokes the DOS interrupt and acts like the intr()
	function except that the flags are not loaded on invocation.
	This function does not create that much overhead as intr(),
	because a) the interrupt number is fixed,& b) it is assumed
	that the stack and its pointer is left unchanged. The subfunction
	AH == 0x4B is performed via intr() as it may violate point b).

	Another difference is the return value: If the carry flag is set on
	return of the interrupt, the return value is the OS error code,
	otherwise 0 (zero). This is useable for most of the functions of
	the DOS API.
	If the OS returned the error state (Carry flag set), but the
	AX register is zero, 0x31 is returned. This number is marked
	"reserved" in the list of the error numbers.

	The DOSREGS structure is an overlayed union of struct REGPACK to
	simplify to access the byte portions of each register. It may be
	passed instead of the struct REGPACK.

	Input:
		rp must be != NULL

	Return:
		OS error code, or 0 (zero) if success
		*rp updated

	Note:
		The reason to include this function into the core section of
		the CLib is derived mostly from the point of view to centralize
		DOS API calls, but remain as portable as possible. The overhead
		the intr() function produces seems to add a point, too.

		On the other hand, if all calls to the DOS API are made through
		this function, this is function is not only "just another way
		to invoke an interrupt" but can be imagined as central system
		call. Then one could port this function by interpreting the
		register values and mapping them to other ways to perform the
		requested functionality.

		Another point is that this function can be adjusted to handle
		different DOS versions, e.g. to simulate function 0x6C, which
		is nearly the direct corresponding function for the POSIX open()
		function.

		To use REGPACK as the register structure may seem to be overestimated,
		though, there is no register [except SS:SP, CS:IP] that is never
		used as an argument, even BP is used: DOS-5F-05!
		On the other hand, to use _callDOS() with function DOS-33-05 (Get
		Boot Drive) requires to allocate a full REGPACK structure,
		to setup and read the register via memory operations,& to do the
		overhead within _callDOS() itself, as it loads/saves all registers,
		not only AX. Instead, the "immediate registers" would provide a
		much more efficient code:
			_AX = 0x3305;
			geninterrupt(0x21);
			if(_AL == 0xFF) error();
			drive = _DL;
		In the view of portablility, the _callDOS() variant is best;
		when always using the _callDOS() function an optimizer can be
		written, that identifies simple _callDOS() calls. In view of
		this case, try to keep any assignments to the REGPACK structure
		passed to _callDOS() near the call of _callDOS.

	Conforms to:
		<none>

	See also:
		intr(), _intr(), struct REGPACK, DOSREGS

	Target compilers:
		Borland C (using inline assembly)

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.3  1998/01/29 07:09:32  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <dos.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: _CALLDOS.C 1.3 1998/01/29 07:09:32 ska Exp $";
#endif

_CLibFunc int _callDOS(struct REGPACK far * const r)
{
	assert(r != NULL);

	if(((DOSREGS * const)r)->h.ah == 0x4B) {	/* special handling */
		/* invoke _intr() */
		return (_intr(INT_DOS, r) & FLAG_CARRY)? (r->r_ax? r->r_ax: 0x31): 0;
	}

	asm {
		push es		 /* save registers that should not be touched */
		push ds
		push bp

		lds bx, r	 /* DS:BX := pointer to the registers */

		mov ax, ds:[bx]
		mov cx, ds:[bx+4]
		mov dx, ds:[bx+6]
		mov bp, ds:[bx+8]
		mov si, ds:[bx+10]
		mov di, ds:[bx+12]
		mov es, ds:[bx+16]
		push Word Ptr ds:[bx+2]		 /* BX */
		mov ds, ds:[bx+14]
		pop bx
		stc					/* Note on Win95 API: Ralf Brown's interrupt
								list: Carry should be set to ensure
								compatibly with pre-DOS7. Existing API
								functions should ignore the carry. */

		int 21h		 /* Invoke DOS API */

		push bp
		mov bp, sp
		mov bp, [bp+2]	 /* get original frame pointer */

		push ds
		push bx
		lds bx, r

		mov Word Ptr ds:[bx], ax
		pop Word Ptr ds:[bx+2]			 /* BX */
		mov Word Ptr ds:[bx+4], cx
		mov Word Ptr ds:[bx+6], dx
		mov Word Ptr ds:[bx+10], si
		mov Word Ptr ds:[bx+12], di
		pop Word Ptr ds:[bx+14]			 /* DS */
		mov Word Ptr ds:[bx+16], es
		pop Word Ptr ds:[bx+8]			 /* BP */

		pop cx			 /* BP is already restored */

		pushf			 /* save the flags */
		pop Word ptr ds:[bx+18]

		pop ds			 /* restore the registers */
		pop es

		jc error1
	}
	return 0;			/* no error status */
error1:
	asm {
		or ax, ax
		jne error2
		mov ax, 31h
	}
error2:
	/* Default return value is the register AX */
	return _AX;		/* Shut warning "function should return a value" */
}
