/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FOREACH.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	This is an utility for Micro-C only, it was coded to not use another
	library or source code.

		FOREACH [{option}] [file | pattern] {<command token>}

	Options:
		/e:<ext>[{,<ext>}]	- ignore all extensions that are not listed here
		/i		- ignore critical errors, e.g. "Drive not ready"
		/t:<token delimiters>	- specify the list of characters that
				delimit two tokens. If the very first character is 's',
				the character means "any whitespaces".
				If the <token delimiters> list is empty, each line is
				interpreted as a filename awholely.
				Default: "s" --> delimit at whitespaces
		/f:<file>	- the command tokens are read out of this file.
				In this mode the file is read into the memory and
				this memory image is dumped into the batch file. All
				characters are copied, except the replaced placeholders.
				If this option is present, no "<command token>" arguments
				are read from the command line.
		/a:<attributes>	- Instead of reading the filenames out of a file
				they are created by a pattern match.
				The first non-option argument is used as a pattern.
				If the "/f" option is present, all non-option arguments
				are used as patterns.
				The attributes consists of:
					a - archive attribute
					d - directory
					h - hidden attrbute
					l - label
					r - read only
					s - system
		/m:<string>	- Makefile mode
				In this mode the file is read until a line beginning
				with <string> is found. The remaining characters of
				this line and any concated lines are tokenized and
				treated as the filenames.
				Lines can be concated by placing a '\\' character as
				the very last character onto the line.
				After processing this line the program terminates.
		/o:[+]outfile	- specifies an output file.
				Instead of creating and executing a batchfile all
				matches are dumped into this file. No additional
				lines or characters are written, neither the "@echo off"
				nor the newline.
				If the plus sing is present, the file is appended
				rather than overwritten.

	It reads the file, breaks all lines into whitespace separated tokens
	and interpretes them as filenames. The "/E" switch causes to ignore
	all filenames that have a different extension as the given ones

	Patterns can consist of any DOS-like wildcards with three exceptions:
	a) If a pattern reads "*", it's replaced by "*.*".
	b) Each filename component may contain wildcards.
	c) The drive letter '?' matches any drive through C-Z; '*' matches A-Z.

	The command tokens may consist of any arguments, they will be joined
	together delimited by a single space, then scanned for placeholders,
	after substituting the placeholders the line(s) is written into a
	temporary batchfile and that is passed to system() function.
	If dumped into a batchfile, the command tokens are automatically
	preceeded by the line
		"@echo off"
	and an additional newline character is appended.

	You may surround an argument containing whitespaces by double
	quotes, e.g.: "/m:SRC =".

	Placeholder:
		%!****!
	The **** stands for any combination of the following identifiers:
		a	the filename, as written
		A	the fully-qualified filename (equal to DPF)
		f	name & extension of filename, as written
		F	name & extension of filename in 8.3 format with dot if present
		d	drive from filename
		D	drive from fully-qualified filename
		p	path from file, not trailing \, but root as \
		P	path from fully-qualified filename, as p
		n	name from filename
		N	name from filename with trailing dot
		e	extension from filename
		E	extension from filename with leading dot
		:	if placed behind d or D, colon if there was a drive letter
			inserted
		\	if placed behind p or P, backslash if there was a path
			inserted and it was not root
		/	any backslashes are converted into forward slashes
		#	newline character
		@	tabulator character
	The sequence: NE is interpreted as Ne
	The sequence: ne is interpreted as nE

	File Revision:    Revision 1.5  1998/02/11 07:40:46  ska
*/

//#define TEST

#include <stdio.h>
#ifdef _MICROC_
#include <file.h>
#define const
#endif

#define NODE_STRING 0xff
#define NODE_PLACEHOLDER 0x00

static char const rcsid[] = 
	"$Id: FOREACH.C 1.5 1998/02/11 07:40:46 ska Exp ska $";

char drive, *path, *name, *ext;	/* relative filename portions */
char adrive, *apath;	/* fully-qualified ones */
char odrv;	/* original path/drive */

char appName[9];
char *batName;	/* temporary batch file */

FILE *ofile;	/* output file */

char buf[2048];			/* The line buffer */
char hbuf[1024];		/* temporary buffer */
char *cwd[26];			/* current working directories */
		/* If a removeble drive is specified and function wants to
			access this drive, set_drive() will work, but any other
			access results into a crictical error. To prevent this
			cwd[] stores whether or not this drive is accessable by
			storing the current working directory of the particular
			drive. The cwd is retreived exactly once. */
char errcwd[] = "";		/* erroreous cwd. Returned by getcwd() if
			a drive is inaccessable. Because of its contents one
			can use it as a substitute for "root directory" in
			case of failure. */

struct ICMDLINE {		/* intermediate command line */
	struct ICMDLINE *nxt;
	int i_node;
	char i_line[1];
};

struct ICMDLINE head;					/* this element ignored! */
struct ICMDLINE *act;	/* last entry */

struct EXTENSION {
	struct EXTENSION *nxt;
	char *ext_name;
} *ext_head = NULL;		/* all accepted extensions */

static char whitespaces[] = " \t\n\r";
char *tokens = "s";
int attr = 0;

#ifdef TEST
void Xkbget(void)
{	puts("Hit 'q' to quit; any key to proceed");
	if(kbget() == 'q')
		error(NULL);
}
#define kbget Xkbget
char *Xmalloc(int size)
{	
	printf("coreleft = %u;; allocate = %u;; left = %u\n", coreleft(), size, coreleft() - size);
//	kbget();
	return malloc(size);
}
#define malloc Xmalloc
char *Xstrdup(char *s)
{	
	printf("coreleft = %u;; allocate = %u;; left = %u\n", coreleft(), strlen(s), coreleft() - strlen(s));
	//kbget();
	return strdup(s);
}
#define strdup Xstrdup
#endif /* TEST */

register error(unsigned args)
{	char *a;

	a = nargs() * 2 + &args;

	if(ofile) fclose(ofile);
	if(args) {
		_format_(a, buf);
		fputs(appName, stderr);
		fputs(": Error: ", stderr);
		fputs(buf, stderr);
		putc('\n', stderr);
		exit(30);
	}
	exit(0);
}

/* Allocate memory and return an error on failure */
char *dupstr(char *s)
{	char *h;

	if(!(h = strdup(s)))
		error("Memory overflow");
	return h;
}

/* Allocate memory and return an error on failure */
char *getmem(unsigned size)
{	char *h;

	if(!(h = malloc(size)))
		error("Memory overflow");
	return h;
}

/* Add an accepted extension */
void addExt(char *ext)
{	struct EXTENSION *h;

	h = getmem(sizeof(struct EXTENSION));
	h->nxt = ext_head;
	h->ext_name = dupstr(ext);
	ext_head = h;
}

int extValid(char *ext)
{	struct EXTENSION *act;

	if(!(act = ext_head)) return 1;

	if(!ext) ext = "";
	do if(strcmp(ext, act->ext_name) == 0) return 1;
	while(act = act->nxt);
	return 0;
}


void nxtFNam(char *digit)		/* generate the next filename */
{	
	++digit;
loop:
	if(!isxdigit(*--digit))
		error("Cannot find a non-existing temporary batchfile name");
	++*digit;
	if(*digit == '9' + 1)
		*digit = 'A';
	else if(*digit == 'F' + 1) {
		*digit = '0';
		goto loop;
	}
}

char *getcwd(int drv)
{	char **p;

	if(!*(p = &cwd[drv])) {
		set_drive(drv);
		if(get_drive() == drv && !getdir(hbuf + 1)) {
			*hbuf = '\\';
				/* root dir -> the "\" is already there */
			*p = dupstr(hbuf[1] == '\\'? &hbuf[1]: hbuf);
		} else *p = errcwd;
		set_drive(odrv);
	}
	return *p;
}

int slashPut(char *s, FILE *f)
{	int retVal;
	char *p, *q;

	q = p = dupstr(s);
	while(q = strchr(q, '\\'))
		*q = '/';
	retVal = fputs(p, f);
	free(p);
	return retVal;
}

/* Split the filename and invoke the command */
void handleFName(char *fnam)
{	char *p, *opa, *fn, *ffile, *fFile;
	struct ICMDLINE *q;
	char hdrive[2];
	FILE *f;				/* temporary batch file */
	int slash;
#ifdef TEST
int cnt;
cnt = 0;

printf("\nFree core %u\n", coreleft());
#endif 

	fn = dupstr(fnam);

	if(fnam[1] == ':') {	/* drive spec */
		adrive = toupper(drive = *fnam);
		fnam += 2;
	}
	else {
		drive = 0;
		adrive = odrv + 'A';
	}

	opa = getcwd(adrive - 'A');
	if(!opa[1])	/* root dir --> "" */
		opa = "";			/* on failure --> root dir */


#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	p = strchr(fnam, '\0');
	while(--p >= fnam && *p != '/' && *p != '\\');
	++p;
	if(p != fnam) {			/* path portion */
		p[-1] = 0;
		path = dupstr(fnam);
		if(*fnam != '/' && *fnam != '\\' && *fnam) {
			concat(apath = getmem(strlen(path) + strlen(opa) + 2)
			, opa, "\\", path);
		}
		else apath = dupstr(path);	/* absolute path incl root dir */
	}
	else {
		path = NULL;
		apath = dupstr(opa);
	}
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 

	p = strrchr(fnam = p, '.');
	ffile = dupstr(fnam);
	if(p != fnam && p) *p = 0;
	name = dupstr(fnam);
	if(strlen(name) > 8)
		name[8] = 0;
	if(p != fnam && p && p[1]) {
		ext = dupstr(p + 1);
		if(strlen(ext) > 3)
			ext[3] = 0;
		concat(fFile = getmem(13), name, ".", ext);
	}
	else {
		ext = NULL;
		if(p) concat(fFile = getmem(10), name, ".");
		else fFile = dupstr(name);
	}

#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	if(extValid(ext)) {

		if(ofile) f = ofile;
		else {
			if((f = fopen(batName, "wt")) == NULL)
				error("Cannot create temporary batch file: %s", batName);

#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 

			fputs("@echo off\n", f);
		}

		q = head;
		hdrive[1] = 0;

#define addStr(a) fputs((a), f)
#define addCh(c) putc((c), f)
#define xaddStr(a) (slash == '/'? slashPut((a),f): fputs((a),f))
		while(q = q->nxt)
			if(q->i_node == NODE_STRING) 
				addStr(q->i_line);
			else {				/* placeholder */
				slash = '\\';
				p = q->i_line - 1;
				while(*++p) switch(*p) {
				case '@':	/* tabulator character */
					addCh('\t');
					break;
				case '#':	/* newline character */
					addCh('\n');
					break;
				case '/':	/* convert backslashes to forward slashes */
					slash = '/';
					break;
				case 'a':	/* the filename */
					xaddStr(fn);
					break;
//				case 'A':	/* the fully-qualified filename */
//					replaced by DPf
				case 'f':	/* name & extension of filename */
					addStr(ffile);
					break;
				case 'F':	/* name & extension of filename in 8.3 */
					addStr(fFile);
					break;
				case 'd':	/* drive from filename */
					if(drive) {
						addCh(drive);
						goto addColon;
					}
					break;
				case 'D':	/* drive from fully-qualified filename */
					addCh(adrive);
				addColon:
					if(p[1] == ':') addCh(':');
					break;
				case 'p':	/* path from file, not trailing \, but root as \ */
					if(path) {
						xaddStr(path);
						if(!*path || p[1] == '\\') addCh(slash);
					}
					break;
				case 'P':	/* path from fully-qualified filename, as p */
					if(apath) {
						xaddStr(apath);
						if(!*apath || p[1] == '\\') addCh(slash);
					}
					break;
				case 'n':	/* name from filename */
					addStr(name);
					break;
				case 'N':	/* name from filename with trailing dot */
					addStr(name);
					addCh('.');
					break;
				case 'E':	/* extension from filename with leading dot */
					addCh('.');
				case 'e':	/* extension from filename */
					if(ext) addStr(ext);
					break;
				//case ':':	break;
				//case '\':	break;
				//default: break;
				}
				break;
			}

		if(!ofile) {
			if(putc('\n', f) != '\n') {
				fclose(f);
#ifndef TEST
				unlink(batName);
#endif 
				error("Cannot write temporary batch file: %s", batName);
			}
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
			fclose(f);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
			system(batName);
#ifndef TEST
			unlink(batName);
#else 
			p = strrchr(batName, '.') - 1;
			nxtFNam(p);
#endif 
		}
	}

#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	free(fn);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	free(fFile);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	free(ffile);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	if(path) free(path);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	free(apath);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	free(name);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
#endif 
	if(ext) free(ext);
#ifdef TEST
printf("#%u: Free core %u\n", ++cnt, coreleft());
kbget();
#endif 
}


void handleTokens(char *p)
{	
	if(tokens) {
		if(p = strtok(p, tokens))
			do handleFName(p);
			while(p = strtok(NULL, tokens));
	}
	else handleFName(p);
}


void handlePattern(void)
{	int len;

	if((len = strlen(buf) + 1) > sizeof(buf) / 2)
		error("Filename pattern matching result too long");
	memcpy(&buf[sizeof(buf) / 2], buf, len);
	handleFName(&buf[sizeof(buf) / 2]);
}

void handleProbedPattern(void)
{	struct FF_block ff;

	if(findfirst(buf, ff, attr) == 0)
		handlePattern();
}


void handleDirEnd(char *p)
{
	if(attr & DIRECTORY) {
		if(p > buf && p[-1] == '\\') {
			/* root drive match?? */
			if(p == buf || p[-2] == ':' && p == &buf[3]) {	/* yes */
				/* match! */
				handlePattern();
				return;
			}
			else p[-1] = '\0';		/* the backslash would cause findfirst()
										to fail */
		}
		handleProbedPattern();
	}
}

char *skipDelim(char *pattern)				/* skip filename delimiter */
{	while(*pattern == '\\')
		++pattern;
	return pattern;
}

void searchPath(char *p, char *pattern)
{	char *begin;
	char *top;					/* current lowest path component;	*/
	struct FF_block ff;
	int eop;					/* end of pattern */

top = getmem(128);
free(top);

	top = p;
	begin = NULL;

loop:		/* skip path delimiters && "." directories */
	pattern = skipDelim(pattern);
	if(!*pattern) {		/* end of pattern --> treat as "." */
		handleDirEnd(p);
		return;
	}
	if(*pattern == '.') switch(pattern[1]) {
	case '\\':
		++pattern;
		goto loop;
	case '\0':
		handleDirEnd(p);
		return;
	case '.':
		if(!pattern[2]) {			/* probe updir */
			if(attr & DIRECTORY) {
				strcpy(p, "..");
				handleProbedPattern();
			}
			return;
		}
		if(pattern[2] == '\\') {		/* up one dir */
			if(begin) {					/* at least one is possible */
				p = begin;
				/* up begin; begin[-1] == '\\' */
				--begin;
				while(--begin > top && *begin != '\\');
				if(begin == top) begin = NULL;
				else ++begin;
			}
			else {						/* no upping possible */
				memcpy(p, "..\\", 3);
				p += 3;
				pattern += 3;
			}
			goto loop;
		}
		break;
	}

	begin = p;			/* start of current filename component */
	while(*pattern && *pattern != '\\') {
		*p++ = *pattern++;
	}
	eop = *pattern == '\0';

	*p = '\0';
	if(strpbrk(begin, "?*")) {		/* wildcard pattern matching */
		/* If we shall match "*" --> we'll search for "*.*" */
		if(begin[0] == '*' && begin[1] == '\0')
			memcpy(&begin[1], ".*", 3);
		if(findfirst(buf, ff, eop? attr: attr | DIRECTORY) == 0) do {
			/* we do NOT match "." and ".."! */
			if((ff.FF_attrib & DIRECTORY) && ff.FF_name[0] == '.'
			 && (ff.FF_name[1] == '\0'
			  || ff.FF_name[1] == '.' && ff.FF_name[2] == '\0'))
			  	continue;
			strcpy(begin, ff.FF_name);
			if(eop) handlePattern();
			else if(ff.FF_attrib & DIRECTORY) {
				/* within a pattern -> next component match */
				p = strchr(begin, '\0');
				*p = '\\';
				*++p = '\0';
				searchPath(p, pattern);
			}
		} while(findnext(ff) == 0);
	}
	else if(eop)
		handleProbedPattern();		/* end of filename pattern found */
	else { 					/* in the middle of a filename pattern */
		*p++ = '\\';
		goto loop;
	}
}

void searchRoot(char *p, char *pattern)		/* start a search at the root path */
{	if(*pattern == '\\')		/* start at root */
		*p++ = '\\';
	searchPath(p, pattern);
}

void searchDrive(int from, int to, char *pattern)
{	int drive;

	drive = get_drive();
	from = toupper(from) - 'A';
	to = toupper(to) - 'A';
	buf[1] = ':';
	while(from <= to) {
		set_drive(from);
		if(get_drive() == from) {
			set_drive(drive);
			*buf = from + 'A';
			searchRoot(&buf[2], pattern);
		}
		++from;
	}
}


void mkNode(int node, char *s)
{
	if(!s || !*s) return;
	(act = act->nxt = getmem(sizeof(struct ICMDLINE) + strlen(s)))->i_node = node;
	strcpy(act->i_line, s);
}

/* Decompose the execution lines into the intermediate command line
	describing where to insert the replacements for the placeholders.
*/
void decomposeArgLine(char *p)
{	char *s, *q, *h;

	q = p;
	while(s = strstr(p, "%!")) {
		if(s[2] == '!') {	/* ignore this placeholder */
			s[2] = '\0';
			mkNode(NODE_STRING, q);
			q = p = s + 3;		/* skip %!! */
		}
		else {				/* check for a valid placeholder */
			*s = '\0';
			mkNode(NODE_STRING, q);
			p = s + 2;
			h = hbuf - 1;
			do {
				switch(*++h = *p) {
				case 'd': case 'D':
					if(strchr("pPnNfF", p[1]))
						*++h = ':';
					break;
				case 'p': case 'P':
					if(strchr("nNfF", p[1]))
						*++h = '\\';
					break;
				case 'N':
					if(p[1] == 'E') {
						++p;
						*++h = 'e';
					}
					break;
				case 'n':
					if(p[1] == 'e') {
						++p;
						*++h = 'E';
					}
					break;
				case 'A':
					*h = 'D';
					*++h = ':';
					*++h = 'P';
					*++h = '\\';
					*++h = 'F';
					break;
				case 'e': case 'E': case ':': case '\\': case '@':
				case '#': case 'a':
					break;
				default:
					error("Character: '%c' is invalid within a placeholder", *p);

				}
				if(h >= &hbuf[sizeof(hbuf)])
					error("Placeholder longer than %u bytes", sizeof(hbuf));
			} while(*++p != '!');
			q = ++p;
			*++h = '\0';
			mkNode(NODE_PLACEHOLDER, hbuf);
		}
	}
	mkNode(NODE_STRING, q);
	act->nxt = NULL;
}

void hlpScreen(void)
{
	puts("Useage: FOREACH [{options}] [file | pattern] {[commands]}");
	puts("\nOptions (in any order):");
	puts("/e:<ext>[{,<ext>}]	- enumerate all valid extensions");
	puts("/i	- ignore critical errors, e.g. \"Drive not ready\"");
	puts("/t:<token delimiters>");
	puts("/f:<file>	- the command tokens are read out of this file");
	puts("/a:<attributes>	- initiate pattern matching");
	puts("/m:<string>	- Makefile mode");
	puts("/o:[+]outfile	- specifies an output file");
	puts("\nFor more information see the header of the source file!");

	exit(127);
}

char *arg(char *p)			/* check for non-empty argument */
{
	if(!strchr(":=", p[2]) || !p[3])
		error("Argument missing or empty in: %s", p);
	return &p[3];
}

char *arg1(char *p)			/* check for argument */
{
	if(!strchr(":=", p[2]))
		error("Argument missing in: %s", p);
	return &p[3];
}

int scanAttr(char *p)
{	int attr;

	attr = ARCHIVE;
	while(*p) switch(toupper(*p++)) {
	case 'A': break;
	case 'D': attr |= DIRECTORY; break;
	case 'S': attr |= SYSTEM; break;
	case 'H': attr |= HIDDEN; break;
	case 'L':
	case 'V': attr |= VOLUME; break;
	case 'R': attr |= READONLY; break;
	default: error("Invalid attribute character %c", p[-1]);
	}

	return attr;
}

FILE *openFile(char *fnam)
{	FILE *f;

	if(strcmp(fnam, "-") == 0)
		return stdin;

	if((f = fopen(fnam, "rt")) == NULL)
		error("Cannot open file %s", fnam);
	return f;
}

/* Retrieve the application's name */
void getAppName(char *fnam)
{	char *p;

	p = strchr(fnam, '\0');
	while(--p >= fnam && !strchr("/\\:", *p))
		if(*p == '.')
			*p = '\0';
	memcpy(appName, p + 1, 8);
	appName[8] = '\0';		/* for names longer than 8 chars */
}

/*
	Micro-C does not detect quoted arguments, e.g.:
		FOREACH "/m=SRC =" makefile
	would be impossible to write.

	This function searches for such arguments and rejoins them
	together.
*/
void preprocess(char **argv)
{	char *p, **to;

	to = argv;
	while(*++to = *++argv) if(**to == '"') {
		p = ++*to;			/* skip leading quote */
		while((p = strchr(p, '"')) == NULL && argv[1] != NULL) {
			p = strchr(*to, '\0');
			*p = ' ';
			++argv;
		}
		if(p) {
			if(p[1])
				memmove(p, &p[1], strlen(&p[1]) + 1);
			else
				*p = '\0';
		}
	}
}

/* The critical error handler just always returns "fail" */
void critError(void)
asm {
	push ds
	mov ax, cs
	mov ds, ax
	mov dx, Offset critErrIgn
	mov ax, 2524h
	int 21h
	pop ds

	jmp short critEnd

critErrIgn:		;; This is the critical error handler :-)
	mov ax, 3
	iret

critEnd:
}

main(int argc, char **argv)
{	FILE *f;			/* The filename file */
	FILE *f1;			/* command file */
	char *p, *q;
	int ch, crit;
	char *cmdFile, *mak, **attrArgv;

#ifdef TEST
puts("\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");
#endif 

	if(argc <= 1) hlpScreen();

	preprocess(argv);	/* rejoin quoted arguments */

	cmdFile = mak = NULL;
	act = head;
	crit = 0;
	memset(cwd, 0, sizeof(cwd));

	if(getcwd(odrv = get_drive()) == errcwd)
		error("Cannot retreive current directory");

	getAppName(argv[0]);

	while((p = *++argv) && *p == '/') switch(p[1]) {
	case 'e': case 'E':		/* valid extensions */
		if(!strchr(":=", p[2]) || !(p = strtok(p + 3, ",")))
			error("Argument missing in: %s", p);
		do addExt(p);
		while(p = strtok(NULL, ","));
		break;
	case 'f': case 'F':		/* command file */
		cmdFile = arg(p);
		break;
	case 'a': case 'A':		/* attribute match */
		attr = scanAttr(arg1(p));
		break;
	case 't': case 'T':		/* tokenize */
		tokens = arg1(p);
		break;
	case 'm': case 'M':		/* makefile switch */
		mak = arg(p);
		break;
	case 'i': case 'I':		/* install critical error handler */
		crit = 1;
		break;
	case 'o': case 'O':		/* outfile */
		if(ofile) error("Only one outputfile can be opened");
		p = arg(p);
		if(*p == '+') ofile = fopen(p + 1, "at");
		else if(strcmp(p, "-") == 0)
			ofile = stdout;
		else ofile = fopen(p, "wt");
		if(!ofile) error("Cannot open output file: \"%s\"", p);
		break;
	case '?': case 'h': case 'H':
		hlpScreen();
	default:				/* Unknown switch */
		error("Unknown switch: %s", *argv);
	}

	if(crit) critError();		/* install critical error handler */

	/* Make filename of temporary batchfile */
	if(!getenv("TEMP", buf) && !getenv("TMP", buf) && !getenv("TEMPDIR", buf)) {
		*buf = odrv + 'A';
		buf[1] = ':';
		strcpy(&buf[2], cwd[odrv]);
	}
	p = strchr(buf, '\0');
	if(p[-1] != '\\') *p++ = '\\';
	strcpy(p, "FOR$0000.BAT");
	p += 7;							/* Place of least-significant digit */
	while((f = fopen(buf, "rb")) != NULL) {		/* already existing */
		fclose(f);
		nxtFNam(p);
	}
	batName = dupstr(buf);

	if(mak && attr)
		error("Pattern matching and Makefile processing cannot be combined");

	if(!*argv)
		error("Missing input file or file pattern");
	if(!attr)
		f = openFile(*argv);
	else attrArgv = &argv[-1];
	++argv;

	if(!cmdFile) {			/* clobber the rest of the command line as command */
		if(!*argv)	error("Missing command to be invoked");
		p = buf;
		do p = stpcpy(p, *argv), *p++ = ' ', *argv = NULL;
		while(*++argv);
		p[-1] = 0;
		decomposeArgLine(buf);
	}
	else {					/* read the command file */
		f1 = openFile(cmdFile);
		if(f1 == stdin && f == f1)
			error("The stdin cannot be read as input and command file simultaneously");
		while(fgets(buf, sizeof(buf) - 1, f1)) {
			strcat(buf, "\n");				/* Micro-C strips the '\n' */
			decomposeArgLine(buf);
		}
		fclose(f1);
	}


	if(*tokens == 's') {
		p = getmem(strlen(&tokens[1]) + sizeof(whitespaces));
		strcat(tokens = strcpy(p, &tokens[1]), whitespaces);
	}
	if(!*tokens) tokens = NULL;

	if(attr) {
		while(q = p = *++attrArgv) if(*p) {
			while(q = strchr(q, '/'))
				*q = '\\';
			if(p[1] == ':') {
				if(*p == '?') searchDrive('C', 'Z', &p[2]);
				else if(*p == '*') searchDrive('A', 'Z', &p[2]);
				else if(isalpha(*p)) searchDrive(*p, *p, &p[2]);
				else error("Invalid drive specification in search pattern: %s", p);
			}
			else searchRoot(buf, p);
		}
	}
	else if(!mak) {
		while(fgets(buf, sizeof(buf), f))
			handleTokens(buf);
	}
	else {							/* Make file processing */
		/* Skip to line */
		while(fgets(buf, sizeof(buf), f))
			if(memcmp(buf, mak, strlen(mak)) == 0) {
			/* found, process the rest of the line */
				q = buf + strlen(mak);
				do {
					/* Check for continueous line */
					p = strchr(q, '\0');
					ch = 0;					/* don't read next line */
					if(p[-1] == '\\') {				/* continueous line */
						--p;
						ch = 1;						/* read next line */
						if(tokens && !strchr(tokens, p[-1])) {		/* need a character from next line */
							do {
								while(p < buf + sizeof(buf) - 1 && !strchr(tokens, (ch = getc(f))) && ch != EOF)
									*p++ = ch;
								if(ch == '\n') {
									if(p[-1] == '\\') {
										--p;
										continue;
									}
									ch = 0;			/* don't read next line */
								}
								break;
							} while(1);
						}
						*p = '\0';
					}
					handleTokens(q);
				} while(ch && fgets(q = buf, sizeof(buf), f));
				error(NULL);
			}
		/* not found -> error */
		error("Cannot find Makefile line starting with: %s", mak);
	}

	error(NULL);
}
