/************************************************************************
* Author      : Tiago Dionizio <tiago.dionizio@gmail.com>               *
* Library     : lgzip - a gzip file access binding for Lua 5            *
*               based on liolib.c from Lua 5.0 library                  *
* Agena port:   Alexander Walz, http://agena.sourceforge.net            *
*                                                                       *
* Permission is hereby granted, free of charge, to any person obtaining *
* a copy of this software and associated documentation files (the       *
* "Software"), to deal in the Software without restriction, including   *
* without limitation the rights to use, copy, modify, merge, publish,   *
* distribute, sublicense, and/or sell copies of the Software, and to    *
* permit persons to whom the Software is furnished to do so, subject to *
* the following conditions:                                             *
*                                                                       *
* The above copyright notice and this permission notice shall be        *
* included in all copies or substantial portions of the Software.       *
*                                                                       *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  *
* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  *
* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     *
* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                *
************************************************************************/

#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <zlib.h>

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"

#ifndef int64_t
#define int64_t long long int
#endif
#ifndef off64_t
#define off64_t int64_t
#endif


#define gzip_c
#define LUA_LIB

#define AGENA_LIBVERSION	"GZIP 0.1.3 binding as of January 11, 2023, based on Tiago Dionizio\'s lgzip 0.3\npackage for Lua 5.0\n"

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_LZLIBLIBNAME "gzip"
LUALIB_API int (luaopen_gzip) (lua_State *L);
#endif


/*
** {======================================================
** FILE Operations
** =======================================================
*/


#define FILEHANDLE		"gzipfile"


static int pushresult (lua_State *L, int i, const char *filename) {
  int en = errno;  /* calls to Lua API may change this value */
  if (i) {
    lua_pushboolean(L, 1);
    return 1;
  }
  else {
    lua_pushnil(L);
    if (filename)
      lua_pushfstring(L, "%s: %s", filename, strerror(en));
    else
      lua_pushfstring(L, "%s", strerror(errno));
    lua_pushnumber(L, errno);
    return 3;
  }
}


static gzFile *gztopfile (lua_State *L, int findex) {
  gzFile*f = (gzFile*)luaL_checkudata(L, findex, FILEHANDLE);
  if (f == NULL) luaL_argerror(L, findex, "bad file");
  return f;
}


static gzFile gztofile (lua_State *L, int findex) {
  gzFile*f = gztopfile(L, findex);
  if (*f == NULL)
    luaL_error(L, "Error in " LUA_QS " package: attempt to use a closed file", "gzip");
  return *f;
}



/*
** When creating file handles, always creates a `closed' file handle
** before opening the actual file; so, if there is a memory error, the
** file is not left opened.
*/
static gzFile *newfile (lua_State *L) {
  gzFile *pf = (gzFile*)lua_newuserdata(L, sizeof(gzFile));
  *pf = NULL;  /* file handle is currently `closed' */
  luaL_getmetatable(L, FILEHANDLE);
  lua_setmetatable(L, -2);
  return pf;
}


static int aux_close (lua_State *L) {
  int nargs, i, c;
  nargs = lua_gettop(L);
  c = 0;
  if (nargs < 1)
    luaL_error(L, "Error in " LUA_QS ": need at least one file handle.", "gzip.close");
  for (i=1; i <= nargs; i++) {
    gzFile f = gztofile(L, i);
    int ok = (gzclose(f) == Z_OK);
    if (ok) {
      c++;
      lua_getglobal(L, "gzip");
      lua_getfield(L, -1, "openfiles");
      if (lua_istable(L, -1)) {  /* avoid segmentation faults */
        lua_pushfstring(L, "gzipfile(%p)", f);
        lua_pushnil(L);
        lua_rawset(L, -3);
      }
      agn_poptoptwo(L);  /* delete "gzip" and "openfiles" */
      *(gzFile*)lua_touserdata(L, i) = NULL;  /* mark file as closed */
    }
  }
  return c;
}


static int gzip_close (lua_State *L) {
  return pushresult(L, aux_close(L), NULL);
}


static int io_gc (lua_State *L) {
  gzFile *f = gztopfile(L, 1);
  if (*f != NULL)  /* ignore closed files */
    aux_close(L);
  return 0;
}


static int io_tostring (lua_State *L) {
  char buff[128];
  gzFile *f = gztopfile(L, 1);
  if (*f == NULL)
    strcpy(buff, "closed");
  else
    sprintf(buff, "%p", lua_touserdata(L, 1));
  lua_pushfstring(L, "gzipfile(%s)", buff);
  return 1;
}


static int gzip_open (lua_State *L) {
  const char *filename = luaL_checkstring(L, 1);
  const char *mode = luaL_optstring(L, 2, "rb");
  gzFile *pf = newfile(L);
  *pf = gzopen(filename, mode);
  if (*pf == NULL)
    return pushresult(L, 0, filename);
  else {
    /* enter new open file to global io.openfiles table */
    lua_getglobal(L, "gzip");
    lua_getfield(L, -1, "openfiles");
    if (lua_istable(L, -1)) {  /* 0.20.2, avoid segmentation faults */
      lua_pushfstring(L, "gzipfile(%p)", *pf);
      lua_createtable(L, 2, 0);
      lua_pushstring(L, filename);
      lua_rawseti(L, -2, 1);
      lua_pushstring(L, mode);
      lua_rawseti(L, -2, 2);
      lua_rawset(L, -3);
    }
    agn_poptoptwo(L);  /* delete "io" and "openfiles" */
    return 1;
  }
}


static int gzip_readline (lua_State *L);

static void aux_lines (lua_State *L, int idx, int close) {
  lua_pushliteral(L, FILEHANDLE);
  lua_rawget(L, LUA_REGISTRYINDEX);
  lua_pushvalue(L, idx);
  lua_pushboolean(L, close);  /* close/not close file when finished */
  lua_pushcclosure(L, gzip_readline, 3);
}


static int gzip_lines (lua_State *L) {
  int type;
  type = lua_type(L, 1);
  if (type == LUA_TSTRING) {
    const char *filename = agn_tostring(L, 1);
    gzFile *pf = newfile(L);
    *pf = gzopen(filename, "rb");
    luaL_argcheck(L, *pf, 1,  strerror(errno));
    aux_lines(L, lua_gettop(L), 1);
  } else if (type == LUA_TUSERDATA) {
    gztofile(L, 1);  /* check that it's a valid file handle */
    aux_lines(L, 1, 0);
  } else
    luaL_error(L, "Error in " LUA_QS ": file handle or file name expected, got %s.",
      "gzip.lines", luaL_typename(L, 1));  /* 2.9.4 */
  return 1;
}


/*
** {======================================================
** READ
** =======================================================
*/


static int test_eof (lua_State *L, gzFile f) {
  lua_pushlstring(L, NULL, 0);
  return (gzeof(f) != 1);
}


static int read_line (lua_State *L, gzFile f) {
  luaL_Buffer b;
  luaL_buffinit(L, &b);
  for (;;) {
    size_t l;
    char *p = luaL_prepbuffer(&b);
    if (gzgets(f, p, LUAL_BUFFERSIZE) == NULL) {  /* eof? */
      luaL_pushresult(&b);  /* close buffer */
      return (lua_strlen(L, -1) > 0);  /* check whether read something */
    }
    l = strlen(p);
    if (l == 0 || !(p[l-1] == '\n' || p[l-1] == '\r'))
      luaL_addsize(&b, l);
    else {
      if (l > 1 && p[l-2] == '\r')  /* added in 0.5.4 for correctly reading DOS files on UNIX platforms */
        luaL_addsize(&b, l - 2);  /* do not include `eol' */
      else
        luaL_addsize(&b, l - 1);  /* do not include `eol' */
      luaL_pushresult(&b);  /* close buffer */
      return 1;  /* read at least an `eol' */
    }
  }
}


static int read_chars (lua_State *L, gzFile f, size_t n) {
  size_t rlen;  /* how much to read */
  size_t nr;  /* number of chars actually read */
  luaL_Buffer b;
  luaL_buffinit(L, &b);
  rlen = LUAL_BUFFERSIZE;  /* try to read that much each time */
  do {
    char *p = luaL_prepbuffer(&b);
    if (rlen > n) rlen = n;  /* cannot read more than asked */
    nr = gzread(f, p, rlen);
    luaL_addsize(&b, nr);
    n -= nr;  /* still have to read `n' chars */
  } while (n > 0 && nr == rlen);  /* until end of count or eof */
  luaL_pushresult(&b);  /* close buffer */
  return (n == 0 || lua_strlen(L, -1) > 0);
}


static int g_read (lua_State *L, gzFile f, int first) {
  int nargs = lua_gettop(L) - 1;
  int success;
  int n;
  if (nargs == 0) {  /* no arguments? */
    success = read_line(L, f);
    n = first+1;  /* to return 1 result */
  }
  else {  /* ensure stack space for all results and for auxlib's buffer */
    luaL_checkstack(L, nargs+LUA_MINSTACK, "too many arguments");
    success = 1;
    for (n = first; nargs-- && success; n++) {
      if (lua_type(L, n) == LUA_TNUMBER) {
        size_t l = (size_t)lua_tonumber(L, n);
        success = (l == 0) ? test_eof(L, f) : read_chars(L, f, l);
      }
      else {
        const char *p = lua_tostring(L, n);
        luaL_argcheck(L, p && p[0] == '*', n, "invalid option");
        switch (p[1]) {
          case 'l':  /* line */
            success = read_line(L, f);
            break;
          case 'a':  /* file */
            read_chars(L, f, ~((size_t)0));  /* read MAX_SIZE_T chars */
            success = 1; /* always success */
            break;
          default:
            return luaL_argerror(L, n, "invalid format");
        }
      }
    }
  }
  if (!success) {
    lua_pop(L, 1);  /* remove last result */
    lua_pushnil(L);  /* push nil instead */
  }
  return n - first;
}


static int gzip_read (lua_State *L) {
  return g_read(L, gztofile(L, 1), 2);
}


static int gzip_readline (lua_State *L) {
  gzFile f = *(gzFile*)lua_touserdata(L, lua_upvalueindex(2));
  if (f == NULL)  /* file is already closed? */
    luaL_error(L, "Error in " LUA_QS " package: file is already closed", "gzip");
  if (read_line(L, f)) return 1;
  else {  /* EOF */
    if (lua_toboolean(L, lua_upvalueindex(3))) {  /* generator created file? */
      lua_settop(L, 0);
      lua_pushvalue(L, lua_upvalueindex(2));
      aux_close(L);  /* close it */
    }
    return 0;
  }
}

/* }====================================================== */


static int g_write (lua_State *L, gzFile f, int arg) {
  int nargs = lua_gettop(L) - 1;
  int status = 1;
  for (; nargs--; arg++) {
    if (lua_type(L, arg) == LUA_TNUMBER) {
      /* optimization: could be done exactly as for strings */
      status = status &&
          gzprintf(f, LUA_NUMBER_FMT, lua_tonumber(L, arg)) > 0;
    }
    else {
      size_t l;
      const char *s = luaL_checklstring(L, arg, &l);
      status = status && (gzwrite(f, (char*)s, l) == l);
    }
  }
  return pushresult(L, status, NULL);
}


static int gzip_write (lua_State *L) {
  return g_write(L, gztofile(L, 1), 2);
}


static int gzip_seek (lua_State *L) {
  static const int mode[] = {SEEK_SET, SEEK_CUR};
  static const char *const modenames[] = {"set", "cur", NULL};
  gzFile f = gztofile(L, 1);
  int op = luaL_checkoption(L, 2, "cur", modenames);
  long offset = luaL_optlong(L, 3, 0);
  op = gzseek(f, offset, mode[op]);
  if (op == -1)
    return pushresult(L, 0, NULL);  /* error */
  else {
    lua_pushnumber(L, op);
    return 1;
  }
}


static int gzip_sync (lua_State *L) {
  return pushresult(L, gzflush(gztofile(L, 1), Z_FINISH) == Z_OK, NULL);
}


/* import gzip;

str := io.readfile('lvm.c');

str := strings.repeat('alex', 100);

s := size str;

for i to 9 do
   a, b := gzip.deflate(str, i);
   c, d := gzip.inflate(a, s)
   print(i, size str, b, d)
od; */

/* See: https://www.experts-exchange.com/articles/3189/In-Memory-Compression-and-Decompression-Using-ZLIB.html
        https://stackoverflow.com/questions/7540259/deflate-and-inflate-zlib-h-in-c */

/* The function compresses a string. The return is the compressed string and its size, in this order. */
static int gzip_deflate (lua_State *L) {  /* 2.12.0 RC 1 */
  z_stream defstream;
  size_t l, ldest, t;
  int n16kBlocks, nErr, level, stats;
  char *dest;
  const char *s = agn_checklstring(L, 1, &l);
  level = agnL_optinteger(L, 2, 9);
  stats = agnL_optboolean(L, 3, 0);
  if (level < 1)
    luaL_error(L, "Error in " LUA_QS ": invalid compression level, must be in [1, 9].", "gzip.deflate");
  n16kBlocks = (l + 16383)/16384;  /* round up any fraction of a block */
  ldest = l + 6 + (n16kBlocks*5);
  dest = agn_malloc(L, ldest, "gzip.deflate", NULL);
  /* zlib struct */
  defstream.zalloc = Z_NULL;
  defstream.zfree = Z_NULL;
  defstream.opaque = Z_NULL;
  defstream.avail_in = l + 0;          /* size of input, string + no terminator */
  defstream.next_in = (Bytef *)s;      /* input char array */
  defstream.avail_out = (uInt)ldest;   /* size of output */
  defstream.next_out = (Bytef *)dest;  /* output char array */
  /* the actual compression work. */
  nErr = deflateInit(&defstream, level);
  if (nErr != Z_OK) {
    deflateEnd(&defstream);
    xfree(dest);
    luaL_error(L, "Error in " LUA_QS ": could not initialise zlib.", "gzip.deflate");
  }
  nErr = deflate(&defstream, Z_FINISH);
  if (nErr != Z_STREAM_END) {
    deflateEnd(&defstream);
    xfree(dest);
    luaL_error(L, "Error in " LUA_QS ": could not compress.", "gzip.deflate");
  }
  deflateEnd(&defstream);
  t = defstream.total_out;
  lua_pushlstring(L, dest, t);
  lua_pushinteger(L, t);  /* compressed size in bytes */
  if (stats) {
    lua_pushnumber(L, (lua_Number)t/(lua_Number)((t > l)*8 + l));  /* compression rate: size of deflated string/size of original string with correction for short strings; 2.36.3 */
    lua_pushinteger(L, l);           /* size of input string 2.36.3 */
    lua_pushinteger(L, ldest);       /* allocated bytes 2.36.3 */
    lua_pushinteger(L, n16kBlocks);  /* number of 16K blocks 2.36.3 */
  }
  xfree(dest);
  return 2 + 4*stats;
}


/* The function uncompresses a inflated string str. The size of the original uncompressed string is given
   as the second argument n, which by default is size(str) + 16383)/16384. The return is the uncompressed string. */
static int gzip_inflate (lua_State *L) {  /* 2.12.0 RC 1 */
  z_stream defstream;
  size_t l, ldest;
  int n16kBlocks, nErr;
  char *dest;
  const char *s = agn_checklstring(L, 1, &l);
  n16kBlocks = (l + 16383)/16384;  /* round up any fraction of a block */
  ldest = 100*(l + 6 + (n16kBlocks*5));
  ldest = agnL_optinteger(L, 2, ldest);  /* original uncompressed size of string */
  if (ldest < 0)
    luaL_error(L, "Error in " LUA_QS ": second argument too small.", "gzip.inflate");
  dest = agn_malloc(L, ldest, "gzip.inflate", NULL);
  /* zlib struct */
  defstream.zalloc = Z_NULL;
  defstream.zfree = Z_NULL;
  defstream.opaque = Z_NULL;
  defstream.avail_in = l + 0;          /* size of input, string + no terminator */
  defstream.next_in = (Bytef *)s;      /* input char array */
  defstream.avail_out = (uInt)ldest;   /* size of output */
  defstream.next_out = (Bytef *)dest;  /* output char array */
  /* the actual compression work. */
  nErr = inflateInit(&defstream);
  if (nErr != Z_OK) {
    inflateEnd(&defstream);
    xfree(dest);
    luaL_error(L, "Error in " LUA_QS ": could not initialise zlib.", "gzip.inflate");
  }
  nErr = inflate(&defstream, Z_FINISH);
  if (nErr != Z_STREAM_END) {
    inflateEnd(&defstream);
    xfree(dest);
    luaL_error(L, "Error in " LUA_QS ": could not decompress.", "gzip.inflate");
  }
  inflateEnd(&defstream);
  lua_pushlstring(L, dest, defstream.total_out);
  lua_pushinteger(L, defstream.total_out);
  xfree(dest);
  return 2;
}


/* }====================================================== */

static const luaL_reg flib[] = {
  {"close",   gzip_close},
  {"lines",   gzip_lines},
  {"read",    gzip_read},
  {"seek",    gzip_seek},
  {"sync",    gzip_sync},
  {"write",   gzip_write},
  {"__gc", io_gc},
  {"__tostring", io_tostring},
  {NULL, NULL}
};

static const luaL_reg iolib[] = {
  {"deflate", gzip_deflate},
  {"inflate", gzip_inflate},
  {"flush",   gzip_sync},
  {"lines",   gzip_lines},
  {"close",   gzip_close},
  {"open",    gzip_open},
  {"read",    gzip_read},
  {"seek",    gzip_seek},
  {"sync",    gzip_sync},
  {"write",   gzip_write},
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  luaL_newmetatable(L, FILEHANDLE);  /* create metatable for file handles */
  lua_pushvalue(L, -1);  /* push metatable */
  lua_setfield(L, -2, "__index");  /* metatable.__index = metatable */
  luaL_register(L, NULL, flib);  /* file methods */
}


LUALIB_API int luaopen_gzip (lua_State *L) {
  createmetatable(L);
  /* create (private) environment (with fields IO_INPUT, IO_OUTPUT, __close) */
  lua_createtable(L, 2, 1);
  lua_replace(L, LUA_ENVIRONINDEX);
  /* open library */
  luaL_register(L, AGENA_LZLIBLIBNAME, iolib);
  lua_rawsetstringstring(L, -1, "initstring", AGENA_LIBVERSION);
  /* table for information on all open files */
  lua_newtable(L);
  lua_setfield(L, -2, "openfiles");  /* table for information on all open files */
  lua_pushliteral (L, "_COPYRIGHT");
  lua_pushliteral (L, "Copyright (C) 2003-2008 Tiago Dionizio");
  lua_settable (L, -3);
  lua_pushliteral (L, "_DESCRIPTION");
  lua_pushliteral (L, "Agena interface to access gzip files, based on lgzip 0.3 for Lua 5.1");
  lua_settable (L, -3);
  lua_pushliteral (L, "_VERSION");
  lua_pushliteral (L, "lgzip 0.4");  /* OOP-enabled */
  lua_settable (L, -3);
  return 1;
}

