/*
** $Id: bytes.c,v 1.67 2005/08/26 17:36:32 roberto Exp $
** Bit and Byte Twiddling Library
** See Copyright Notice in agena.h
*/

#include <stdlib.h>
#include <math.h>
#include <stdint.h>  /* for UINT32_MAX */
#include <string.h>
#include <limits.h>  /* for CHAR_BIT */

#define bytes_c
#define LUA_LIB

#include "agena.h"
#include "lstate.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agnconf.h"
#include "sunpro.h"
#include "charbuf.h"
#include "agncmpt.h"
#include "long.h"

/* For unknown reasons I am too tired to clarify, we import the *GET_*, *SET_*, *INSERT*, *EXTRACT*,
   etc. Sun Microsystems macros from the following header file instead of agnhlps.h: */
#include "agnhlps.h"
#include "sunpro.h"

#define AGENA_LIBVERSION  "bytes 1.4.0 for Agena as of April 17, 2022\n"


/* Converts an integer to a `floating point byte`, represented as (eeeeexxx), where the real value
   is (1xxx) * 2^(eeeee - 1) if eeeee <> 0 and (xxx) otherwise. This function is used to transport
   numbers to the Lua/Agena virtual machine. See also: bytes.fpbtoint */
static int bytes_inttofpb (lua_State *L) {  /* 2.3.1 */
  int x;
  x = agn_checknonnegint(L, 1);  /* 2.9.8 */
  lua_pushinteger(L, agn_int2fb(L, x));
  return 1;
}


/* Converts a `floating point byte` generated by bytes.inttofpb back. This function is used to
   evaluate numbers transported to the Lua/Agena virtual machine. Please note that
   bytes.inttofpb(bytes.fpbtoint(x)) does not return x. */
static int bytes_fpbtoint (lua_State *L) {  /* 2.3.1 */
  int x;
  x = agn_checkinteger(L, 1);
  lua_pushinteger(L, agn_fb2int(L, x));
  return 1;
}


static int bytes_numhigh (lua_State *L) {  /* 2.4.5, based on the 0.27.0 operator */
  LUAI_UINT32 hx;  /* numrange is [0, 4294967295] */
  GET_HIGH_WORD(hx, agn_checknumber(L, 1));
  lua_pushnumber(L, hx);
  return 1;
}


static int bytes_numlow (lua_State *L) {  /* 2.4.5, based on the 0.27.0 operator */
  LUAI_UINT32 lx;  /* numrange is [0, 4294967295] */
  GET_LOW_WORD(lx, agn_checknumber(L, 1));
  lua_pushnumber(L, lx);
  return 1;
}


static int bytes_setnumhigh (lua_State *L) {  /* 2.4.5, based on the 0.27.0 operator, 2.10.0 hardening, patched 2.13.0 */
  double x;
  uint32_t hx;  /* numrange is [0, 4294967295] */
  int32_t lx;
  x = agn_checknumber(L, 1);
  hx = luaL_checkuint32_t(L, 2);  /* 2.16.8 fix, 2.16.11 improvement */
  GET_LOW_WORD(lx, x);
  x = 0;  /* explicitly zero x to prevent round-off errors */
  INSERT_WORDS(x, hx, lx);
  lua_pushnumber(L, x);
  return 1;
}


static int bytes_setnumlow (lua_State *L) {  /* 2.4.5, based on the 0.27.0 operator, 2.10.0 hardening, patched 2.13.0 */
  double x;
  uint32_t hx;  /* numrange is [0, 4294967295] */
  int32_t lx;
  x = agn_checknumber(L, 1);
  lx = luaL_checkuint32_t(L, 2);  /* 2.16.8 fix, 2.16.11 improvement */
  GET_HIGH_WORD(hx, x);
  x = 0;  /* explicitly zero x to prevent round-off errors */
  INSERT_WORDS(x, hx, lx);
  lua_pushnumber(L, x);
  return 1;
}


static int bytes_numwords (lua_State *L) {  /* 2.10.0 */
  uint32_t hx, lx;  /* numrange is [0, 4294967295] */
  int32_t ux;  /* 2.17.1 fix */
  lua_Number x = agn_checknumber(L, 1);
  EXTRACT_WORDS(hx, lx, x);
  lua_pushnumber(L, hx);
  lua_pushnumber(L, lx);
  ux = ((hx >> 20) & 0x7ff) - 0x3ff;  /* unbiased exponent, equals math.exponent(bytes_numhigh(x)) - 1, except 0 -> -1023,  2.16.11 */
  lua_pushnumber(L, ux);
  return 3;
}


static int bytes_setnumwords (lua_State *L) {  /* 2.13.0 */
  uint32_t hx, lx;  /* numrange is [0, 4294967295] */
  double x = 0;
  hx = luaL_checkuint32_t(L, 1);  /* 2.16.8 fix, 2.16.12 improvement */
  lx = luaL_checkuint32_t(L, 2);  /* 2.16.8 fix, 2.16.12 improvement */
  INSERT_WORDS(x, hx, lx);
  lua_pushnumber(L, x);
  return 1;
}


/* Returns a sequence with all the bytes of a number x in Little Endian order. */
static int bytes_tobytes (lua_State *L) {  /* 2.6.1, extended 2.9.0; rewritten 2.9.4; changed 2.17.4 */
  union ldshape d;
  lua_Number x;
  int s, i, tolittle;
  const int E[] = { sizeof(int32_t), sizeof(lua_Number), sizeof(float), sizeof(uint16_t), SIZEOFLDBL};
  getdlongnum(L, d, 1, "bytes.tobytes");
  s = luaL_optint(L, 2, lua_isnumber(L, 1) ? sizeof(lua_Number) : SIZEOFLDBL);
  tolittle = agnL_optboolean(L, 3, 1);  /* new 2.17.4 */
  (void)tolittle;  /* not used on Little Endian systems */
  if (!tools_isintenum(fabs(s), E, sizeof(E)/sizeof(*E)))  /* 2.38.3 fix */
    luaL_error(L, "Error in " LUA_QS ": second argument must either be %d, +/-%d, %d or %d.",
      "bytes.tobytes", sizeof(uint16_t), sizeof(int32_t), sizeof(lua_Number), SIZEOFLDBL);
  agn_createseq(L, s);
  if (s == -4) s = 10004;  /* 2.25.5, to prevent compiler warnings in Debian Bullseye */
  if (s == -2) s = 10002;  /* 2.38.3, to prevent compiler warnings in Debian Bullseye */
  switch (s) {
    case sizeof(lua_Number): {
      /* using tools_double2uint does not work GCC 4.5.2, but not in GCC 4.8.1; tuned 2.14.9 */
#if BYTE_ORDER == BIG_ENDIAN
      if (tolittle) x = tools_tolittleendian(x);  /* 2.14.9 fix */
#endif
      x = d.d;
      unsigned char *src = (unsigned char *)&x;
      for (i=0; i < s; i++) {
        agn_seqsetinumber(L, -1, i + 1, src[i]);
      }
      break;
    }
    case sizeof(uint16_t): {  /* added 2.18.2 */
      uint16_t y = d.d;
      #if BYTE_ORDER == BIG_ENDIAN
      if (tolittle) tools_swapuint16_t(&y);
      #endif
      unsigned char *src = (unsigned char *)&y;
      for (i=0; i < s; i++) {
        agn_seqsetinumber(L, -1, i + 1, src[i]);
      }
      break;
    }
    case sizeof(uint32_t): {  /* changed 2.17.4 */
      uint32_t y = d.d;
      #if BYTE_ORDER == BIG_ENDIAN
      if (tolittle) tools_swapuint32_t(&y);
      #endif
      unsigned char *src = (unsigned char *)&y;
      for (i=0; i < s; i++) {
        agn_seqsetinumber(L, -1, i + 1, src[i]);
      }
      break;
    }
#ifndef __ARMCPU
    case SIZEOFLDBL: {  /* 2.35.0 */
      for (i=0; i < SIZEOFLDBL; i++) {
      #if BYTE_ORDER == BIG_ENDIAN
        agn_seqsetinumber(L, -1, i + 1, d.c[SIZEOFLDBL - i - 1]);
      #else
        agn_seqsetinumber(L, -1, i + 1, d.c[i]);
      #endif
      }
      break;
    }
#endif
    case 10002: {  /* added 2.38.3 */
      int16_t y = d.d;
      s = sizeof(int16_t);
      #if BYTE_ORDER == BIG_ENDIAN
      if (tolittle) tools_swapint16_t(&y);  /* 2.14.9 fix */
      #endif
      unsigned char *src = (unsigned char *)&y;  /* tuned 2.14.9 */
      for (i=0; i < s; i++) {
        agn_seqsetinumber(L, -1, i + 1, src[i]);
      }
      break;
    }
    case 10004: {  /* added 2.17.4 */
      int32_t y = d.d;
      s = sizeof(int32_t);
      #if BYTE_ORDER == BIG_ENDIAN
      if (tolittle) tools_swapint32_t(&y);  /* 2.14.9 fix */
      #endif
      unsigned char *src = (unsigned char *)&y;  /* tuned 2.14.9 */
      for (i=0; i < s; i++) {
        agn_seqsetinumber(L, -1, i + 1, src[i]);
      }
      break;
    }
    default:
      lua_assert(0);
  }
  return 1;
}

/* Takes a sequence r of numbers representing bytes and converts it into an Agena number. Regardless of your platform,
   the order of bytes in r is assumed to be Little Endian.

   See: http://stackoverflow.com/questions/2182002/convert-big-endian-to-little-endian-in-c-without-using-provided-func,
   answer by dreamlax. */

static int bytes_tonumber (lua_State *L) {  /* 2.6.1 */
  int i, size, isfloat;
  const int E[] = { sizeof(int32_t), sizeof(lua_Number), sizeof(float), sizeof(uint16_t), SIZEOFLDBL};
  luaL_argcheck(L, lua_isseq(L, 1), 1, "sequence expected");
  isfloat = agnL_optboolean(L, 2, 0);
  size = agn_seqsize(L, 1);
  if (!tools_isintenum(size, E, sizeof(E)/sizeof(*E)))
    luaL_error(L, "Error in " LUA_QS ": expected a sequence of %d, %d, %d or %d integers.",
      "bytes.tonumber", sizeof(uint16_t), sizeof(int32_t), sizeof(lua_Number), SIZEOFLDBL);
  union {
    lua_Number x;
    long double ld;
    uint16_t us;
    int32_t i;  /* 2.18.2 */
    float f;
    unsigned char c[size];
  } dst;
  for (i=0; i < size; i++) {
#if BYTE_ORDER != BIG_ENDIAN
    dst.c[i] = (unsigned char)agn_seqgetinumber(L, 1, i + 1);
#else
    dst.c[size - 1 - i] = (unsigned char)agn_seqgetinumber(L, 1, i + 1);
#endif
  }
#ifndef __ARMCPU  /* 2.37.1 */
  if (size == SIZEOFLDBL) {
    createdlong(L, dst.ld);
#else
  if (0) {
    lua_assert(0);
#endif
  } else {
    if (isfloat && size == sizeof(float))
      lua_pushnumber(L, dst.f);
    else if (size == sizeof(uint16_t))
      lua_pushnumber(L, dst.us);
    else if (size == sizeof(lua_Number))
      lua_pushnumber(L, dst.x);
    else
      lua_pushnumber(L, dst.i);
  }
  return 1;
}


/* returns the bits of an unsigned byte as a sequence; rewritten 2.17.4 */
static int bytes_tobinary (lua_State *L) {
  lua_Integer x;
  unsigned char ax;
  int i, c, flag;
  x = agn_checknonnegint(L, 1);
  c = 0;
  flag = 1;
  agn_createseq(L, CHAR_BIT);
  for (i=0; i < CHAR_BIT; i++) {
    ax = (x & (1 << (CHAR_BIT - i - 1))) != 0;
    if (flag && ax == 0)
      continue;
    else
      flag = 0;
    agn_seqsetinumber(L, -1, ++c, ax);
  }
  if (c == 0)  /* x == 0 ?  Then sequence is still empty. */
    agn_seqsetinumber(L, -1, 1, 0);
  return 1;
}


/* Casts the given number x internally to a signed or unsigned n-bit integer, and returns it as a number. */
static int bytes_castint (lua_State *L) {  /* 2.10.0 */
  lua_Number x;
  int n;
  x = agn_checknumber(L, 1);
  n = agn_checkinteger(L, 2);
  switch (n) {
    case -64: x = (int64_t)x;  break;
    case -32: x = (int32_t)x;  break;
    case -16: x = (int16_t)x;  break;
    case  -8: x = (int8_t)x;   break;
    case  64: x = (uint64_t)x; break;
    case  32: x = (uint32_t)x; break;
    case  16: x = (uint16_t)x; break;
    case   8: x = (uint8_t)x;  break;
    default:
      luaL_error(L, "Error in " LUA_QS ": unexpected second argument.", "bytes.castint");
  }
  lua_pushnumber(L, x);
  return 1;
}


/* Counts the number of one bits in the unsigned 32-bit integer x and returns it as an integer. */
static int bytes_onebits (lua_State *L) {  /* 2.11.3 */
  register uint32_t n;
  n = luaL_checkuint32_t(L, 1);  /* 2.16.8 fix, 2.16.12 improvement */
#ifdef __GNUC__
  lua_pushnumber(L, __builtin_popcount(n));
#else
  lua_pushnumber(L, tools_onebits(n));
#endif
  return 1;
}


/* Returns the number of leading zeros in the unsigned 32-bit integer n, and the modified value of x after this operation,
   where all bits starting with the first non-zero bit in n are set to 1. 2.11.3 */
static int bytes_leadzeros (lua_State *L) {
  register uint32_t r, n;
  uint32_t x;  /* cannot put result into register, changed to uint32_t 2.16.8 */
  n = luaL_checkuint32_t(L, 1);  /* 2.16.8 fix, 2.16.12 improvement */
  r = tools_clz32(n, &x);
  lua_pushnumber(L, r);
  lua_pushnumber(L, x);
  return 2;
}


/* A lua_Unsigned with its first LUA_NBITS bits equal to 1. (Shift must be made in two parts to avoid problems when
   LUA_NBITS is equal to the number of bits in a lua_Unsigned.) */
#define ALLONES    (~(((~(lua_Unsigned)0) << (LUA_NBITS - 1)) << 1))

/* macro to trim extra bits */
#define trim(x)    ((x) & ALLONES)

/* builds a number with 'n' ones (1 <= n <= LUA_NBITS) */
#define mask(n)    (~((ALLONES << 1) << ((n) - 1)))

static int bytes_trailzeros (lua_State *L) {  /* 2.14.10 */
  register uint32_t r, n;
  uint32_t x;  /* cannot put result into register, changed to uint32_t 2.16.8 */
  n = luaL_checkuint32_t(L, 1);  /* 2.16.8 fix, 2.16.12 improvement */
  r = tools_ctz32(n, &x);
  lua_pushnumber(L, r);  /* 2.16.8 fix */
  lua_pushnumber(L, x | (r ? mask(r) : 0));  /* 2.39.12 fix */
  return 2;
}


/* Returns the position of the most significant bit (msb) in an unsigned 32-bit integer, counting from bit number 1. */
static int bytes_mostsigbit (lua_State *L) {  /* 2.12.3 */
  lua_pushnumber(L, tools_msb(luaL_checkuint32_t(L, 1)));  /* 2.16.8 fix, 2.16.12 improvement */
  return 1;
}


/* Returns the position of the least significant bit in an unsigned 32-bit integer, counting from bit number 1. */
static int bytes_leastsigbit (lua_State *L) {  /* 2.12.3 */
  lua_pushnumber(L, tools_lsb(luaL_checkuint32_t(L, 1)));  /* 2.16.8 fix, 2.16.12 improvement */
  return 1;
}


/* Reverses all the bits in an unsigned 32-bit integer, flipping all bits from 0 to 1 and vice versa. */
static int bytes_reverse (lua_State *L) {  /* 2.12.3 */
  register uint32_t r, n;
  n = luaL_checkuint32_t(L, 1);  /* 2.16.8 fix, 2.16.12 improvement */
  if (n < 1)
    r = 0;
  else
    r = tools_flip(n);
  lua_pushnumber(L, r);  /* 2.16.8 fix */
  return 1;
}


#define checkcast(L, n) (ieee_double_shape_type *)luaL_checkudata(L, n, "cast")

/* Creates a userdata structure of type 'cast' that stores the Agena number 0 and its integer representation as two unsigned 32-bit integers.
    See also: bytes.getdouble, bytes.gethigh, bytes.getlow, bytes.getuints. */
static int bytes_cast (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d;
  int nargs;
  lua_Number f;
  uint32_t hx, lx;
  f = 0; hx = 0; lx = 0;
  nargs = lua_gettop(L);
  /* we have to check the function arguments first, for newuserdata changes the stack */
  if (nargs == 1) {
    f = agn_checknumber(L, 1);
  } else if (nargs == 2) {
    hx = luaL_checkuint32_t(L, 1);  /* 2.16.8 fix, 2.16.12 improvement */
    lx = luaL_checkuint32_t(L, 2);  /* 2.16.8 fix, 2.16.12 improvement */
  }
  d = (ieee_double_shape_type *)lua_newuserdata(L, sizeof(ieee_double_shape_type));
  if (nargs == 2) {
    d->parts.msw = hx;
    d->parts.lsw = lx;
  } else if (nargs == 1)
    d->value = f;
  else {
    d->parts.msw = 0;
    d->parts.lsw = 0;
    d->value = 0.0;
  }
  luaL_checkstack(L, 4, "not enough stack space");  /* 4.7.1 fix */
  luaL_getmetatable(L, "cast");
  lua_setmetatable(L, -2);
  lua_pushstring(L, "cast");
  agn_setutype(L, -2, -1);
  agn_poptop(L);
  lua_pushnumber(L, d->parts.msw);
  lua_pushnumber(L, d->parts.lsw);
  lua_pushnumber(L, d->value);
  return 4;
}


/* Sets the higher and lower unsigned 32-bit integer elements of doublecast userdata a. The return is the corresponding floating point representation, i.e. a number. */
static int bytes_setwords (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  d->parts.msw = luaL_checkuint32_t(L, 2);  /* 2.16.8 fix, 2.16.12 improvement */
  d->parts.lsw = luaL_checkuint32_t(L, 3);  /* 2.16.8 fix, 2.16.12 improvement */
  lua_pushnumber(L, d->value);
  return 1;
}


/* Sets the higher bytes unsigned 32-bit integer component in cast userdata a and returns the corresponding floating point representation, i.e. a number. */
static int bytes_sethigh (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  d->parts.msw = luaL_checkuint32_t(L, 2);  /* 2.16.8 fix, 2.16.12 improvement */
  lua_pushnumber(L, d->value);
  return 1;
}


/* Sets the lower bytes unsigned 32-bit integer component in cast userdata a and returns the corresponding floating point representation, i.e. a number. */
static int bytes_setlow (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  d->parts.lsw = luaL_checkuint32_t(L, 2);  /* 2.16.8 fix, 2.16.12 improvement */
  lua_pushnumber(L, d->value);
  return 1;
}


/* Sets the floating point element of doublecast userdata a and returns the higher and lower unsigned 32-bit integer representations, in this order. */
static int bytes_setdouble (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  d->value = agn_checknumber(L, 2);
  lua_pushnumber(L, d->parts.msw);
  lua_pushnumber(L, d->parts.lsw);
  return 2;
}


/* Returns the floating point element of doublecast userdata a, i.e. a number. */
static int bytes_getdouble (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  lua_pushnumber(L, d->value);
  return 1;
}


/* Returns the higher and lower unsigned 32-bit integer representations of a number from doublecast userdata a. */
static int bytes_getwords (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  lua_pushnumber(L, d->parts.msw);
  lua_pushnumber(L, d->parts.lsw);
  return 2;
}


/* Returns the unbiased exponent from doublecast userdata a, equals math.exponent(bytes_gethigh(a)) - 1, except for bytes_gethigh(a) = 0, where it returns -1023. See: tools_isint in agnhlps.c. */
static int bytes_getunbiased (lua_State *L) {  /* 2.16.11 */
  ieee_double_shape_type *d = checkcast(L, 1);
  /* d->parts.msw is already represented by a Uint32_t, so no need to cast to uint32_t (again) */
  int32_t ux = ((d->parts.msw >> 20) & 0x7ff) - 0x3ff;  /* do NOT cast to Uint32_t ! 2.16.12 fix */
  lua_pushnumber(L, ux);
  return 1;
}


/* Returns the higher unsigned 32-bit integer representation of a number from doublecast userdata a. */
static int bytes_gethigh (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  lua_pushnumber(L, d->parts.msw);
  return 1;
}


/* Returns the lower unsigned 32-bit integer representation of a number from doublecast userdata a. */
static int bytes_getlow (lua_State *L) {  /* 2.13.0 */
  ieee_double_shape_type *d = checkcast(L, 1);
  lua_pushnumber(L, d->parts.lsw);
  return 1;
}


static int mt_uniongc (lua_State *L) {  /* please do not forget to garbage collect deleted userdata */
  (void)checkcast(L, 1);
  lua_pushnil(L);                  /* 2.15.1, delete metattable */
  lua_setmetatable(L, 1);
  agn_setutypestring(L, 1, NULL);  /* 2.15.1 delete user-defined type */
  return 0;
}


static int mt_union2string (lua_State *L) {  /* at the console, the array is formatted as follows: */
  ieee_double_shape_type *d = checkcast(L, 1);
  if (agn_getutype(L, 1)) {
    lua_pushfstring(L, "(%f : %u, %u)", d->value, d->parts.msw, d->parts.lsw);
    lua_concat(L, 2);
  } else
    luaL_error(L, "Error in " LUA_QS ": invalid cast union.", "bytes.__tostring");
  return 1;
}


#define checkieee(L, n) (double_ieee754 *)luaL_checkudata(L, n, "ieee")

/* Creates a userdata structure of type 'cast' that stores the Agena number 0 and its integer representation as two unsigned 32-bit integers.
   See also: bytes.getdouble, bytes.gethigh, bytes.getlow, bytes.getuints. */
static int bytes_ieee (lua_State *L) {  /* 2.32.2 */
  double_ieee754 *d;
  int nargs;
  lua_Number v;
  uint64_t sign, exponent, mantissa_high, mantissa_low;  /* we use uint64_t's as they are used in double_ieee754 structures */
  v = 0; sign = 0; exponent = 0; mantissa_high = 0; mantissa_low = 0;
  nargs = lua_gettop(L);
  /* we have to check the function arguments first, for newuserdata changes the stack */
  if (nargs == 1) {
    v = agn_checknumber(L, 1);
  } else if (nargs == 4) {
    sign          = luaL_checkuint32_t(L, 1);
    exponent      = luaL_checkuint32_t(L, 2);
    mantissa_high = luaL_checkuint32_t(L, 3);
    mantissa_low  = luaL_checkuint32_t(L, 4);
  }
  d = (double_ieee754 *)lua_newuserdata(L, sizeof(double_ieee754));
  if (nargs == 4) {
    d->c.sign          = sign;
    d->c.exponent      = exponent;
    d->c.mantissa_high = mantissa_high;
    d->c.mantissa_low  = mantissa_low;
  } else if (nargs == 1)
    d->v = v;
  else {
    d->c.sign          = 0;
    d->c.exponent      = 0;
    d->c.mantissa_high = 0;
    d->c.mantissa_low  = 0;
    d->v = 0.0;
  }
  luaL_getmetatable(L, "ieee");
  lua_setmetatable(L, -2);
  lua_pushstring(L, "ieee");
  agn_setutype(L, -2, -1);
  agn_poptop(L);
  luaL_checkstack(L, 5, "not enough stack space");
  lua_pushnumber(L, d->c.sign);
  lua_pushnumber(L, d->c.exponent);
  lua_pushnumber(L, d->c.mantissa_high);
  lua_pushnumber(L, d->c.mantissa_low);
  lua_pushnumber(L, d->v);
  return 6;
}


/* Sets the floating point element of ieee754 userdata a and returns its components sign, biased exponent,
   high and low parts of mantisse, in this order. */
static int bytes_setieeedouble (lua_State *L) {  /* 2.32.2 */
  double_ieee754 *d = checkieee(L, 1);
  d->v = agn_checknumber(L, 2);
  luaL_checkstack(L, 4, "not enough stack space");
  lua_pushnumber(L, d->c.sign);
  lua_pushnumber(L, d->c.exponent);
  lua_pushnumber(L, d->c.mantissa_high);
  lua_pushnumber(L, d->c.mantissa_low);
  return 4;
}



/* Returns the floating point element of ieee754 userdata a, i.e. a number. */
static int bytes_getieeedouble (lua_State *L) {  /* 2.32.2 */
  double_ieee754 *d = checkieee(L, 1);
  lua_pushnumber(L, d->v);
  return 1;
}


#define ieee754_pushcomponents(L,d) { \
  luaL_checkstack(L, 5, "not enough stack space"); \
  lua_pushnumber(L, d->v); \
  lua_pushnumber(L, d->c.sign); \
  lua_pushnumber(L, d->c.exponent); \
  lua_pushnumber(L, d->c.mantissa_high); \
  lua_pushnumber(L, d->c.mantissa_low); \
}

static int bytes_getieee (lua_State *L) {  /* 2.32.2 */
  int nargs, nres, i;
  double_ieee754 *d = checkieee(L, 1);
  nargs = lua_gettop(L);
  if (nargs == 1) {
    ieee754_pushcomponents(L, d);
    nres = 5;
  } else {
    luaL_checkstack(L, nargs - 1, "too many arguments");
    nres = 0;
    for (i=2; i <= nargs && agn_isstring(L, i); i++) {  /* 3.7.6 fix */
      const char *option = agn_tostring(L, i);
      if (tools_streq(option, "double")) {
        lua_pushnumber(L, d->v);
      } else if (tools_streqx(option, "expo", "exponent", NULL)) {
        lua_pushnumber(L, d->c.exponent);
      } else if (tools_streq(option, "high")) {
        lua_pushnumber(L, d->c.mantissa_low);
      } else if (tools_streq(option, "low")) {
        lua_pushnumber(L, d->c.mantissa_high);
      } else if (tools_streq(option, "signbit")) {
        lua_pushinteger(L, d->c.sign);
      } else {
        luaL_error(L, "Error in " LUA_QS ": unknown option " LUA_QS ".", "bytes.getieee", option);
      }
      /* do not call lua_settop as it would corrupt the argument stack since we have already pushed values */
      nres++;
    }
  }
  return nres;
}


static int bytes_setieee (lua_State *L) {  /* 2.32.2 */
  int nargs, i;
  double_ieee754 *d = checkieee(L, 1);
  nargs = lua_gettop(L);
  for (i=2; i <= nargs && lua_ispair(L, nargs); i++) {
    agn_pairgeti(L, i, 1);  /* get left value, set to stack index -2 */
    agn_pairgeti(L, i, 2);  /* get right value, set to stack index  -1 */
    if (agn_isstring(L, -2)) {
      const char *option = agn_tostring(L, -2);
      if (tools_streq(option, "double")) {
        d->v = agn_checknumber(L, -1);
      } else if (tools_streqx(option, "expo", "exponent", NULL)) {
        d->c.exponent = agn_checkuint32_t(L, -1);
      } else if (tools_streq(option, "low")) {
        d->c.mantissa_low = agn_checkuint32_t(L, -1);
      } else if (tools_streq(option, "high")) {
        d->c.mantissa_high = agn_checkuint32_t(L, -1);
      } else if (tools_streq(option, "signbit")) {
        uint32_t s = agn_checkuint32_t(L, -1);
        d->c.sign = s > 1 ? 1 : s;
      } else {
        agn_poptoptwo(L);
        luaL_error(L, "Error in " LUA_QS ": unknown option " LUA_QS ".", "bytes.setieee", option);
      }
    }
    /* do not call lua_settop as it would corrupt the argument stack since we have already pushed values */
    agn_poptoptwo(L);
  }
  ieee754_pushcomponents(L, d);
  return 5;
}


static int bytes_setieeesignbit (lua_State *L) {  /* 2.32.2 */
  int n;
  double_ieee754 *d = checkieee(L, 1);
  d->c.sign = agn_checkuint32_t(L, 2);
  if ( (n = lua_gettop(L) > 2) ) {
    ieee754_pushcomponents(L, d);
  }
  return 5*n;
}


static int bytes_setieeeexpo (lua_State *L) {  /* 2.32.2 */
  int n;
  double_ieee754 *d = checkieee(L, 1);
  d->c.exponent = agn_checkuint32_t(L, 2);
  if ( (n = lua_gettop(L) > 2) ) {
    ieee754_pushcomponents(L, d);
  }
  return 5*n;
}


static int bytes_setieeehigh (lua_State *L) {  /* 2.32.2 */
  int n;
  double_ieee754 *d = checkieee(L, 1);
  d->c.mantissa_high = agn_checkuint32_t(L, 2);
  if ( (n = lua_gettop(L) > 2) ) {
    ieee754_pushcomponents(L, d);
  }
  return 5*n;
}


static int bytes_setieeelow (lua_State *L) {  /* 2.32.2 */
  int n;
  double_ieee754 *d = checkieee(L, 1);
  d->c.mantissa_low = agn_checkuint32_t(L, 2);
  if ( (n = lua_gettop(L) > 2) ) {
    ieee754_pushcomponents(L, d);
  }
  return 5*n;
}


static int ieee754_uniongc (lua_State *L) {  /* please do not forget to garbage collect deleted userdata */
  (void)checkieee(L, 1);
  lua_pushnil(L);                  /* delete metattable */
  lua_setmetatable(L, 1);
  agn_setutypestring(L, 1, NULL);  /* delete user-defined type */
  return 0;
}


static int ieee754_union2string (lua_State *L) {  /* at the console, the array is formatted as follows: */
  double_ieee754 *d = checkieee(L, 1);
  if (agn_getutype(L, 1)) {
    lua_pushfstring(L, "(%f : %u, %u, %u, %u)", d->v, d->c.sign, d->c.exponent, d->c.mantissa_high, d->c.mantissa_low);
    lua_concat(L, 2);
  } else
    luaL_error(L, "Error in " LUA_QS ": invalid ieee754 union.", "bytes.__tostring");
  return 1;
}


/* }====================================================== */


/*
** {======================================================
** PACK/UNPACK, taken from Lua 5.3.5
** =======================================================
*/

#define MAX_SIZET2  ((size_t)(~(size_t)0))

#define MAXSIZE2  \
  (sizeof(size_t) < sizeof(int) ? MAX_SIZET2 : (size_t)(INT_MAX))

/* value used for padding */
#if !defined(LUAL_PACKPADBYTE)
#define LUAL_PACKPADBYTE    0x00
#endif

/* maximum size for the binary representation of an integer */
#define MAXINTSIZE  16

/* number of bits in a character */
#define NB  CHAR_BIT

/* mask for one character (NB 1's) */
#define MC  ((1 << NB) - 1)

/* size of a lua_Integer */
#define SZINT  ((int)sizeof(lua_Integer))


/* dummy union to get native endianness */
static const union {
  int dummy;
  char little;  /* true iff machine is little endian */
} nativeendian = {1};


/* dummy structure to get native alignment requirements */
struct cD {
  char c;
  union { double d; void *p; lua_Integer i; lua_Number n; } u;
};

#define MAXALIGN  (offsetof(struct cD, u))


/*
** Union for serializing floats
*/
typedef union Ftypes {
  float f;
  double d;
  lua_Number n;
  char buff[5 * sizeof(lua_Number)];  /* enough for any float type */
} Ftypes;


/* information to pack/unpack stuff */
typedef struct Header {
  lua_State *L;
  int islittle;
  int maxalign;
} Header;


/* options for pack/unpack */
typedef enum KOption {
  Kint,    /* signed integers */
  Kuint,  /* unsigned integers */
  Kfloat,  /* floating-point numbers */
  Kchar,  /* fixed-length strings */
  Kstring,  /* strings with prefixed length */
  Kzstr,  /* zero-terminated strings */
  Kpadding,  /* padding */
  Kpaddalign,  /* padding for alignment */
  Knop    /* no-op (configuration or spaces) */
} KOption;


/* Read an integer numeral from string 'fmt' or return 'df' if there is no numeral */
static int digit (int c) { return '0' <= c && c <= '9'; }

static int getnum (const char **fmt, int df) {
  if (!digit(**fmt))  /* no number? */
    return df;  /* return default value */
  else {
    int a = 0;
    do {
      a = a*10 + (*((*fmt)++) - '0');
    } while (digit(**fmt) && a <= ((int)MAXSIZE2 - 9)/10);
    return a;
  }
}


/* Read an integer numeral and raises an error if it is larger than the maximum size for integers. */
static int getnumlimit (Header *h, const char **fmt, int df) {
  int sz = getnum(fmt, df);
  if (sz > MAXINTSIZE || sz <= 0)
    return luaL_error(h->L, "integral size (%d) out of limits [1,%d]",
                            sz, MAXINTSIZE);
  return sz;
}


/* Initialize Header */
static void initheader (lua_State *L, Header *h) {
  h->L = L;
  h->islittle = nativeendian.little;
  h->maxalign = 1;
}


/* Read and classify next option. 'size' is filled with option's size. */
static KOption getoption (Header *h, const char **fmt, int *size) {
  int opt = *((*fmt)++);
  *size = 0;  /* default */
  switch (opt) {
    case 'b': *size = CHARSIZE; return Kint;
    case 'B': *size = CHARSIZE; return Kuint;
    case 'h': *size = sizeof(short); return Kint;
    case 'H': *size = sizeof(short); return Kuint;
    case 'l': *size = sizeof(long); return Kint;
    case 'L': *size = sizeof(long); return Kuint;
    case 'j': *size = sizeof(lua_Integer); return Kint;
    case 'J': *size = sizeof(lua_Integer); return Kuint;
    case 'T': *size = sizeof(size_t); return Kuint;
    case 'f': *size = sizeof(float); return Kfloat;
    case 'd': *size = sizeof(double); return Kfloat;
    case 'n': *size = sizeof(lua_Number); return Kfloat;
    case 'i': *size = getnumlimit(h, fmt, sizeof(int)); return Kint;
    case 'I': *size = getnumlimit(h, fmt, sizeof(int)); return Kuint;
    case 's': *size = getnumlimit(h, fmt, sizeof(size_t)); return Kstring;
    case 'c':
      *size = getnum(fmt, -1);
      if (*size == -1)
        luaL_error(h->L, "missing size for format option 'c'");
      return Kchar;
    case 'z': return Kzstr;
    case 'x': *size = 1; return Kpadding;
    case 'X': return Kpaddalign;
    case ' ': break;
    case '<': h->islittle = 1; break;
    case '>': h->islittle = 0; break;
    case '=': h->islittle = nativeendian.little; break;
    case '!': h->maxalign = getnumlimit(h, fmt, MAXALIGN); break;
    default: luaL_error(h->L, "invalid format option '%c'", opt);
  }
  return Knop;
}


/* Read, classify, and fill other details about the next option. 'psize' is filled with option's size, 'notoalign' with its
   alignment requirements. Local variable 'size' gets the size to be aligned. (Kpadal option always gets its full alignment,
   other options are limited by the maximum alignment ('maxalign'). Kchar option needs no alignment despite its size. */
static KOption getdetails (Header *h, size_t totalsize,
                           const char **fmt, int *psize, int *ntoalign) {
  KOption opt = getoption(h, fmt, psize);
  int align = *psize;  /* usually, alignment follows size */
  if (opt == Kpaddalign) {  /* 'X' gets alignment from following option */
    if (**fmt == '\0' || getoption(h, fmt, &align) == Kchar || align == 0)
      luaL_argerror(h->L, 1, "invalid next option for option 'X'");
  }
  if (align <= 1 || opt == Kchar)  /* need no alignment? */
    *ntoalign = 0;
  else {
    if (align > h->maxalign)  /* enforce maximum alignment */
      align = h->maxalign;
    if ((align & (align - 1)) != 0)  /* is 'align' not a power of 2? */
      luaL_argerror(h->L, 1, "format asks for alignment not power of 2");
    *ntoalign = (align - (int)(totalsize & (align - 1))) & (align - 1);
  }
  return opt;
}


/* Pack integer 'n' with 'size' bytes and 'islittle' endianness. The final 'if' handles the case when 'size' is larger than
   the size of a Lua integer, correcting the extra sign-extension bytes if necessary (by default they would be zeros). */
static void packint (lua_State *L, Charbuf *b, lua_Unsigned n,
                     int islittle, int size, int neg) {
  char *buff = charbuf_growsize(L, b, size);
  int i;
  buff[islittle ? 0 : size - 1] = (char)(n & MC);  /* first byte */
  for (i = 1; i < size; i++) {
    n >>= NB;
    buff[islittle ? i : size - 1 - i] = (char)(n & MC);
  }
  if (neg && size > SZINT) {  /* negative number need sign extension? */
    for (i = SZINT; i < size; i++)  /* correct extra bytes */
      buff[islittle ? i : size - 1 - i] = (char)MC;
  }
  charbuf_addsize(b, size);  /* add result to buffer */
}


/* Copy 'size' bytes from 'src' to 'dest', correcting endianness if given 'islittle' is different from native endianness. */
static void copywithendian (volatile char *dest, volatile const char *src,
                            int size, int islittle) {
  if (islittle == nativeendian.little) {
    while (size-- != 0)
      *(dest++) = *(src++);
  }
  else {
    dest += size - 1;
    while (size-- != 0)
      *(dest--) = *(src++);
  }
}


static int bytes_pack (lua_State *L) {  /* 2.14.4 */
  Charbuf b;
  Header h;
  const char *fmt = luaL_checkstring(L, 1);  /* format string */
  int arg = 1;  /* current argument to pack */
  size_t totalsize = 0;  /* accumulate total size of result */
  initheader(L, &h);
  lua_pushnil(L);  /* mark to separate arguments from string buffer */
  charbuf_init(L, &b, 0, L->buffersize, 0);  /* 2.34.9 adaption */
  while (*fmt != '\0') {
    int size, ntoalign;
    KOption opt = getdetails(&h, totalsize, &fmt, &size, &ntoalign);
    totalsize += ntoalign + size;
    while (ntoalign-- > 0)
      charbuf_appendchar(L, &b, LUAL_PACKPADBYTE);  /* fill alignment */
    arg++;
    switch (opt) {
      case Kint: {  /* signed integers */
        lua_Integer n = luaL_checkinteger(L, arg);
        if (size < SZINT) {  /* need overflow check? */
          lua_Integer lim = (lua_Integer)1 << ((size * NB) - 1);
          luaL_argcheck(L, -lim <= n && n < lim, arg, "integer overflow");
        }
        packint(L, &b, (lua_Unsigned)n, h.islittle, size, (n < 0));
        break;
      }
      case Kuint: {  /* unsigned integers */
        lua_Integer n = luaL_checkinteger(L, arg);
        if (size < SZINT)  /* need overflow check? */
          luaL_argcheck(L, (lua_Unsigned)n < ((lua_Unsigned)1 << (size * NB)),
                           arg, "unsigned overflow");
        packint(L, &b, (lua_Unsigned)n, h.islittle, size, 0);
        break;
      }
      case Kfloat: {  /* floating-point options */
        volatile Ftypes u;
        char *buff = charbuf_growsize(L, &b, size);
        lua_Number n = luaL_checknumber(L, arg);  /* get argument */
        if (size == sizeof(u.f)) u.f = (float)n;  /* copy it into 'u' */
        else if (size == sizeof(u.d)) u.d = (double)n;
        else u.n = n;
        /* move 'u' to final result, correcting endianness if needed */
        copywithendian(buff, u.buff, size, h.islittle);
        charbuf_addsize(&b, size);
        break;
      }
      case Kchar: {  /* fixed-size string */
        size_t len;
        const char *s = luaL_checklstring(L, arg, &len);
        luaL_argcheck(L, len <= (size_t)size, arg,
                         "string longer than given size");
        charbuf_append(L, &b, s, len);  /* add string */
        while (len++ < (size_t)size)  /* pad extra space */
          charbuf_appendchar(L, &b, LUAL_PACKPADBYTE);
        break;
      }
      case Kstring: {  /* strings with length count */
        size_t len;
        const char *s = luaL_checklstring(L, arg, &len);
        luaL_argcheck(L, size >= (int)sizeof(size_t) ||
                         len < ((size_t)1 << (size * NB)),
                         arg, "string length does not fit in given size");
        packint(L, &b, (lua_Unsigned)len, h.islittle, size, 0);  /* pack length */
        charbuf_append(L, &b, s, len);
        totalsize += len;
        break;
      }
      case Kzstr: {  /* zero-terminated string */
        size_t len;
        const char *s = luaL_checklstring(L, arg, &len);
        luaL_argcheck(L, tools_strlen(s) == len, arg, "string contains zeros");  /* 2.17.8 tweak */
        charbuf_append(L, &b, s, len);
        charbuf_appendchar(L, &b, '\0');  /* add zero at the end */
        totalsize += len + 1;
        break;
      }
      case Kpadding: charbuf_appendchar(L, &b, LUAL_PACKPADBYTE);  /* FALLTHROUGH */
      case Kpaddalign: case Knop:
        arg--;  /* undo increment */
        break;
    }
  }
  lua_pushlstring(L, b.data, b.size);
  charbuf_free(&b);
  return 1;
}


static int bytes_packsize (lua_State *L) {  /* 2.14.4 */
  Header h;
  const char *fmt = luaL_checkstring(L, 1);  /* format string */
  size_t totalsize = 0;  /* accumulate total size of result */
  initheader(L, &h);
  while (*fmt != '\0') {
    int size, ntoalign;
    KOption opt = getdetails(&h, totalsize, &fmt, &size, &ntoalign);
    size += ntoalign;  /* total space used by option */
    luaL_argcheck(L, totalsize <= MAXSIZE2 - size, 1,
                     "format result too large");
    totalsize += size;
    switch (opt) {
      case Kstring:  /* strings with length count */
      case Kzstr:    /* zero-terminated string */
        luaL_argerror(L, 1, "variable-length format");
        /* call never return, but to avoid warnings: *//* FALLTHROUGH */
      default:  break;
    }
  }
  lua_pushinteger(L, (lua_Integer)totalsize);
  return 1;
}


/* Unpack an integer with 'size' bytes and 'islittle' endianness. If size is smaller than the size of a Lua integer and integer
   is signed, must do sign extension (propagating the sign to the higher bits); if size is larger than the size of a Lua integer,
   it must check the unread bytes to see whether they do not cause an overflow. */
static lua_Integer unpackint (lua_State *L, const char *str,
                              int islittle, int size, int issigned) {
  lua_Unsigned res = 0;
  int i;
  int limit = (size  <= SZINT) ? size : SZINT;
  for (i = limit - 1; i >= 0; i--) {
    res <<= NB;
    res |= (lua_Unsigned)(unsigned char)str[islittle ? i : size - 1 - i];
  }
  if (size < SZINT) {  /* real size smaller than lua_Integer? */
    if (issigned) {  /* needs sign extension? */
      lua_Unsigned mask = (lua_Unsigned)1 << (size*NB - 1);
      res = ((res ^ mask) - mask);  /* do sign extension */
    }
  }
  else if (size > SZINT) {  /* must check unread bytes */
    int mask = (!issigned || (lua_Integer)res >= 0) ? 0 : MC;
    for (i = limit; i < size; i++) {
      if ((unsigned char)str[islittle ? i : size - 1 - i] != mask)
        luaL_error(L, "%d-byte integer does not fit into Lua Integer", size);
    }
  }
  return (lua_Integer)res;
}


static int bytes_unpack (lua_State *L) {  /* 2.14.4 */
  Header h;
  const char *fmt = luaL_checkstring(L, 1);
  size_t ld;
  const char *data = luaL_checklstring(L, 2, &ld);
  size_t pos = (size_t)tools_posrelat(luaL_optinteger(L, 3, 1), ld) - 1;
  int n = 0;  /* number of results */
  luaL_argcheck(L, pos <= ld, 3, "initial position out of string");
  initheader(L, &h);
  while (*fmt != '\0') {
    int size, ntoalign;
    KOption opt = getdetails(&h, pos, &fmt, &size, &ntoalign);
    if ((size_t)ntoalign + size > ~pos || pos + ntoalign + size > ld)
      luaL_argerror(L, 2, "data string too short");
    pos += ntoalign;  /* skip alignment */
    /* stack space for item + next position */
    luaL_checkstack(L, 2, "too many results");
    n++;
    switch (opt) {
      case Kint:
      case Kuint: {
        lua_Integer res = unpackint(L, data + pos, h.islittle, size,
                                       (opt == Kint));
        lua_pushinteger(L, res);
        break;
      }
      case Kfloat: {
        volatile Ftypes u;
        lua_Number num;
        copywithendian(u.buff, data + pos, size, h.islittle);
        if (size == sizeof(u.f)) num = (lua_Number)u.f;
        else if (size == sizeof(u.d)) num = (lua_Number)u.d;
        else num = u.n;
        lua_pushnumber(L, num);
        break;
      }
      case Kchar: {
        lua_pushlstring(L, data + pos, size);
        break;
      }
      case Kstring: {
        size_t len = (size_t)unpackint(L, data + pos, h.islittle, size, 0);
        luaL_argcheck(L, pos + len + size <= ld, 2, "data string too short");
        lua_pushlstring(L, data + pos + size, len);
        pos += len;  /* skip string */
        break;
      }
      case Kzstr: {
        size_t len = (int)tools_strlen(data + pos);  /* 2.17.8 tweak */
        lua_pushlstring(L, data + pos, len);
        pos += len + 1;  /* skip string plus final '\0' */
        break;
      }
      case Kpaddalign: case Kpadding: case Knop:
        n--;  /* undo increment */
        break;
    }
    pos += size;
  }
  lua_pushinteger(L, pos + 1);  /* next position */
  return n + 1;
}


/* bytes.rotate32 (x, n)

   Rotates the bits in the 32-bit integer x n displacements to the right if n >= 0, or n places to the left if n < 0.
   The return is a 32-bit integer.

   Internally the function uses unsigned 32-bit integers by default. You can change this to signed integers by
   calling the `environ.kernel` function with the 'signedbits' option.

   See also: <<<< and >>>> operators.

   The implementation in lvm.c somehow does not work with 32-bit integers, so we use code published on:
   https://en.wikipedia.org/wiki/Circular_shift#Implementing_circular_shifts */

static int bytes_rotate32 (lua_State *L) {  /* 2.14.5 */
  int n;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t x;
    x = luaL_checkint32_t(L, 1);
    n = luaL_checkinteger(L, 2);
    lua_pushnumber(L, (n < 0) ? tools_rotl32(x, -n) : tools_rotr32(x, n));
  } else {
    uint32_t x;
    x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    n = luaL_checkinteger(L, 2);   /* 2.16.11 improvement */
    lua_pushnumber(L, (n < 0) ? tools_rotl32(x, -n) : tools_rotr32(x, n));
  }
  return 1;
}


#define aux_shift32(t,x,n,r) { \
  if (abs(n) >= LUA_NBITS) \
    (r) = 0; \
  else \
    (r) = ((n) < 0) ? t((x) << (-(n))) : t(tools_slo32((x),(n))); \
}

/* bytes.shift32 (x, n)

   Shifts the bits in the 32-bit integer x n displacements to the left if n < 0, and to the right if n > 0.

   Internally the function uses unsigned 32-bit integers by default. You can change this to signed integers by
   calling the `environ.kernel` function with the 'signedbits' option.

   See also: bytes.arshift32. */

static int bytes_shift32 (lua_State *L) {  /* 2.14.5 */
  int32_t n;
  lua_Number res;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t x;
    x = luaL_checkint32_t(L, 1);
    n = luaL_checkinteger(L, 2);
    aux_shift32((int32_t), x, n, res);
  } else {
    uint32_t x;
    x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    n = luaL_checkinteger(L, 2);   /* 2.16.11 improvement */
    aux_shift32((uint32_t), x, n, res);
  }
  lua_pushnumber(L, res);
  return 1;
}

#define aux_xorshift32(t,x,n,r) { \
  if (abs(n) >= LUA_NBITS) \
    (r) = 0; \
  else \
    (r) = (x) ^ ((n) < 0) ? t((x) << (-(n))) : t((x) >> (n)); \
}

static int bytes_xorshift32 (lua_State *L) {  /* 2.17.1 */
  int32_t n;
  lua_Number res;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t x;
    x = luaL_checkint32_t(L, 1);
    n = luaL_checkinteger(L, 2);
    aux_xorshift32((int32_t), x, n, res);
  } else {
    uint32_t x;
    x = luaL_checkuint32_t(L, 1);
    n = luaL_checkinteger(L, 2);
    aux_xorshift32((uint32_t), x, n, res);
  }
  lua_pushnumber(L, res);
  return 1;
}


/* bytes.arshift32 (x, n)

   Returns the 32-bit unsigned integer x shifted `n' bits to the right. The number `n' may be any representable integer. Negative displacements
   shift to the left.

   This shift operation is what is called arithmetic shift. Vacant bits on the left are filled with copies of the higher bit of x; vacant bits on
   the right are filled with zeros. In particular, displacements with absolute values higher than 31 result in zero or 0xFFFFFFFF (all original
   bits are shifted out).

   Signed 32-bit integers x are not supported. See also: bytes.shift32.

   This code complies with the examples given in https://en.wikipedia.org/wiki/Arithmetic_shift [1].
   Lua's 5.2+ bits.arsghift function often does not.

   left shift  0b10111 -> 0b101110. "The empty position in the least significant bit is filled with a zero." [1]
   right shift 0b10111 -> 0b1011.   "The empty position in the most significant bit is filled with a copy of the original MSB." [1] */
static int bytes_arshift32 (lua_State *L) {  /* 2.14.5, taken partially from Lua 5.3.5, lbitlib.c file */
  if (L->settings & 1) {  /* signed bits operation ?  New in 2.31.0 */
    int32_t x; int32_t n;
    x = luaL_checkint32_t(L, 1);
    n = luaL_checkint32_t(L, 2);
    if (n == 0) {
      /* push back x */
    } else if (n < 0) {  /* shift to the left */
      n = -n;
      if (n >= LUA_NBITS) x = 0;
      else x <<= n;
      x = trim(x);
    } else {  /* shift to the right */
      if (n >= LUA_NBITS)
        x = ALLONES;  /* like in Lua */
      else {  /* we preserve the sign with negative x, see Rupert Tombs' sar.c 2020 implementation, MIT licence */
        const int logical = (((int32_t)(-1)) >> 1) > 0;                                 \
        int32_t fixu = -(logical & (x < 0));                                         \
        int32_t fix = *(int32_t*) &fixu;                                                 \
        x = (x >> n) | (fix ^ (fix >> n));
      }
    }
    lua_pushnumber(L, x);
  } else {
    uint32_t x; int32_t n;
    x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    n = luaL_checkint32_t(L, 2);  /* 2.16.11 improvement */
    if (n == 0)
      lua_pushnumber(L, x);
    else if (n < 0) {  /* shift to the left; don't check this: n < 0 || !(x & ((uint32_t)1 << (LUA_NBITS - 1)))) as in Lua 5.3.5 */
      n = -n;
      if (n >= LUA_NBITS) x = 0;
      else x <<= n;
      x = trim(x);
      lua_pushnumber(L, x);
    } else {  /* shift to the right; n > 0, https://stackoverflow.com/questions/7622/are-the-shift-operators-arithmetic-or-logical-in-c, #13, answered by John Scipione */
      if (n >= LUA_NBITS)
        x = ALLONES;  /* like in Lua */
      else {  /* as published on StackOverflow, see link above */
        if (x < 0)
          x = x >> n | ~(~0U >> n);  /* add signal bit */
        else
          x = x >> n;
      }
      lua_pushnumber(L, (int32_t)x);
    }
  }
  return 1;
}


/* Taken from Lua 5.3.5, file lbitlib.c. */

/* bytes.extract32 (n, field [, width])

   Returns the unsigned number formed by the bits field to field + width - 1 from n. Bits are numbered from 0
   (least significant) to 31 (most significant). All accessed bits must be in the range [0, 31].

   The default for width is 1.

   Signed 32-bit integers n are not supported. See also: bytes.replace32. */

#define pushunsigned(L,n)  lua_pushinteger(L, (lua_Integer)(n))
#define checkunsigned(L,i)  ((lua_Unsigned)luaL_checkinteger(L,i))

/* Get field and width arguments for field-manipulation functions, checking whether they are valid. ('luaL_error' called without
   'return' to avoid later warnings about 'width' being used uninitialized.) */

static int fieldargs (lua_State *L, int farg, int *width) {
  lua_Integer f = luaL_checkinteger(L, farg);
  lua_Integer w = luaL_optinteger(L, farg + 1, 1);
  luaL_argcheck(L, 0 <= f, farg, "field cannot be negative");
  luaL_argcheck(L, 0 < w, farg + 1, "width must be positive");
  if (f + w > LUA_NBITS)
    luaL_error(L, "trying to access non-existent bits");
  *width = (int)w;
  return (int)f;
}

static int bytes_extract32 (lua_State *L) {  /* 2.14.5, taken from Lua 5.3.5, lbitlib.c file */
  int w;
  lua_Unsigned r = trim(checkunsigned(L, 1));
  int f = fieldargs(L, 2, &w);
  r = (r >> f) & mask(w);
  pushunsigned(L, r);
  return 1;
}


/* bytes.replace32 (n, v, field [, width])

   Returns a copy of n, an unsigned 32-bit integer, with the bits field to field + width - 1 replaced by the value v.

   Signed 32-bit integers n are not supported.

   See bytes.extract32 for details about field and width. */
static int bytes_replace32 (lua_State *L) {  /* 2.14.5, taken from Lua 5.3.5, lbitlib.c file */
  int w;
  lua_Unsigned r = trim(checkunsigned(L, 1));
  lua_Unsigned v = trim(checkunsigned(L, 2));
  int f = fieldargs(L, 3, &w);
  lua_Unsigned m = mask(w);
  r = (r & ~(m << f)) | ((v & m) << f);
  pushunsigned(L, r);
  return 1;
}


static int bytes_not32 (lua_State *L) {  /* 2.14.5, idea taken from Lua 5.2 */
  lua_Number res;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t r = ~luaL_checkint32_t(L, 1);
    res = trim(r);
  } else {
    uint32_t r = ~luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    res = trim(r);
  }
  lua_pushnumber(L, res);
  return 1;
}


static int bytes_and32 (lua_State *L) {  /* 2.14.5, idea taken from Lua 5.2 */
  lua_Number res;
  int i;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t r;
    r = ~(int32_t)0;
    for (i=1; i <= lua_gettop(L); i++)
      r &= luaL_checkint32_t(L, i);
    res = trim(r);
  } else {
    uint32_t r;
    r = ~(uint32_t)0;
    for (i=1; i <= lua_gettop(L); i++)
      r &= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    res = trim(r);
  }
  lua_pushnumber(L, res);
  return 1;
}


static int bytes_or32 (lua_State *L) {  /* 2.14.5, idea taken from Lua 5.2 */
  lua_Number res;
  int i;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t r;
    r = (int32_t)0;
    for (i=1; i <= lua_gettop(L); i++)
      r |= luaL_checkint32_t(L, i);
    res = trim(r);
  } else {
    uint32_t r;
    r = (uint32_t)0;
    for (i=1; i <= lua_gettop(L); i++)
      r |= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    res = trim(r);
  }
  lua_pushnumber(L, res);
  return 1;
}


static int bytes_xor32 (lua_State *L) {  /* 2.14.5, idea taken from Lua 5.2 */
  lua_Number res;
  int i;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t r;
    r = (int32_t)0;
    for (i=1; i <= lua_gettop(L); i++)
      r ^= luaL_checkint32_t(L, i);
    res = trim(r);
  } else {
    uint32_t r;
    r = (uint32_t)0;
    for (i=1; i <= lua_gettop(L); i++)
      r ^= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    res = trim(r);
  }
  lua_pushnumber(L, res);
  return 1;
}


static int bytes_nor32 (lua_State *L) {  /* 2.14.5, idea taken from Lua 5.2 */
  int i;
  int32_t r = (int32_t)0;  /* only signed operations make sense here */
  for (i=1; i <= lua_gettop(L); i++)
    r |= luaL_checkint32_t(L, i);
  lua_pushnumber(L, ~(r));
  return 1;
}


static int bytes_xnor32 (lua_State *L) {  /* 2.14.5, idea taken from Lua 5.2 */
  int i;
  int32_t r = (int32_t)0;  /* only signed operations make sense here */
  for (i=1; i <= lua_gettop(L); i++)
    r ^= luaL_checkint32_t(L, i);
  lua_pushnumber(L, ~(r));
  return 1;
}


static int bytes_nand32 (lua_State *L) {  /* 2.14.5, idea taken from Lua 5.2 */
  int i;
  int32_t r = ~(int32_t)0;  /* only signed operations make sense here */
  for (i=1; i <= lua_gettop(L); i++)
    r &= luaL_checkint32_t(L, i);
  lua_pushnumber(L, ~(r));
  return 1;
}


/* Trims extra bits. UNDOCUMENTED */
static int bytes_trim32 (lua_State *L) {  /* 2.14.6, based on bytes.not32 */
  lua_Number res;
  if (L->settings & 1) {  /* signed bits operation ? */
    int32_t r = luaL_checkint32_t(L, 1);
    res = trim(r);
  } else {
    uint32_t r = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    res = trim(r);
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Returns a number with 'n' binary ones. */
static int bytes_mask32 (lua_State *L) {  /* 2.14.6, based on bytes.not32 */
  uint32_t r = luaL_checkuint32_t(L, 1);   /* 2.16.12 improvement */
  lua_pushnumber(L, mask(r));
  return 1;
}


/* Returns the Binary coded decimal (BCD) representation of the non-negative integer `n`. From left to right, each decimal
   digit is converted to a four-bit representation (0 = 0b0000, 9 = 0b1001), and the resulting bit sequence is then returned
   as one decimal integer, e.g. decimal 102 = 0001 0000 0010 -> BCD 258.
   By default, if only n is given, the function converts the decimal integer to BCD. If `true` is passed as a second argument,
   `n` is converted from BCD to its decimal integer representation. */
static int bytes_bcd (lua_State *L) {  /* 2.14.6, based on bytes.not32; this should fix wrong results in PowerPC Linux 2.14.9 */
  /* 2.14.9: changed from int32_t to uint32_t */
  uint32_t v, r;
  v = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
  r = 0;
  if (agnL_optboolean(L, 2, 0)) {  /* bcd -> int, fixed 2.14.9, taken from:
    https://stackoverflow.com/questions/45894333/how-to-convert-binary-coded-decimal-to-int-in-c, by leyanpan, answer #3 */
    uint32_t p = 1;
    while (v) {
      r += (v & 0xf) * p;  /* v & 0xf represents the respective digit, 0xf is a four-bit mask */
      v >>= 4;
      p *= 10;
    }
  } else {  /* int -> bcd, taken from: https://stackoverflow.com/questions/13247647/convert-integer-from-pure-binary-to-bcd */
    char i;
    for (i=0; v; i++) {
      ((char*)&r)[i/2] |= (i & 1) ? (v % 10) << 4 : (v % 10) & 0xf;  /* 2.14.9 modification */
      v /= 10;
    }
#if BYTE_ORDER == BIG_ENDIAN
    tools_swapuint32_t(&r);  /* 2.14.9 fix */
#endif
  }
  lua_pushnumber(L, r);
  return 1;
}


/* Adds two or more numbers a, b, ... using 4-byte unsigned integer arithmetic. The return is an integer.
   You can switch from unsigned to signed arithmetic by setting environ.kernel(signedbits = true), and from
   signed to unsigned arithmetic by environ.kernel(signedbits = false). */
static int bytes_add32 (lua_State *L) {  /* 2.14.7 */
  int32_t i, nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": need at least two arguments.", "bytes.add32");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t x = luaL_checkint32_t(L, 1);
    for (i=2; i <= nargs; i++) x += luaL_checkint32_t(L, i);
    res = x;
  } else {
    uint32_t x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    for (i=2; i <= nargs; i++) x += luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    res = x;
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Subtracts two or more numbers a, b, ... using 4-byte unsigned integer arithmetic. The return is an integer.
   You can switch from unsigned to signed arithmetic by setting environ.kernel(signedbits = true), and from
   signed to unsigned arithmetic by environ.kernel(signedbits = false). */
static int bytes_sub32 (lua_State *L) {  /* 2.14.7 */
  int32_t i, nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": need at least two arguments.", "bytes.sub32");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t x = luaL_checkint32_t(L, 1);
    for (i=2; i <= nargs; i++) x -= luaL_checkint32_t(L, i);
    res = x;
  } else {
    uint32_t x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    for (i=2; i <= nargs; i++) x -= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    res = x;
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Multiplies two or more numbers a, b, ... using 4-byte unsigned integer arithmetic. The return is an integer.
   You can switch from unsigned to signed arithmetic by setting environ.kernel(signedbits = true), and from
   signed to unsigned arithmetic by environ.kernel(signedbits = false). */
static int bytes_mul32 (lua_State *L) {  /* 2.14.7 */
  int32_t i, nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": need at least two arguments.", "bytes.mul32");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t x = luaL_checkint32_t(L, 1);
    for (i=2; i <= nargs; i++) x *= luaL_checkint32_t(L, i);
    res = x;
  } else {
    uint32_t x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    for (i=2; i <= nargs; i++) x *= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    res = x;
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Multiplies two numbers a, b, and adds further numbers c, ... using 4-byte unsigned integer arithmetic. The return
   is the integer a*b + c + ...
   You can switch from unsigned to signed arithmetic by setting environ.kernel(signedbits = true), and from
   signed to unsigned arithmetic by environ.kernel(signedbits = false). */
static int bytes_muladd32 (lua_State *L) {  /* 2.16.4 */
  int32_t i, nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs < 3)
    luaL_error(L, "Error in " LUA_QS ": need at least three arguments.", "bytes.muladd32");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t x = luaL_checkint32_t(L, 1);
    x *= luaL_checkint32_t(L, 1);  /* = a * b */
    for (i=3; i <= nargs; i++) x += luaL_checkint32_t(L, i);  /* a * b + c + ... */
    res = x;
  } else {
    uint32_t x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    x *= luaL_checkuint32_t(L, 1);  /* = a * b */
    for (i=3; i <= nargs; i++) x += luaL_checkuint32_t(L, i);  /* a * b + c + ..., 2.16.12 improvement */
    res = x;
  }
  lua_pushnumber(L, res);
  return 1;
}


/* 2.16.2, Multiplies two or more 32-bit integers x, y, ... in 64-bit space, with a final succeeding n-bit shift.
   If n < 0 then a left-shift is conducted, otherwise a right-shift. For no shift, pass 0 for n.
   Idea taken from WinFract.h (WGENERAL.ASM) */
static int bytes_mul32shift (lua_State *L) {
  int32_t i, nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs < 3)
    luaL_error(L, "Error in " LUA_QS ": need at least three arguments.", "bytes.mul32shift");
  if (L->settings & 1) {  /* signed operation ? */
    int64_t x = luaL_checkint32_t(L, 1);
    for (i=2; i < nargs; i++) x *= luaL_checkint32_t(L, i);
    aux_shift32((int32_t), x, luaL_checkint32_t(L, i), res);
  } else {
    uint64_t x = luaL_checkuint32_t(L, 1);  /* 2.16.11 improvement */
    for (i=2; i < nargs; i++) x *= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    aux_shift32((uint32_t), x, luaL_checkint32_t(L, i), res);
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Divides two or more numbers a, b, ... using 4-byte unsigned integer arithmetic. The return is an integer.
   You can switch from unsigned to signed arithmetic by setting environ.kernel(signedbits = true), and from
   signed to unsigned arithmetic by environ.kernel(signedbits = false). */
static int bytes_div32 (lua_State *L) {  /* 2.14.7 */
  int32_t i, nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": need at least two arguments.", "bytes.div32");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t x = luaL_checkint32_t(L, 1);
    for (i=2; i <= nargs; i++) x /= luaL_checkint32_t(L, i);
    res = x;
  } else {
    uint32_t x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    for (i=2; i <= nargs; i++) x /= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    res = x;
  }
  lua_pushnumber(L, res);
  return 1;
}


#ifndef BIT
#define BIT(nr)         (1UL << (nr))
#endif

#ifndef BIT_ULL
#define BIT_ULL(nr)     (1ULL << (nr))
#endif

#define xint32divmod(t,n,d) { \
  t q, mask; \
  q = 0; r = 0; \
  if (!d) { \
    n = 0; \
    r = 0; \
  } else if (d == 2) { \
    r = n & 1; \
    n >>= 1; \
  } else if (d == 16) { \
    r = n & 0xf; \
    n >>= 4; \
  } else if ((uint32_t)n <= 0xffffffff) { \
    t u = n; \
    r = u % d; \
    n = u / d; \
  } else { \
    for (mask = BIT_ULL(31); mask; mask >>= 1) { \
      r <<= 1; \
      if (n & mask) r |= 1; \
      if (r >= d) { \
        r -= d; \
        q |= mask; \
      } \
    } \
    n = q; \
  } \
}

/* Returns the quotient and remainder of the 4-byte division a/b. */
static int bytes_divmod32 (lua_State *L) {  /* 2.16.4 */
  int32_t nargs, r;
  lua_Number r0, r1;
  nargs = lua_gettop(L);
  if (nargs != 2)  /* 2.16.8 fix */
    luaL_error(L, "Error in " LUA_QS ": need two arguments.", "bytes.divmod32");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t n, d;
    n = luaL_checkint32_t(L, 1);
    d = luaL_checkint32_t(L, 2);
    xint32divmod(int32_t, n, d);
    r0 = n;
    r1 = (!r) ? AGN_NAN : r;
  } else {
    uint32_t n, d;
    n = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    d = luaL_checkuint32_t(L, 2);  /* 2.16.12 improvement */
    xint32divmod(uint32_t, n, d);
    if (!n && !r) {
      r0 = AGN_NAN; r1 = AGN_NAN;
    } else {
      r0 = n; r1 = r;
    }
  }
  lua_pushnumber(L, r0);
  lua_pushnumber(L, r1);
  return 2;
}


/* 2.16.2, Divides two or more 32-bit integers x, y, ... in 64-bit space, with a final succeeding n-bit shift.
   If n < 0 then a left-shift is conducted, otherwise a right-shift. For no shift, pass 0 for n.
   Idea taken from WinFract.h (WGENERAL.ASM) */
static int bytes_div32shift (lua_State *L) {  /* 2.16.2 */
  int32_t i, nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs < 3)
    luaL_error(L, "Error in " LUA_QS ": need at least three arguments.", "bytes.div32shift");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t x = luaL_checkint32_t(L, 1);
    for (i=2; i < nargs; i++) x /= luaL_checkint32_t(L, i);
    aux_shift32((int32_t), x, luaL_checkint32_t(L, i), res);
  } else {
    uint32_t x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    for (i=2; i < nargs; i++) x /= luaL_checkuint32_t(L, i);  /* 2.16.12 improvement */
    aux_shift32((uint32_t), x, luaL_checkint32_t(L, i), res);
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Take the modulus of two numbers a, b using 4-byte unsigned integer arithmetic. The return is the integer
   a % b. You can switch from unsigned to signed arithmetic by setting environ.kernel(signedbits = true),
   and from signed to unsigned arithmetic by environ.kernel(signedbits = false). */
static int bytes_mod32 (lua_State *L) {  /* 2.14.7 */
  int32_t nargs;
  lua_Number res;
  nargs = lua_gettop(L);
  if (nargs != 2)  /* 2.16.8 fix */
    luaL_error(L, "Error in " LUA_QS ": need two arguments.", "bytes.mod32");
  if (L->settings & 1) {  /* signed operation ? */
    int32_t x, y;
    x = luaL_checkint32_t(L, 1);
    y = luaL_checkint32_t(L, 2);
    res = (y == 0) ? AGN_NAN : x % y;
  } else {  /* unsigned mode */
    uint32_t x, y;
    x = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
    y = luaL_checkuint32_t(L, 2);  /* 2.16.12 improvement */
    res = (y == 0) ? AGN_NAN : x % y;
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Converts a number to its signed or unsigned 4-byte integer representation. Note that very large values
   (positive or negative) might overflow, e.g. bytes.numto32(2^32+1) -> 1. The result may differ across
   platforms in overflow situations.

   You can switch from unsigned to signed arithmetic by setting environ.kernel(signedbits = true), and
   from signed to unsigned arithmetic by environ.kernel(signedbits = false).

   Signed 32-bit integers are in the range from -2,147,483,648 = -2^31 through 2,147,483,647 = 2^31-1. 
   Unsigned 32-bit integers are between (including) 0 and 4294967295 = 2^32-1. */
static int bytes_numto32 (lua_State *L) {  /* 2.14.8 */
  lua_Number res;
  if (L->settings & 1) {  /* signed operation ? */
    res = (int32_t)luaL_checkint32_t(L, 1);  /* if number is out-of-range, clamps between -2^31 .. 2^31-1. */
  } else {  /* unsigned mode */
    res = (uint32_t)luaL_checkuint32_t(L, 1);
  }
  lua_pushnumber(L, res);
  return 1;
}


/* Determines the parity of the unsigned 4-byte integer x, i.e. the number of 1-bits in x modulo 2.
   Returns 0 if x is of even parity, and 1 in case of odd parity.
   See: https://stackoverflow.com/questions/21617970/how-to-check-if-value-has-even-parity-of-bits-or-odd, answer #5
   and http://graphics.stanford.edu/~seander/bithacks.html#ParityLookUpTable */

#ifndef __GNUC__
static const char ParityTable256[256] = {
#   define P2(n) n, n^1, n^1, n
#   define P4(n) P2(n), P2(n^1), P2(n^1), P2(n)
#   define P6(n) P4(n), P4(n^1), P4(n^1), P4(n)
    P6(0), P6(1), P6(1), P6(0) };
#endif

static int bytes_parity32 (lua_State *L) {  /* 2.14.8 */
  uint32_t v = luaL_checkuint32_t(L, 1);  /* 2.16.12 improvement */
#ifdef __GNUC__
  lua_pushinteger(L, __builtin_parity(v));  /* 2.14.12 */
#else
  if (v < 256) {  /* five percent faster */
    lua_pushinteger(L, ParityTable256[v]);
  } else {
    v ^= v >> 16;  /* table-lookup is a little bit faster than calculation */
    v ^= v >> 8;
    lua_pushinteger(L, ParityTable256[v & 0xff]);
  }
  /* 0 for even parity, 1 for odd parity */
#endif
  return 1;
}


/* On Big Endian systems, converts the number `x` into its Little Endian representation and returns it.
   On Little Endian platforms, just returns `x` unaltered. See also: `bytes.tobig`. */
static int bytes_tolittle (lua_State *L) {  /* 2.14.9, extended 2.27.2 */
  int what;
  lua_Number x = agn_checknumber(L, 1);
  (void)x;
  what = agnL_optinteger(L, 2, 8);
  switch (what) {
    case -4: {  /* 2.31.5 */
#if BYTE_ORDER == BIG_ENDIAN
      int32_t r = (int32_t)x;
      tools_swapint32_t(&r);
      lua_pushnumber(L, r);
#else
      lua_pushvalue(L, 1);
#endif
      break;
    }
    case 4: {
#if BYTE_ORDER == BIG_ENDIAN
      uint32_t r = (uint32_t)x;
      tools_swapuint32_t(&r);
      lua_pushnumber(L, r);
#else
      lua_pushvalue(L, 1);
#endif
      break;
    }
    case 8: default:
#if BYTE_ORDER == BIG_ENDIAN
      lua_pushnumber(L, tools_tolittleendian(x));
#else
      lua_pushvalue(L, 1);
#endif
  }
  return 1;
}


/* On Little Endian systems, converts the number `x` into its Big Endian representation and returns it.
   On Big Endian platforms, just returns `x` unaltered. See also: `bytes.tolittle`. */
static int bytes_tobig (lua_State *L) {  /* 2.14.9, extended 2.27.2 */
  int what;
  lua_Number x = agn_checknumber(L, 1);
  what = agnL_optinteger(L, 2, 8);
  switch (what) {
    case -4: {  /* 2.31.5 */
#if BYTE_ORDER != BIG_ENDIAN
      int32_t r = (int32_t)x;
      tools_swapint32_t(&r);
      lua_pushnumber(L, r);
#else
      lua_pushvalue(L, 1);
#endif
      break;
    }
    case 4: {
#if BYTE_ORDER != BIG_ENDIAN
      uint32_t r = (uint32_t)x;
      tools_swapuint32_t(&r);
      lua_pushnumber(L, r);
#else
      lua_pushvalue(L, 1);
#endif
      break;
    }
    case 8: default:
#if BYTE_ORDER != BIG_ENDIAN
      lua_pushnumber(L, tools_tobigendian(x));
#else
      lua_pushvalue(L, 1);
#endif
  }
  return 1;
}


/* Gets and clears the next bit from the unsigned 4-byte mask, starting with the most significant bit.
   The function returns the modified value of mask and the respective bit position 0 .. 31. */
static inline int aux_getnextbit (uint32_t *mask) {
  int32_t bit;
  if (*mask == 0) return -1;
#ifdef __GNUC__
  bit = 31 - __builtin_clz(*mask);
#else
  int y;
  bit = 31 - tools_clz32(*mask, &y);
  (void)y;
#endif
  *mask &= ~BIT(bit);
  return bit;
}

static int bytes_nextbit (lua_State *L) {    /* 2.16.4 */
  uint32_t mask = luaL_checkuint32_t(L, 1);  /* 2.16.8 fix, 2.16.12 improvement */
  int pos = aux_getnextbit(&mask);
  lua_pushnumber(L, mask);
  lua_pushnumber(L, pos);
  return 2;
}


/* Checks whether the given number is in the range of a signed or an unsigned 4-byte integer and returns `true` or `false`.
   To check in which mode Agena is, check the environ.kernel/signedbits setting. It should usually be unsigned.
   If you are in unsigned mode, the argument should be in the range 0 .. environ.kernel().maxulong = 0 .. 4'294'967'295.
   If you are in signed mode, the argument should be in the range environ.kernel().minlong .. environ.kernel().maxlong =
   -2'147'483'647 .. 2'147'483'647.

   Example:

   > import bytes

   > environ.kernel().signedbits:  # we are in unsigned mode (C uint32_t's).
   false

   > bytes.isint32(4'294'967'295):
   true

   > bytes.isint32(4'294'967'295 + 1):
   false
*/

static int bytes_isint32 (lua_State *L) {  /* 2.16.12, this version is 25 % faster than a simple (lua_Number)((u/int32_t)x) == x query */
  lua_Number x;
  int32_t hx, ux;
  x = agn_checknumber(L, 1);
  if (tools_isfrac(x)) {
    lua_pushfalse(L);
    return 1;
  }
  GET_HIGH_WORD(hx, x);  /* calling a special tools_isint() routine that returns hx, ux is not faster */
  ux = ((((uint32_t)hx) >> 20) & 0x7ff) - 0x3ff;  /* unbiased exponent: math.exponent(x) - 1, except 0 -> -1023 */
  if (L->settings & 1)  /* signed bits operation ? */
    lua_pushboolean(L, ux == -1023 || !(ux > 30));  /* x < -2147483647 or x >= 2147483647 ? */
  else  /*  x < 0 or x > 4294967295 ? */
    lua_pushboolean(L, ux == -1023 || !(hx < 0 || ux > 31));
  return 1;
}


/* Calculates the optimal number of bytes (places) in a C `array` (e.g. a memfile, numarray or even a string) if it
   shall be aligned on the 4- or 8-byte word boundary. 2.26.2 */
static int bytes_optsize (lua_State *L) {
  size_t chunks;
  lua_pushinteger(L, tools_optstrlen(agn_checkposint(L, 1), &chunks));
  (void)chunks;
  return 1;
}


/* Swaps the lower n bytes of the unsigned 4-byte integer x; bytes above those will be discarded. The return is the new integer. */
static int bytes_swaplower (lua_State *L) {  /* 2.38.3 */
  lua_pushnumber(L, tools_swaplower32(agn_checkuint32_t(L, 1), agn_checkinteger(L, 2)));
  return 1;
}


/* Swaps the upper n bytes of the unsigned integer x; bytes below those will be discarded. The return is the new integer. */
static int bytes_swapupper (lua_State *L) {  /* 2.38.3 */
  lua_pushnumber(L, tools_swapupper32(agn_checkuint32_t(L, 1), agn_checkinteger(L, 2)));
  return 1;
}


static int bytes_swap (lua_State *L) {  /* 2.38.3 */
  lua_pushnumber(L, tools_swapu32(agn_checkuint32_t(L, 1)));
  return 1;
}


/* Takes two unsigned 4-byte integers hx and lx and applies one of the following binary operations on them:
   'xor' (the default), 'and', 'or'. See also: `bytes.numwords`. By passing a non-negative mask as the optional fourth argument,
   the mask is applied to the intermediate result, the default is 0xFFFFFFFF.
   If a fifth positive sh integer is given, the intermediate result is right-shifted sh bits; if sh is a negative integer, it is
   left-shifted sh bits. If sh is 0 (the default), there is no shift.
   If a sixth argument n is given, a positive integer, the intermediate result is taken modulus n. The default is 1.

   Thus:
   option = 'or':  (((hx || lx) && mask) >>> sh) % n, if sh > 0,
   option = 'and': (((hx && lx) && mask) >>> sh) % n, if sh > 0,
   option = 'xor': (((hx ^^ lx) && mask) >>> sh) % n, if sh > 0,

   or

   option = 'or':  (((hx || lx) && mask) <<< |sh|) % n, if sh < 0,
   option = 'and': (((hx && lx) && mask) <<< |sh|) % n, if sh < 0,
   option = 'xor': (((hx ^^ lx) && mask) <<< |sh|) % n, if sh < 0.

   based on hashes.interweave, 2.38.3 */
enum WEAVEACTION {AGN_XOR, AGN_AND, AGN_OR};

static int bytes_interweave (lua_State *L) {
  int op;
  uint32_t hx, lx, mask, sh, n;
  static const int action[] = {AGN_XOR, AGN_AND, AGN_OR};
  static const char *const actionnames[] = {"xor", "and", "or", NULL};
  hx = agn_checkuint32_t(L, 1);
  lx = agn_checkuint32_t(L, 2);
  op = luaL_checkoption(L, 3, "xor", actionnames);
  mask = agnL_optnonnegative(L, 4, 0xFFFFFFFF);
  sh = agnL_optinteger(L, 5, 0);  /* shift */
  n = agnL_optinteger(L, 6, 0);   /* C % or Fibonacci modulus */
  switch (action[op]) {
    case AGN_AND: { hx &= lx; break; }
    case AGN_OR:  { hx |= lx; break; }
    case AGN_XOR: { hx ^= lx; break; }
    default:
      luaL_error(L, "Error in " LUA_QS ": unknown option.", "hashes.interweave");
  }
  hx = (mask == 0xFFFFFFFF) ? hx : hx & mask;
  if (sh < 0)
    hx <<= sh;
  else if (sh > 0)
    hx >>= sh;
  if (n > 0) hx %= n;
  else if (n < 0) {  /* Fibonacci modulus, 2.38.3 */
    uint64_t t = (2654435769ULL * (uint64_t)hx) & 0x00000000FFFFFFFF;
    n &= 0x0000001F;  /* make sure shift is 31 or less */
    hx = t >> (32 + n);  /* returns a value in the uint32_t domain, n is negative */
  }
  lua_pushnumber(L, hx);
  return 1;
}


static const struct luaL_Reg cast_lib [] = {  /* metamethods for numeric userdata `n` */
  {"__gc", mt_uniongc},                  /* do not forget garbage collection */
  {"__tostring", mt_union2string},       /* for output at the console, e.g. print(n) */
  {NULL, NULL}
};


static const struct luaL_Reg ieee_lib [] = {  /* metamethods for numeric userdata `n` */
  {"__gc", ieee754_uniongc},             /* do not forget garbage collection */
  {"__tostring", ieee754_union2string},  /* for output at the console, e.g. print(n) */
  {NULL, NULL}
};


static const luaL_Reg byteslib[] = {
  {"add32", bytes_add32},                  /* added on March 26, 2019 */
  {"and32", bytes_and32},                  /* added on January 16, 2019 */
  {"arshift32", bytes_arshift32},          /* added on January 15, 2019 */
  {"bcd", bytes_bcd},                      /* added on March 22, 2019 */
  {"cast", bytes_cast},                    /* added on August 29, 2018 */
  {"castint", bytes_castint},              /* added on March 16, 2017 */
  {"div32", bytes_div32},                  /* added on March 26, 2019 */
  {"divmod32", bytes_divmod32},            /* added on October 04, 2019 */
  {"div32shift", bytes_div32shift},        /* added on September 13, 2019 */
  {"extract32", bytes_extract32},          /* added on January 15, 2019 */
  {"fpbtoint", bytes_fpbtoint},            /* added on November 21, 2014 */
  {"getdouble", bytes_getdouble},          /* added on August 29, 2018 */
  {"gethigh", bytes_gethigh},              /* added on August 29, 2018 */
  {"getlow", bytes_getlow},                /* added on August 29, 2018 */
  {"getunbiased", bytes_getunbiased},      /* added on December 01, 2019 */
  {"getwords", bytes_getwords},            /* added on August 29, 2018 */
  {"ieee", bytes_ieee},                    /* added on October 01, 2022 */
  {"interweave", bytes_interweave},        /* added on April 07, 2023 */
  {"inttofpb", bytes_inttofpb},            /* added on November 21, 2014 */
  {"isint32", bytes_isint32},              /* added on January 04, 2020 */
  {"leadzeros", bytes_leadzeros},          /* added on November 18, 2017 */
  {"leastsigbit", bytes_leastsigbit},      /* added on July 21, 2018 */
  {"mask32", bytes_mask32},                /* added on March 10, 2019 */
  {"mul32", bytes_mul32},                  /* added on March 26, 2019 */
  {"mul32shift", bytes_mul32shift},        /* added on September 13, 2019 */
  {"muladd32", bytes_muladd32},            /* added on October 04, 2019 */
  {"mod32", bytes_mod32},                  /* added on March 26, 2019 */
  {"mostsigbit", bytes_mostsigbit},        /* added on July 21, 2018 */
  {"nand32", bytes_nand32},                /* added on January 17, 2019 */
  {"nextbit", bytes_nextbit},              /* added on October 04, 2019 */
  {"nor32", bytes_nor32},                  /* added on January 17, 2019 */
  {"not32", bytes_not32},                  /* added on January 16, 2019 */
  {"numhigh", bytes_numhigh},              /* 0.27.0 / 2.4.5, function added on March 05, 2015 */
  {"numlow", bytes_numlow},                /* 0.27.0 / 2.4.5, function added on March 05, 2015 */
  {"numto32", bytes_numto32},              /* added on March 28, 2019 */
  {"numwords", bytes_numwords},            /* added on March 05, 2017 */
  {"onebits", bytes_onebits},              /* added on November 18, 2017 */
  {"optsize", bytes_optsize},              /* added on March 21, 2022 */
  {"or32", bytes_or32},                    /* added on January 16, 2019 */
  {"pack", bytes_pack},                    /* added on January 04, 2019 */
  {"packsize", bytes_packsize},            /* added on January 04, 2019 */
  {"parity32", bytes_parity32},            /* added on March 30, 2019 */
  {"replace32", bytes_replace32},          /* added on January 15, 2019 */
  {"reverse", bytes_reverse},              /* added on July 21, 2018 */
  {"rotate32", bytes_rotate32},            /* added on January 14, 2019 */
  {"setdouble", bytes_setdouble},          /* added on August 29, 2018 */
  {"sethigh", bytes_sethigh},              /* added on September 02, 2018 */
  {"setlow", bytes_setlow},                /* added on September 02, 2018 */
  {"setnumhigh", bytes_setnumhigh},        /* 0.27.0 / 2.4.5, function added on March 05, 2015 */
  {"setnumlow", bytes_setnumlow},          /* 0.27.0 / 2.4.5, function added on March 05, 2015 */
  {"setnumwords", bytes_setnumwords},      /* added on August 30, 2018 */
  {"setwords", bytes_setwords},            /* added on August 29, 2018 */
  {"setieeedouble", bytes_setieeedouble},  /* added on October 01, 2022 */
  {"getieeedouble", bytes_getieeedouble},  /* added on October 01, 2022 */
  {"setieee", bytes_setieee},              /* added on October 01, 2022 */
  {"getieee", bytes_getieee},              /* added on October 01, 2022 */
  {"setieeehigh", bytes_setieeehigh},      /* added on October 01, 2022 */
  {"setieeelow", bytes_setieeelow},        /* added on October 01, 2022 */
  {"setieeesignbit", bytes_setieeesignbit},/* added on October 01, 2022 */
  {"setieeeexpo", bytes_setieeeexpo},      /* added on October 01, 2022 */
  {"shift32", bytes_shift32},              /* added on January 15, 2019 */
  {"sub32", bytes_sub32},                  /* added on March 26, 2019 */
  {"swap", bytes_swap},                    /* added on April 06, 2023 */
  {"swaplower", bytes_swaplower},          /* added on April 06, 2023 */
  {"swapupper", bytes_swapupper},          /* added on April 06, 2023 */
  {"tobig", bytes_tobig},                  /* added on April 12, 2019 */
  {"tobinary", bytes_tobinary},            /* added on December 21, 2015 */
  {"tobytes", bytes_tobytes},              /* added on May 24, 2015 */
  {"tolittle", bytes_tolittle},            /* added on April 12, 2019 */
  {"tonumber", bytes_tonumber},            /* added on May 25, 2015 */
  {"trailzeros", bytes_trailzeros},        /* added on April 21, 2019 */
  {"trim32", bytes_trim32},                /* added on March 10, 2019 */
  {"unpack", bytes_unpack},                /* added on January 04, 2019 */
  {"xnor32", bytes_xnor32},                /* added on January 17, 2019 */
  {"xor32", bytes_xor32},                  /* added on January 17, 2019 */
  {"xorshift32", bytes_xorshift32},        /* added on February 06, 2020 */
  {NULL, NULL}
};


/*
** Open bytes library
*/
LUALIB_API int luaopen_bytes (lua_State *L) {
  luaL_newmetatable(L, "cast");         /* metatable for cast userdata, adds it to the registry with key 'cast' */
  luaL_register(L, NULL, cast_lib);     /* assign C metamethods to this metatable */
  luaL_newmetatable(L, "ieee");         /* metatable for cast userdata, adds it to the registry with key 'cast' */
  luaL_register(L, NULL, ieee_lib);     /* assign C metamethods to this metatable */
  luaL_register(L, AGENA_BYTESLIBNAME, byteslib);
  return 1;
}

