/*
* lmapm.c
*
* big-number library for Lua 5.1 based on the MAPM library originally written by Michael C. Ring.

* Luiz Henrique de Figueiredo <lhf@tecgraf.puc-rio.br>
* 03 Apr 2009 00:11:32
*
* This code is hereby placed in the public domain.
*
* MAPM binding for Agena initiated January 17, 2010
*
* For Michael C. Ring's MAPM library see: https://github.com/achan001/MAPM-5 et al.
* For the Lua binding see: https://github.com/LuaDist/mapm/
*/

#include <stdlib.h>
#include "m_apm.h"
#include "m_apm_lc.h"

#define mapm_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "lobject.h"
#include "agenalib.h"
#include "prepdefs.h"  /* FORCE_INLINE */

/* #define AGENA_LIBVERSION	"MAPM 4.9.5a binding, version 2.0.2 as of November 05, 2023\n" */

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(__ANSI__))
#define AGENA_MAPMLIBNAME "mapm"
LUALIB_API int (luaopen_mapm) (lua_State *L);
#endif


/* Converts an object that has already been instantiated to userdata and pushes it onto the stack */
#ifndef lua_boxpointer
#define lua_boxpointer(L, u) \
	(*(void **)(lua_newuserdata(L, sizeof(void *))) = (u))
#endif

#define MYNAME		"mapm"
#define MYVERSION	MYNAME " library for " AGENA_VERSION " / Apr 2009 / "\
			"using MAPM " MAPM_LIB_SHORT_VERSION
#define MYTYPE		"xnumber"


static int DIGITS = 17;
#define MYDIGITS   (DIGITS + 2)
static lua_State *LL = NULL;

void M_apm_log_error_msg (int fatal, char *message) {
#ifdef IGNORE_MAPM_WARNINGS
  if (fatal)
#endif
  luaL_error(LL, "(MAPM) %s.", message);
}


/* Instantiates a new M_APM object, creates a userdata from it, pushes it onto the stack and assigns the mt */
static M_APM Bnew (lua_State *L) {
  M_APM x = m_apm_init();
  lua_boxpointer(L, x);
  lua_setmetatabletoobject(L, -1, MYTYPE, 1);  /* 3.5.0/1 change */
  return x;
}


/* - With a number or string at stack position i, creates a new M_APM userdata object and replaces the value at stack index i
     with this userdata object. The stack is left unchanged. The address of the M_APM ud object is returned.
   - With M_APM userdata, the address of the ud object is returned. */
static M_APM Bget (lua_State *L, int i) {
  LL = L;
  switch (lua_type(L, i)) {
    case LUA_TNUMBER: {
      M_APM x = Bnew(L);
      m_apm_set_double(x, lua_tonumber(L, i));  /* Agena 1.4.3/1.5.0, 2.9.4 change */
      lua_replace(L, i);
      return x;
    }
    case LUA_TSTRING: {
      M_APM x = Bnew(L);
      m_apm_set_string(x, (char*)lua_tostring(L, i));
      lua_replace(L, i);
      return x;
    }
    default:
      return *((void**)luaL_checkudata(L, i, MYTYPE));
  }
  return NULL;
}


/* Applies function f onto the value at stack position 1, and returns the result as a new M_APM userdata object */
static int Bdo0 (lua_State *L, void (*f)(M_APM y, M_APM x)) {
  M_APM a, c;
  a = Bget(L, 1);
  c = Bnew(L);
  f(c, a);
  return 1;
}


static int Bdo1 (lua_State *L, void (*f)(M_APM y, int n, M_APM x)) {
  M_APM a, c;
  int n = agnL_optnonnegint(L, 2, DIGITS);
  a = Bget(L, 1);
  c = Bnew(L);
  f(c, n, a);
  return 1;
}


static int Bdo2 (lua_State *L, void (*f)(M_APM z, M_APM x, M_APM y)) {
  M_APM a, b, c;
  a = Bget(L, 1);
  b = Bget(L, 2);
  c = Bnew(L);
  f(c, a, b);
  return 1;
}


static int Bdo3 (lua_State *L, void (*f)(M_APM z, int n, M_APM x, M_APM y)) {
  M_APM a, b, c;
  int n;
  n = agnL_optnonnegint(L, 3, DIGITS);
  a = Bget(L, 1);
  b = Bget(L, 2);
  c = Bnew(L);
  f(c, n, a, b);
  return 1;
}


static int Bdigits (lua_State *L) {  /* digits([n]) */
  DIGITS = agnL_optnonnegint(L, 1, DIGITS);
  lua_pushinteger(L, DIGITS);  /* changed Agena 0.30.3 */
  return 1;
}


static int Btostring (lua_State *L) {  /* tostring(x,[n,exp]) */
  char *s;
  int n;
  M_APM a = Bget(L, 1);
  n = agnL_optinteger(L, 2, DIGITS);
  if (lua_toboolean(L, 3)) {
    int m = (n < 0) ? m_apm_significant_digits(a) : n;
    s = malloc((m + 16)*sizeof(char));
    if (s != NULL) {
      m_apm_to_string(s, n, a);
    } else {  /* 4.11.5 */
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "mapm tostring mt");
    }
  } else {
    s = m_apm_to_fixpt_stringexp(n, a, '.', 0, 0);
  }
  lua_pushstring(L, s);
  xfree(s);
  return 1;
}


static int mt_tostring (lua_State *L) {
  Btostring(L);
  return 1;
}


/* static int Btonumber (lua_State *L) {  // tonumber(x), extended 2.31.12
  int n = agnL_optposint(L, 2, 20);    // 20 is enough for IEEE doubles
  lua_settop(L, 1);
  lua_pushinteger(L, n);
  lua_pushboolean(L, 1);
  Btostring(L);
  lua_pushnumber(L, lua_tonumber(L, -1));  // Agena 1.4.3/1.5.0, patched 2.9.4
  return 1;
} */


static lua_Number aux_stringtonumber (lua_State *L, M_APM a, int n) {  /* 3.6.1, this version is w/o a call to lua_settop */
  lua_Number x;
  char *s;
  int m = (n < 0) ? m_apm_significant_digits(a) : n;
  s = malloc((m + 16)*sizeof(char));
  if (s != NULL) {
    m_apm_to_string(s, n, a);
  } else {
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "mapm.xtonumber");
  }
  lua_pushstring(L, s);
  x = lua_tonumber(L, -1);  /* does not change the stack */
  xfree(s);  /* 3.6.1 change: call free only _after_ conversion to a number */
  lua_pop(L, 1);  /* pop string */
  return x;
}

static int Btonumber (lua_State *L) {  /* mapm.xtonumber(x), extended 2.31.12, changed 3.6.1 */
  lua_pushnumber(L, aux_stringtonumber(L, Bget(L, 1), agnL_optposint(L, 2, 20)));
  return 1;
}


static int Bnumber (lua_State *L) {  /* mapm.xnumber(x) */
  Bget(L, 1);
  lua_settop(L, 1);
  return 1;
}


static int Bround (lua_State *L) {   /* round(x) */
  return Bdo1(L, m_apm_round);
}


static int Binv (lua_State *L) {     /* inv(x) */
  return Bdo1(L, m_apm_reciprocal);
}

static int mt_recip (lua_State *L) {  /* recip(x) */
  return Bdo1(L, m_apm_reciprocal);
}


static int Bsqrt (lua_State *L) {  /* sqrt(x) */
  return Bdo1(L, m_apm_sqrt);
}

static int mt_sqrt (lua_State *L) {  /* sqrt(x) */
  return Bdo1(L, m_apm_sqrt);
}


static int Bcbrt (lua_State *L) {  /* cbrt(x) */
  return Bdo1(L, m_apm_cbrt);
}


static int Blog (lua_State *L) {  /* ln(x) */
  return Bdo1(L, m_apm_log);
}

static int mt_ln (lua_State *L) {  /* ln(x) */
  return Bdo1(L, m_apm_log);
}


static int Blog10 (lua_State *L) {  /* log10(x) */
  return Bdo1(L, m_apm_log10);
}


static int Bexp (lua_State *L) {  /* exp(x) */
  return Bdo1(L, m_apm_exp);
}

static int mt_exp (lua_State *L) {  /* exp(x) */
  return Bdo1(L, m_apm_exp);
}


static int Bsin (lua_State *L) {  /* sin(x) */
  return Bdo1(L, m_apm_sin);
}

static int mt_sin (lua_State *L) {  /* sin(x) */
  return Bdo1(L, m_apm_sin);
}


static int Bcos (lua_State *L) {  /* cos(x) */
  return Bdo1(L, m_apm_cos);
}

static int mt_cos (lua_State *L) {  /* cos(x) */
  return Bdo1(L, m_apm_cos);
}


static int Btan (lua_State *L) {  /* tan(x) */
  return Bdo1(L, m_apm_tan);
}

static int mt_tan (lua_State *L) {   /* tan(x) */
  return Bdo1(L, m_apm_tan);
}


static int Basin (lua_State *L) {  /* asin(x) */
  return Bdo1(L, m_apm_asin);
}

static int mt_arcsin (lua_State *L) {  /* arcsin(x) */
  return Bdo1(L, m_apm_asin);
}


static int Bacos (lua_State *L) {    /* acos(x) */
  return Bdo1(L, m_apm_acos);
}

static int mt_arccos (lua_State *L) {  /* arccos(x) */
  return Bdo1(L, m_apm_acos);
}


static int Batan (lua_State *L) {    /* atan(x) */
  return Bdo1(L, m_apm_atan);
}

static int mt_arctan (lua_State *L) {  /* arctan(x) */
  return Bdo1(L, m_apm_atan);
}


/* arcsec, arcsecant */
#define basec() { \
  M_APM a, r; \
  a = Bget(L, 1); \
  r = Bnew(L); \
  if (m_apm_compare(a, MM_One) < 0) { \
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'m_apm_arcsec\', |Argument| < 1"); \
    M_set_to_zero(r); \
  } else { \
    m_apm_reciprocal(r, MYDIGITS, a); \
    m_apm_arccos(r, MYDIGITS, r); \
  } \
  return 1;\
}

static int Basec (lua_State *L) {   /* arcsec(x), 3.3.3 */
  basec();
}


static int mt_arcsec (lua_State *L) {  /* arcsec(x), 3.3.3 */
  basec();
}


/* arccsc, arccosecant */
static int Bacsc (lua_State *L) {  /* arccsc(x), 3.3.3 */
  M_APM a, r;
  a = Bget(L, 1);
  r = Bnew(L);
  if (m_apm_compare(a, MM_One) < 0) {      /* |x| < 1 */
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'m_apm_arccsc\', |Argument| < 1");  /* 3.5.2 fix */
    M_set_to_zero(r);
  } else {
    m_apm_reciprocal(r, DIGITS + 2, a);
    m_apm_arcsin(r, DIGITS, r);
  }
  return 1;
}


static int Bsincos (lua_State *L) {  /* sincos(x) */
  M_APM a, s, c;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  s = Bnew(L);
  c = Bnew(L);
  m_apm_sin_cos(s, c, n, a);
  return 2;
}


static int Bsinhcosh (lua_State *L) {  /* sinhcosh(x), 3.5.2 */
  M_APM a, s, c;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  s = Bnew(L);
  c = Bnew(L);
  m_apm_sinh(s, n, a);
  m_apm_cosh(c, n, a);
  return 2;
}


static int Batan2 (lua_State *L) {  /* atan2(y, x) */
  return Bdo3(L, m_apm_atan2);
}


static int Bsinh (lua_State *L) {   /* sinh(x) */
  return Bdo1(L, m_apm_sinh);
}

static int mt_sinh (lua_State *L) {   /* sinh(x) */
  return Bdo1(L, m_apm_sinh);
}


static int Bcosh (lua_State *L) {   /* cosh(x) */
  return Bdo1(L, m_apm_cosh);
}

static int mt_cosh (lua_State *L) {   /* cosh(x) */
  return Bdo1(L, m_apm_cosh);
}


static int Btanh (lua_State *L) {   /* tanh(x) */
  return Bdo1(L, m_apm_tanh);
}

static int mt_tanh (lua_State *L) {   /* tanh(x) */
  return Bdo1(L, m_apm_tanh);
}


static int Basinh (lua_State *L) {  /* asinh(x) */
  return Bdo1(L, m_apm_asinh);
}


static int Bacosh (lua_State *L) {  /* acosh(x) */
  return Bdo1(L, m_apm_acosh);
}


static int Batanh (lua_State *L) {  /* atanh(x) */
  return Bdo1(L, m_apm_atanh);
}


static int Bsec (lua_State *L) {  /* xsec(x) = 1/cos(x), 3.5.2 */
  M_APM a, r, co;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  co = m_apm_init();
  m_apm_cos(co, n + 2, a);
  if (m_apm_compare(co, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xsec\', division by zero");
    M_set_to_zero(r);
  } else {
    m_apm_reciprocal(r, n + 2, co);
  }
  m_apm_free(co);
  return 1;
}


static int Bsech (lua_State *L) {  /* xsech(x) = 1/cosh(x), 3.5.2 */
  M_APM a, r, coh;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  coh = m_apm_init();
  m_apm_cosh(coh, n + 2, a);
  if (m_apm_compare(coh, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xsec\', division by zero");
    M_set_to_zero(r);
  } else {
    m_apm_reciprocal(r, n + 2, coh);
  }
  m_apm_free(coh);
  return 1;
}


static int Bcsc (lua_State *L) {  /* xcsc(x) = 1/sin(x), 3.5.2 */
  M_APM a, r, si;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  si = m_apm_init();
  m_apm_sin(si, n + 2, a);
  if (m_apm_compare(si, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xcsc\', division by zero");
    M_set_to_zero(r);
  } else {
    m_apm_reciprocal(r, n + 2, si);
  }
  m_apm_free(si);
  return 1;
}


static int Bcsch (lua_State *L) {  /* xcsch(x) = 1/sinh(x), 3.5.2 */
  M_APM a, r, sih;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  sih = m_apm_init();
  m_apm_sinh(sih, n + 2, a);
  if (m_apm_compare(sih, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xcsch\', division by zero");
    M_set_to_zero(r);
  } else {
    m_apm_reciprocal(r, n + 2, sih);
  }
  m_apm_free(sih);
  return 1;
}


static int Bcot (lua_State *L) {  /* xcot(x) = -tan(Pi/2 + x), 3.5.2 */
  M_APM a, r, t, ta;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  t = m_apm_init();
  ta = m_apm_init();
  m_apm_add(t, MM_lc_HALF_PI, a);
  m_apm_tan(ta, n + 2, t);
  m_apm_negate(r, ta);
  m_apm_free(t); m_apm_free(ta);
  return 1;
}


static int Bcoth (lua_State *L) {  /* xcoth(x) = 1/tanh(x), 3.5.2 */
  M_APM a, r, tah;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  tah = m_apm_init();
  m_apm_tanh(tah, n + 2, a);
  if (m_apm_compare(tah, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xcoth\', division by zero");
    M_set_to_zero(r);
  } else {
    m_apm_reciprocal(r, n + 2, tah);
  }
  m_apm_free(tah);
  return 1;
}


static int Bsinc (lua_State *L) {  /* xsinc(x), 3.5.2 */
  M_APM a, r, si;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  if (m_apm_compare(a, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xsinc\', division by zero");
    M_set_to_zero(r);
  } else {
    si = m_apm_init();
    m_apm_sin(si, MYDIGITS, a);
    m_apm_divide(r, n, si, a);
    m_apm_free(si);
  }
  return 1;
}


static int mt_sinc (lua_State *L) {  /* 3.5.2 */
  Bsinc(L);
  return 1;
}


static int Bcosc (lua_State *L) {  /* xcosc(x), 3.5.2 */
  M_APM a, r, co;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  if (m_apm_compare(a, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xcosc\', division by zero");
    M_set_to_zero(r);
  } else {
    co = m_apm_init();
    m_apm_cos(co, MYDIGITS, a);
    m_apm_divide(r, n, co, a);
    m_apm_free(co);
  }
  return 1;
}


static int Btanc (lua_State *L) {  /* xtanc(x), 3.5.2 */
  M_APM a, r, ta;
  int n = agnL_optinteger(L, 2, DIGITS);
  a = Bget(L, 1);
  r = Bnew(L);
  if (m_apm_compare(a, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xtanc\', division by zero");
    M_set_to_zero(r);
  } else {
    ta = m_apm_init();
    m_apm_tan(ta, MYDIGITS, a);
    m_apm_divide(r, n, ta, a);
    m_apm_free(ta);
  }
  return 1;
}


static int Babs (lua_State *L) {   /* abs(x) */
  return Bdo0(L, m_apm_absolute_value);
}


static int Bneg (lua_State *L) {   /* neg(x) */
  return Bdo0(L, m_apm_negate);
}

static int mt_unm (lua_State *L) {   /* -(x), 3.5.2 fix */
  return Bneg(L);
}


static int Bfactorial (lua_State *L) {  /* factorial(x) */
  return Bdo0(L, m_apm_factorial);
}


static int Bfloor (lua_State *L) {  /* floor(x) */
  return Bdo0(L, m_apm_floor);
}


static int Bceil (lua_State *L) {   /* ceil(x) */
  return Bdo0(L, m_apm_ceil);
}


/* C trunc, round towards the next integer towards zero */
#define bint() { \
  M_APM a, r; \
  a = Bget(L, 1); \
  r = Bnew(L); \
  if (a->m_apm_sign < 0) \
    m_apm_ceil(r, a); \
  else \
    m_apm_floor(r, a); \
  return 1; \
}

static int Bint (lua_State *L) {   /* int(x), 3.3.3 */
  bint();
}


static int mt_int (lua_State *L) {  /* int(x), 3.3.3 */
  bint();
}


static int Badd (lua_State *L) {    /* add(x, y) */
  return Bdo2(L, m_apm_add);
}

static int mt_add (lua_State *L) {    /* add(x, y), 3.5.2 fix */
  return Bdo2(L, m_apm_add);
}


static int Bsub (lua_State *L) {    /* sub(x, y) */
  return Bdo2(L, m_apm_subtract);
}

static int mt_sub (lua_State *L) {    /* sub(x, y), 3.5.2 fix */
  return Bdo2(L, m_apm_subtract);
}


static int Bmul (lua_State *L) {    /* mul(x, y) */
  return Bdo2(L, m_apm_multiply);
}

static int mt_mul (lua_State *L) {    /* mul(x, y), 3.5.2 fix */
  return Bdo2(L, m_apm_multiply);
}


static int Bdiv (lua_State *L) {    /* div(x, y) */
  return Bdo3(L, m_apm_divide);
}

static int mt_div (lua_State *L) {    /* div(x, y), 3.5.2 fix */
  return Bdo3(L, m_apm_divide);
}


static int Bidiv (lua_State *L) {   /* idiv(x, y) */
  M_APM a, b, p, q;
  a = Bget(L, 1);
  b = Bget(L, 2);
  p = Bnew(L);
  q = Bnew(L);
  m_apm_integer_div_rem(p, q, a, b);
  return 2;
}


static int mt_idiv (lua_State *L) {   /* idiv(x, y), new 3.5.2 */
  M_APM a, b, p, q;
  a = Bget(L, 1);
  b = Bget(L, 2);
  p = Bnew(L);
  q = Bnew(L);
  m_apm_integer_div_rem(p, q, a, b);
  return 2;
}


static int Bmod (lua_State *L) {    /* __mod(x, y) */
  Bidiv(L);
  return 1;
}

static int mt_mod (lua_State *L) {    /* __mod(x, y), 3.5.2 fix */
  Bidiv(L);
  return 1;
}


static int Bpow (lua_State *L) {    /* pow(x, y) */
  return Bdo3(L, m_apm_pow);
}

static int mt_pow (lua_State *L) {    /* pow(x, y), 3.5.2 fix */
  return Bdo3(L, m_apm_pow);
}


static int mt_ipow (lua_State *L) {   /* x ** n, 2.33.1, 3.5.2 fix */
  M_APM x, r;
  int n;
  x = Bget(L, 1);
  n = agn_checkinteger(L, 2);
  r = Bnew(L);
  m_apm_integer_pow(r, DIGITS, x, n);
  return 1;
}


static int Bcompare (lua_State *L) {  /* compare(x, y) */
  M_APM a, b;
  a = Bget(L, 1);
  b = Bget(L, 2);
  lua_pushinteger(L, m_apm_compare(a, b));
  return 1;
}


static int Beq (lua_State *L) {
  M_APM a, b;
  a = Bget(L, 1);
  b = Bget(L, 2);
  lua_pushboolean(L, m_apm_compare(a, b) == 0);
  return 1;
}

static int mt_eq (lua_State *L) {  /* 3.5.2 fix */
  return Beq(L);
}


static int Blt (lua_State *L) {
  M_APM a, b;
  a = Bget(L, 1);
  b = Bget(L, 2);
  lua_pushboolean(L, m_apm_compare(a, b) < 0);
  return 1;
}

static int mt_lt (lua_State *L) {  /* 3.5.2 fix */
  return Blt(L);
}


static int mt_le (lua_State *L) {  /* new 3.5.2 */
  M_APM a, b;
  a = Bget(L, 1);
  b = Bget(L, 2);
  lua_pushboolean(L, m_apm_compare(a, b) <= 0);
  return 1;
}


static int Bsign (lua_State *L) {       /* sign(x) */
  M_APM a = Bget(L, 1);
  lua_pushinteger(L, m_apm_sign(a));
  return 1;
}

static int mt_sign (lua_State *L) {   /* sign mt, 3.5.2 fix */
  M_APM a, r;
  a = Bget(L, 1);
  r = Bnew(L);
  m_apm_set_double(r, (double)m_apm_sign(a));
  return 1;
}


static int mt_abs (lua_State *L) {   /* abs mt */
  return Bdo0(L, m_apm_absolute_value);
}


static int Bexponent (lua_State *L) {   /* exponent(x) */
  M_APM a = Bget(L, 1);
  lua_pushinteger(L, m_apm_exponent(a));
  return 1;
}


static int Bisint (lua_State *L) {      /* isint(x) */
  M_APM a = Bget(L, 1);
  lua_pushboolean(L, m_apm_is_integer(a));
  return 1;
}


static int Biseven (lua_State *L) {     /* iseven(x) */
  M_APM a = Bget(L, 1);
  lua_pushboolean(L, m_apm_is_integer(a) && m_apm_is_even(a));
  return 1;
}

static int mt_even (lua_State *L) {     /* even(x), new 3.5.2 */
  M_APM a;
  a = Bget(L, 1);
  lua_pushboolean(L, m_apm_is_integer(a) && m_apm_is_even(a));
  return 1;
}


static int Bisodd (lua_State *L) {      /* isodd(x) */
  M_APM a = Bget(L, 1);
  lua_pushboolean(L, m_apm_is_integer(a) && m_apm_is_odd(a));
  return 1;
}

static int mt_odd (lua_State *L) {      /* odd(x), new 3.5.2 */
  M_APM a;
  a = Bget(L, 1);
  lua_pushboolean(L, m_apm_is_integer(a) && m_apm_is_odd(a));
  return 1;
}


static int Bdigitsin (lua_State *L) {   /* digitsin(x) */
  M_APM a = Bget(L, 1);
  lua_pushinteger(L, m_apm_significant_digits(a));
  return 1;
}


static int mt_gc (lua_State *L) {  /* this is the better-be-sure-than-sorry GC metamethod */
  if (luaL_isudata(L, 1, MYTYPE)) {
    M_APM a = Bget(L, 1);
    if (a) {  /* added 3.5.4 */
      lua_setmetatabletoobject(L, 1, NULL, 1);
      m_apm_free(a);
      m_apm_trim_mem_usage();  /* 2.9.6 */
    }
  }
  return 0;
}


/* Computes the n-th Chebyshev polynomial of the first kind, evaluated at x, with n a non-negative integer and x a number. 2.33.1
   42 percent faster than the Agena implementation which returns cos(n*arccos(x)):
   mapm.xchebyt := proc(n, x) is
     local t1, t4, t6, t9, t15, t17, t20
     if n :: nonnegint then
        n := mapm.xnumber(n)
     fi;
     if x :: number then
        x := mapm.xnumber(x)
     fi
     t1  := x*x;
     t4  := sqrt(t1 + mapm.one + mapm.two*x);
     t6  := sqrt(t1 + mapm.one - mapm.two*x);
     t9  := n*arccos(mapm.half*(t4 - t6));
     t15 := (mapm.half*(t4 + t6))^mapm.two;
     t17 := sqrt(t15 - mapm.one);
     t20 := n*sign(x)*ln(mapm.half*(t4 + t6) + t17);
     return cos(t9)*cosh(t20)
  end; */
static void m_apm_chebyt (M_APM rr, int places, M_APM n, M_APM x) {  /* 2.33.1, result is in n */
  M_APM a, b, c, d, t4, t6, t9, t15, t17;
  a   = M_get_stack_var();
  b   = M_get_stack_var();
  c   = M_get_stack_var();
  d   = M_get_stack_var();
  t4  = M_get_stack_var();
  t6  = M_get_stack_var();
  t9  = M_get_stack_var();
  t15 = M_get_stack_var();
  t17 = M_get_stack_var();
  /* t1(=a) = x*x; */
  m_apm_multiply(a, x, x);
  /* t4 = sqrt(t1 + 1.0 + 2.0*x);
     t6 = sqrt(t1 + 1.0 - 2.0*x); */
  m_apm_add(c, a, MM_One);
  m_apm_multiply(d, x, MM_Two);
  m_apm_add(a, c, d);
  m_apm_subtract(b, c, d);
  m_apm_sqrt(t4, places, a);
  m_apm_sqrt(t6, places, b);
  /* t9 = n*sun_acos(0.5*(t4 - t6)); */
  m_apm_subtract(a, t4, t6);
  m_apm_divide(b, places, a, MM_Two);
  m_apm_arccos(c, places, b);
  m_apm_multiply(t9, n, c);
  /* t15 = sun_pow(0.5*(t4 + t6), 2.0, 1); */
  m_apm_add(a, t4, t6);
  m_apm_divide(b, places, a, MM_Two);
  m_apm_multiply(t15, b, b);
  /* t17 = sqrt(t15 - 1.0); */
  m_apm_subtract(a, t15, MM_One);
  m_apm_sqrt(t17, places, a);
  /* t20 = n*tools_sign(a)*sun_log(0.5*(t4 + t6) + t17); */
  m_apm_set_double(a, (double)m_apm_sign(x));
  m_apm_multiply(b, n, a);
  m_apm_add(c, t4, t6);
  m_apm_divide(d, places, c, MM_Two);
  m_apm_add(d, d, t17);
  m_apm_log(d, places, d);
  m_apm_multiply(rr, b, d);
  /* sun_cos(t9)*sun_cosh(t20=rr) */
  m_apm_cos(a, places, t9);
  m_apm_cosh(b, places, rr);
  m_apm_multiply(rr, a, b);  /* result is in rr */
  M_restore_stack(9);  /* restore the 9 locals we used here */
}

static int Bchebyt (lua_State *L) {
  return Bdo3(L, m_apm_chebyt);
}


static void m_apm_hypot (M_APM rr, int places, M_APM x, M_APM y) {  /* 2.33.1 */
  M_APM a, b, c;
  a = m_apm_init();  /* don't use M_get_stack_var or Agena will crash */
  b = m_apm_init();
  c = m_apm_init();
  m_apm_multiply(a, x, x);
  m_apm_multiply(b, y, y);
  m_apm_add(c, a, b);
  m_apm_sqrt(rr, places + 2, c);
  m_apm_free(a); m_apm_free(b); m_apm_free(c);
}

static int Bhypot (lua_State *L) {
  return Bdo3(L, m_apm_hypot);
}


static void m_apm_hypot4 (M_APM rr, int places, M_APM x, M_APM y) {  /* 3.3.3 */
  M_APM a, b, c;
  a = m_apm_init();  /* don't use M_get_stack_var or Agena will crash */
  b = m_apm_init();
  c = m_apm_init();
  m_apm_multiply(a, x, x);
  m_apm_multiply(b, y, y);
  m_apm_subtract(c, a, b);
  if (m_apm_compare(c, MM_Zero) < 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'m_apm_hypot4\', |difference| < 0");  /* 3.5.2 fix */
    M_set_to_zero(rr); \
  } else { \
    m_apm_sqrt(rr, places + 2, c);
  }
  m_apm_free(a); m_apm_free(b); m_apm_free(c);
}

static int Bhypot4 (lua_State *L) {
  return Bdo3(L, m_apm_hypot4);
}


static void m_apm_square (M_APM rr, int places, M_APM x) {  /* 2.33.1 */
  m_apm_multiply(rr, x, x);
}

static int mt_square (lua_State *L) {  /* square mt, 2.33.1 */
  return Bdo1(L, m_apm_square);
}

static int Bsquare (lua_State *L) {  /* square mt, 3.3.3 */
  return Bdo1(L, m_apm_square);
}


static void m_apm_cube (M_APM rr, int places, M_APM x) {  /* 2.33.1 */
  m_apm_integer_pow(rr, places + 2, x, 3);
}

static int mt_cube (lua_State *L) {  /* cube mt, 2.33.1 */
  return Bdo1(L, m_apm_cube);
}

static int Bcube (lua_State *L) {  /* cube mt, 3.3.3 */
  return Bdo1(L, m_apm_cube);
}


static int mapm_xfma (lua_State *L) {  /* x*y + z, 2.33.1 */
  M_APM a, x, y, z, r;
  a = m_apm_init();  /* don't use M_get_stack_var or Agena will crash */
  x = Bget(L, 1);
  y = Bget(L, 2);
  z = Bget(L, 3);
  r = Bnew(L);
  m_apm_multiply(a, x, y);
  m_apm_add(r, a, z);
  m_apm_free(a);
  return 1;
}


static int mapm_xterm (lua_State *L) {  /* c*x**n, 2.33.1 */
  M_APM r, a, c, x, n;
  a = m_apm_init();  /* don't use M_get_stack_var or Agena will crash */
  c = Bget(L, 1);
  x = Bget(L, 2);
  n = Bget(L, 3);
  r = Bnew(L);
  m_apm_pow(a, MYDIGITS, x, n);
  m_apm_multiply(r, c, a);
  m_apm_free(a);
  return 1;
}


static int mapm_xlog (lua_State *L) {  /* 2.33.2 */
  int places;
  M_APM r, a, b, x, n;
  a = m_apm_init();  /* don't use M_get_stack_var or Agena will crash */
  b = m_apm_init();
  x = Bget(L, 1);
  n = Bget(L, 2);
  places = agnL_optinteger(L, 3, DIGITS);
  r = Bnew(L);
  m_apm_log(a, places + 2, x);
  m_apm_log(b, places + 2, n);
  m_apm_divide(r, places, a, b);
  m_apm_free(a); m_apm_free(b);
  return 1;
}


static int mapm_xlog2 (lua_State *L) {  /* 2.33.2 */
  int places;
  M_APM r, a, x;
  a = m_apm_init();  /* don't use M_get_stack_var or Agena will crash */
  x = Bget(L, 1);
  places = agnL_optinteger(L, 2, DIGITS);
  r = Bnew(L);
  m_apm_log(a, places + 2, x);
  m_apm_divide(r, places, a, MM_lc_log2);  /* 3.5.2 tuning */
  m_apm_free(a);
  return 1;
}


static int mapm_xexp2 (lua_State *L) {  /* 2.33.2 */
  int places;
  M_APM r, x;
  x = Bget(L, 1);
  places = agnL_optinteger(L, 2, DIGITS);
  r = Bnew(L);
  m_apm_pow(r, places, MM_Two, x);
  return 1;
}


static int mapm_xexp10 (lua_State *L) {  /* 2.33.2 */
  int places;
  M_APM r, x;
  x = Bget(L, 1);
  places = agnL_optinteger(L, 2, DIGITS);
  r = Bnew(L);
  m_apm_pow(r, places, MM_Ten, x);
  return 1;
}


static int mapm_xrandom (lua_State *L) {  /* new 3.5.2 */
  M_APM r = Bnew(L);
  m_apm_get_random(r);
  return 1;
}

/* Example: m_apm_set_random_seed("12345678");

  This function will set the random number generator to a known starting value.

  The char string argument should correspond to any *integer* value between 0 and (1.0E+15 - 1).

  This function can be called at any time, either before or anytime after 'm_apm_get_random'. */
static int mapm_xrandomseed (lua_State *L) {  /* 3.5.2 */
  int i;
  const char *str = agn_checkstring(L, 1);
  for (i=0; i < tools_strlen(str); i++) {
    if (!isdigit(str[i])) luaL_error(L, "Error in " LUA_QS ": character is non-numeric.", "mapm.xrandomseed");
  }
  m_apm_set_random_seed((char *)str);
  return 0;
}


/*** COMPLEX compartment *************************************************************************************/

#define MYCTYPE		"cnumber"

typedef struct {
  M_APM real;
  M_APM imag;
} M_CAPM;


/* auxiliary functions **********************************************************************/

static FORCE_INLINE void m_apm_copysign (M_APM rr, M_APM a, M_APM b) {
  int acomp = m_apm_compare(a, MM_Zero);
  int bcomp = m_apm_compare(b, MM_Zero);
  if ((acomp > 0 && bcomp < 0) || (acomp < 0 && bcomp > 0))
    m_apm_negate(rr, a);
  else
    m_apm_copy(rr, a);
}


static void m_apm_csgn (M_APM rr, M_APM a, M_APM b) {
  int acomp = m_apm_compare(a, MM_Zero);
  int bcomp = m_apm_compare(b, MM_Zero);
  if (acomp > 0 || (acomp == 0 && bcomp > 0))
    m_apm_set_double(rr, 1.0);
  else if (acomp < 0 || (acomp == 0 && bcomp < 0))
    m_apm_set_double(rr, -1.0);
  else
    m_apm_set_double(rr, 0.0);
}

/********************************************************************************************/


static M_CAPM *Cnew (lua_State *L) {
  M_CAPM *x = lua_newuserdata(L, sizeof(M_CAPM) + 2*sizeof(M_APM));
  x->real = m_apm_init();
  x->imag = m_apm_init();
  lua_setmetatabletoobject(L, -1, MYCTYPE, 1);  /* 3.5.1 change */
  return x;
}


#define Cgetrealimag(L,idx,re,im) { \
  M_CAPM *_x = (M_CAPM *)luaL_checkudata(L, idx, MYCTYPE); \
  (re) = _x->real; \
  (im) = _x->imag; \
}

#define Cgetreal(L,idx,re) { \
  M_CAPM *_x = (M_CAPM *)luaL_checkudata(L, idx, MYCTYPE); \
  (re) = _x->real; \
}

#define Cgetimag(L,idx,im) { \
  M_CAPM *_x = (M_CAPM *)luaL_checkudata(L, idx, MYCTYPE); \
  (im) = _x->imag; \
}


#define setcomponent(L,x,idx) { \
  switch (lua_type(L, idx)) { \
    case LUA_TNUMBER: { \
      m_apm_set_double((x), lua_tonumber(L, idx)); \
      break; \
    } \
    case LUA_TSTRING: { \
      m_apm_set_string((x), (char*)lua_tostring(L, idx)); \
      break; \
    } \
    default: { \
      if (luaL_isudata(L, idx, MYTYPE)) { \
        m_apm_copy((x), *((void**)lua_touserdata(L, idx))); \
      } else { \
        luaL_argerror(L, idx, "wrong type of argument"); \
      } \
    } \
  } \
}

static int Cnumber (lua_State *L) {  /* mapm.cnumber(x, y) */
  LL = L;
  M_CAPM *x = Cnew(L);  /* pushes new M_CAPM userdata onto the stack top */
  setcomponent(L, x->real, 1);
  setcomponent(L, x->imag, 2);
  return 1;
}


/* mode = 1: one string, mode = 2 two strings */
static int Ctostring (lua_State *L, int idx, int mode) {  /* tostring(x,[n,exp]) */
  char *s;
  int n, bool;
  M_APM re, im;
  Cgetrealimag(L, idx, re, im);  /* 3.5.1 fix */
  if (idx < 0) {
    n = DIGITS; bool = 0;
  } else {
    n = agnL_optinteger(L, idx + 1, DIGITS);
    bool = lua_toboolean(L, idx + 2);
  }
  luaL_checkstack(L, mode == 1 ? 4 : 2, "not enough stack space");  /* 3.5.3 fix */
  /* real part */
  if (bool) {
    int m = (n < 0) ? m_apm_significant_digits(re) : n;
    s = malloc((m + 16)*sizeof(char));
    if (s != NULL) {
      m_apm_to_string(s, n, re);
    } else {  /* 4.11.5 fix */
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "mapm tostring mt");
    }
  } else
      s = m_apm_to_fixpt_stringexp(n, re, '.', 0, 0);
  lua_pushstring(L, s);
  xfree(s);
  /* imaginary part */
  if (mode == 1) lua_pushstring(L, (m_apm_compare(im, MM_Zero) < 0) ? "" : "+");  /* 3.5.2 fix for negative imags */
  if (bool) {
    int m = (n < 0) ? m_apm_significant_digits(im) : n;
    s = malloc((m + 16)*sizeof(char));
    if (s != NULL) {
      m_apm_to_string(s, n, im);
    } else {
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "mapm tostring mt");
    }
  } else
    s = m_apm_to_fixpt_stringexp(n, im, '.', 0, 0);
  if (mode == 1) {
    lua_pushstring(L, s);
    lua_pushstring(L, "*I");
    lua_concat(L, 4);
  } else
    lua_pushstring(L, s);
  xfree(s);
  return mode;
}


static int mapm_ctostring (lua_State *L) {
  int mode = (lua_gettop(L) == 1) ? 1 : 2;
  Ctostring(L, 1, mode);
  return mode;
}


#define IEEEPRECISION  21
static FORCE_INLINE void Ctorealnumbers (lua_State *L, int idx, lua_Number *real, lua_Number *imag) {
  char *s;
  int overflow;
  M_APM re, im;
  Cgetrealimag(L, idx, re, im);  /* 3.5.1 fix */
  /* real part */
  s = m_apm_to_fixpt_stringexp(IEEEPRECISION, re, '.', 0, 0);
  *real = luaL_str2d(L, s, &overflow);
  xfree(s);
  /* imaginary part */
  s = m_apm_to_fixpt_stringexp(IEEEPRECISION, im, '.', 0, 0);
  *imag = luaL_str2d(L, s, &overflow);
  xfree(s);
}


static int Ctonumber (lua_State *L) {  /* tocnumber(x) */
  lua_Number re, im;
  Ctorealnumbers(L, 1, &re, &im);
  lua_pushnumber(L, re);
  lua_pushnumber(L, im);
  return 2;
}


static int Ctocomplex (lua_State *L) {  /* mapm.ctocomplex(x) */
  lua_Number re, im;
  Ctorealnumbers(L, 1, &re, &im);
  agn_pushcomplex(L, re, im);
  return 1;
}


static int mt_ctostring (lua_State *L) {
  Ctostring(L, 1, 1);
  return 1;
}


static int mt_creal (lua_State *L) {  /* real(x) */
  M_APM a, r;
  Cgetreal(L, 1, a);
  r = Bnew(L);
  /* we duplicate explicitly for the return not to be GC'd when the original value is being GC'd */
  m_apm_copy(r, a);
  return 1;
}


static int mt_cimag (lua_State *L) {  /* imag(x) */
  M_APM b, r;
  Cgetimag(L, 1, b);
  r = Bnew(L);
  /* we duplicate explicitly for the return not to be GC'd when the original value is being GC'd */
  m_apm_copy(r, b);
  return 1;
}


static int mt_cunm (lua_State *L) {  /* 3.5.2 fix */
  M_APM a, b;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  m_apm_negate(z->real, a);
  m_apm_negate(z->imag, b);
  return 1;
}


static int mt_cadd (lua_State *L) {  /* 3.5.2 fix */
  M_APM a, b, c, d;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  Cgetrealimag(L, 2, c, d);
  m_apm_add(z->real, a, c);
  m_apm_add(z->imag, b, d);
  return 1;
}


static int mt_csub (lua_State *L) {  /* 3.5.2 fix */
  M_APM a, b, c, d;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  Cgetrealimag(L, 2, c, d);
  m_apm_subtract(z->real, a, c);
  m_apm_subtract(z->imag, b, d);
  return 1;
}


static int Cmul (lua_State *L) {
  M_APM a, b, c, d, t0, t1;
  M_CAPM *z = Cnew(L);
  t0 = M_get_stack_var();
  t1 = M_get_stack_var();
  Cgetrealimag(L, 1, a, b);
  Cgetrealimag(L, 2, c, d);
  m_apm_multiply(t0, a, c);
  m_apm_multiply(t1, b, d);
  m_apm_subtract(z->real, t0, t1);
  m_apm_multiply(t0, a, d);
  m_apm_multiply(t1, b, c);
  m_apm_add(z->imag, t0, t1);
  M_restore_stack(2);  /* restore the locals we used here */
  return 1;
}

static int mt_cmul (lua_State *L) {  /* 3.5.2 fix */
  return Cmul(L);
}


static int Cdiv (lua_State *L) {
  M_APM a, b, c, d;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  Cgetrealimag(L, 2, c, d);
  if (m_apm_compare(c, MM_Zero) == 0 && m_apm_compare(d, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'/\', division by zero");  /* 3.5.2 fix */
    M_set_to_zero(z->real);
    M_set_to_zero(z->imag);
  } else {
    M_APM t2, t3, t5, t6, t7, t8;
    t2 = M_get_stack_var();
    t3 = M_get_stack_var();
    t5 = M_get_stack_var();
    t6 = M_get_stack_var();
    t7 = M_get_stack_var();
    t8 = M_get_stack_var();
    m_apm_multiply(t2, c, c);
    m_apm_multiply(t3, d, d);
    m_apm_add(t6, t2, t3);
    m_apm_divide(t5, MYDIGITS, MM_One, t6);
    m_apm_multiply(t2, a, c);
    m_apm_multiply(t3, b, d);
    m_apm_multiply(t7, t2, t5);
    m_apm_multiply(t8, t3, t5);
    m_apm_add(z->real, t7, t8);
    m_apm_multiply(t2, b, c);
    m_apm_multiply(t3, a, d);
    m_apm_multiply(t7, t2, t5);
    m_apm_multiply(t8, t3, t5);
    m_apm_subtract(z->imag, t7, t8);
    M_restore_stack(6);
  }
  return 1;
}

static int mt_cdiv (lua_State *L) {  /* 3.5.2 fix */
  return Cdiv(L);
}


static int Cpow (lua_State *L) {
  M_APM x0, y0, x, y;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, x0, y0);
  Cgetrealimag(L, 2, x, y);
  if (m_apm_compare(x0, MM_Zero) == 0 && m_apm_compare(y0, MM_Zero) == 0 && m_apm_compare(x, MM_Zero) <= 0) {
    if (m_apm_compare(y, MM_Zero) == 0) {
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'^\', argument is out-of-range");  /* 3.5.2 fix */
    }
    M_set_to_zero(z->real);
    M_set_to_zero(z->imag);
  } else {
    M_APM absa = M_get_stack_var();
    /* absa = sun_hypot(x0, y0); */
    m_apm_hypot(absa, MYDIGITS, x0, y0);
    /* if (absa == 0.0) { */
    if (m_apm_compare(absa, MM_Zero) == 0) {
      M_set_to_zero(z->real);
      M_set_to_zero(z->imag);
      M_restore_stack(1);
    } else {
      M_APM arga, r, r0, theta, t0, si, co;
      arga  = M_get_stack_var();
      r     = M_get_stack_var();
      r0    = M_get_stack_var();
      theta = M_get_stack_var();
      t0    = M_get_stack_var();
      si    = M_get_stack_var();
      co    = M_get_stack_var();
      /* arga = sun_atan2(y0, x0); */
      m_apm_arctan2(arga, MYDIGITS, y0, x0);
      /* r = sun_pow(absa, x, 0); */
      m_apm_pow(r, MYDIGITS, absa, x);
      /* theta = x*arga; */
      m_apm_multiply(theta, x, arga);
      /* if (y != 0.0) { */
      if (m_apm_compare(y, MM_Zero) != 0) {
        /* r = r*sun_exp(-y*arga); */
        m_apm_multiply(t0, y, arga);
        m_apm_negate(r0, t0);
        m_apm_exp(t0, MYDIGITS, r0);
        /* r = r*t0; */
        m_apm_multiply(r0, r, t0);
        m_apm_copy(r, r0);
        /* theta = theta + y*sun_log(absa); */
        m_apm_log(t0, MYDIGITS, absa);
        m_apm_multiply(r0, y, t0);
        m_apm_add(t0, theta, r0);
        m_apm_copy(theta, t0);
      }
      /* sun_sincos(theta, &si, &co);  r*co, r*si; */
      m_apm_sin_cos(si, co, MYDIGITS, theta);
      m_apm_multiply(z->real, r, co);
      m_apm_multiply(z->imag, r, si);
      M_restore_stack(8);
    }
  }
  return 1;
}

static int mt_cpow (lua_State *L) {  /* 3.5.2 fix */
  return Cpow(L);
}


static int Cipow (lua_State *L) {
  M_APM a, b, x, y, n, t1, t2, t5, t6, t7, t8, t9;
  M_CAPM *z = Cnew(L);
  t1 = M_get_stack_var();
  t2 = M_get_stack_var();
  t5 = M_get_stack_var();
  t6 = M_get_stack_var();
  t7 = M_get_stack_var();
  t8 = M_get_stack_var();
  t9 = M_get_stack_var();
  x  = M_get_stack_var();
  y  = M_get_stack_var();
  n  = M_get_stack_var();
  Cgetrealimag(L, 1, a, b);
  m_apm_multiply(t1, a, a);
  m_apm_multiply(t2, b, b);
  m_apm_add(x, t1, t2);
  m_apm_set_double(n, agn_checknumber(L, 2));
  m_apm_divide(y, MYDIGITS, n, MM_Two);
  m_apm_pow(t5, MYDIGITS, x, y);
  m_apm_arctan2(t6, MYDIGITS, b, a);
  m_apm_multiply(t7, t6, n);
  m_apm_sin_cos(t9, t8, MYDIGITS, t7);
  m_apm_multiply(z->real, t5, t8);
  m_apm_multiply(z->imag, t5, t9);
  M_restore_stack(10);
  return 1;
}

static int mt_cipow (lua_State *L) {  /* 3.5.2 fix */
  return Cipow(L);
}


static int mt_crecip (lua_State *L) {
  M_APM c, d;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, c, d);
  if (m_apm_compare(c, MM_Zero) == 0 && m_apm_compare(d, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'recip\', division by zero");  /* 3.5.2 fix */
    M_set_to_zero(z->real);
    M_set_to_zero(z->imag);
  } else {
    M_APM t2, t3, t5, t6;
    t2 = M_get_stack_var();
    t3 = M_get_stack_var();
    t5 = M_get_stack_var();
    t6 = M_get_stack_var();
    m_apm_multiply(t2, c, c);
    m_apm_multiply(t3, d, d);
    m_apm_add(t6, t2, t3);
    m_apm_divide(t5, MYDIGITS, MM_One, t6);
    m_apm_multiply(z->real, c, t5);
    m_apm_multiply(t6, d, t5);
    m_apm_negate(z->imag, t6);
    M_restore_stack(4);
  }
  return 1;
}


static int Cfma (lua_State *L) {
  M_APM a, b, c, d, e, f, t0, t1, t2;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  Cgetrealimag(L, 2, c, d);
  Cgetrealimag(L, 3, e, f);
  t0 = M_get_stack_var();
  t1 = M_get_stack_var();
  t2 = M_get_stack_var();
  /* real: a*c - b*d + e */
  m_apm_multiply(t0, a, c);
  m_apm_multiply(t1, b, d);
  m_apm_subtract(t2, t0, t1);
  m_apm_add(z->real, t2, e);
  /* imag: a*d + b*c + f */
  m_apm_multiply(t0, a, d);
  m_apm_multiply(t1, b, c);
  m_apm_add(t2, t0, t1);
  m_apm_add(z->imag, t2, f);
  M_restore_stack(3);
  return 1;
}


static int mt_cabs (lua_State *L) {
  M_APM a, b, r;
  r = Bnew(L);
  Cgetrealimag(L, 1, a, b);
  m_apm_hypot(r, DIGITS, a, b);
  return 1;
}


static int Cargument (lua_State *L) {
  M_APM a, b, r;
  r = Bnew(L);
  Cgetrealimag(L, 1, a, b);
  m_apm_atan2(r, DIGITS, b, a);
  return 1;
}


static int mt_csgn (lua_State *L) {
  M_APM a, b, r;
  r = Bnew(L);
  Cgetrealimag(L, 1, a, b);
  m_apm_csgn(r, a, b);
  return 1;
}


static int mt_csquare (lua_State *L) {
  M_APM a, b, t0, t1;
  M_CAPM *z;
  z = Cnew(L);
  t0 = M_get_stack_var();
  t1 = M_get_stack_var();
  Cgetrealimag(L, 1, a, b);
  m_apm_multiply(t0, a, a);
  m_apm_multiply(t1, b, b);
  m_apm_subtract(z->real, t0, t1);
  m_apm_multiply(t0, a, b);
  m_apm_multiply(t1, b, a);
  m_apm_add(z->imag, t0, t1);
  M_restore_stack(2);  /* restore the locals we used here */
  return 1;
}


static int mt_ccube (lua_State *L) {
  M_APM a, b, t0, t1, x, y;
  M_CAPM *z;
  z = Cnew(L);
  t0 = M_get_stack_var();
  t1 = M_get_stack_var();
  x = M_get_stack_var();
  y = M_get_stack_var();
  Cgetrealimag(L, 1, a, b);
  m_apm_multiply(t0, a, a);
  m_apm_multiply(t1, b, b);
  m_apm_subtract(x, t0, t1);
  m_apm_multiply(t0, a, b);
  m_apm_multiply(t1, b, a);
  m_apm_add(y, t0, t1);
  m_apm_multiply(t0, x, a);
  m_apm_multiply(t1, y, b);
  m_apm_subtract(z->real, t0, t1);
  m_apm_multiply(t0, x, b);
  m_apm_multiply(t1, y, a);
  m_apm_add(z->imag, t0, t1);
  M_restore_stack(4);  /* restore the locals we used here */
  return 1;
}


static int mt_cln (lua_State *L) {
  M_APM a, b;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  if (m_apm_compare(a, MM_Zero) == 0 && m_apm_compare(b, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'ln\', argument is zero");  /* 3.5.2 fix */
    M_set_to_zero(z->real);
    M_set_to_zero(z->imag);
  } else {
    M_APM t, t0, t1;
    t0 = M_get_stack_var();
    t1 = M_get_stack_var();
    t  = M_get_stack_var();
    m_apm_multiply(t0, a, a);
    m_apm_multiply(t1, b, b);
    m_apm_add(t, t0, t1);
    m_apm_log(t, MYDIGITS, t);
    m_apm_divide(z->real, MYDIGITS, t, MM_Two);
    m_apm_arctan2(z->imag, MYDIGITS, b, a);
    M_restore_stack(3);
  }
  return 1;
}


static int mt_cexp (lua_State *L) {
  M_APM a, b, t, si, co;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  si = M_get_stack_var();
  co = M_get_stack_var();
  t  = M_get_stack_var();
  m_apm_exp(t, MYDIGITS, a);
  m_apm_sin_cos(si, co, MYDIGITS, b);
  m_apm_multiply(z->real, t, co);
  m_apm_multiply(z->imag, t, si);
  M_restore_stack(3);
  return 1;
}


static int mt_csqrt (lua_State *L) {
  M_APM a, b;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  if (m_apm_compare(b, MM_Zero) == 0) {
    if (m_apm_compare(a, MM_Zero) < 0) {
      M_APM ta;
      M_set_to_zero(z->real);
      ta = M_get_stack_var();
      m_apm_negate(ta, a);
      m_apm_sqrt(z->imag, MYDIGITS, ta);
    } else {
      m_apm_sqrt(z->real, MYDIGITS, a);
      M_set_to_zero(z->imag);
    }
  } else {
    M_APM ta, t1, t2, t4, t6, t10, t, x, y;
    ta  = M_get_stack_var();
    t1  = M_get_stack_var();
    t2  = M_get_stack_var();
    t4  = M_get_stack_var();
    t6  = M_get_stack_var();
    t10 = M_get_stack_var();
    t   = M_get_stack_var();
    x   = M_get_stack_var();
    y   = M_get_stack_var();
    m_apm_multiply(t1, a, a);
    m_apm_multiply(t2, b, b);
    m_apm_add(t, t1, t2);
    m_apm_sqrt(t4, MYDIGITS, t);
    m_apm_multiply(x, t4, MM_Two);
    m_apm_multiply(y, a, MM_Two);
    m_apm_add(t, x, y);
    m_apm_sqrt(t6, MYDIGITS, t);
    m_apm_subtract(t, x, y);
    m_apm_sqrt(t10, MYDIGITS, t);
    /* tools_csgn(b, -a)*t10/2; */
    m_apm_negate(ta, a);
    m_apm_csgn(t, b, ta);
    m_apm_multiply(t1, t, t10);
    m_apm_divide(z->real, MYDIGITS, t6, MM_Two);
    m_apm_divide(z->imag, MYDIGITS, t1, MM_Two);
    M_restore_stack(9);  /* restore the locals we used here */
  }
  return 1;
}


static int mt_csin (lua_State *L) {
  M_APM a, b, si, co, sih, coh;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  si  = M_get_stack_var();
  co  = M_get_stack_var();
  sih = M_get_stack_var();
  coh = M_get_stack_var();
  m_apm_sin_cos(si, co, MYDIGITS, a);
  m_apm_sinh(sih, MYDIGITS, b);
  m_apm_cosh(coh, MYDIGITS, b);
  m_apm_multiply(z->real, si, coh);
  m_apm_multiply(z->imag, co, sih);
  M_restore_stack(4);
  return 1;
}


static int mt_ccos (lua_State *L) {
  M_APM a, b, sir, si, co, sih, coh;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  si  = M_get_stack_var();
  co  = M_get_stack_var();
  sih = M_get_stack_var();
  coh = M_get_stack_var();
  sir = M_get_stack_var();
  m_apm_sin_cos(si, co, MYDIGITS, a);
  m_apm_sinh(sih, MYDIGITS, b);
  m_apm_cosh(coh, MYDIGITS, b);
  m_apm_multiply(z->real, co, coh);
  m_apm_negate(sir, si);
  m_apm_multiply(z->imag, sir, sih);
  M_restore_stack(5);
  return 1;
}


static int mt_ctan (lua_State *L) {
  M_APM a, b, re, im, si, co, sih, coh, den;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  re  = M_get_stack_var();
  im  = M_get_stack_var();
  si  = M_get_stack_var();
  co  = M_get_stack_var();
  sih = M_get_stack_var();
  coh = M_get_stack_var();
  den = M_get_stack_var();
  m_apm_multiply(re, a, MM_Two);
  m_apm_multiply(im, b, MM_Two);
  m_apm_sin_cos(si, co, MYDIGITS, re);
  m_apm_sinh(sih, MYDIGITS, im);
  m_apm_cosh(coh, MYDIGITS, im);
  m_apm_add(den, co, coh);
  if (m_apm_compare(den, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'tan\', argument is out-of-range");  /* 3.5.2 fix */
    M_set_to_zero(z->real);
    M_set_to_zero(z->imag);
  } else {
    m_apm_divide(z->real, MYDIGITS, si, den);
    m_apm_divide(z->imag, MYDIGITS, sih, den);
  }
  M_restore_stack(7);
  return 1;
}


static int mt_csinh (lua_State *L) {
  M_APM a, b, si, co, sih, coh;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  si  = M_get_stack_var();
  co  = M_get_stack_var();
  sih = M_get_stack_var();
  coh = M_get_stack_var();
  m_apm_sin_cos(si, co, MYDIGITS, b);
  m_apm_sinh(sih, MYDIGITS, a);
  m_apm_cosh(coh, MYDIGITS, a);
  m_apm_multiply(z->real, sih, co);
  m_apm_multiply(z->imag, coh, si);
  M_restore_stack(4);
  return 1;
}


static int mt_ccosh (lua_State *L) {
  M_APM a, b, si, co, sih, coh;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  si  = M_get_stack_var();
  co  = M_get_stack_var();
  sih = M_get_stack_var();
  coh = M_get_stack_var();
  m_apm_sin_cos(si, co, MYDIGITS, b);
  m_apm_sinh(sih, MYDIGITS, a);
  m_apm_cosh(coh, MYDIGITS, a);
  m_apm_multiply(z->real, coh, co);
  m_apm_multiply(z->imag, sih, si);
  M_restore_stack(4);
  return 1;
}


static int mt_ctanh (lua_State *L) {
  M_APM a, b, si, co, sih, coh, t4, t6, t8, t9;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  si  = M_get_stack_var();
  co  = M_get_stack_var();
  sih = M_get_stack_var();
  coh = M_get_stack_var();
  t4  = M_get_stack_var();
  t6  = M_get_stack_var();
  t8  = M_get_stack_var();
  t9  = M_get_stack_var();
  m_apm_sin_cos(si, co, MYDIGITS, b);
  m_apm_sinh(sih, MYDIGITS, a);
  m_apm_cosh(coh, MYDIGITS, a);
  m_apm_multiply(t4, sih, sih);
  m_apm_multiply(t6, co, co);
  m_apm_add(t8, t4, t6);
  m_apm_divide(t9, MYDIGITS, MM_One, t8);
  m_apm_multiply(t4, sih, coh);
  m_apm_multiply(t6, si, co);
  m_apm_multiply(z->real, t9, t4);
  m_apm_multiply(z->imag, t9, t6);
  M_restore_stack(8);
  return 1;
}


/* -I*arcsin(I*z) */
static void m_apm_carcsinh (M_APM rx, M_APM ry, M_APM a, M_APM b) {
  if (m_apm_compare(b, MM_Zero) == 0) {
    m_apm_asinh(rx, MYDIGITS, a);
    M_set_to_zero(ry);
  } else {
    M_APM b2, t3, t4, t5, t6, t8, t10, t12, z0, z1, z2;
    t3  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t6  = M_get_stack_var();
    t8  = M_get_stack_var();
    t10 = M_get_stack_var();
    t12 = M_get_stack_var();
    z0  = M_get_stack_var();
    z1  = M_get_stack_var();
    z2  = M_get_stack_var();
    b2  = M_get_stack_var();
    m_apm_multiply(t3, a, a);
    m_apm_multiply(t4, b, b);
    /* z1 = t3 + t4 + 1.0; */
    m_apm_add(z0, t3, t4);
    m_apm_add(z1, MM_One, z0);
    /* t6 = sqrt(z1 + 2.0*b); */
    m_apm_multiply(b2, b, MM_Two);
    m_apm_add(t3, z1, b2);
    m_apm_sqrt(t6, MYDIGITS, t3);
    /* t8 = sqrt(z1 - 2.0*b); */
    m_apm_subtract(z0, z1, b2);
    m_apm_sqrt(t8, MYDIGITS, z0);
    /* z2 = 0.5*t6 + 0.5*t8; */
    m_apm_divide(z0, MYDIGITS, t6, MM_Two);
    m_apm_divide(z1, MYDIGITS, t8, MM_Two);
    m_apm_add(z2, z0, z1);
    /* t10 = tools_square(z2); */
    m_apm_multiply(t10, z2, z2);
    /* t12 = sqrt(t10 - 1.0); */
    m_apm_subtract(t3, t10, MM_One);
    m_apm_sqrt(t12, MYDIGITS, t3);
    /* rx = tools_csgn(a, b)*sun_log(z2 + t12); */
    m_apm_csgn(t3, a, b);
    m_apm_add(t4, z2, t12);
    m_apm_log(t5, MYDIGITS, t4);
    m_apm_multiply(t10, t3, t5);  /* rx */
    /* z2 = 0.5*t6 - 0.5*t8; */
    m_apm_subtract(z2, z0, z1);
    /* tools_adjust(t18, 1, AGN_EPSILON, -1); */
    /* ry = sun_asin(z2); */
    m_apm_asin(t12, MYDIGITS, z2);  /* ry */
    /* z->real = copysign(rx, a); */
    m_apm_copysign(rx, t10, a);
    /* z->imag = copysign(ry, b); */
    m_apm_copysign(ry, t12, b);
    M_restore_stack(11);
  }
}

static int Carcsinh (lua_State *L) {
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  m_apm_carcsinh(z->real, z->imag, a, b);
  return 1;
}


static int Carccosh (lua_State *L) {
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  if (m_apm_compare(b, MM_Zero) == 0 && m_apm_compare(a, MM_One) >= 0) {
    m_apm_acosh(z->real, MYDIGITS, a);
    M_set_to_zero(z->imag);
  } else {
    M_APM u, v, t2, t6, t7, t8, t9, t11, t13, t15, t9h, t11h, t16;
    t2  = M_get_stack_var();
    t6  = M_get_stack_var();
    t7  = M_get_stack_var();
    t8  = M_get_stack_var();
    t9  = M_get_stack_var();
    t11 = M_get_stack_var();
    t13 = M_get_stack_var();
    t15 = M_get_stack_var();
    t9h = M_get_stack_var();
    t11h = M_get_stack_var();
    t16 = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    /* t2 = tools_csgn(b, 1 - a); */
    m_apm_subtract(u, MM_One, a);
    m_apm_csgn(t2, b, u);
    /* t6 = a*a; */
    m_apm_multiply(t6, a, a);
    /* t7 = b*b; */
    m_apm_multiply(t7, b, b);
    /* t8 = t6 + t7 + 1.0; */
    m_apm_add(u, t6, t7);
    m_apm_add(t8, u, MM_One);
    /* t9 = sqrt(2.0*a + t8); */
    m_apm_multiply(u, MM_Two, a);
    m_apm_add(v, u, t8);
    m_apm_sqrt(t9, MYDIGITS, v);
    /* t11 = sqrt(-2.0*a + t8); */
    m_apm_negate(v, u);
    m_apm_add(u, v, t8);
    m_apm_sqrt(t11, MYDIGITS, u);
    /* t13 = sun_pow(0.5*(t9 + t11), 2.0, 1); */
    m_apm_add(u, t9, t11);
    m_apm_divide(v, MYDIGITS, u, MM_Two);
    m_apm_multiply(t13, v, v);
    /* t15 = sqrt(t13 - 1.0); */
    m_apm_subtract(u, t13, MM_One);
    m_apm_sqrt(t15, MYDIGITS, u);
    /* t9h = 0.5*t9; */
    m_apm_divide(t9h, MYDIGITS, t9, MM_Two);
    /* t11h = 0.5*t11; */
    m_apm_divide(t11h, MYDIGITS, t11, MM_Two);
    /* t16 = t9h - t11h; */
    m_apm_subtract(t16, t9h, t11h);
    /* real -t2*tools_csgn(-b, a)*sun_log(t9h + t11h + t15) */
    m_apm_negate(u, b);
    m_apm_csgn(v, u, a);
    m_apm_multiply(t7, t2, v);
    m_apm_negate(t6, t7);  /* t6 = -t2*tools_csgn(-b, a) */
    /* sun_log(t9h + t11h + t15) */
    m_apm_add(u, t9h, t11h);
    m_apm_add(v, u, t15);
    m_apm_log(t7, MYDIGITS, v);
    m_apm_multiply(z->real, t6, t7);
    /* imag: t2*sun_acos(t16)); */
    m_apm_acos(u, MYDIGITS, t16);
    m_apm_multiply(z->imag, t2, u);
    M_restore_stack(13);
  }
  return 1;
}


static int Carctanh (lua_State *L) {
  M_APM a, b, absa;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  absa = m_apm_init();
  m_apm_absolute_value(absa, a);
  if (m_apm_compare(b, MM_Zero) == 0 && m_apm_compare(absa, MM_One) <= 0) {
    m_apm_atanh(z->real, MYDIGITS, a);
    M_set_to_zero(z->imag);
    m_apm_free(absa);
  } else {
    M_APM t1, t2, t3, t4, t6, u, v, w;
    m_apm_free(absa);
    t1 = M_get_stack_var();
    t2 = M_get_stack_var();
    t3 = M_get_stack_var();
    t4 = M_get_stack_var();
    t6 = M_get_stack_var();
    u  = M_get_stack_var();
    v  = M_get_stack_var();
    w  = M_get_stack_var();
    /* t1 = a + 1.0; */
    m_apm_add(t1, a, MM_One);
    /* t2 = t1*t1; */
    m_apm_multiply(t2, t1, t1);
    /* t3 = b*b; */
    m_apm_multiply(t3, b, b);
    /* t4 = a - 1.0; */
    m_apm_subtract(t4, a, MM_One);
    /* t6 = t4*t4; */
    m_apm_multiply(t6, t4, t4);
    /* im = 0.5*(sun_atan2(b, t1) - sun_atan2(-b, 1.0 - a)); */
    /* u = sun_atan2(b, t1) */
    m_apm_atan2(u, MYDIGITS, b, t1);
    /* v = sun_atan2(-b, 1.0 - a) */
    m_apm_negate(t1, b);
    m_apm_subtract(w, MM_One, a);
    m_apm_atan2(v, MYDIGITS, t1, w);
    m_apm_subtract(w, u, v);
    m_apm_divide(u, MYDIGITS, w, MM_Two);
    if (m_apm_compare(a, MM_Zero) > 0 && m_apm_compare(b, MM_Zero) == 0) {
      m_apm_negate(v, u);
      m_apm_copy(z->imag, v);
    } else
      m_apm_copy(z->imag, u);
    /* 0.25*sun_log((t2 + t3)/(t6 + t3)) */
    m_apm_add(u, t2, t3);
    m_apm_add(v, t6, t3);
    m_apm_divide(w, MYDIGITS, u, v);
    m_apm_log(u, MYDIGITS, w);
    m_apm_divide(z->real, MYDIGITS, u, MM_Four);
    M_restore_stack(8);
  }
  return 1;
}


static int mt_carcsin (lua_State *L) {
  M_APM a, b, x, y;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  x = m_apm_init();
  y = m_apm_init();
  m_apm_carcsinh(x, y, b, a);
  if (m_apm_compare(b, MM_Zero) == 0 && m_apm_compare(a, MM_One) >= 0) {
    M_APM t = m_apm_init();
    m_apm_negate(t, x);
    m_apm_copy(x, t);
    m_apm_free(t);
  }
  m_apm_copy(z->real, y);
  m_apm_copy(z->imag, x);
  m_apm_free(x); m_apm_free(y);
  return 1;
}


static int mt_carccos (lua_State *L) {
  M_APM a, b, t, x, y;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  x = m_apm_init();
  y = m_apm_init();
  t = m_apm_init();
  m_apm_carcsinh(x, y, b, a);
  if (m_apm_compare(b, MM_Zero) != 0 || m_apm_compare(a, MM_One) <= 0) {
    m_apm_negate(t, x);
    m_apm_copy(x, t);
  }
  /* y = PIO2 - y; */
  m_apm_subtract(z->real, MM_HALF_PI, y);
  m_apm_copy(z->imag, x);
  m_apm_free(x); m_apm_free(y); m_apm_free(t);
  return 1;
}


static int mt_carctan (lua_State *L) {
  M_APM a, b, t3, t5, t6, t9, u, v, w, m1;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  t3 = M_get_stack_var();
  t5 = M_get_stack_var();
  t6 = M_get_stack_var();
  t9 = M_get_stack_var();
  u  = M_get_stack_var();
  v  = M_get_stack_var();
  w  = M_get_stack_var();
  m1 = M_get_stack_var();
  m_apm_negate(m1, MM_One);
  /* t3 = b + 1.0; */
  m_apm_add(t3, b, MM_One);
  /* t5 = a*a; */
  m_apm_multiply(t5, a, a);
  /* t6 = t3*t3; */
  m_apm_multiply(t6, t3, t3);
  /* t9 = tools_square(b - 1.0); */
  m_apm_subtract(u, b, MM_One);
  m_apm_multiply(t9, u, u);
  /* if (a == 0 || a == -0) { */
  if (m_apm_compare(a, MM_Zero) == 0) {
    /* if ((b == 1 || b == -1)) { */
    m_apm_absolute_value(v, b);
    if (m_apm_compare(v, MM_One) == 0) {  /* undefined */
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'arctan\', |imag| is one");  /* 3.5.2 fix */
      M_set_to_zero(z->real);
      M_set_to_zero(z->imag);
      return 1;
    /* else if (b < -1) */
    } else if (m_apm_compare(b, m1) < 0) {
      /* r = -PI*0.5; */
      m_apm_negate(z->real, MM_HALF_PI);
      goto lcatan1;
    }
  }
  /* r = sun_atan2(a, 1.0 - b)*0.5 - sun_atan2(-a, t3)*0.5; */
  m_apm_subtract(u, MM_One, b);
  m_apm_atan2(v, MYDIGITS, a, u);
  m_apm_divide(u, MYDIGITS, v, MM_Two);
  m_apm_negate(v, a);
  m_apm_atan2(w, MYDIGITS, v, t3);
  m_apm_divide(v, MYDIGITS, w, MM_Two);
  m_apm_subtract(z->real, u, v);
lcatan1:
  /* i = sun_log((t5 + t6)/(t5 + t9))*0.25; */
  m_apm_add(u, t5, t6);
  m_apm_add(v, t5, t9);
  m_apm_divide(w, MYDIGITS, u, v);
  m_apm_log(u, MYDIGITS, w);
  m_apm_divide(z->imag, MYDIGITS, u, MM_Four);
  M_restore_stack(8);
  return 1;
}


static int mt_carcsec (lua_State *L) {
  M_APM a, b, e, t, u, v, x, y;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  x = M_get_stack_var();
  y = M_get_stack_var();
  e = M_get_stack_var();
  t = M_get_stack_var();
  u = M_get_stack_var();
  v = M_get_stack_var();
  /* e = x*x + y*y; */
  m_apm_multiply(t, a, a);
  m_apm_multiply(u, b, b);
  m_apm_add(e, t, u);
  /* t = a/e; u = (-b)/e; */
  m_apm_divide(t, MYDIGITS, a, e);
  m_apm_divide(u, MYDIGITS, b, e);
  m_apm_negate(v, u);
  m_apm_copy(u, v);
  /* tools_casinh(u, t, &x, &y); */
  m_apm_carcsinh(x, y, u, t);
  if (m_apm_compare(u, MM_Zero) != 0 || m_apm_compare(t, MM_One) > 0) {
    m_apm_negate(t, x);
    m_apm_copy(x, t);
  }
  /* y = PIO2 - y; */
  m_apm_subtract(z->real, MM_HALF_PI, y);
  m_apm_copy(z->imag, x);
  M_restore_stack(6);
  return 1;
}


static int mt_ceq (lua_State *L) {  /* 3.5.2 fix */
  M_APM a, b, c, d;
  Cgetrealimag(L, 1, a, b);
  Cgetrealimag(L, 2, c, d);
  lua_pushboolean(L, m_apm_compare(a, c) == 0 && m_apm_compare(b, d) == 0);
  return 1;
}


static int mt_cgc (lua_State *L) {  /* this is the better-be-sure-than-sorry GC metamethod */
  if (luaL_isudata(L, 1, MYCTYPE)) {
    M_CAPM *a = (M_CAPM *)lua_touserdata(L, 1);
    if (a) {
      lua_setmetatabletoobject(L, 1, NULL, 1);  /* 3.5.1 change */
      m_apm_free(a->real);
      m_apm_free(a->imag);
      m_apm_trim_mem_usage();
    }
  }
  return 0;
}


static int Csinc (lua_State *L) {  /* 3.5.2 */
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if ((a) == 0 && (b) == 0) */
  if (m_apm_compare(a, MM_Zero) == 0 && m_apm_compare(b, MM_Zero) == 0) {
    m_apm_copy(z->real, MM_One);
    M_set_to_zero(z->imag);
  } else {
    M_APM t3, t4, t5, t7, t8, t12, t13, si, co, sih, coh;
    t3  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t7  = M_get_stack_var();
    t8  = M_get_stack_var();
    t12 = M_get_stack_var();
    t13 = M_get_stack_var();
    si  = M_get_stack_var();
    co  = M_get_stack_var();
    sih = M_get_stack_var();
    coh = M_get_stack_var();
    /* sun_sincos(a, &si, &co); */
    m_apm_sin_cos(si, co, MYDIGITS, a);
    /* luai_numsinhcosh(b, &sih, &coh); */
    m_apm_sinh(sih, MYDIGITS, b);
    m_apm_cosh(coh, MYDIGITS, b);
    /* t3 = si*coh; */
    m_apm_multiply(t3, si, coh);
    /* t12 = co*sih; */
    m_apm_multiply(t12, co, sih);
    /* t4 = a*a; */
    m_apm_multiply(t4, a, a);
    /* t5 = b*b; */
    m_apm_multiply(t5, b, b);
    /* t7 = 1/(t4 + t5); */
    m_apm_add(si, t4, t5);
    m_apm_reciprocal(t7, MYDIGITS, si);
    /* t8 = a*t7; */
    m_apm_multiply(t8, a, t7);
    /* t13 = b*t7; */
    m_apm_multiply(t13, b, t7);
    /* im = t12*t8 - t3*t13; */
    m_apm_multiply(si, t12, t8);
    m_apm_multiply(co, t3, t13);
    m_apm_subtract(z->imag, si, co);
    /* re = t3*t8 + t12*t13 */
    m_apm_multiply(si, t3, t8);
    m_apm_multiply(co, t12, t13);
    m_apm_add(z->real, si, co);
    M_restore_stack(11);
  }
  return 1;
}


static int mt_csinc (lua_State *L) {  /* 3.5.2 */
  Csinc(L);
  return 1;
}


static int Ccosc (lua_State *L) {  /* 3.5.2 */
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if ((a) == 0 && (b) == 0) */
  if (m_apm_compare(a, MM_Zero) == 0 && m_apm_compare(b, MM_Zero) == 0) {
    M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xcosc\', argument is zero");
    M_set_to_zero(z->real);
    M_set_to_zero(z->imag);
  } else {
    M_APM t3, t4, t5, t7, t8, t12, t13, si, co, sih, coh;
    t3  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t7  = M_get_stack_var();
    t8  = M_get_stack_var();
    t12 = M_get_stack_var();
    t13 = M_get_stack_var();
    si  = M_get_stack_var();
    co  = M_get_stack_var();
    sih = M_get_stack_var();
    coh = M_get_stack_var();
    /* sun_sincos(a, &si, &co); */
    m_apm_sin_cos(si, co, MYDIGITS, a);
    /* luai_numsinhcosh(b, &sih, &coh); */
    m_apm_sinh(sih, MYDIGITS, b);
    m_apm_cosh(coh, MYDIGITS, b);
    /* t3 = co*coh; */
    m_apm_multiply(t3, co, coh);
    /* t12 = si*sih; */
    m_apm_multiply(t12, si, sih);
    /* t4 = a*a; */
    m_apm_multiply(t4, a, a);
    /* t5 = b*b; */
    m_apm_multiply(t5, b, b);
    /* t7 = 1/(t4 + t5); */
    m_apm_add(si, t4, t5);
    m_apm_reciprocal(t7, MYDIGITS, si);
    /* t8 = a*t7; */
    m_apm_multiply(t8, a, t7);
    /* t13 = b*t7; */
    m_apm_multiply(t13, b, t7);
    /* im = -t12*t8 - t3*t13; */
    m_apm_multiply(si, t12, t8);
    m_apm_negate(t4, si);
    m_apm_multiply(co, t3, t13);
    m_apm_subtract(z->imag, t4, co);
    /* re = t3*t8 + t12*t13 */
    m_apm_multiply(si, t3, t8);
    m_apm_multiply(co, t12, t13);
    m_apm_subtract(z->real, si, co);
    M_restore_stack(11);
  }
  return 1;
}


static int Ctanc (lua_State *L) {  /* 3.5.2 */
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if ((a) == 0 && (b) == 0) */
  if (m_apm_compare(b, MM_Zero) == 0) {
    if (m_apm_compare(a, MM_Zero) == 0) {
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.xtanc\', argument is zero");
      M_set_to_zero(z->real);
    } else {
      M_APM u = M_get_stack_var();
      m_apm_tan(u, MYDIGITS, a);
      m_apm_divide(z->real, MYDIGITS, u, a);
      M_restore_stack(1);
    }
    M_set_to_zero(z->imag);
  } else {
    M_APM t2, t3, t4, t5, t6, t8, t10, t11, t13, t14, t17, t19, u, v;
    t2  = M_get_stack_var();
    t3  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t6  = M_get_stack_var();
    t8  = M_get_stack_var();
    t10 = M_get_stack_var();
    t11 = M_get_stack_var();
    t13 = M_get_stack_var();
    t14 = M_get_stack_var();
    t17 = M_get_stack_var();
    t19 = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    /* t2 = cos(a); */
    m_apm_sin_cos(u, t2, MYDIGITS, a);
    /* t3 = sin(a)*t2; */
    m_apm_multiply(t3, u, t2);
    /* t4 = t2*t2; */
    m_apm_multiply(t4, t2, t2);
    /* t5 = sinh(b); */
    m_apm_sinh(t5, MYDIGITS, b);
    /* t6 = t5*t5; */
    m_apm_multiply(t6, t5, t5);
    /* t8 = 1/(t4+t6); */
    m_apm_add(u, t4, t6);
    m_apm_reciprocal(t8, MYDIGITS, u);
    /* t10 = a*a; */
    m_apm_multiply(t10, a, a);
    /* t11 = b*b; */
    m_apm_multiply(t11, b, b);
    /* t13 = 1/(t10+t11); */
    m_apm_add(u, t10, t11);
    m_apm_reciprocal(t13, MYDIGITS, u);
    /* t14 = t8*a*t13; */
    m_apm_multiply(u, t8, a);
    m_apm_multiply(t14, u, t13);
    /* t17 = t5*cosh(b); */
    m_apm_cosh(u, MYDIGITS, b);
    m_apm_multiply(t17, t5, u);
    /* t19 = t8*b*t13; */
    m_apm_multiply(u, t8, b);
    m_apm_multiply(t19, u, t13);
    /* re = t3*t14+t17*t19 */
    m_apm_multiply(u, t3, t14);
    m_apm_multiply(v, t17, t19);
    m_apm_add(z->real, u, v);
    /* im = t17*t14-t3*t19 */
    m_apm_multiply(u, t17, t14);
    m_apm_multiply(v, t3, t19);
    m_apm_subtract(z->imag, u, v);
    M_restore_stack(14);
  }
  return 1;
}


static int Ccsc (lua_State *L) {  /* 3.5.2 */
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if (b == 0) */
  if (m_apm_compare(b, MM_Zero) == 0) {
    /* agn_pushcomplex(L, 1/sun_sin(x), 0); */
    M_APM si = M_get_stack_var();
    m_apm_sin(si, MYDIGITS, a);
    if (m_apm_compare(si, MM_Zero) == 0) {
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.ccsc\', division by zero");  /* 3.5.3 improvement */
      M_set_to_zero(z->real);
      M_set_to_zero(z->imag);
    } else {
      m_apm_reciprocal(z->real, MYDIGITS, si);
      M_set_to_zero(z->imag);
    }
    M_restore_stack(1);
  } else {
    M_APM t1, t2, t4, t5, t7, t8, t9, t10, t13, u, v, w;
    t1  = M_get_stack_var();
    t2  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t7  = M_get_stack_var();
    t8  = M_get_stack_var();
    t9  = M_get_stack_var();
    t10 = M_get_stack_var();
    t13 = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    w   = M_get_stack_var();
    /* sun_sincos(a, &t1, &t7); */
    m_apm_sin_cos(t1, t7, MYDIGITS, a);
    /* luai_numsinhcosh(b, &t9, &t2); */
    m_apm_sinh(t9, MYDIGITS, b);
    m_apm_cosh(t2, MYDIGITS, b);
    /* t4 = t1*t1; */
    m_apm_multiply(t4, t1, t1);
    /* t5 = t2*t2; */
    m_apm_multiply(t5, t2, t2);
    /* t8 = t7*t7; */
    m_apm_multiply(t8, t7, t7);
    /* t10 = t9*t9; */
    m_apm_multiply(t10, t9, t9);
    /* t13 = 1/(t4*t5 + t8*t10); */
    m_apm_multiply(u, t4, t5);
    m_apm_multiply(v, t8, t10);
    m_apm_add(w, u, v);
    m_apm_reciprocal(t13, MYDIGITS, w);
    /* re = t1*t2*t13 */
    m_apm_multiply(u, t1, t2);
    m_apm_multiply(z->real, u, t13);
    /* im = -t7*t9*t13 */
    m_apm_multiply(u, t7, t9);
    m_apm_multiply(v, u, t13);
    m_apm_negate(z->imag, v);
    M_restore_stack(12);
  }
  return 1;
}


static int Csec (lua_State *L) {  /* 3.5.2 */
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if (b == 0) */
  if (m_apm_compare(b, MM_Zero) == 0) {
    /* agn_pushcomplex(L, 1/sun_sin(x), 0); */
    M_APM co = M_get_stack_var();
    m_apm_cos(co, MYDIGITS, a);
    if (m_apm_compare(co, MM_Zero) == 0) {
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.csec\', division by zero");  /* 3.5.3 improvement */
      M_set_to_zero(z->real);
      M_set_to_zero(z->imag);
    } else {
      m_apm_reciprocal(z->real, MYDIGITS, co);
      M_set_to_zero(z->imag);
    }
    M_restore_stack(1);
  } else {
    M_APM t1, t2, t4, t5, t7, t8, t9, t10, t13, u, v, w;
    t1  = M_get_stack_var();
    t2  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t7  = M_get_stack_var();
    t8  = M_get_stack_var();
    t9  = M_get_stack_var();
    t10 = M_get_stack_var();
    t13 = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    w   = M_get_stack_var();
    /* sun_sincos(a, &t7, &t1); */
    m_apm_sin_cos(t7, t1, MYDIGITS, a);
    /* luai_numsinhcosh(b, &t9, &t2); */
    m_apm_sinh(t9, MYDIGITS, b);
    m_apm_cosh(t2, MYDIGITS, b);
    /* t4 = t1*t1; */
    m_apm_multiply(t4, t1, t1);
    /* t5 = t2*t2; */
    m_apm_multiply(t5, t2, t2);
    /* t8 = t7*t7; */
    m_apm_multiply(t8, t7, t7);
    /* t10 = t9*t9; */
    m_apm_multiply(t10, t9, t9);
    /* t13 = 1/(t4*t5 + t8*t10); */
    m_apm_multiply(u, t4, t5);
    m_apm_multiply(v, t8, t10);
    m_apm_add(w, u, v);
    m_apm_reciprocal(t13, MYDIGITS, w);
    /* re = t1*t2*t13 */
    m_apm_multiply(u, t1, t2);
    m_apm_multiply(z->real, u, t13);
    /* im = t7*t9*t13 */
    m_apm_multiply(u, t7, t9);
    m_apm_multiply(z->imag, u, t13);
    M_restore_stack(12);
  }
  return 1;
}


static int Ccot (lua_State *L) {  /* 3.5.2 */
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if (b == 0) */
  if (m_apm_compare(b, MM_Zero) == 0) {
    /* agn_pushcomplex(L, -sun_tan(PIO2 + x), 0); */
    M_APM u, v;
    u = M_get_stack_var();
    v = M_get_stack_var();
    m_apm_add(u, MM_lc_HALF_PI, a);
    m_apm_tan(v, MYDIGITS, u);
    m_apm_negate(z->real, v);
    M_set_to_zero(z->imag);
    M_restore_stack(2);
  } else {
    M_APM t1, t4, t5, t6, t8, co, coh, u, v;
    t1  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t6  = M_get_stack_var();
    t8  = M_get_stack_var();
    co  = M_get_stack_var();
    coh = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    /* sun_sincos(a, &t1, &co); */
    m_apm_sin_cos(t1, co, MYDIGITS, a);
    /* luai_numsinhcosh(b, &t5, &coh); */
    m_apm_sinh(t5, MYDIGITS, b);
    m_apm_cosh(coh, MYDIGITS, b);
    /* t4 = t1*t1; */
    m_apm_multiply(t4, t1, t1);
    /* t6 = t5*t5; */
    m_apm_multiply(t6, t5, t5);
    /* t8 = 1/(t4 + t6); */
    m_apm_add(u, t4, t6);
    m_apm_reciprocal(t8, MYDIGITS, u);
    /* re = t1*co*t8 */
    m_apm_multiply(u, t1, co);
    m_apm_multiply(z->real, u, t8);
    /* im = -t5*coh*t8 */
    m_apm_multiply(u, t5, coh);
    m_apm_multiply(v, u, t8);
    m_apm_negate(z->imag, v);
    M_restore_stack(9);
  }
  return 1;
}


static int Ccoth (lua_State *L) {  /* 3.5.2 */
  M_APM a, b;
  M_CAPM *z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if (b == 0) */
  if (m_apm_compare(b, MM_Zero) == 0) {
    /* agn_pushcomplex(L, 1/luai_numtanh(x), 0); */
    M_APM u;
    u = M_get_stack_var();
    m_apm_tanh(u, MYDIGITS, a);
    if (m_apm_compare(u, MM_Zero) == 0) {
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.ccoth\', division by zero");  /* 3.5.3 improvement */
      M_set_to_zero(z->real);
      M_set_to_zero(z->imag);
    } else {
      m_apm_reciprocal(z->real, MYDIGITS, u);
      M_set_to_zero(z->imag);
    }
    M_restore_stack(1);
  } else {
    M_APM t1, t2, t4, t5, t6, t7, t9, t11, t12, t14, t15, t20, u, v, w;
    t1  = M_get_stack_var();
    t2  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t6  = M_get_stack_var();
    t7  = M_get_stack_var();
    t9  = M_get_stack_var();
    t11 = M_get_stack_var();
    t12 = M_get_stack_var();
    t14 = M_get_stack_var();
    t15 = M_get_stack_var();
    t20 = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    w   = M_get_stack_var();
    /* sun_sincos(b, &t14, &t5); */
    m_apm_sin_cos(t14, t5, MYDIGITS, b);
    /* luai_numsinhcosh(a, &t1, &t2); */
    m_apm_sinh(t1, MYDIGITS, a);
    m_apm_cosh(t2, MYDIGITS, a);
    /* t4 = t1*t1; */
    m_apm_multiply(t4, t1, t1);
    /* t6 = t5*t5; */
    m_apm_multiply(t6, t5, t5);
    /* t7 = t4+t6; */
    m_apm_add(t7, t4, t6);
    /* t9 = t2*t2; */
    m_apm_multiply(t9, t2, t2);
    /* t11 = t7*t7; */
    m_apm_multiply(t11, t7, t7);
    /* t12 = 1/t11; */
    m_apm_reciprocal(t12, MYDIGITS, t11);
    /* t15 = t14*t14; */
    m_apm_multiply(t15, t14, t14);
    /* t20 = 1/t7/(t4*t9*t12 + t15*t6*t12); */
    /* v = t4*t9*t12 */
    m_apm_multiply(u, t4, t9);
    m_apm_multiply(v, u, t12);
    /* w = t15*t6*t12 */
    m_apm_multiply(u, t15, t6);
    m_apm_multiply(w, u, t12);
    /* u = (t4*t9*t12 + t15*t6*t12) */
    m_apm_add(u, v, w);
    /* v = 1/t7 */
    m_apm_reciprocal(v, MYDIGITS, t7);
    /* t20 = v/u */
    m_apm_divide(t20, MYDIGITS, v, u);
    /* re = t1*t2*t20 */
    m_apm_multiply(u, t1, t2);
    m_apm_multiply(z->real, u, t20);
    /* im = -t14*t5*t20 */
    m_apm_multiply(u, t14, t5);
    m_apm_multiply(v, u, t20);
    m_apm_negate(z->imag, v);
    M_restore_stack(15);
  }
  return 1;
}


static int Ccsch (lua_State *L) {
  M_APM a, b, t1, t2, t4, t5, t7, t8, t9, t10, t13, u, v, w;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if (b == 0) */
  if (m_apm_compare(b, MM_Zero) == 0) {
    u = M_get_stack_var();
    m_apm_sinh(u, MYDIGITS, a);
    if (m_apm_compare(u, MM_Zero) == 0) {
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.ccsch\', division by zero");  /* 3.5.3 improvement */
      M_set_to_zero(z->real);
      M_set_to_zero(z->imag);
    } else {
      m_apm_reciprocal(z->real, MYDIGITS, u);
      M_set_to_zero(z->imag);
    }
    M_restore_stack(1);
  } else {
    t1  = M_get_stack_var();
    t2  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t7  = M_get_stack_var();
    t8  = M_get_stack_var();
    t9  = M_get_stack_var();
    t10 = M_get_stack_var();
    t13 = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    w   = M_get_stack_var();
    /* t1 = sinh(a); t2 = cos(b); t7 = cosh(a); t9 = sin(b); */
    m_apm_sin_cos(t9, t2, MYDIGITS, b);
    m_apm_sinh(t1, MYDIGITS, a);
    m_apm_cosh(t7, MYDIGITS, a);
    /* t4 = t1*t1; */
    m_apm_multiply(t4, t1, t1);
    /* t5 = t2*t2; */
    m_apm_multiply(t5, t2, t2);
    /* t8 = t7*t7; */
    m_apm_multiply(t8, t7, t7);
    /* t10 = t9*t9; */
    m_apm_multiply(t10, t9, t9);
    /* t13 = 1/(t4*t5+t8*t10); */
    m_apm_multiply(u, t4, t5);
    m_apm_multiply(v, t8, t10);
    m_apm_add(w, u, v);
    m_apm_reciprocal(t13, MYDIGITS, w);
    /* re = t1*t2*t13 */
    m_apm_multiply(u, t1, t2);
    m_apm_multiply(z->real, u, t13);
    /* im = -t7*t9*t13; */
    m_apm_multiply(u, t7, t9);
    m_apm_multiply(v, u, t13);
    m_apm_negate(z->imag, v);
    M_restore_stack(12);
  }
  return 1;
}


static int Csech (lua_State *L) {
  M_APM a, b, t1, t2, t4, t5, t7, t8, t9, t10, t13, u, v, w;
  M_CAPM *z;
  z = Cnew(L);
  Cgetrealimag(L, 1, a, b);
  /* if (b == 0) */
  if (m_apm_compare(b, MM_Zero) == 0) {
    u = M_get_stack_var();
    m_apm_cosh(u, MYDIGITS, a);
    if (m_apm_compare(u, MM_Zero) == 0) {
      M_apm_log_error_msg(M_APM_RETURN, (char *)"\'mapm.csech\', division by zero");  /* 3.5.3 improvement */
      M_set_to_zero(z->real);
      M_set_to_zero(z->imag);
    } else {
      m_apm_reciprocal(z->real, MYDIGITS, u);
      M_set_to_zero(z->imag);
    }
    M_restore_stack(1);
  } else {
    t1  = M_get_stack_var();
    t2  = M_get_stack_var();
    t4  = M_get_stack_var();
    t5  = M_get_stack_var();
    t7  = M_get_stack_var();
    t8  = M_get_stack_var();
    t9  = M_get_stack_var();
    t10 = M_get_stack_var();
    t13 = M_get_stack_var();
    u   = M_get_stack_var();
    v   = M_get_stack_var();
    w   = M_get_stack_var();
    /* t1 = cosh(a); t2 = cos(b); t7 = sinh(a); t9 = sin(b); */
    m_apm_sin_cos(t9, t2, MYDIGITS, b);
    m_apm_sinh(t7, MYDIGITS, a);
    m_apm_cosh(t1, MYDIGITS, a);
    /* t4 = t1*t1; */
    m_apm_multiply(t4, t1, t1);
    /* t5 = t2*t2; */
    m_apm_multiply(t5, t2, t2);
    /* t8 = t7*t7; */
    m_apm_multiply(t8, t7, t7);
    /* t10 = t9*t9; */
    m_apm_multiply(t10, t9, t9);
    /* t13 = 1/(t4*t5+t8*t10); */
    m_apm_multiply(u, t4, t5);
    m_apm_multiply(v, t8, t10);
    m_apm_add(w, u, v);
    m_apm_reciprocal(t13, MYDIGITS, w);
    /* re = t1*t2*t13 */
    m_apm_multiply(u, t1, t2);
    m_apm_multiply(z->real, u, t13);
    /* im = -t7*t9*t13; */
    m_apm_multiply(u, t7, t9);
    m_apm_multiply(v, u, t13);
    m_apm_negate(z->imag, v);
    M_restore_stack(12);
  }
  return 1;
}


/* The metemethods ************************************************************************/

static const struct luaL_Reg mapm_lib [] = {  /* metamethods for real mapm numbers */
  {"__abs", mt_abs},
  {"__sign", mt_sign},
  {"__unm", mt_unm},
  {"__add", mt_add},
  {"__sub", mt_sub},
  {"__mul", mt_mul},
  {"__div", mt_div},
  {"__intdiv", mt_idiv},
  {"__eq",	mt_eq},
  {"__lt",	mt_lt},
  {"__le",	mt_le},
  {"__mod", mt_mod},
  {"__pow", mt_pow},
  {"__ipow", mt_ipow},
  {"__square", mt_square},
  {"__cube", mt_cube},
  {"__recip", mt_recip},
  {"__sqrt", mt_sqrt},
  {"__ln", mt_ln},
  {"__exp", mt_exp},
  {"__sin", mt_sin},
  {"__cos", mt_cos},
  {"__tan", mt_tan},
  {"__arcsec", mt_arcsec},
  {"__arcsin", mt_arcsin},
  {"__arccos", mt_arccos},
  {"__arctan", mt_arctan},
  {"__sinc", mt_sinc},
  {"__sinh", mt_sinh},
  {"__cosh", mt_cosh},
  {"__tanh", mt_tanh},
  {"__int", mt_int},
  {"__even", mt_even},   /* __even(x) */
  {"__odd",	mt_odd},   /* __odd(x) */
  {"__gc",	mt_gc},
  {"__tostring", mt_tostring},  /* __tostring(x) */
  {NULL, NULL}
};


static const struct luaL_Reg mapm_clib [] = {  /* metamethods for complex mapm numbers */
  {"__abs",    mt_cabs},
  {"__sign",   mt_csgn},
  {"__add",    mt_cadd},  /* __add(x, y) */
  {"__sub",    mt_csub},  /* __sub(x, y) */
  {"__mul",    mt_cmul},  /* __mul(x, y) */
  {"__div",    mt_cdiv},  /* __div(x, y) */
  {"__recip",  mt_crecip},
  {"__pow",    mt_cpow},  /* __pow(x, y) */
  {"__ipow",   mt_cipow}, /* __ipow(x, y) */
  {"__square", mt_csquare},
  {"__cube",   mt_ccube},
  {"__sqrt",   mt_csqrt},
  {"__ln",     mt_cln},
  {"__exp",    mt_cexp},
  {"__sin",    mt_csin},
  {"__cos",    mt_ccos},
  {"__tan",    mt_ctan},
  {"__sinc",   mt_csinc},
  {"__sinh",   mt_csinh},
  {"__cosh",   mt_ccosh},
  {"__tanh",   mt_ctanh},
  {"__arcsin", mt_carcsin},
  {"__arccos", mt_carccos},
  {"__arctan", mt_carctan},
  {"__arcsec", mt_carcsec},
  {"__gc",	   mt_cgc},
  {"__real",	 mt_creal},
  {"__imag",	 mt_cimag},
  {"__tostring", mt_ctostring},
  {"__unm",    mt_cunm},  /* __unm(x) */
  {"__eq",	   mt_ceq},   /* __eq(x, y) */
#ifdef XXX
  {"__mod",    Bmod},  /* __mod(x, y) */
  {"__int",    mt_int},
  {"__lt",	   Blt},   /* __lt(x, y) */
#endif
  {NULL, NULL}
};


static const luaL_Reg mapmlib[] = {
  {"xabs", Babs},
  {"xarccos", Bacos},
  {"xarccosh", Bacosh},
  {"xadd", Badd},
  {"xarccsc", Bacsc},
  {"xarcsec", Basec},
  {"xarcsin", Basin},
  {"xarcsinh", Basinh},
  {"xarctan", Batan},
  {"xarctan2", Batan2},
  {"xarctanh", Batanh},
  {"xcbrt", Bcbrt},
  {"xceil", Bceil},
  {"xchebyt", Bchebyt},
  {"xcompare", Bcompare},
  {"xcos", Bcos},
  {"xcosc", Bcosc},
  {"xcosh", Bcosh},
  {"xcot", Bcot},
  {"xcoth", Bcoth},
  {"xcsc", Bcsc},
  {"xcsch", Bcsch},
  {"xcube", Bcube},
  {"xdigits", Bdigits},
  {"xdigitsin", Bdigitsin},
  {"xdiv", Bdiv},
  {"xexp", Bexp},
  {"xexp2", mapm_xexp2},
  {"xexp10", mapm_xexp10},
  {"xexponent", Bexponent},
  {"xfactorial", Bfactorial},
  {"xfloor", Bfloor},
  {"xfma", mapm_xfma},
  {"xhypot", Bhypot},
  {"xhypot4", Bhypot4},
  {"xidiv", Bidiv},
  {"xint", Bint},
  {"xinv", Binv},
  {"xiseven", Biseven},
  {"xisint", Bisint},
  {"xisodd", Bisodd},
  {"xln", Blog},  /* changed 0.30.3 */
  {"xlog", mapm_xlog},
  {"xlog2", mapm_xlog2},
  {"xlog10", Blog10},
  {"xmod", Bmod},
  {"xmul", Bmul},
  {"xneg", Bneg},
  {"xnumber", Bnumber},
  {"xpow",	Bpow},
  {"xrandom", mapm_xrandom},
  {"xrandomseed", mapm_xrandomseed},
  {"xrecip", Binv},
  {"xround", Bround},
  {"xsec", Bsec},
  {"xsech", Bsech},
  {"xsign", Bsign},
  {"xsin", Bsin},
  {"xsinc", Bsinc},
  {"xsincos", Bsincos},
  {"xsinhcosh", Bsinhcosh},
  {"xsinh", Bsinh},
  {"xsquare", Bsquare},
  {"xsqrt", Bsqrt},
  {"xsub", Bsub},
  {"xtan", Btan},
  {"xtanc", Btanc},
  {"xtanh", Btanh},
  {"xterm", mapm_xterm},
  {"xtonumber", Btonumber},
  {"xtostring", Btostring},
  /* complex compartment */
  {"cnumber", Cnumber},
  {"ctocomplex", Ctocomplex},
  {"ctonumber", Ctonumber},
  {"ctostring", mapm_ctostring},
  {"carccosh", Carccosh},
  {"carcsinh", Carcsinh},
  {"carctanh", Carctanh},
  {"cfma", Cfma},
  {"cargument", Cargument},
  {"ccsc", Ccsc},
  {"ccsch", Ccsch},
  {"csec", Csec},
  {"csech", Csech},
  {"ccot", Ccot},
  {"ccoth", Ccoth},
  {"csinc", Csinc},
  {"ccosc", Ccosc},
  {"ctanc", Ctanc},
  {NULL, NULL}
};


static void clearmapm (void) {  /* 2.9.6 */
  m_apm_free_all_mem();
}


static int nopened = 0;

LUALIB_API int luaopen_mapm (lua_State *L) {
  if (nopened++ == 0) atexit(clearmapm);  /* only register once, 3.6.1 */
  luaL_newmetatable(L, MYTYPE);         /* metatable for userdata, adds it to the registry with key 'mapm' */
  luaL_register(L, NULL, mapm_lib);     /* assign C metamethods to this metatable */
  luaL_newmetatable(L, MYCTYPE);        /* metatable for long userdata, adds it to the registry with key 'mapm' */
  luaL_register(L, NULL, mapm_clib);    /* assign C metamethods to this metatable */
  luaL_register(L, AGENA_MAPMLIBNAME, mapmlib);  /* leaves the package table on the top of the stack */
  lua_pushliteral(L, "version");
  lua_pushliteral(L, MYVERSION);
  lua_settable(L, -3);
  lua_pushliteral(L, "__index");
  lua_pushvalue(L, -2);
  lua_settable(L, -3);
  return 1;
}

