/* inifile library
   taken from: https://gitlab.com/iniparser/iniparser

   Copyright (c) 2000-2024 by Nicolas Devillard and many contributors
   MIT License

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
   DEALINGS IN THE SOFTWARE.

   Agena binding written by Alexander Walz, see MIT copyright notice above.
   Additions and changes have all been marked in the source code.

   @file    dictionary.h
   @author  N. Devillard
   @brief   Implements a dictionary for string variables.

   This module implements a simple dictionary object, i.e. a list
   of string/string associations. This object is useful to store e.g.
   information retrieved from a configuration file (ini files).
*/

/*---------------------------------------------------------------------------
                                Includes
 ---------------------------------------------------------------------------*/

#ifndef iniparse_h
#define iniparse_h

#include <stdio.h>
#include <stdint.h>

/*---------------------------- Defines -------------------------------------*/

#define ASCIILINESZ       (1024*8)
#define INI_INVALID_KEY   ((char*)-1)

#define BUCKETSLOTS  4
#define MODULO       (BUCKETSLOTS - 1)
#define DICTMINSZ    (32*BUCKETSLOTS)  /* must be a multiple of BUCKETSLOTS */

/*---------------------------------------------------------------------------
                                New types
 ---------------------------------------------------------------------------*/

/**
  @brief    Dictionary object

  This object contains a list of string/string associations. Each
  association is identified by a unique string key. Looking up values
  in the dictionary is speeded up by the use of a (hopefully collision-free)
  hash function.

  With the Agena 4.6.7 port, internal memory management changed: Items are not
  stored consecutively, but depending on their hash value taken modulo
  `bucketslots`. This allows for much faster read (3x) and write (4x) access,
  at least.
*/
typedef struct dictionary {
  unsigned int n;            /* number of entries in dictionary */
  unsigned int *nextfree;    /* position of the next slot to occupy, 4.6.7 */
  unsigned int bucketslots;  /* slots per bucket, must be a multiple of 4, 4.6.8 */
  size_t   size;             /* storage size */
  char     **val;            /* list of string values */
  char     **key;            /* list of string keys */
  unsigned int *hash;        /* list of hash values for keys */
} dictionary;


/*---------------------------------------------------------------------------
                            Function prototypes
 ---------------------------------------------------------------------------*/

/**
  @brief    Compute the hash key for a string.
  @param    key     Character string to use for key.
  @return   1 unsigned int on at least 32 bits.

  This hash function has been taken from an Article in Dr Dobbs Journal.
  This is normally a collision-free function, distributing keys evenly.
  The key is stored anyway in the struct so that collision can be avoided
  by comparing the key itself in last resort.
*/
unsigned int dictionary_hash (const char *key);


/**
  @brief    Create a new dictionary object.
  @param    size    Optional initial size of the dictionary.
  @return   1 newly allocated dictionary object.

  This function allocates a new dictionary object of given size and returns
  it. If you do not know in advance (roughly) the number of entries in the
  dictionary, give size=0.
*/
dictionary *dictionary_new (size_t size);


/**
  @brief    Delete a dictionary object
  @param    d   dictionary object to deallocate.
  @return   void

  Deallocate a dictionary object and all memory associated to it.
*/
void dictionary_del (dictionary *vd, int xfreeme);


/**
  @brief    Get a value from a dictionary.
  @param    d       dictionary object to search.
  @param    key     Key to look for in the dictionary.
  @param    def     Default value to return if key not found.
  @return   1 pointer to internally allocated character string.
  @return rc:   -1 if key could not be found, 0 if key actually denotes a section name,
                +1 if key is a key name of a key-value pair. (Agena extension)

  This function locates a key in a dictionary and returns a pointer to its
  value, or the passed 'def' pointer if no such key can be found in
  dictionary. The returned character pointer points to data internal to the
  dictionary object, you should not try to free it or modify it.
*/
const char *dictionary_get (dictionary *d, const char *key, const char *def, int *rc);


/**
  @brief    Set a value in a dictionary.
  @param    d       dictionary object to modify.
  @param    key     Key to modify or add.
  @param    val     Value to add.
  @return   int     0 if Ok, anything else otherwise

  If the given key is found in the dictionary, the associated value is
  replaced by the provided one. If the key cannot be found in the
  dictionary, it is added to it.

  It is Ok to provide a NULL value for val, but NULL values for the dictionary
  or the key are considered as errors: the function will return immediately
  in such a case.

  Notice that if you dictionary_set a variable to NULL, a call to
  dictionary_get will return a NULL value: the variable will be found, and
  its value (NULL) is returned. In other words, setting the variable
  content to NULL is equivalent to deleting the variable from the
  dictionary. It is not possible (in this implementation) to have a key in
  the dictionary without value.

  This function returns non-zero in case of failure.
*/
int dictionary_set (dictionary *vd, const char *key, const char *val);


/**
  @brief    Delete a key in a dictionary
  @param    d       dictionary object to modify.
  @param    key     Key to remove.
  @return   void

  This function deletes a key in a dictionary. Nothing is done if the
  key cannot be found.
*/
int dictionary_unset (dictionary *d, const char *key);


/**
  @brief    Dump a dictionary to an opened file pointer.
  @param    d   Dictionary to dump
  @param    f   Opened file pointer.
  @return   void

  Dumps a dictionary onto an opened file pointer. Key pairs are printed out
  as @c [Key]=[Value], one per line. It is Ok to provide stdout or stderr as
  output file pointers.
*/
void dictionary_dump (dictionary *d, FILE *out);


/* ######################################################################## */

/**
   @file    iniparser.h
   @author  N. Devillard
   @brief   Parser for ini files.
*/

/**
  @brief    Configure a function to receive the error messages.
  @param    errback  Function to call.

  By default, the error will be printed on stderr. If a null pointer is passed
  as errback the error callback will be switched back to default.
 */
void iniparser_set_error_callback (int (*errback)(const char *, ...));


/**
  @brief    Get number of sections in a dictionary
  @param    d   Dictionary to examine
  @return   int Number of sections found in dictionary

  This function returns the number of sections found in a dictionary.
  The test to recognize sections is done on the string stored in the
  dictionary: a section name is given as "section" whereas a key is
  stored as "section:key", thus the test looks for entries that do not
  contain a colon.

  This clearly fails in the case a section name contains a colon, but
  this should simply be avoided.

  This function returns -1 in case of error.
*/
int iniparser_getnsec (dictionary *d);


/**
  @brief    Get name for section n in a dictionary.
  @param    d   Dictionary to examine
  @param    n   Section number (from 0 to nsec-1).
  @return   Pointer to char string

  This function locates the n-th section in a dictionary and returns
  its name as a pointer to a string statically allocated inside the
  dictionary. Do not free or modify the returned string!

  This function returns NULL in case of error.
*/
const char *iniparser_getsecname (dictionary *d, int n);


/**
  @brief    Save a dictionary to a loadable ini file
  @param    d   Dictionary to dump
  @param    f   Opened file pointer to dump to

  This function dumps a given dictionary into a loadable ini file.
  It is Ok to specify @c stderr or @c stdout as output files.

  All values are quoted, these characters are escaped:

  - ' : the quote character (e.g. "String with \"Quotes\"")
  - \ : the backslash character (e.g. "C:\\tmp")
*/
void iniparser_dump_ini (dictionary *d, FILE *f);


/**
  @brief    Save a dictionary section to a loadable ini file
  @param    d   Dictionary to dump
  @param    s   Section name of dictionary to dump
  @param    f   Opened file pointer to dump to

  This function dumps a given section of a given dictionary into a loadable ini
  file.  It is Ok to specify @c stderr or @c stdout as output files.
*/
void iniparser_dumpsection_ini (dictionary *d, const char *s, FILE *f);


/**
  @brief    Dump a dictionary to an opened file pointer.
  @param    d   Dictionary to dump.
  @param    f   Opened file pointer to dump to.

  This function prints out the contents of a dictionary, one element by
  line, onto the provided file pointer. It is OK to specify @c stderr
  or @c stdout as output files. This function is meant for debugging
  purposes mostly.
*/
void iniparser_dump (dictionary *d, FILE *f);


/**
  @brief    Get the number of keys in a section of a dictionary.
  @param    d   Dictionary to examine
  @param    s   Section name of dictionary to examine
  @return   Number of keys in section
*/
int iniparser_getsecnkeys (dictionary *d, const char *s);


/**
  @brief    Get the number of keys in a section of a dictionary.
  @param    d    Dictionary to examine
  @param    s    Section name of dictionary to examine
  @param    keys Already allocated array to store the keys in
  @return   The pointer passed as `keys` argument or NULL in case of error

  This function queries a dictionary and finds all keys in a given section.
  The keys argument should be an array of pointers which size has been
  determined by calling `iniparser_getsecnkeys` function prior to this one.

  Each pointer in the returned char pointer-to-pointer is pointing to
  a string allocated in the dictionary; do not free or modify them.
*/
const char **iniparser_getseckeys (dictionary *d, const char *s, const char **keys);


/**
  @brief    Get the string associated to a key
  @param    d       Dictionary to search
  @param    key     Key string to look for
  @param    def     Default value to return if key not found.
  @return   pointer to statically allocated character string

  This function queries a dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the pointer passed as 'def' is returned.
  The returned char pointer is pointing to a string allocated in
  the dictionary, do not free or modify it.
*/
const char *iniparser_getstring (dictionary *d, const char *key, const char *def);


/**
  @brief    Get the string associated to a key, convert to an int
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   integer

  This function queries a dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.

  Supported values for integers include the usual C notation
  so decimal, octal (starting with 0) and hexadecimal (starting with 0x)
  are supported. Examples:

  - "42"      ->  42
  - "042"     ->  34 (octal -> decimal)
  - "0x42"    ->  66 (hexa  -> decimal)

  Warning: the conversion may overflow in various ways. Conversion is
  totally outsourced to strtol(), see the associated man page for overflow
  handling.

  Credits: Thanks to A. Becker for suggesting strtol()
*/
int iniparser_getint (dictionary *d, const char *key, int notfound);


/**
  @brief    Get the string associated to a key, convert to an long int
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   integer

  This function queries a dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.

  Supported values for integers include the usual C notation
  so decimal, octal (starting with 0) and hexadecimal (starting with 0x)
  are supported. Examples:

  - "42"      ->  42
  - "042"     ->  34 (octal -> decimal)
  - "0x42"    ->  66 (hexa  -> decimal)

  Warning: the conversion may overflow in various ways. Conversion is
  totally outsourced to strtol(), see the associated man page for overflow
  handling.
*/
long int iniparser_getlongint (dictionary *d, const char *key, long int notfound);


/**
  @brief    Get the string associated to a key, convert to an int64_t
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   integer

  This function queries a dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.

  Supported values for integers include the usual C notation
  so decimal, octal (starting with 0) and hexadecimal (starting with 0x)
  are supported. Examples:

  - "42"      ->  42
  - "042"     ->  34 (octal -> decimal)
  - "0x42"    ->  66 (hexa  -> decimal)

  Warning: the conversion may overflow in various ways. Conversion is
  totally outsourced to strtoimax(), see the associated man page for overflow
  handling.

  This function is useful on 32bit architectures where `long int` is only
  32bit.
*/
int64_t iniparser_getint64 (dictionary *d, const char *key, int64_t notfound);


/**
  @brief    Get the string associated to a key, convert to an uint64_t
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   integer

  This function queries a dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.

  Supported values for integers include the usual C notation
  so decimal, octal (starting with 0) and hexadecimal (starting with 0x)
  are supported. Examples:

  - "42"      ->  42
  - "042"     ->  34 (octal -> decimal)
  - "0x42"    ->  66 (hexa  -> decimal)

  Warning: the conversion may overflow in various ways. Conversion is
  totally outsourced to strtoumax(), see the associated man page for overflow
  handling.

  This function is useful on 32bit architectures where `long int` is only
  32bit.
*/
uint64_t iniparser_getuint64 (dictionary *d, const char *key, uint64_t notfound);


/**
  @brief    Get the string associated to a key, convert to a double
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   double

  This function queries a dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.
*/
double iniparser_getdouble (dictionary *d, const char *key, double notfound);


/**
  @brief    Get the string associated to a key, convert to a boolean
  @param    d Dictionary to search
  @param    key Key string to look for
  @param    notfound Value to return in case of error
  @return   integer

  This function queries a dictionary for a key. A key as read from an
  ini file is given as "section:key". If the key cannot be found,
  the notfound value is returned.

  A true boolean is found if one of the following is matched:

  - A string starting with 'y'
  - A string starting with 'Y'
  - A string starting with 't'
  - A string starting with 'T'
  - A string starting with '1'

  A false boolean is found if one of the following is matched:

  - A string starting with 'n'
  - A string starting with 'N'
  - A string starting with 'f'
  - A string starting with 'F'
  - A string starting with '0'

  The notfound value returned if no boolean is identified, does not
  necessarily have to be 0 or 1.
*/
int iniparser_getboolean (dictionary *d, const char *key, int notfound);


/**
  @brief    Set an entry in a dictionary.
  @param    ini     Dictionary to modify.
  @param    entry   Entry to modify (entry name)
  @param    val     New value to associate to the entry.
  @return   int     0 if Ok, -1 otherwise.

  If the given entry can be found in the dictionary, it is modified to
  contain the provided value. If it cannot be found, the entry is created.
  It is Ok to set val to NULL.
*/
int iniparser_set (dictionary *ini, const char *entry, const char *val);


/**
  @brief    Delete an entry in a dictionary
  @param    ini     Dictionary to modify
  @param    entry   Entry to delete (entry name)

  If the given entry can be found, it is deleted from the dictionary.
*/
void iniparser_unset (dictionary *ini, const char *entry);


/**
  @brief    Finds out if a given entry exists in a dictionary
  @param    ini     Dictionary to search
  @param    entry   Name of the entry to look for
  @return   integer 1 if entry exists, 0 otherwise

  Finds out if a given entry exists in the dictionary. Since sections
  are stored as keys with NULL associated values, this is the only way
  of querying for the presence of sections in a dictionary.
*/
int iniparser_find_entry (dictionary *ini, const char *entry) ;


/**
  @brief    Parse an ini file and return an allocated dictionary object
  @param    ininame Name of the ini file to read.
  @return   Pointer to newly allocated dictionary

  This is the parser for ini files. This function is called, providing
  the name of the file to be read. It returns a dictionary object that
  should not be accessed directly, but through accessor functions
  instead.

  Iff the value is a quoted string it supports some escape sequences:

  - \" or ' : the quote character
  (e.g. 'String with "Quotes"' or "String with 'Quotes'")
  - \ : the backslash character (e.g. "C:\tmp")

  Escape sequences always start with a backslash. Additional escape sequences
  might be added in the future. Backslash characters must be escaped. Any other
  sequence then those outlined above is invalid and may lead to unpredictable
  results.

  The returned dictionary must be freed using iniparser_freedict().
*/
dictionary *iniparser_load (dictionary *d, const char *ininame, int xfreeme);


/**
  @brief    Parse an ini file and return an allocated dictionary object
  @param    in File to read.
  @param    ininame Name of the ini file to read (only used for nicer error messages)
  qparam    free dictionary itself, to be set to 0 if used in the userdata context
  @return   Pointer to newly allocated dictionary

  This is the parser for ini files. This function is called, providing
  the file to be read. It returns a dictionary object that should not
  be accessed directly, but through accessor functions instead.

  Iff the value is a quoted string it supports some escape sequences:

  - \" or ' : the quote character
  (e.g. 'String with "Quotes"' or "String with 'Quotes'")
  - \ : the backslash character (e.g. "C:\tmp")

  Escape sequences always start with a backslash. Additional escape sequences
  might be added in the future. Backslash characters must be escaped. Any other
  sequence then those outlined above is invalid and may lead to unpredictable
  results.

  The returned dictionary must be freed using iniparser_freedict().
*/
dictionary *iniparser_load_file (dictionary *dict, FILE *in, const char *ininame, int xfreeme);


/**
  @brief    Free all memory associated to an ini dictionary
  @param    d Dictionary to free
  qparam    free dictionary itself, to be set to 0 if used in the userdata context

  Free all memory associated to an ini dictionary.
  It is mandatory to call this function before the dictionary object
  gets out of the current context.
*/
void iniparser_freedict (dictionary *d);

#endif

