
{ Retrieve Message by ID String }
function FetchMessage(const ID : String) : string;
var
    S : String;
begin
    if (Verbose > 0) then begin
        S := LookupASCIIZ(Messages, 'VERB_' + ID);
        if S <> '' then
            FetchMessage := S
        else
           FetchMessage := LookupASCIIZ(Messages, ID);
    end else
        FetchMessage := LookupASCIIZ(Messages, ID);
end;

{ Parses a translation message using provided string data }
function ParseKeyValue(const KeyValue : String; Data : String) : String;
var
    I, P, C, H : integer;
    S, R, X : String;
begin
    S := KeyValue;
    I := 0;
    P := 1;
    R := '';
    while I < Length(S) do begin
        Inc(I);
        if S[I] <> '%' then begin
            R := R + S[I];
        end else if (I + 1 <= Length(S)) and (S[I + 1] = '_') then begin
            Inc(I);
            R := R + #32;
        end else if (I + 1 <= Length(S)) and (S[I + 1] = 'r') then begin
            Inc(I);
            R := R + CRLF;
        end else if (I + 1 <= Length(S)) and (S[I + 1] = '%') then begin
            Inc(I);
            R := R + '%';
        end else begin
            H := Pos(S[I + 1], '0123456789ABCDEF');
            if (I + 1 <= Length(S)) and (H >= 1) then begin
                C := 0;
                P := 1;
                Inc(I);
                while (C < H - 1) and (P <= Length(Data)) do begin
                    if Data[P] = FormatDelim then Inc(C);
                    Inc(P);
                end;
            end;
            if P <= Length(Data) then begin
                X := Copy(Data, P, Length(Data));
                C := Pos(FormatDelim, X) - 1;
                if C < 0 then C := Length(X);
                X := Copy(X, 1, C);
                R := R + X;
                P := P + Length(X) + 1;
            end;
        end
    end;
    ParseKeyValue := R;
end;

function ParseMessage(const ID : String; Data : String) : String;
begin
    ParseMessage := ParseKeyValue(KeyValueOnly(FetchMessage(ID)), Data);
end;

procedure ShowBanner(AClass : byte); forward;

procedure ShowText(AClass : byte; const AText : String);
var
    I : integer;
begin
    {
    Sometimes this called on a per/character basis. when that occurs filter
    out any CR characters, then treat any LF as a CRLF
    }
    if WantBanner then ShowBanner(AClass);
{   if (AClass > mcCritical) and ((Verbose < 0)) then exit;
    if (AClass >= mcVerbose) and (Verbose < 1) then exit;}
    if AClass > (Verbose + 2) then exit;
    if AText = CR_char then exit;
    if AText = LF_char then begin
        if IsRedirectedOutput then begin
            StdOutChr(CR_char);
            StdOutChr(LF_char);
        end else
            WriteLn;
        exit;
    end;
    If IsRedirectedOutput then begin
        for I := 1 to Length(AText) do
            StdOutChr(AText[I]);
    end else
        Write(AText);
end;

procedure ShowTextLn(AClass : byte; const AText : String);
begin
    ShowText(AClass, AText + CRLF);
end;

procedure MaybeTextLn(AClass : byte; const AText : String);
begin
    if AText <> '' then
        ShowText(AClass, AText + CRLF);
end;

procedure ShowBanner(AClass : byte);
var
    X : Byte;
begin
    if AClass > (Verbose + 2) then exit;
    if not WantBanner then exit;
    WantBanner := False;
    X := TextAttr;
    ThemeColor(clBanner);
    ShowTextLn(AClass, ParseMessage('TITLE', Title + FormatDelim + Version));
    ShowText(AClass, ParseMessage('COPYRIGHT', ''));
    TextAttr := X;
    ShowTextLn(AClass, '');
    ShowTextLn(AClass, '');
end;

{ Display all help messages }
procedure ShowHelp;
var
    I : integer;
    S, H : String;
begin
{    ShowTextLn(mcGeneral, ''); }
    ShowTextLn(mcGeneral, ParseMessage('USAGE', GetExeBase));
    ShowTextLn(mcGeneral, '');
    I := 0;
    repeat
        Str(I, S);
        S := 'HELP_' + S;
        H := FetchMessage(S);
        if (H <> '') then begin
            Inc(I);
            ShowTextLn(mcGeneral, ParseKeyValue(KeyValueOnly(H), SwitchChar));
        end;
    until H = '';
end;

{ Display an error message and possibly terminate program }
procedure ShowError(ID, Data : String; ErrorCode : Byte; Terminate : boolean );
var
    X : byte;
begin
    if ErrorCode = 0 then Exit;
    if (Pos(Chr(ErrorCode), FatalErrors) > 0) then
        Terminate := True;
    if ID = '' then begin
            ID := 'ER_' + IntStr(ErrorCode);
            if FetchMessage(ID) = '' then
                ID := 'ER_UNK';
        end;
    X := TextAttr;
    if Terminate then begin
        ThemeColor(clFatalLabel);
        ShowText(mcAlways, ParseMessage('FATAL', ''));
        ThemeColor(clFatalText);
        ShowText(mcAlways, ParseMessage(ID, Data));
        TextAttr := X;
        ShowTextLn(mcAlways, '');
    end else begin
        ThemeColor(clWarningLabel);
        ShowText(mcCritical, ParseMessage('ERROR', ''));
        ThemeColor(clWarningText);
        ShowText(mcCritical, ParseMessage(ID, Data));
        TextAttr := X;
        ShowTextLn(mcCritical, '');
    end;
    if Terminate then begin
        CleanUp;
        Halt(ErrorCode);
    end;
end;

procedure MissingSpecial(Special : byte);
begin
    if Special = 0 then exit;
    ShowError('BAD_MCO', SwitchChar + FormatDelim + spSwitches[Special], erCommand_Line_Error, True);
end;

procedure NeedHelp;
begin
    ShowError('NEEDHELP', SwitchChar + FormatDelim  +SwitchChar, erNo_Command_Line, True);
end;

procedure LoadLanguage(ALanguage : String);
var
    NLS  : String;
    Name : String;
    F : PDiskFile;
    P : pointer;
    S, C, D : word;
    Z : boolean;
begin
    Language := UCase(ALanguage);

    if MessagesSize > 0 then
        FreeMem(Messages, MessagesSize);
    Messages := @BuiltInEnglish;
    MessagesSize := 0;

    if ALanguage = '*' then
        ALanguage := GetEnv('LANG');
    if ALanguage = '' then ALanguage := 'EN';
    NLS := GetEnv('NLSPATH');
    if NLS = '' then NLS := GetExePath;
    Name := TailDelim(NLS) + GetExeBase + '.' + ALanguage;
    if not FileExists(Name) then
        Name := TailDelim(GetExePath) + GetExeBase + '.' + ALanguage;
    if not FileExists(Name) then
        Name := TailDelim(GetExePath) + TailDelim('NLS') + GetExeBase + '.' + ALanguage;
    if not FileExists(Name) then
        exit;
    P := nil;
    CheckMemory(Sizeof(TDiskFile));
    F := New(PDiskFile, Create(nil));
    F^.SetMode(flRead);
    if F^.Result = 0 then F^.Assign(Name);
    if F^.Result = 0 then F^.Reset;
    if F^.Result = 0 then begin
        MessagesSize := F^.FileSize;
        CheckMemory(MessagesSize);
        GetMem(P, MessagesSize);
        F^.BlockRead(P^, MessagesSize, C);
        if (F^.Result <> 0) or (MessagesSize <> C) then begin
            FreeMem(P, MessagesSize);
            MessagesSize := 0;
            P := nil;
        end;
        { need to compress CRLF and blank lines, and convert CRLF to nulls.
         Size will be less than original size. So, just wasting a few ram bytes
         squeezing the data. }
        D := 0;
        Z := True;
        for S := 0 to C do begin
            if (PCharArray(P)^[S] = #13) or (PCharArray(P)^[S] = #10) then begin
                if not Z then begin
                    PCharArray(P)^[D] := #0;
                    inc(D);
                end;
                Z := True;
            end else begin
                PCharArray(P)^[D] := PCharArray(P)^[S];
                inc(D);
                Z := False;
            end;
        end;
        PCharArray(P)^[D] := #0;
    end;
    if F^.Result = 0 then F^.Close;
    Dispose(F, Destroy);
    if Assigned(P) then
        Messages := P;

    Stamper.Date := KeyValueOnly(FetchMessage('DATE'));
    Stamper.Time := KeyValueOnly(FetchMessage('TIME'));
    Stamper.AM := KeyValueOnly(FetchMessage('AM'));
    Stamper.PM := KeyValueOnly(FetchMessage('PM'));
    Stamper.Stamp := KeyValueOnly(FetchMessage('STAMP'));

    Bytes := KeyValueOnly(FetchMessage('BYTES'));
    KBytes := KeyValueOnly(FetchMessage('KBYTES'));
    MBytes := KeyValueOnly(FetchMessage('MBYTES'));

end;

function AttribStr ( AAttrib : Word ) : String;
begin
    AttribStr :=
        WhichStr(AAttrib and faDirectory = faDirectory, '.', 'd') +
        WhichStr(AAttrib and faReadOnly  = faReadOnly,  '.', 'r') +
        WhichStr(AAttrib and faArchive   = faArchive,   '.', 'a') +
        WhichStr(AAttrib and faSystem    = faSystem,    '.', 's') +
        WhichStr(AAttrib and faHidden    = faHidden,    '.', 'h');
end;

{
; Used for date time stamps
; DATE = %0 4 digit Year, %1 2 digit Month, %2 2 digit Day,
;        %3 2 digit year, %4 1-2 digit Month, %5 1-2 digit day
DATE=%1/%2/%0
; TIME = %0 Hour (24 hour), %1 Minute, %2 Second
;        %3 12 hour clock, %4 AM/PM
TIME=%3:%1 %4
AM=am
PM=pm
; STAMP = %0 date, %1 time
STAMP=%0 @ %1
}
function DateStr(Year, Month, Day : word) : string;
begin
    DateStr := ParseKeyValue(Stamper.Date,
        IntStr(Year) + FormatDelim + ZPad(IntStr(Month), 2) + FormatDelim + ZPad(IntStr(Day), 2)
        + FormatDelim + Copy(IntStr(Year), 3, 2) + FormatDelim + IntStr(Month) + FormatDelim + IntStr(Day)
    );
end;

function TimeStr(Hour, Min, Sec : word) : string;
var
    H : word;
    APM : string;
begin
    if (Hour = 24) or (Hour < 12) then begin
        APM := Stamper.AM;
        if (Hour = 24) or (Hour = 0) then
            H := 12
        else
            H := Hour;
    end else begin
        if Hour > 12 then
            H := Hour - 12
        else
            H := Hour;
        APM := Stamper.PM;
    end;

    TimeStr := ParseKeyValue(Stamper.Time,
        ZPad(IntStr(Hour),2) + FormatDelim + ZPad(IntStr(Min),2) + FormatDelim +
        ZPad(IntStr(Sec),2) + FormatDelim + IntStr(H) + FormatDelim + APM
    );
end;

function StampStr ( AStamp : LongInt ) : String;
var
    DT : TDateTime;
begin
    UnPackTime(AStamp, DT);
    StampStr := ParseKeyValue(Stamper.Stamp,
        DateStr(DT.Year, DT.Month, DT.Day) + FormatDelim +
        TimeStr(DT.Hour, DT.Min, DT.Sec)
    );
end;

function SizeStr(ASize : LongInt) : String;
var
    Factor : String;
begin
    Factor := Bytes;
    if (ASize >= 1024) and (ASize mod 1024 = 0) then begin
        ASize := ASize div 1024;
        Factor := KBytes;
        if (ASize >= 1024) and (ASize mod 1024 = 0) then begin
            ASize := ASize div 1024;
            Factor := MBytes;
        end;
    end;
    SizeStr := ParseKeyValue(Factor, IntStr(ASize));
end;

