{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$IFDEF SECTINT}

var
    MouseLag : LongInt;

{ Mouse functions }
function MouseButtons : word; { Mouse button count or Zero if not present }
procedure SetMouseFactor ( XScale, YScale : byte );

{ Mouse Status Testing }
function MouseEvent ( var Event : TEvent) : boolean;
function MouseClick ( var Event : TEvent) : boolean;
procedure SetMouseBounds (x1, y1, x2, y2 : integer);
procedure SetMousePosition(x, y : integer);

{ Mouse Pointer Functons }
procedure MouseMove(Where : TPoint);
procedure MouseHide;
procedure MouseShow;

procedure MouseDefault(Visible : boolean);

{$ENDIF}

{$IFDEF SECTIMP}
var
    MouseButtonCount    : word;

const
    MouseDoubleDelay    : word = 5;
    MouseScaleFactor    : word = $0000;
    MouseMaxX           : integer = 1024;
    MouseMaxY           : integer = 768;
    MouseMinX           : integer = 0;
    MouseMinY           : integer = 0;
    LastMousePos        : TPoint = (X:0; Y:0);
    LastMouseButton     : word = 0;
    LastMouseChange     : array[0..15] of LongInt =
        (0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);

function ResetMouse : word; assembler;
asm
    xor         ax, ax
    int         $33
    cmp         ax, 0
    je          @@Done
    cmp         bx, 0
    je          @@OneButton
    cmp         bx, $ffff
    je          @@TwoButton
    jmp         @@SetButtons
@@OneButton:
    mov         bx, 1
    jmp         @@SetButtons
@@TwoButton:
    mov         bx, 2
@@SetButtons:
    mov         ax, bx
@@Done:
    mov         MouseButtonCount, ax
end;

procedure MouseHandler ( Flags, CS, IP, AX, BX, CX, DX, SI,
    DI, DS, ES, BP : word); interrupt;
var
    Event   : TEvent;
    I       : byte;
    Trigger : boolean;
begin
    asm
        pushf
        cli
    end;
    {
        AX = Event Flag
        BX = Button State
        CX = X coor - ignored
        DX = Y Coor - ignored
        SI = X Movement
        DI = Y Movement
        DS = Mouse Driver Data Segment
    }
    Trigger           := False;
    Event.Kind        := evNull;
    Event.Buttons     := BX;
    Event.Modifiers   := ShiftState;
    Event.Position.X  := LastMousePos.X + SI shr Lo(MouseScaleFactor);
    Event.Position.Y  := LastMousePos.Y + DI shr Hi(MouseScaleFactor);
    if MoveEvent.Kind = evMouseMove then begin
        Event.Movement.X  := Event.Movement.X + SI;
        Event.Movement.Y  := Event.Movement.Y + DI;
    end else begin
        Event.Movement.X  := SI;
        Event.Movement.Y  := DI;
    end;
    if Event.Position.X < MouseMinX then  Event.Position.X := MouseMinX;
    if Event.Position.Y < MouseMinY then  Event.Position.Y := MouseMinY;
    if Event.Position.X > MouseMaxX then  Event.Position.X := MouseMaxX;
    if Event.Position.Y > MouseMaxY then  Event.Position.Y := MouseMaxY;

    if (DI <> 0) or (SI <> 0) then begin
        Event.Kind := Event.Kind or evMouseMove;
    end;
    if (Event.Buttons <> LastMouseButton) then
        begin
            Trigger := True;
            for I := 0 to 15 do
                if LastMouseButton and (1 shl I) <> Event.Buttons and (1 shl I) then
                    case Event.Buttons and (1 shl I) = (1 shl I) of
                        False : Event.Kind := Event.Kind or evMouseRelease;
                        True  : begin
                            Event.Kind := Event.Kind or evMouseClick;
                            if TimerTick - LastMouseChange[I] <= MouseDoubleDelay then
                                Event.Kind := Event.Kind or evMouseDouble;
                            LastMouseChange[I] := TimerTick;
                        end;
                    end;
        end;
    LastMousePos := Event.Position;
    LastMouseButton := Event.Buttons;
    if Trigger then begin
        PutEvent(Event);
        MoveEvent.Kind := evNull;
        MouseLag := 0;
    end else if (Event.Kind = evMouseMove) then begin
        MoveEvent := Event;
        Inc(MouseLag);
    end;

    asm { read/reset motion counters and exit }
        popf
        mov     ax, $0b
        int     $33
        mov     sp, bp
        pop     bp
        pop     es
        pop     ds
        pop     di
        pop     si
        pop     dx
        pop     cx
        pop     bx
        pop     ax
        retf
    end;

end;

procedure SwapMouseHandler(var Mask : word; var Handler : Pointer); assembler;
asm
      mov       ax, $14
      les       di, Mask
      mov       cx, es:[di]
      les       di, Handler
      mov       dx, es:[di]
      mov       es, es:[di + 2]
      int       $33
      mov       bx, es
      les       di, Handler
      mov       es:[di], dx
      mov       es:[di + 2], bx
      les       di, Mask
      mov       es:[di], cx
end;

function MouseButtons : word;
begin
    MouseButtons := MouseButtonCount;
end;

procedure SetMouseFactor ( XScale, YScale : byte );
begin
    MouseScaleFactor := (YScale Shl 8) or XScale;
end;

function MouseEvent ( var Event : TEvent) : boolean;
begin
    MouseEvent := Event.Kind and evMouse <> 0;
end;

function MouseClick ( var Event : TEvent) : boolean;
begin
    MouseClick := Event.Kind and evMouseClick = evMouseClick;
end;

procedure SetMousePosition(x, y : integer);
begin
    LastMousePos.X := X;
    LastMousePos.Y := Y;
end;

procedure SetMouseBounds (x1, y1, x2, y2 : integer);
begin
    MouseMinX := x1;
    MouseMinY := y1;
    MouseMaxX := x2;
    MouseMaxY := y2;
end;

procedure MouseMove(Where : TPoint);
begin
    if Assigned(MousePtr) then
        Video^.SpriteMove(MousePtr, Where.X, Where.Y);
end;

procedure MouseHide;
begin
    if Assigned(MousePtr) then
        Video^.SpriteHide(MousePtr);
end;

procedure MouseShow;
begin
    if Assigned(MousePtr) then
        Video^.SpriteShow(MousePtr);
end;

{$I MOUSEPTR.INC}

procedure MouseDefault(Visible : boolean);
var
    I, J, C : word;
begin { Default Mouse is 8x8 Pixels and 16 colors }
    if not Assigned(MousePtr) then begin
        MousePtr := Video^.NewSprite(8, 8,
            High(DefaultMousePtrImgs) - Low(DefaultMousePtrImgs) + 1, true);
        if not Assigned(MousePtr) then Exit;
        MousePtr^.Kind  := skMouse;
        MousePtr^.Level := slMouse;
        MousePtr^.Animate := -1; { infinite }
        for I := Low(DefaultMousePtrImgs) to High(DefaultMousePtrImgs) do begin
            Move(DefaultMousePtrMask, MousePtr^.Sprites^[I].BMask^.ImageData, 8);
            for J := 0 to 63 do begin
                C := DefaultMousePtrImgs[I][J];
                case Video^.Colors of
                    2 : if C > 6 then C := 1 else C := 0;
                    4 : C := C div 4;
                { Colors >= 16, then were fine }
                end;
                Video^.ImagePutPixel(MousePtr^.Sprites^[I].Image, J mod 8, J div 8, C);
            end;
            { if ImageCompress then
                Video^.ImageImplode(MousePtr^.Sprites^[I].Image); }
        end;
        Video^.SpriteSort;
    end;
    if Assigned(MousePtr) then begin
        SetMouseBounds(0,0,Video^.Width - 1, Video^.Height - 1);
        SetMousePosition(Video^.Width div 2, Video^.Height div 2);
        Video^.SpriteMove(MousePtr, Video^.Width div 2, Video^.Height div 2);
    end;
    Video^.SpriteSetVisible(MousePtr, Visible);
end;


{$ENDIF}