{
    BSD 3-Clause License
    Copyright (c) 2021, Jerome Shidel
    All rights reserved.
}

{$I INFERNO.DEF}
unit FFmtPCX; { Simple PCX Graphic Image File Format  }

interface

uses Inferno;

var
    PCX_Handler : PFormatHandler;

implementation

type
    PCXFileHeader = record
        Mfg             : byte; { 10 = ZSoft }
        Version         : byte; { 5 is latest, we will try using 6 }
        Encoding        : byte; { 1 is RLE }
        BitsPerPixel    : byte;
        Margin          : TArea; { Left, Top, Right Bottom word margins of pic }
        HDPI, VDPI      : word;  { Resolution of image }
        PaletteSmall    : array [0..15] of TRGBPalette; { 16 color images }
        Reserved1       : byte;
        Planes          : byte;
        BytesPerLine    : word; { Always even, use right/left margin to adjust }
        PaletteType     : word; { 1 color/bw, 2 grayscale }
        HScreen, VScreen: word; { Screen Size }
        Reserved2       : array[0..53] of byte;
        { Palette Table for version 5+ }
        VGAPaletteID    : byte; { 0x0C, then table follows }
        Palettes        : TRGBPalettes;
    end;

function PCX_Save(FileName : String; P : Pointer; Size : LongInt) : integer; far;
var
    FH : PCXFileHeader;
    Image : PImage;
begin
    Image := P;
    FileName := FileBase(FileName) + 'PCX';
    ClearError;
    FillChar(FH, Sizeof(FH), 0);
    with FH do begin
        Mfg             := 10;
        Version         := 6;
        BitsPerPixel    := 8;
        Margin.Left     := 0; { 0 or 1, IDK , offsets or what? }
        Margin.Top      := 0;
        Margin.Right    := 0;
        Margin.Bottom   := 0;
        HDPI            := 75;
        VDPI            := 75;
        Planes          := 1;
        PaletteType     := 1;
        HScreen         := Image^.Width;  { unsure... }
        VScreen         := Image^.Height; { unsure... }
        VGAPaletteID    := $0c;
        BytesPerLine := Image^.Width;
        { Add code to edit make bytes even!! }
        { if Image^.Width mod 2 <> 0 then
            BytesPerLine := Image^.Width + 1; }
        Video^.GetPalettes(Palettes);
        move(Palettes, PaletteSmall, Sizeof(PaletteSmall));
        if FileSave(FileName, @FH, Sizeof(FH)) then
            FileAppend(FileName, @Image^.ImageData, Video^.ImageSizeData(Image));
    end;
    PCX_Save := GetError;
end;

function PCX_Check(FileName : String) : boolean; far;
var
    H : ^PCXFileHeader;
    F : boolean;
begin
    F := False;
    ClearError;
    H := FileRead(FileName, 0, Sizeof(PCXFileHeader));
    if Assigned(H) then begin
        F := (H^.Mfg = 10) and (H^.Version >= 5) and (H^.Encoding = 0) and
            (H^.BitsPerPixel = 8) and (H^.VGAPaletteID = $0c);
        FreeMem(H, Sizeof(PCXFileHeader));
    end;
    (* if F then begin
        with H^ do begin
            FileLog('', Filename );
            FileLog('', 'Mfg: ' + IntStr(Mfg) );
            FileLog('', 'Ver: ' + IntStr(Version) );
            FileLog('', 'Encoding: ' + IntStr(Encoding) );
            FileLog('', 'BitsPerPixel: ' + IntStr(BitsPerPixel) );
            FileLog('', 'Margins: ' +
                IntStr(Margin.Left) + ',' +
                IntStr(Margin.Top) + ',' +
                IntStr(Margin.Right) + ',' +
                IntStr(Margin.Bottom)
            );
            FileLog('', 'DPI: ' + IntStr(HDPI) + 'x' + IntStr(VDPI) );
            { PaletteSmall }
            { Reserve1 }
            FileLog('', 'Planes: ' + IntStr(Planes) );
            FileLog('', 'BytesPerLine: ' + IntStr(BytesPerLine) );
            FileLog('', 'PaletteType: ' + IntStr(PaletteType) );
            FileLog('', 'Screen: ' + IntStr(HScreen) + 'x' + IntStr(VScreen) );
            { Reserved2 }
            FileLog('', 'VGAPalettes: ' + ByteHex(PaletteType) );
        end;
    end; *)

    (*
        Mfg             : byte; { 10 = ZSoft }
        Version         : byte; { 5 is latest, we will try using 6 }
        Encoding        : byte; { 1 is RLE }
        BitsPerPixel    : byte;
        Margin          : TArea; { Left, Top, Right Bottom word margins of pic }
        HDPI, VDPI      : word;  { Resolution of image }
        PaletteSmall    : array [0..15] of TRGBPalette; { 16 color images }
        Reserved1       : byte;
        Planes          : byte;
        BytesPerLine    : word; { Always even, use right/left margin to adjust }
        PaletteType     : word; { 1 color/bw, 2 grayscale }
        HScreen, VScreen: word; { Screen Size }
        Reserved2       : array[0..53] of byte;
        { Palette Table for version 5+ }
        VGAPaletteID    : byte; { 0x0C, then table follows }
        Palettes        : TRGBPalettes;
    *)
    PCX_Check := NoError and F;
    ClearError;
end;

function PCX_Process(var P : Pointer; var Size : word) : integer; far;
begin
    ClearError;
    SetError(erOperation_Not_Supported);
    { Perform data conversion, free old pointer memory, return new pointer
      and it's size. Caller will not free old pointer memory for you! }
    if Assigned(P) then begin
        FreeMem(P, Size);
        P := nil;
    end;
    PCX_Process := GetError;
end;

function PCX_Load(FileName : String; var P : Pointer; var Size : LongInt) : integer; far;
var
    H : ^PCXFileHeader;
    Image : PImage;
    PP, SP : TRGBPalettes;
begin
    image := nil;
    ClearError;
    H := FileRead(FileName, 0, Sizeof(PCXFileHeader));
    if Assigned(H) then begin
        PP := H^.Palettes;
        Image := Video^.NewImage(H^.HScreen, H^.VScreen);
        if Assigned(Image) then begin
            P := FileRead(Filename, Sizeof(PCXFileHeader), Video^.ImageSizeOf(Image) - 4 );
            if Assigned(P) then begin
                move(P^, Image^.ImageData, Video^.ImageSizeData(Image));
                FreeMem(P, Video^.ImageSizeData(Image));
                Video^.GetPalettes(SP);
                Video^.ImageRemap(Image, SP, PP, FormatPaletteMode);
                if (FormatPaletteMode = ipmComplete) or (FormatPaletteMode = ipmOverride) then
                    Video^.SetPalettes(PP);
            end else
                Video^.FreeImage(Image);
        end;
    end;
    if IsError then Video^.FreeImage(Image);
    if Assigned(H) then
        FreeMem(H,  Sizeof(PCXFileHeader));
    P := Image;
    PCX_Load := GetError;
end;

begin
    PCX_Handler := New(PFormatHandler);
    with PCX_Handler^ do begin
        Kind   := ffImage;
        UID    := 'IMGPCX';
        Compat := 0;
        Exts   := 'PCX';
        Check  := PCX_Check;
        Process:= PCX_Process;
        { Save   := PCX_Save; }
        Save   := nil;
        Load   := PCX_Load;
    end;
    RegisterFileFormat(PCX_Handler);
end.