/* This is part of the NEWTRACK eyetracking software, (c) 2005 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

#ifdef MAKE_CONVERTER	/* only compile when creating pcx2tpl binary */
/* convert PCX image to PCX template */
/* compile: gcc -DMAKE_CONVERTER -s -o pcx2tpl pcx*.c fileread.c */

#include "tracker.h"		/* include all needed headers */

#include "vesa.h"		/* for lfbSel, pixelspace, spacePIX, ... */
				/* (the vesa FUNCTIONS are not used!) */

#define MARG 10			/* upper and left margin (image/canvas) */

#ifndef __DJGPP__		/* only for DJGPP */
#define O_BINARY 0		/* no-op */
#endif



/* write data to a file, split into several chunks if needed */
ssize_t writeLoop(int fd, const char *buf, size_t count)
{
  size_t left = count;
  char * ptr = (char *)buf;		/* no longer const */
  ssize_t done;
  while (1) {
      done = write(fd, ptr, left);
      if (done < 0) return done;	/* error */
      if (done == left) return count;	/* completed */
      left -= done;
      ptr += done;
  }
} /* writeLoop */



/* compress the current pixelspace to a PCX body data stream */
int writePCXStream(int handle, int xres, int yres)
{
  sint32 x, y, n;		/* coordinates, stream size */
  uint8 * streambuf;		/* output buffer */
  uint8 * p;
  int status;

  streambuf = malloc (vesamode.width * vesamode.height);
  if (streambuf == NULL) {
      printf("Out of memory!\n");
      return -1;
  }
  n = 0;
  p = streambuf;

  /* printf("Compressing %d*%d image.\n", xres, yres); */
  for (y = 0; y < yres; y++) {
      int rep;					/* repeat count */
      uint8 * row = &spacePIX(MARG,y+MARG);	/* pointer to byte line */
      x = 0;
      do {
          rep = 1;
          while ( ((x+rep) < xres) && (row[x+rep] == row[x]) && (rep < 63) ) {
              rep++;
          }					/* collect compressibles */
          if (rep == 1) {			/* single pixel */
              if (row[x] > 0xbf) {
                  /* could overflow: p[0] = 0xc1; p[1] = row[x]; p++; */
                  p[0] = 0;			/* just store as 0 */
              } else {
                  p[0] = row[x];		/* store pixel */
              }
          } else {				/* several pixels */
              p[0] = 0xc0 + rep;		/* RLE compressed: rep...  */
              p[1] = row[x];			/* ...pixels of this color */
              p++;
          }
          p++;					/* point after last byte */
          x += rep;				/* scan on */
      } while (x < xres);			/* process line */
  } /* y loop */
  n = (p - streambuf);				/* amount of data */

  status = writeLoop(handle, streambuf, n);	/* write to file */
  free(streambuf);
  return status;
} /* writePCXStream */



int main (int argc, char * argv[])
{
  uint32 histogram[256];	/* color frequencies */
  uint8 * pcxcontents = NULL;   /* PCX file contents */
  char templatename[256];	/* output PCX file name */
  uint8 pcxheader[128];		/* the outfile header is from the infile */
  uint8 pcxpalette[769];	/* the palette is based on the infile */
  uint32 pcxsize = 0;           /* size of PCX file contents */
  uint32 i;
  int handle;			/* file handle */
  int best, nareas, minY, maxY;	/* analysis results */

  XGFX = XGFX_HI+20;		/* just use hardcoded canvas size here */
  YGFX = YGFX_HI+20;		/* YGFX is the same as vesamode.width */

  pixelspace = malloc (vesamode.width * vesamode.height);
      /* the variable itself is global in vesa.h */
  if (pixelspace == NULL) return 1;

  for (i = 0; i < (vesamode.width * vesamode.height); i++)
      pixelspace[i] = 255;	/* "nothing else on screen" */
				/* (no analyzed PCX, that is) */

  if (argc > 1) {		/* user specified input file name */
      char * extn = strstr(argv[1], ".pcx");
      if (extn == NULL)
          extn = strstr(argv[1], ".PCX");
      if (extn == NULL) {
          printf("Input file name must end with .pcx\n");
          printf("Usage: %s [input.pcx [output.pcx]]\n", argv[0]);
          printf("To generate a template out of a (max 256 color) PCX file.\n");
          printf("If no output name given, ?-tpl.pcx is used for input ?.pcx.\n");
          return 1;
      }
      pcxsize = fileread("PCX", argv[1], &pcxcontents);
          /* pcxcontents sent as NULL, so fileread allocates a buffer */
      if (argc > 2) {		/* user specified a template name */
          strncpy(templatename, argv[2], 255);
          templatename[255] = '\0';
      } else {			/* make template name from input name */
          extn[0] = '\0';	/* chop off extension */
          strncpy(templatename, argv[1], 240);
          templatename[240] = '\0';
          strcat(templatename, "-tpl.pcx");	/* bad luck for short names */
      }
  } else {
      pcxsize = fileread("PCX", NULL, &pcxcontents);
          /* name sent as NULL, so fileread prompts the user */
      strcpy(templatename,"template.pcx");	/* default name */
  }

  if (pcxsize < (128+768)) {
      printf("Input file too small, probably no 256 color PCX.\n");
      return 1;
  }

  memcpy(pcxheader, &pcxcontents[0], 128);	/* grab 128 header bytes */
  if (pcxcontents[pcxsize-769] != 12) {		/* no 256 color palette? */
      for (i = 0; i < 256; i++) {
          pcxpalette[1+(i*3)] = pcxpalette[1+(i*3)] = pcxpalette[1+(i*3)] = i;
      }
  } else {					/* keep original palette */
      memcpy(pcxpalette, &pcxcontents[pcxsize-769], 769); /* 1+(3*256) bytes */
  }

  pcxpalette[0] = 12;				/* magic value */
  for (i = 0; i < 64; i++) {			/* replace first 64 colors */
      int c85[4] = { 0, 3*85, 85, 2*85 };	/* intense colors first */
#ifdef CLASSIC_PALETTE
      pcxpalette[1+(i*3)] = c85[(i>>4)&3];
      pcxpalette[2+(i*3)] = c85[(i>>2)&3];
      pcxpalette[3+(i*3)] = c85[ i    &3];
#else	/* new palette: first 8 colors are most intense (0/100%), */
	/* others have one or more RGB components closer to 50%.  */
      pcxpalette[1+(i*3)] = c85[((i>>2)&1) | ((i>>5)&1)<<1];
      pcxpalette[2+(i*3)] = c85[((i>>1)&1) | ((i>>4)&1)<<1];
      pcxpalette[3+(i*3)] = c85[( i    &1) | ((i>>3)&1)<<1];
#endif
  }

  /* *** we load the image invisibly, no VESA needed here */
  /* *** check validity, area count, size, layout, object placement */
  if ( (i=pcxshow (pcxsize, pcxcontents, MARG /* x margin */,
      MARG /* margin, neg.: from bottom */,
      pixelspace, histogram, 0 /* invisible */)) != 0) {
      printf("PCX file load error %d.\n", i);
      printf("Input file must be 1 plane, 8 bits per pixel.\n");
      printf("Maximum size: %d*%d pixels.\n",
          XGFX-(MARG+MARG), YGFX-(MARG+MARG));
      return 128+i;
  }

  /* *** */
  best = findmax (histogram, 256);	/* find most used color index */  
  nareas = imagechunker (best, 0 /* dont visualize result */,
      &minY, &maxY); /* uses pixelspace as input and output */
  /* *** */

  printf("%d areas found on canvas rows %d-%d, background %d\n",
      nareas, minY-MARG, maxY-MARG, best);
  if ((nareas < 1) || (nareas > 64)) {
      printf("Sorry, cannot create template file with %d areas.\n", nareas);
      return 128;
  }

  handle = open (templatename, O_CREAT | O_EXCL | O_WRONLY,
      S_IRUSR | S_IWUSR ); /* must be writeable and will be writeable */
      /* if file has been there before, we refuse to open it. */
  if (handle < 0) {
      printf("Cannot create template output file %s\n", templatename);
      printf("If the file already existed, it will NOT be overwritten.\n");
      return 2;
  }
  close(handle);

  handle = open (templatename, O_WRONLY | O_APPEND | O_BINARY);
  if (handle < 0)
      return 2; /* should not happen */

  i = writeLoop(handle, pcxheader, 128);
  if (i < 128) {
      printf("PCX file header write error %d.\n", i);
      return 3;
  }

  /* *** */
#undef GETw
#define GETw(n) ( ((uint16)(pcxheader[n])) | ((uint16)(pcxheader[n+1])<<8) )
  i = writePCXStream(handle, 1+GETw(8)-GETw(4), 1+GETw(10)-GETw(6));
#undef GETw
  /* *** */
  if (i < 0) {
      printf("PCX file data write error %d.\n", i);
      return 3;
  }

  i = writeLoop(handle, pcxpalette, 769);
  if (i < 128) {
      printf("PCX file palette write error %d.\n", i);
      return 3;
  }

  close(handle);

  { /* variable scope: fileinfo */
      struct stat fileinfo;

      if (stat (templatename, &fileinfo)) {
          return 2; /* should not happen */
      } else {
          if ( (S_IWUSR & fileinfo.st_mode) && S_ISREG(fileinfo.st_mode) ) {
              i = fileinfo.st_size;
          } else {
              return 2; /* should not happen */
          } /* no writeableable regular file */
      } /* file exists */
  } /* variable scope: fileinfo */

  if (i == 0) {
      printf("Deleting empty file again.\n");
      if (unlink (templatename))
          printf("... even that failed!?");
      return 2;
  }

  printf("Template written to %s file, size %d bytes.\n", templatename, i);
  return 0;

} /* main */

#endif /* MAKE_CONVERTER */

