/* This is part of the NEWTRACK eyetracking software, (c) 2004 by  */
/* Eric Auer. NEWTRACK is free software; you can redistribute it   */
/* and modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2  */
/* of the License, or (at your option) any later version.          */
/*     NEWTRACK is distributed in the hope that it will be useful, */
/*     but WITHOUT ANY WARRANTY; without even the implied warranty */
/*     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.     */
/*     See the GNU General Public License for more details.        */
/* You should have received a copy of the GNU General Public       */
/* License (license.txt) along with this program; if not, check    */
/* www.gnu.org or write to the Free Software Foundation, Inc., 59  */
/* Temple Place, Suite 330, Boston, MA  02111-1307 USA.            */

/* detecting which word is being looked at */
/* UPDATE 10/2004: check for pixelspace area information bitmap, too */
/* UPDATE  4/2005: Right word edges were wrong by 1 char, fixed!     */
/* UPDATE  4/2005: Fix handling of word boxes in context of INVSPACE */
/* UPDATE  4/2005: Allow font zooming for VESA mode, with FONT_ZOOM  */

#include "tracker.h"		/* include all needed headers */

#if GRAPHICS
#include "vesa.h"
#endif

#include "script.h"		/* script processing specific things */

/* #define SINGLEFUZZ 2 */	/* if there is only one line of text,    */
	/* pretend that the next line would be SINGLEFUZZ lines below,   */
	/* in other words: SINGLEFUZZ-1 empty lines between the one line */
	/* and an imaginary next line... If not defined, this is checked */
	/* by comparing get_line_ypos for line 0 and line "1", better.   */



/* browse through stimulus and create the INFO WORD log lines:   */
/* INFO WORD areanumber leftpix toppix rightpix bottompix string */
/* This assumes a virtual 8x16 font, as does get_pixel_xy etc.!  */
void describe_stimulus (int usedlines)	/* Also initializes area list! */
{
  uint32 timestamp, firsttime;			/* dummy, for logging */
  int line, x, y, areanum, yfuzz;
  char descline[200];
  char * thisline;

  for (areanum = 0; areanum < MAX_AREAS; areanum++) {
      areas[areanum].theWord[0] = '\0';		/* flush list */
      areas[areanum].miny = -10;
      areas[areanum].maxy = -10;
      areas[areanum].pminy = -10;
      areas[areanum].pmaxy = -10;
      /* X values are not marked as invalid */
      areas[areanum].dwelltime = 0;
      areas[areanum].avgxsum = 0;
      areas[areanum].avgysum = 0;
  } /* flush area definitions */
  timestamp = firsttime = 0;			/* dummy, for logging */

  y = get_line_ypos (0, usedlines, &x);
#ifdef SINGLEFUZZ
  yfuzz  = (usedlines > 1) ?
      (get_line_ypos (1, usedlines, &x) - 1) : (y + SINGLEFUZZ - 1);
#else
  yfuzz  = get_line_ypos (1, usedlines, &x) - 1;
#endif
  if ( (y > 0) && ((yfuzz - y) > 0) ) {		/* big line spacing? */
      yfuzz = (yfuzz - y) * 16 / 4;		/* extra margin */
      /* margin is on top / bottom of word, 1..2 times 16/4 "pixels"   */
      /* i.e. if between each 2 lines of text there is one line empty, */
      /* 1/4 of the bottom of it is counted to the lower text line and */
      /* 1/4 of the top of it counts to the upper text line. 1/2 of    */
      /* the empty line counts as really empty space between the text. */
  } else {
      yfuzz = 0;
  } /* no vertically enlarged bounding boxes */

  areanum = 0;
  for (line = 0; line < usedlines; line++) {
      char theword[100];
      int x0, i;
      int xfuzz;	/* *** fix 4/2005: INVSPACE things *** */
#if GRAPHICS	/* new 4/2005: FONT_ZOOM processing */
      int fontzoom = (lfbSel > 0) ? FONT_ZOOM : 1;	/* only for VESA */
#endif

      thisline = get_line (line);		/* get contents */
      y = get_line_ypos (line, usedlines, &x);	/* get position */
      if (thisline == NULL)
          continue;

      x0 = x;					/* word start */
      i = 0;
      xfuzz = 4;
      theword[i] = '\0';
      for ( ; ; thisline++) {		/* scan this line */

          if ( (*thisline == ' ') || (*thisline == INVSPACE) ||
               (!*thisline) ) {		/* end of word - or end of line! */

              /* did we reach the end of a word? */
              /* *** this uses hardcoded 8x16 VIRTUAL font size for conv.    */
              /* *** from char coords to pixel coords in TEXT AND VESA mode! */
              if (i) {				/* any nonspace met yet? */

                  areas[areanum].minx = x0;
                  areas[areanum].maxx = x-1;
                  areas[areanum].miny = y;
                  areas[areanum].maxy = y;
                  areas[areanum].pminx = (x0 * 8) - xfuzz;
                      /* can be < 0. */
                      /* no "xfuzz" if word "started with" an invspace */
                      /* (xfuzz initially 4, updated after each word!) */
                  xfuzz = (*thisline == INVSPACE) ? 0 : 4;
                      /* if space is not invisible, enlarge area by xfuzz */
                      /* (and enlarge the next area to the left later...) */
                  areas[areanum].pmaxx = ((x-1) * 8) + (8-1) + xfuzz;
                      /* could be > maximum coordinate    */
                      /* no "xfuzz" if we hit an invspace */
                      /* *** FIXED 4/2005: The 10/2004 version used *** */
                      /* *** (x * 8) + (8-1) + xfuzz, but (x-1) as  *** */
                      /* *** with maxx has to be used instead of x! *** */
                  areas[areanum].pminy = (y * 16) - yfuzz;
                  areas[areanum].pmaxy = (y * 16) + (16-1) + yfuzz;
                  sprintf (areas[areanum].theWord, "%.75s", theword);
                  sprintf (descline, "INFO WORD %d %d %d %d %d %s", areanum,
#if GRAPHICS	/* new 4/2005: FONT_ZOOM processing */
                      fontzoom * areas[areanum].pminx,
                      fontzoom * areas[areanum].pminy,
                      fontzoom * areas[areanum].pmaxx,
                      fontzoom * areas[areanum].pmaxy,
#else
                      areas[areanum].pminx, areas[areanum].pminy,
                      areas[areanum].pmaxx, areas[areanum].pmaxy,
#endif
                      theword);
                  LOG_STRING (descline);	/* log word description */
                  i = 0;			/* reset nonspace count */
                  theword[i] = '\0';
                  areanum++;
#if GRAPHICS
                  if (areanum == MAX_AREAS - ((lfbSel > 0) ? 256 : 0)) {
#else
                  if (areanum == MAX_AREAS) {
#endif
                      LOG_STRING ("* AREA LIST FULL *");
                      areanum--;		/* overwrite last entry */
                  } /* area list overflow */

              } /* anything to log */

              if (*thisline == ' ') {
                  x++;		/* not logged but affects position... */
              }
              x0 = x;		/* next word starts after last space */
              			/* *** fix 4/2005: INVSPACE things *** */
              if (!*thisline)
                  break;	/* done with this line! */

          } else {		/* else it is a visible part of a word */

              theword[i] = *thisline;
              theword[i+1] = '\0';
              i++;
              x++;

          } /* visible part of a word */

      } /* loop inside line */

  } /* loop over lines */

#if GRAPHICS
          if (lfbSel > 0) { /* in graphics mode? */
              for (x = 1; x < 255; x++) {
                  sprintf (areas[(MAX_AREAS - 256) + x].theWord, "PCX/%d", x);
              } /* use this range for PCX area numbers. No coordinates. */
          }
#endif /* GRAPHICS */

} /* describe_stimulus */



/* this step only gives an "for this moment" decision        */
/* ... so it does NOT directly create:                       */
/* ENTER WORD areanumber avgx avgy x y string/(graphical)    */
/* LEAVE WORD areanum avgx avgy x y dwelltime soundtimeindex */
/* input is in virtual pixels, as returned by get_pixel_xy() */
/* return value is an area number or -1                      */
int analyze_focus(int usedlines, int xeye, int yeye)	/* pixel coords! */
{
  int rows, cols, ypos, yend, a, yfuzz, bigyfuzz;
  int pixel_x, pixel_y, scan;
  int looked_line;		/* line which matches eyey best */
 				/* "looked_col" not used - using area list */
#if 0 				/* not needed when using pixel coords */
!  static int last_used_y = -10;	/* not reset between trials... */
#endif

  if ((xeye < 0) || (yeye < 0))
      return -1;		/* outside screen -> outside all areas */

#if GRAPHICS			/* added 10/2004 */
  if (lfbSel > 0) {		/* are we in graphics mode? */
      int real_x, real_y;	/* actual VESA canvas coordinates */

      real_x = FONT_ZOOM * xeye;	/* in sync with REAL screen.c font! */
      real_y = FONT_ZOOM * yeye;	/* in sync with REAL screen.c font! */
          /* real font size is 8x16 zoomed by FONT_ZOOM x FONT_ZOOM */

      if ((real_x >= XGFX) || (real_y >= YGFX))
          return -1;

      a = spacePIX(real_x, real_y);

      if ((a > 0) && (a < 255)) {	/* neither background nor reserved */
          return ((MAX_AREAS-256) + a);
          /* return special value for PCX area number. Has priority. */
      } /* ... else continue with text mode analysis */
  } /* in graphics mode */
#endif /* GRAPHICS */

  pixel_x = xeye;
  pixel_y = yeye;
  xeye /= 8;			/* assuming virtual 8x16 font, rounding down */
  yeye /= 16;			/* assuming virtual 8x16 font, rounding down */

  rows = get_resolution_y (&cols, &a);	/* get screen geometry (ignore a) */

  ypos = get_line_ypos (0, usedlines, &a);		/* ignore xpos */
  yend = get_line_ypos (usedlines - 1, usedlines, &a);

#ifdef SINGLEFUZZ
  yfuzz = (usedlines > 1) ?
      (get_line_ypos (1, usedlines, &a) - ypos) : SINGLEFUZZ;
#else
  yfuzz = get_line_ypos (1, usedlines, &a) - ypos;
  if (yfuzz < 1) yfuzz = 1;		/* security check 10/2004 */
      /* yfuzz is now distance from each used line to the next, */
      /* can be 1, 2 or 3, for 0, 1 or 2 empty lines in between */
#endif

  bigyfuzz = (yfuzz > 1) ? 2 : 1;	/* max. lines outside used area */
  if (ypos < bigyfuzz)
      bigyfuzz = ypos;			/* reduce bigyfuzz if near border */
  if ((yend + bigyfuzz) >= rows)
      bigyfuzz = (rows-1) - yend;	/* ...same... */

  /* are there chances that we look at any word at all? */
  if ( (yeye >= (ypos - bigyfuzz)) && (yeye <= (yend + bigyfuzz)) ) {
      looked_line = 0;
      if (yeye > ypos)			/* if at / below topmost used line */
          looked_line = (yeye + ((yfuzz == 3) ? 1 : 0) - ypos) / yfuzz;
          /* do some rounding: if yfuzz is 3, lines next to line count, too */
      if (yeye > yend)
          looked_line = usedlines - 1;	/* beyond last is same as last */

#if 0				/* not needed anymore - using pixel coords! */
!     if ( (yfuzz == 2) && ((yeye - ypos) & 1) ) {
!         /* between 2 lines with yfuzz 2: Deciding "as we did before" */
!         if (last_used_y == (looked_line+1)) {	/* should we round UP ? */
!             looked_line = last_used_y;
!         } else if (last_used_y != looked_line) {
!             looked_line = -1;
!         } /* rounding DOWN was wrong, too */
!     } /* dynamic "snap to grid" for yfuzz 2 */
#endif

  } else {
      looked_line = -1;
  }

  if (looked_line >= 0) {	/* Line is known. Now check the X position. */
#if 0				/* no longer used for pixel coord version */
!     int xpos, xlen;
!
!     ypos = get_line_ypos (0, usedlines, &xpos);
!     a = (xpos > 0) ? (xpos-1) : xpos;	/* accept 1 char too far left */
!     xlen = cols - (xpos+xpos);	/* based on "centered" assumption */
!
!     if ( (xeye >= a) && (xeye <= (xpos + xlen)) ) {
#endif

          for (scan = 0; scan < MAX_AREAS; scan++) {
              if ( (areas[scan].pminy <= pixel_y) &&
                   (areas[scan].pmaxy >= pixel_y) &&
                   (areas[scan].pminx <= pixel_x) &&
                   (areas[scan].pmaxx >= pixel_x) ) {
                  return scan;		/* we found the right word */
              } /* did we find THE word? */
              if (areas[scan].miny == -10)
                  break;		/* end of list marker reached */
          } /* look in area list */

#if 0				/* no longer used for pixel coord version */
!     } /* X is in promising range */
#endif

  } /* Y in promising range) */

  return -1;	/* eye cursor is not inside any word on screen */

} /* analyze_focus */



char * word_for_area(int area)				/* guess what :-) */
{

  if ( (area < 0) || (area >= MAX_AREAS) )
      return NULL;

  return (areas[area].theWord[0] != '\0') ? areas[area].theWord : NULL;

} /* word_for_area */

